/*-
 * Copyright (c) 2005 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipastat_st.c,v 1.2 2011/01/23 18:42:35 simon Exp $";
#endif /* !lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipastat_log.h"
#include "ipastat_rules.h"
#include "ipastat_st.h"
#include "ipastat_main.h"

const struct st_list *global_st_list;	/* global { st_list } */

signed char	debug_st_null;		/* debug_st_null parameter. */

/* Built-in "null" statistics system. */
struct st_list st_list_null = STAILQ_HEAD_INITIALIZER(st_list_null);

/* List of all used "st_list" parameters. */
struct st_sets st_sets = SLIST_HEAD_INITIALIZER(st_sets);

/* List of all statistics modules. */
struct st_mod_list st_mod_list;

/* Current optional statistics name. */
struct opt_st *cur_opt_st = NULL;

/* List of all opt_st structures. */
struct opt_st_list opt_st_list = STAILQ_HEAD_INITIALIZER(opt_st_list);

/*
 * Find a statistics module by configuration prefix.
 */
struct st_mod *
st_mod_by_prefix(const char *prefix)
{
	struct st_mod *st_mod;

	SLIST_FOREACH(st_mod, &st_mod_list, link)
		if (st_mod->ipa_st_mod->conf_prefix != NULL &&
		    strcmp(st_mod->ipa_st_mod->conf_prefix, prefix) == 0)
			break;
	return (st_mod);
}

/*
 * Find a statistics module by a name.
 */
struct st_mod *
st_mod_by_name(const char *name)
{
	struct st_mod *st_mod;

	SLIST_FOREACH(st_mod, &st_mod_list, link)
		if (strcmp(st_mod->ipa_st_mod->st_name, name) == 0)
			break;
	return (st_mod);
}

/*
 * Release memory held by st_set, including struct st_set{}.
 */
static void
free_st_set(struct st_set *set)
{
	struct st_elem *st, *st_next;

	STAILQ_FOREACH_SAFE(st, &set->list, link, st_next)
		mem_free(st, m_anon);
	mem_free(set, m_anon);
}

/*
 * Release memory held by all st_set.
 */
void
free_st_lists(void)
{
	struct st_set *set, *set_next;

	SLIST_FOREACH_SAFE(set, &st_sets, link, set_next)
		free_st_set(set);
}

/*
 * Pre-initialize all st_mods.
 */
int
pre_init_st_mods(void)
{
	const struct st_mod *st_mod;

	SLIST_FOREACH(st_mod, &st_mod_list, link)
		if (st_mod->ipa_st_mod->st_pre_init() < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: st_pre_init failed",
			    st_mod->mod_file);
			return (-1);
		}
	return (0);
}

/*
 * Initialize all st_mods.
 */
int
init_st_mods(void)
{
	const struct st_mod *st_mod;

	SLIST_FOREACH(st_mod, &st_mod_list, link)
		if (st_mod->ipa_st_mod->st_init() < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: st_init failed",
			    st_mod->mod_file);
			return (-1);
		}
	return (0);
}

/*
 * Deinitialize all st_mods.
 */
int
deinit_st_mods(void)
{
	const struct st_mod *st_mod;
	int rv;

	rv = 0;
	SLIST_FOREACH(st_mod, &st_mod_list, link)
		if (st_mod->ipa_st_mod->st_deinit() < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: st_deinit failed",
			    st_mod->mod_file);
			rv = -1;
		}
	return (rv);
}

/*
 * Initialize one rule in statistics systems.
 */
int
st_init_rule(const struct rule *rule)
{
	if (!STAILQ_EMPTY(rule->st_list)) {
		const struct st_elem *st;

		STAILQ_FOREACH(st, rule->st_list, link) {
			if (st->ipa_st_mod->st_init_rule == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
				    "st_init_rule: module does not support "
				    "rules", st->mod_file, rule->name);
				return (-1);
			}
			if (st->ipa_st_mod->st_init_rule(rule->no,
			    rule->name) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
				    "st_init_rule failed", st->mod_file,
				    rule->name);
				return (-1);
			}
		}
	} else if (debug_st_null)
		logdbg("rule %s: uses built-in \"null\" statistics system",
		    rule->name);
	return (0);
}

/*
 * Deinitialize one rule in statistics systems, which it uses.
 */
int
st_deinit_rule(const struct rule *rule)
{
	const struct st_elem *st;
	int rv;

	rv = 0;
	STAILQ_FOREACH(st, rule->st_list, link)
		if (st->ipa_st_mod->st_deinit_rule != NULL &&
		    st->ipa_st_mod->st_deinit_rule(rule->no) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "st_deinit_rule failed", st->mod_file, rule->name);
			rv = -1;
		}
	return (rv);
}

int
st_get_rules_list(const struct st_list *st_list, unsigned int *n,
    struct ipa_entity_desc **buf_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(st_list)) {
		/* "null" system: no records and always successful. */
		*n = 0;
		*buf_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, st_list, link)
		if (st->ipa_st_mod->st_get_rules_list != NULL) {
			if (st->ipa_st_mod->st_get_rules_list(x_pat,
			    x_reg_ptr, m_result, n, buf_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: "
			    "st_get_rules_list failed", st->mod_file);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "st_get_rules_list: no statistics system "
	    "is able to return rules list");
	return (-1);
}

int
st_get_rule_stat(const struct rule *rule, const ipa_tm *tm1, const ipa_tm *tm2,
    int exact, unsigned int *n, struct ipa_rule_stat **buf_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(rule->st_list)) {
		/* "null" system: no records and always successful. */
		*n = 0;
		*buf_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, rule->st_list, link)
		if (st->ipa_st_mod->st_get_rule_stat != NULL) {
			if (st->ipa_st_mod->st_get_rule_stat(rule->no, tm1,
			    tm2, exact, m_result, n, buf_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "st_get_rule_stat failed", st->mod_file,
			    rule->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s: st_get_rule_stat: no statistics "
	    "system is able to return statistics for rule", rule->name);
	return (-1);
}

int
st_get_rule_info(const struct rule *rule, char **info_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(rule->st_list)) {
		/* "null" system: no info. */
		*info_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, rule->st_list, link)
		if (st->ipa_st_mod->st_get_rule_info != NULL) {
			if (st->ipa_st_mod->st_get_rule_info(rule->no,
			    m_result, info_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "st_get_rule_info failed", st->mod_file,
			    rule->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s: st_get_rule_info: no statistics "
	    "system is able to return info for rule", rule->name);
	return (-1);
}

#ifdef WITH_LIMITS
/*
 * Initialize one limit in statistics systems.
 */
int
st_init_limit(const struct rule *rule, const struct limit *limit)
{
	if (!STAILQ_EMPTY(limit->st_list)) {
		const struct st_elem *st;

		STAILQ_FOREACH(st, limit->st_list, link) {
			if (st->ipa_st_mod->st_init_limit == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "limit %s: st_init_limit: module does "
				    "not support limits", st->mod_file,
				    rule->name, limit->name);
				return (-1);
			}
			if (st->ipa_st_mod->st_init_limit(rule->no, rule->name,
			    limit->no, limit->name) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "limit %s: st_init_limit failed",
				    st->mod_file, rule->name, limit->name);
				return (-1);
			}
		}
	} else if (debug_st_null)
		logdbg("rule %s, limit %s: uses built-in \"null\" "
		    "statistics system", rule->name, limit->name);
	return (0);
}

/*
 * Deinitialize one limit in statistics systems, which it uses.
 */
int
st_deinit_limit(const struct rule *rule, const struct limit *limit)
{
	const struct st_elem *st;
	int rv;

	rv = 0;
	STAILQ_FOREACH(st, limit->st_list, link)
		if (st->ipa_st_mod->st_deinit_limit != NULL &&
		    st->ipa_st_mod->st_deinit_limit(rule->no, limit->no) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s: "
			    "st_deinit_limit failed", st->mod_file, rule->name,
			    limit->name);
			rv = -1;
		}
	return (rv);
}

int
st_get_limits_list(const struct rule *rule, unsigned int *n,
    struct ipa_entity_desc **buf_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(rule->st_list)) {
		/* "null" system: no records and always successful. */
		*n = 0;
		*buf_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, rule->st_list, link)
		if (st->ipa_st_mod->st_get_limits_list != NULL) {
			if (st->ipa_st_mod->st_get_limits_list(rule->no,
			    x_pat, x_reg_ptr, m_result, n, buf_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "st_get_limits_list failed", st->mod_file,
			    rule->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s: st_get_limits_list: no statistics "
	    "system is able to return list of limits", rule->name);
	return (-1);
}

int
st_get_limit_stat(const struct rule *rule, const struct limit *limit,
    const ipa_tm *tm1, const ipa_tm *tm2, unsigned int *n,
    struct ipa_limit_state **buf_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(limit->st_list)) {
		/* "null" system: no records and always successful. */
		*n = 0;
		*buf_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, limit->st_list, link)
		if (st->ipa_st_mod->st_get_limit_stat != NULL) {
			if (st->ipa_st_mod->st_get_limit_stat(rule->no,
			    limit->no, tm1, tm2, m_result, n, buf_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s: "
			    "st_get_limit_stat failed", st->mod_file,
			    rule->name, limit->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s, limit %s: st_get_limit_stat: "
	    "no statistics system is able to return statistics for limit",
	    rule->name, limit->name);
	return (-1);
}

int
st_get_limit_info(const struct rule *rule, const struct limit *limit,
    char **info_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(limit->st_list)) {
		/* "null" system: no info. */
		*info_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, limit->st_list, link)
		if (st->ipa_st_mod->st_get_limit_info != NULL) {
			if (st->ipa_st_mod->st_get_limit_info(rule->no,
			    limit->no, m_result, info_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s: "
			    "st_get_limit_info failed", st->mod_file,
			    rule->name, limit->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s, limit %s: st_get_limit_info: "
	    "no statistics system is able to return info for limit",
	    rule->name, limit->name);
	return (-1);
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
/*
 * Initialize one threshold in statistics systems.
 */
int
st_init_threshold(const struct rule *rule, const struct threshold *threshold)
{
	if (!STAILQ_EMPTY(threshold->st_list)) {
		const struct st_elem *st;

		STAILQ_FOREACH(st, threshold->st_list, link) {
			if (st->ipa_st_mod->st_init_threshold == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "threshold %s: st_init_threshold: "
				    "module does not support thresholds",
				    st->mod_file, rule->name, threshold->name);
				return (-1);
			}
			if (st->ipa_st_mod->st_init_threshold(rule->no,
			    rule->name, threshold->no, threshold->name) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "threshold %s: st_init_threshold failed",
				    st->mod_file, rule->name, threshold->name);
				return (-1);
			}
		}
	} else if (debug_st_null)
		logdbg("rule %s, threshold %s: uses built-in \"null\" "
		    "statistics system", rule->name, threshold->name);
	return (0);
}

/*
 * Deinitialize one threshold in statistics systems, which it uses.
 */
int
st_deinit_threshold(const struct rule *rule, const struct threshold *threshold)
{
	const struct st_elem *st;
	int rv;

	rv = 0;
	STAILQ_FOREACH(st, threshold->st_list, link)
		if (st->ipa_st_mod->st_deinit_threshold != NULL &&
		    st->ipa_st_mod->st_deinit_threshold(rule->no,
		    threshold->no) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: st_deinit_threshold failed",
			    st->mod_file, rule->name, threshold->name);
			rv = -1;
		}
	return (rv);
}

int
st_get_thresholds_list(const struct rule *rule, unsigned int *n,
    struct ipa_entity_desc **buf_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(rule->st_list)) {
		/* "null" system: no records and always successful. */
		*n = 0;
		*buf_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, rule->st_list, link)
		if (st->ipa_st_mod->st_get_thresholds_list != NULL) {
			if (st->ipa_st_mod->st_get_thresholds_list(rule->no,
			    x_pat, x_reg_ptr, m_result, n, buf_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "st_get_thresholds_list failed", st->mod_file,
			    rule->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s: st_get_thresholds_list: no statistics "
	    "system is able to return list of thresholds", rule->name);
	return (-1);
}

int
st_get_threshold_stat(const struct rule *rule,
    const struct threshold *threshold, int *reged,
    struct ipa_threshold_state *buf)
{
	const struct st_elem *st;
	int rv;

	if (STAILQ_EMPTY(threshold->st_list)) {
		/* "null" system: no statistics for threshold. */
		buf->thr = 0;
		return (0);
	}

	STAILQ_FOREACH(st, threshold->st_list, link)
		if (st->ipa_st_mod->st_get_threshold_stat != NULL) {
			rv = st->ipa_st_mod->st_get_threshold_stat(rule->no,
			    threshold->no, buf);
			if (rv >= 0) {
				*reged = (rv > 0);
				return (0);
			}
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: st_get_threshold_stat failed",
			    st->mod_file, rule->name, threshold->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: st_get_threshold_stat: "
	    "no statistics system is able to return statistics for threshold",
	    rule->name, threshold->name);
	return (-1);
}

int
st_get_threshold_info(const struct rule *rule,
    const struct threshold *threshold, char **info_ptr)
{
	const struct st_elem *st;

	if (STAILQ_EMPTY(threshold->st_list)) {
		/* "null" system: no info. */
		*info_ptr = NULL;
		return (0);
	}

	STAILQ_FOREACH(st, threshold->st_list, link)
		if (st->ipa_st_mod->st_get_threshold_info != NULL) {
			if (st->ipa_st_mod->st_get_threshold_info(rule->no,
			    threshold->no, m_result, info_ptr) == 0)
				return (0);
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: st_get_threshold_info failed",
			    st->mod_file, rule->name, threshold->name);
			return (-1);
		}
	logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: st_get_threshold_info: "
	    "no statistics system is able to return info for threshold",
	    rule->name, threshold->name);
	return (-1);
}
#endif /* WITH_THRESHOLDS */

/*
 * Add optional statistics name: -q -s name
 */
int
opt_st_add(char *st_names)
{
	struct opt_st *opt_st;

	if (*st_names == '\0') {
		cur_opt_st = NULL;
		return (0);
	}
	opt_st = mem_malloc(sizeof(*opt_st), m_anon);
	if (opt_st == NULL) {
		logmsgx(IPA_LOG_ERR, "opt_st_add: mem_malloc failed");
		return (-1);
	}
	opt_st->st_names = st_names;
	STAILQ_INSERT_TAIL(&opt_st_list, opt_st, link);
	cur_opt_st = opt_st;
	return (0);
}

/*
 * Parse names of statistics system given in command query.
 */
int
opt_st_parse(void)
{
	const struct st_mod *st_mod;
	const char *st_name;
	struct opt_st *opt_st;
	struct st_set *set;
	struct st_elem *st;
	struct st_list *list;
	char *ptr;

	STAILQ_FOREACH(opt_st, &opt_st_list, link) {
		set = NULL;
		list = NULL;

		for (ptr = opt_st->st_names; ptr != NULL;) {
			st_name = ptr;
			ptr = strchr(ptr, ' ');
			if (ptr != NULL) {
				*ptr++ = '\0';
				for (; *ptr == ' '; ++ptr)
					;
			}
			/* Handle "null" statistics system. */
			if (strcmp(st_name, "null") == 0) {
				if (list == NULL && ptr == NULL) {
					list = &st_list_null;
					continue;
				}
				logmsgx(IPA_LOG_ERR, "opt_st_parse: built-in "
				    "statistics system \"null\" cannot be used "
				    "together with another statistics systems");
				return (-1);
			}

			st_mod = st_mod_by_name(st_name);
			if (st_mod == NULL) {
				logmsgx(IPA_LOG_ERR, "opt_st_parse: "
				    "cannot find module with \"%s\" statistics "
				    "system name", st_name);
				return (-1);
			}

			if (set != NULL) {
				/* We already have set. */
				STAILQ_FOREACH(st, list, link)
					if (strcmp(st_name,
					    st->ipa_st_mod->st_name) == 0) {
						logmsgx(IPA_LOG_ERR,
						    "opt_st_parse: duplicated "
						    "statistics system name "
						    "\"%s\"", st_name);
						return (-1);
					}
			} else {
				/* Create new set for st_list parameter. */
				set = mem_malloc(sizeof(*set), m_anon);
				if (set == NULL) {
					logmsgx(IPA_LOG_ERR, "opt_st_parse: "
					    "mem_malloc failed");
					return (-1);
				}
				list = &set->list;
				STAILQ_INIT(list);
			}

			/* Add new st element to st_list. */
			st = mem_malloc(sizeof(*st), m_anon);
			if (st == NULL) {
				logmsgx(IPA_LOG_ERR, "opt_st_parse: "
				    "mem_malloc failed");
				return (-1);
			}
			st->ipa_st_mod = st_mod->ipa_st_mod;
			st->mod_file = st_mod->mod_file;

			STAILQ_INSERT_TAIL(list, st, link);
		}

		opt_st->st_list = list;
		if (set != NULL)
			SLIST_INSERT_HEAD(&st_sets, set, link);
	}

	return (0);
}

/*
 * Release memory previously allocated by all opt_st_add() calls.
 */
void
opt_st_free(void)
{
	struct opt_st *opt_st, *opt_st_next;

	STAILQ_FOREACH_SAFE(opt_st, &opt_st_list, link, opt_st_next)
		mem_free(opt_st, m_anon);
}
