/*-
 * Copyright (c) 2005 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipastat_rules.c,v 1.3.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipastat_log.h"

#include "ipastat_conf.h"
#include "ipastat_main.h"
#include "ipastat_rules.h"
#include "ipastat_st.h"

#ifndef RULES_HASH_BUCKETS
# define RULES_HASH_BUCKETS 64		/* Must be power of 2. */
#endif

unsigned int	nrules = 0;		/* Number of rules. */

signed char	dynamic_rules;		/* dynamic_rules parameter. */

struct opt_rule	*cur_opt_rule = NULL;	/* Current optional rule. */

ipa_mzone	*rule_mzone;		/* Mzone for all struct rule{}. */

ipa_mzone	*rulepat_mzone;		/* Mzone for all struct rulepat{}. */

/* List of all rules. */
struct rules_list rules_list = STAILQ_HEAD_INITIALIZER(rules_list);

/* List of rulepats. */
struct rulepats_list rulepats_list = STAILQ_HEAD_INITIALIZER(rulepats_list);

/* List of all rules names in the command line. */
struct opt_rules_list opt_rules_list = STAILQ_HEAD_INITIALIZER(opt_rules_list);

SLIST_HEAD(rules_hash, rule);

static struct rules_hash rules_hash[RULES_HASH_BUCKETS];

/*
 * Initialize one rule.
 */
static int
init_rule(const struct opt_rule *opt_rule)
{
	struct rule *rule;

	rule = opt_rule->rule;
	if (!rule->inited) {
		rule->inited = 1;
		if (st_init_rule(rule) < 0) {
			logbt("init_rule");
			return (-1);
		}
	}
	return (0);
}

/*
 * Initialize rules and limits or thresholds.
 */
int
init_rules(void)
{
	const struct opt_rule *opt_rule;

	STAILQ_FOREACH(opt_rule, &opt_rules_list, link) {
		if (init_rule(opt_rule) < 0)
			goto failed;
#ifdef WITH_LIMITS
		if (has_opt_limits)
			if (init_limits(opt_rule) < 0)
				goto failed;
#endif
#ifdef WITH_THRESHOLDS
		if (has_opt_thresholds)
			if (init_thresholds(opt_rule) < 0)
				goto failed;
#endif
	}
	return (0);

failed:
	logbt("init_rules");
	return (-1);
}

/*
 * Release memory used by one rule.
 */
static void
free_rule(struct rule *rule)
{
#ifdef WITH_LIMITS
	free_limits(&rule->limits);
#endif
#ifdef WITH_THRESHOLDS
	free_thresholds(&rule->thresholds);
#endif
	if (rule->free_mask & RULE_FREE_NAME)
		mem_free(rule->name, m_anon);
	mem_free(rule->info, m_result);
	mzone_free(rule_mzone, rule);
}

/*
 * Call free_rule() for each rule.
 */
void
free_rules(void)
{
	struct rule *rule, *rule_next;

	STAILQ_FOREACH_SAFE(rule, &rules_list, list, rule_next)
		free_rule(rule);
	STAILQ_INIT(&rules_list);
}

/*
 * Deinitialize one rule.
 */
static int
deinit_rule(const struct opt_rule *opt_rule)
{
	struct rule *rule;

	rule = opt_rule->rule;
	if (rule->inited) {
		rule->inited = 0;
		if (st_deinit_rule(rule) < 0) {
			logbt("deinit_rule");
			return (-1);
		}
	}
	return (0);
}

/*
 * Deinitialize previously initialized rules, limits or thresholds.
 */
int
deinit_rules(void)
{
	struct opt_rule *opt_rule;
	int rv;

	rv = 0;
	STAILQ_FOREACH(opt_rule, &opt_rules_list, link) {
		if (deinit_rule(opt_rule) < 0)
			rv = -1;
#ifdef WITH_LIMITS
		if (has_opt_limits)
			if (deinit_limits(opt_rule) < 0)
				rv = -1;
#endif
#ifdef WITH_THRESHOLDS
		if (has_opt_thresholds)
			if (deinit_thresholds(opt_rule) < 0)
				rv = -1;
#endif
	}
	if (rv != 0)
		logbt("deinit_rules");
	return (rv);
}

/*
 * Return name hash value for the given name;
 */
static unsigned int
rule_name_hash(const char *name)
{
	unsigned int value;

	for (value = 0; *name != '\0'; ++name)
		value += (unsigned char)*name;
	return (value);
}

#define rules_hash_bucket(x) ((x) & (RULES_HASH_BUCKETS - 1))

/*
 * Add rule to rules_hash.
 */
void
rules_hash_add(struct rule *rule)
{
	rule->name_hash = rule_name_hash(rule->name);
	SLIST_INSERT_HEAD(&rules_hash[rules_hash_bucket(rule->name_hash)],
	    rule, hlink);
}

void
rules_hash_init(void)
{
	struct rules_hash *hash;

	for (hash = rules_hash; hash < rules_hash + RULES_HASH_BUCKETS; ++hash)
		SLIST_INIT(hash);
}

/*
 * Return pointer to rule with the given name.
 */
struct rule *
rule_by_name(const char *name)
{
	const struct rules_hash *hash;
	struct rule *rule;
	unsigned int name_hash;

	name_hash = rule_name_hash(name);
	hash = &rules_hash[rules_hash_bucket(name_hash)];

	SLIST_FOREACH(rule, hash, hlink)
		if (rule->name_hash == name_hash &&
		    strcmp(rule->name, name) == 0)
			break;

	return (rule);
}

struct rule *
alloc_rule(char *name)
{
	struct rule *rule;

	rule = mzone_alloc(rule_mzone);
	if (rule == NULL) {
		xlogmsgx(IPA_LOG_ERR, "alloc_rule: mzone_alloc failed");
		return (NULL);
	}
	rule->name = name;
	rule->info = NULL;
	rule->no = nrules++;
	rule->st_list = NULL;
	rule->inited = 0;

	rules_hash_add(rule);
	STAILQ_INSERT_TAIL(&rules_list, rule, list);

#ifdef WITH_LIMITS
	STAILQ_INIT(&rule->limits);
#endif
#ifdef WITH_THRESHOLDS
	STAILQ_INIT(&rule->thresholds);
#endif

	return (rule);
}

/*
 * Release memory used by all rulepats.
 */
void
free_rulepats(void)
{
	struct rulepat *rulepat;

	STAILQ_FOREACH(rulepat, &rulepats_list, link) {
		mem_free(rulepat->pat, m_parser);
		regfree(&rulepat->re);
#ifdef WITH_LIMITS
		free_limits(&rulepat->limits);
#endif
#ifdef WITH_THRESHOLDS
		free_thresholds(&rulepat->thresholds);
#endif
	}
	mzone_deinit(rulepat_mzone);
}

/*
 * Inherit settings from rulepat from modules for rule.
 */
int
mod_rule_inherit(const struct rulepat *rulepat, const struct rule *rule)
{
	const struct st_mod *st_mod;

	SLIST_FOREACH(st_mod, &st_mod_list, link)
		if (st_mod->ipa_st_mod->conf_inherit != NULL &&
		    st_mod->ipa_st_mod->conf_inherit(rulepat->no, rule->no,
		    rule->name) < 0) {
			xlogmsgx(IPA_LOG_ERR, "module %s: "
			    "conf_inherit(%s, %s) failed", st_mod->mod_file,
			    parser_stringify(rulepat->pat), rule->name);
			return (-1);
		}
	return (0);
}

/*
 * Inherit settings from rulepat{} in rule{}.
 */
static int
rule_inherit_rulepat(const struct rulepat *rulepat, struct rule *rule)
{
	if (rule->st_list == NULL)
		rule->st_list = rulepat->st_list;
#ifdef WITH_LIMITS
	if (STAILQ_EMPTY(&rule->limits) &&
	    !STAILQ_EMPTY(&rulepat->limits))
		if (copy_limits(rule, &rulepat->limits) < 0) {
			xlogmsgx(IPA_LOG_ERR, "rule %s: cannot copy all "
			    "limits from rulepat %s", rule->name,
			    parser_stringify(rulepat->pat));
			return (-1);
		}
#endif
#ifdef WITH_THRESHOLDS
	if (STAILQ_EMPTY(&rule->thresholds) &&
	    !STAILQ_EMPTY(&rulepat->thresholds))
		if (copy_thresholds(rule, &rulepat->thresholds) < 0) {
			xlogmsgx(IPA_LOG_ERR, "rule %s: cannot copy all "
			    "thresholds from rulepat %s", rule->name,
			    parser_stringify(rulepat->pat));
			return (-1);
		}
#endif
	if (mod_rule_inherit(rulepat, rule) < 0) {
		xlogmsgx(IPA_LOG_ERR, "rule %s: cannot inherit settings "
		    "from rulepat %s {} from some module", rule->name,
		    parser_stringify(rulepat->pat));
		return (-1);
	}
	return (0);
}

/*
 * Inherit settings from all rulepats in rule{}.
 * Return:
 *   0 -- settings were copied, but matched rulepat had check_next == 0;
 *   1 -- settings were copied and matched rulepat had check_next == 1;
 *  -1 -- some error occurred.
 */
static int
rule_inherit_rulepats(struct rule *rule)
{
	const struct rulepat *rulepat;

	STAILQ_FOREACH(rulepat, &rulepats_list, link) {
		if (regexec_simple(&rulepat->re, rule->name) != 0)
			continue;
		if (rule_inherit_rulepat(rulepat, rule) < 0)
			return (-1);
		if (rulepat->check_next == 0)
			return (1);
	}
	return (0);
}

/*
 * Inherit settings from global{} in rule{}.
 */
static void
rule_inherit_global(struct rule *rule)
{
	if (rule->st_list == NULL)
		rule->st_list = global_st_list;
}

/*
 * Inherit settings for a rule from a matched rule pattern,
 * then from global{}.
 */
static int
rule_inherit(struct rule *rule)
{
	switch (rule_inherit_rulepats(rule)) {
	case 0:
		rule_inherit_global(rule);
		break;
	case -1:
		xlogmsgx(IPA_LOG_ERR, "rule %s: cannot inherit settings "
		    "from rule patterns", rule->name);
		return (-1);
	}
	return (0);
}

int
rules_inherit(void)
{
	struct rule *rule;

	STAILQ_FOREACH(rule, &rules_list, list)
		if (rule_inherit(rule) < 0)
			return (-1);
	return (0);
}

/*
 * Set default settings and inherit settings from global{} for rulepat{}.
 */
void
rulepats_inherit(void)
{
	struct rulepat *rulepat;

	STAILQ_FOREACH(rulepat, &rulepats_list, link)
		if (rulepat->check_next <= 0) {
			rulepat->check_next = 0;
			if (rulepat->st_list == NULL)
				rulepat->st_list = global_st_list;
		}
}

/*
 * Add one optional rule given in the -q -r option.
 */
int
opt_rule_add(const char *name)
{
	struct opt_rule *opt_rule;

	opt_rule = mem_malloc(sizeof(*opt_rule), m_anon);
	if (opt_rule == NULL) {
		logbt("opt_rule_add");
		return (-1);
	}

	opt_rule->name = name;
	opt_rule->opt_st = cur_opt_st;
	opt_rule->type = OPT_RULE_RULE;
	opt_rule->data = NULL;
#ifdef WITH_LIMITS
	STAILQ_INIT(&opt_rule->opt_limits);
#endif
#ifdef WITH_THRESHOLDS
	STAILQ_INIT(&opt_rule->opt_thresholds);
#endif
	STAILQ_INSERT_TAIL(&opt_rules_list, opt_rule, link);
	cur_opt_rule = opt_rule;

	return (0);
}

/*
 * Parse all rules, limits and thresholds names given in the
 * command line.
 */
int
opt_rules_parse(void)
{
	struct opt_rule *opt_rule;
	struct rule *rule;

	STAILQ_FOREACH(opt_rule, &opt_rules_list, link) {
		rule = rule_by_name(opt_rule->name);
		if (rule == NULL) {
			/* This rule is not given in a configuration file. */
			if (!dynamic_rules) {
				logmsgx(IPA_LOG_ERR, "opt_rules_parse: "
				    "unknown rule %s", opt_rule->name);
				return (-1);
			}
			rule = alloc_rule((char *)opt_rule->name);
			if (rule == NULL)
				goto failed;
			rule->free_mask = 0;
		}

		if (opt_rule->opt_st != NULL)
			rule->st_list = opt_rule->opt_st->st_list;

		if (rule_inherit(rule) < 0)
			goto failed;

		opt_rule->rule = rule;

#ifdef WITH_LIMITS
		if (opt_limits_parse(opt_rule) < 0)
			goto failed;
#endif
#ifdef WITH_THRESHOLDS
		if (opt_thresholds_parse(opt_rule) < 0)
			goto failed;
#endif
	}
	return (0);

failed:
	logbt("opt_rules_parse");
	return (-1);
}

/*
 * Release memory used by all opt_rule structures.
 */
void
opt_rules_free(void)
{
	struct opt_rule *opt_rule, *opt_rule_next;

	STAILQ_FOREACH_SAFE(opt_rule, &opt_rules_list, link, opt_rule_next) {
#ifdef WITH_LIMITS
		opt_limits_free(&opt_rule->opt_limits);
#endif
#ifdef WITH_THRESHOLDS
		opt_thresholds_free(&opt_rule->opt_thresholds);
#endif
		mem_free(opt_rule, m_anon);
	}
}
