/*-
 * Copyright (c) 1999-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *   @(#)$Id: ipa_rules.h,v 1.2 2011/01/23 18:42:35 simon Exp $
 */

#ifndef IPA_RULES_H
#define IPA_RULES_H

#include "ipa_limits.h"
#include "ipa_thresholds.h"

#ifndef RULEPAT_NSIZE
# define RULEPAT_NSIZE	10
#endif
#ifndef RULEPAT_NALLOC
# define RULEPAT_NALLOC	10
#endif

/*
 * rulepat{} section.
 */
struct rulepat {
	STAILQ_ENTRY(rulepat) link;	/* For list building. */

	char		*pat;		/* Regular expression string. */
	regex_t		re;		/* Compiled regular expression. */
	unsigned int	no;		/* Ordinal number of this section. */

	const struct tevent *update_tevent; /* { update_time } */
	const struct tevent *append_tevent; /* { append_time } */

	const struct worktime *worktime;/* { worktime } */

	const struct ac_list *ac_list;	/* { ac_list } */
	const struct db_list *db_list;	/* { db_list } */

	signed char	debug_exec;	/* { debug_exec } */
	signed char	check_next;	/* { check_next_rulepat } */

#ifdef WITH_LIMITS
	signed char	debug_limit;	/* { debug_limit } */
	signed char	debug_limit_init; /* { debug_limit_init } */
	struct limits_list limits;	/* All { limit {}} */
#endif

#ifdef WITH_THRESHOLDS
	signed char	debug_threshold; /* { debug_threshold } */
	signed char	debug_threshold_init; /* { debug_threshold_init } */
	struct thresholds_list thresholds; /* All { threshold {}} */
#endif

#ifdef CTL_CHECK_CREDS
	const struct ctl_acl_class *ctl_rule_acl; /* { ctl_rule_acl } */
#endif

	struct cmds_rule rc[2];		/* { startup } and { shutdown } */
};

/*
 * List of all rulepat{} sections.
 */
STAILQ_HEAD(rulepats_list, rulepat);

extern struct rulepats_list rulepats_list;

extern ipa_mzone *rulepat_mzone;

#ifdef WITH_RULES

struct rule;

/*
 * Element in reverse list for "ac_gather_*" parameters.
 */
struct acg {
	SLIST_ENTRY(acg) link;		/* Link for list of acg for rule. */
	struct rule	*rule;		/* Pointer to rule, in which current
					   rule matches ac_gather_* value. */
	char		addition;	/* If non-zero, then add statistics. */
};

#define ACG_NSIZE	RULE_NSIZE
#define ACG_NALLOC	RULE_NALLOC

extern ipa_mzone *acg_mzone;

SLIST_HEAD(acg_list, rule);

extern signed char has_ac_gather;
extern struct acg_list acg_list;

extern int	init_acg(struct rule *);

#endif /* WITH_RULES */

#ifndef RULE_NSIZE
# define RULE_NSIZE	30
#endif

#ifndef RULE_NALLOC
# define RULE_NALLOC	20
#endif

#define RULE_FLAG_ACTIVE	0x01	/* Rule is active. */
#define RULE_FLAG_QUEUED	0x02	/* Rule is in [in]active queue. */
#ifdef WITH_LIMITS
# define RULE_FLAG_FREE_LIMITS	0x04	/* Free memory in limits. */
#endif
#ifdef WITH_THRESHOLDS
# define RULE_FLAG_FREE_THRESHOLDS 0x08	/* Free memory in thresholds. */
#endif
#define RULE_FLAG_DYNAMIC	0x10	/* Dynamic rule. */
#define RULE_FLAG_IPACTL	0x20	/* Dynamic rule created by ipactl. */

#define RULE_IS_ACTIVE(x)	((x)->rule_flags & RULE_FLAG_ACTIVE)
#define RULE_IS_INACTIVE(x)	(!RULE_IS_ACTIVE(x))
#define RULE_IS_QUEUED(x)	((x)->rule_flags & RULE_FLAG_QUEUED)

#define RULE_SET_ACTIVE(x)	((x)->rule_flags |= RULE_FLAG_ACTIVE)
#define RULE_SET_INACTIVE(x)	((x)->rule_flags &= ~RULE_FLAG_ACTIVE)
#define RULE_SET_QUEUED(x)	((x)->rule_flags |= RULE_FLAG_QUEUED)

#define RULE_IS_DYNAMIC(x)	((x)->rule_flags & RULE_FLAG_DYNAMIC)
#define RULE_SET_DYNAMIC(x)	((x)->rule_flags |= RULE_FLAG_DYNAMIC)

/*
 * rule{} section.
 */
struct rule {
	TAILQ_ENTRY(rule) queue;	/* Active/inactive rules queue link. */
	LIST_ENTRY(rule) hlink;		/* One hash bucket link. */

	unsigned int	no;		/* Rule ordinal number. */
	unsigned int	orderno;	/* Ordinal number in queues. */

	unsigned int	check_sec;	/* Time when to check rule. */
	unsigned int	append_sec;	/* Time when to append new record. */
	unsigned int	inactive_sec;	/* Time when to set rule inactive. */

	uint64_t	cnt;		/* Counter. */
	uint64_t	cnt_neg;	/* Negative counter. */
	char		newstat;	/* Newstat flag for rule. */

	unsigned int	rule_flags;	/* ORed RULE_FLAG_xxx */

	const struct tevent *update_tevent; /* { update_time } */
	const struct tevent *append_tevent; /* { append_time } */
	const struct worktime *worktime;/* { worktime } */

	const struct ac_list *ac_list;	/* { ac_list } */
	const struct db_list *db_list;	/* { db_list } */

#ifdef WITH_RULES
	char		*acg_add_pat;	/* { ac_gather_add } */
	char		*acg_sub_pat;	/* { ac_gather_sub } */
	regex_t		acg_add_re;	/* Compiled acg_add_pat. */
	regex_t		acg_sub_re;	/* Compiled acg_sub_pat. */
	SLIST_ENTRY(rule) acg_link;	/* Link rules with "ac_gather_*". */
	SLIST_HEAD(EMPTY, acg) acgs;	/* List of reverse "ac_gather_*". */
#endif

	signed char	debug_exec;	/* { debug_exec } */

#ifdef WITH_LIMITS
	signed char debug_limit;	/* { debug_limit } */
	signed char debug_limit_init;	/* { debug_limit_init } */
	struct limits_list limits;	/* All { limit {}} */
#endif

#ifdef WITH_THRESHOLDS
	signed char debug_threshold;	/* { debug_threshold } */
	signed char debug_threshold_init; /* { debug_threshold_init } */
	struct thresholds_list thresholds; /* All { threshold {}} */
#endif

#ifdef CTL_CHECK_CREDS
	const struct ctl_acl_class *ctl_rule_acl; /* All { ctl_rule_acl } */
#endif

	TAILQ_ENTRY(rule) list;		/* All rules list. */

	char		*name;		/* Name of this rule. */
	char		*info;		/* { info } */
	unsigned int	name_hash;	/* Hashed value of rule's name. */

#ifdef WITH_AUTORULES
	unsigned int	autoruleno;	/* Autorule number of this rule. */
#endif

	struct cmds_rule rc[2];		/* { startup } and { shutdown } */
};

/*
 * List of all rules.
 */
TAILQ_HEAD(rules_list, rule);

/*
 * Active/inactive rules queue.
 */
TAILQ_HEAD(rules_queue, rule);

extern unsigned int nstatrules;

extern signed char keep_rules_order;

extern struct rules_list rules_list;
extern ipa_mzone *rule_mzone;

extern unsigned int rules_inactive_check_sec;

extern struct rules_queue rules_active;
extern struct rules_queue rules_inactive;

extern int	set_rule_active(struct rule *);
extern int	set_rule_inactive(struct rule *);
extern void	queue_active_rule(struct rule *);
extern void	sort_inactive_rules(void);
extern int	check_inactive_rules(void);

extern int	init_rules(int);
extern int	deinit_rules(void);
extern int	deinit_rule(struct rule *);
extern void	free_rule(struct rule *);
extern void	free_rules(void);
extern void	rule_init_cmds(struct rule *);
extern int	rule_inherit(struct rule *);

extern int	rule_add_chunk(struct rule *, const uint64_t *);
extern int	rule_sub_chunk(struct rule *, const uint64_t *);

extern struct rule *rule_by_name(const char *);
extern void	rules_hash_init(void);
extern void	rules_hash_add(struct rule *);
extern void	rules_hash_rem(struct rule *);
extern int	rules_hash_is_empty(void);

extern void	set_rules_for_check(void);

extern int	mod_set_rule_active(struct rule *, int);

extern void	rulepats_inherit(void);
extern void	free_rulepats(void);

extern int	mod_rule_inherit(const struct rulepat *, const struct rule *);

#ifdef WITH_RULES
extern void	init_active_rules(void);
extern int	rules_inherit(void);
#endif

#endif /* !IPA_RULES_H */
