/*-
 * Copyright (c) 1999-2004, 2009 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_main.c,v 1.4.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#ifdef WITH_PTHREAD
# include <pthread.h>
#endif

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_log.h"
#include "ipa_time.h"

#include "ipa_cmd.h"
#include "ipa_conf.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

#ifndef WCOREDUMP
# define WCOREDUMP(x) (0)
#endif

char		debug = 0;		/* -d */

uid_t		myuid;			/* Current process UID. */
gid_t		mygid;			/* Current process GID. */

const uint64_t	uint64_zero = 0;
const uint64_t	uint64_max = UINT64_MAX;

ipa_mem_type	*m_anon;		/* Anonymous memory allocations. */
ipa_mem_type	*m_tmp;			/* Temporary memory allocations. */

sigset_t	sigmask;		/* Signal mask. */
sigset_t	zeromask;		/* Zero mask for unblocking signals. */

int		stdout_read;		/* Pipe's read-end for stdout. */
int		stderr_read;		/* Pipe's read-end for stderr. */

char		dump_flag = 0;		/* Set if ctl command "dump". */
char		freeze_flag = 0;	/* Set if ctl command "freeze". */

static time_t	newdaytime;		/* Time of a new day. */

unsigned int	main_check_sec;		/* Time in seconds since midnight
					   when to wakeup. */

static fd_set	readfdset;		/* Read descriptors set. */
static int	maxfd;			/* Maximum #descr + 1 in readfdset. */

/*
 * All sig_*() functions write one character to sig_pipe, since its
 * write-end is marked as non-blocking, they cannot block in write().
 * According to SUSv3 a signal handler can assign value only to
 * a variable of type 'volatile sig_atomic_t'.
 */

/* Pipe for IPC from async signals handlers. */
static int	sig_pipe[2];

/* Set if need to check all active rules. */
volatile sig_atomic_t need_check_flag = 0;

/* Different flags for signal handlers. */
static volatile sig_atomic_t signal_flag = 0;
static volatile sig_atomic_t reconf_flag = 0;
static volatile sig_atomic_t shutdown_flag = 0;
static volatile sig_atomic_t sigchld_flag = 0;

/* ac_get_rule_stat() generation counter. */
static unsigned int stat_generation;

#ifdef WITH_PTHREAD
int
Sigprocmask(int how, const sigset_t *set, sigset_t *oset)
{
	int error;

	error = pthread_sigmask(how, set, oset);
	if (error != 0) {
		errno = error;
		return (-1);
	}
	return (0);
}
#endif

/*
 * Mark a file descriptor as non-blocking.
 */
int
fd_set_nonblock(int fd)
{
	int val;

	val = fcntl(fd, F_GETFL, 0);
	if (val < 0) {
		logmsg(IPA_LOG_ERR, "fd_set_nonblock: fcntl(%d, F_GETFL)", fd);
		return (-1);
	}
	if (fcntl(fd, F_SETFL, val | O_NONBLOCK) < 0) {
		logmsg(IPA_LOG_ERR, "fd_set_nonblock: fcntl(%d, F_SETFL)", fd);
		return (-1);
	}
	return (0);
}

static int
init_sig_pipe(void)
{
	static char called_before = 0;

	if (called_before) {
		/* Close previously opened pipe. */
		if (close(sig_pipe[0]) < 0 || close(sig_pipe[1]) < 0) {
			logmsg(IPA_LOG_ERR, "init_sig_pipe: close");
			return (-1);
		}
	} else
		called_before = 1;

	/* Create new pipe for IPC from signal handlers. */
	if (pipe(sig_pipe) < 0) {
		logmsg(IPA_LOG_ERR, "init_sig_pipe: pipe");
		return (-1);
	}

	/* Mark write- and read-ends as non-blocking. */
	if (fd_set_nonblock(sig_pipe[0]) < 0 ||
	    fd_set_nonblock(sig_pipe[1]) < 0) {
		logbt("init_sig_pipe");
		return (-1);
	}

	return (0);
}

/*
 * Initialize everything.
 */
int
init_all(void)
{
	if (init_time_values() < 0)
		goto failed;

	/* Flush mask of read-ready descriptors. */
	FD_ZERO(&readfdset);
	maxfd = 0;

	/* Initialize signal IPC pipe, add read-end to readfdset. */
	if (init_sig_pipe() < 0)
		goto failed;
	FD_SET(sig_pipe[0], &readfdset);
	if (maxfd < sig_pipe[0])
		maxfd = sig_pipe[0];

	/* Set pipes for stdout and stderr om readfdset. */
	FD_SET(stdout_read, &readfdset);
	if (maxfd < stdout_read)
		maxfd = stdout_read;
	if (!debug) {
		FD_SET(stderr_read, &readfdset);
		if (maxfd < stderr_read)
			maxfd = stderr_read;
	}

	/* Initialize ctl. */
	if (ctl_enable) {
		if (init_ctl() < 0)
			goto failed;
		FD_SET(ctl_listenfd, &readfdset);
		if (ctl_listenfd > maxfd)
			maxfd = ctl_listenfd;
	}

	++maxfd;

	if (pre_init_ac_mods() < 0)
		goto failed;
	if (pre_init_db_mods() < 0)
		goto failed;

#ifdef WITH_AUTORULES
	if (init_autorules() < 0)
		goto failed;
#endif

#ifdef WITH_RULES
	/* Initialize active rules queue. */
	init_active_rules();

	/* Initialize acg_mzone, needed only if there are static rules. */
	if (!SLIST_EMPTY(&acg_list)) {
		acg_mzone = mzone_init(MZONE_NAME(acg), "Reverse ac_gather_*",
		    0, sizeof(struct acg), ACG_NSIZE, ACG_NALLOC);
		if (acg_mzone == NULL)
			goto failed;
		has_ac_gather = 1;
	} else {
		acg_mzone = NULL;
		has_ac_gather = 0;
	}

	/* Initialize static rules, limits and thresholds. */
	if (init_rules(1) < 0)
		goto failed;

	if (acg_mzone != NULL) {
		/*
		 * If any rule was not caught by ac_gather_* and
		 * there no autorules, then can deinitialize acg_mzone.
		 */
		if (mzone_is_empty(acg_mzone) && nautorules == 0) {
			mzone_deinit(acg_mzone);
			acg_mzone = NULL;
		}
	}
#endif /* WITH_RULES */

	if (init_ac_mods() < 0)
		goto failed;
	if (init_db_mods() < 0)
		goto failed;

	return (0);

failed:
	logbt("init_all");
	return (-1);
}

/*
 * Deinitialize everything ignoring errors.  Do the same steps as in
 * init_all(), but in reverse order.
 */
int
deinit_all(void)
{
	int rv;

	rv = 0;
	if (deinit_rules() < 0)
		rv = -1;
#ifdef WITH_AUTORULES
	if (deinit_autorules() < 0)
		rv = -1;
#endif
	if (deinit_ac_mods() < 0)
		rv = -1;
	if (deinit_db_mods() < 0)
		rv = -1;
	if (deinit_ctl() < 0)
		rv = -1;
	if (rv < 0) {
		logbt("deinit_all");
		return (-1);
	}

	/* Validate internal structures after deinitialization. */
	if (!TAILQ_EMPTY(&rules_active)) {
		logmsgx(IPA_LOG_ERR, "internal error: deinit_all: "
		    "rules_active is not empty");
		return (-1);
	}
	if (!TAILQ_EMPTY_HEAD_VALID(&rules_active)) {
		logmsgx(IPA_LOG_ERR, "internal error: deinit_all: empty "
		    "rules_active list is in inconsistent state");
		return (-1);
	}
	if (!TAILQ_EMPTY(&rules_inactive)) {
		logmsgx(IPA_LOG_ERR, "internal error: deinit_all: "
		    "rules_inactive is not empty");
		return (-1);
	}
	if (!TAILQ_EMPTY_HEAD_VALID(&rules_inactive)) {
		logmsgx(IPA_LOG_ERR, "internal error: deinit_all: empty "
		    "rules_inactive list is in inconsistent state");
		return (-1);
	}
#ifdef WITH_ANY_LIMITS
	if (!wpid_hash_is_empty()) {
		logmsgx(IPA_LOG_ERR, "internal error: deinit_all: wpid_hash "
		    "is not empty");
		return (-1);
	}
#endif
	return (0);
}

/*
 * Free all resources allocated during work and validate internal
 * structures after releasing memory.
 */
int
free_all(void)
{
	unsigned int n;
	int rv;

	if (shell_path != shell_path_default)
		mem_free(shell_path, m_parser);
	if (shell_arg1 != shell_arg1_default)
		mem_free(shell_arg1, m_parser);

	free_rules();
	if (!TAILQ_EMPTY(&rules_list)) {
		logmsgx(IPA_LOG_ERR, "internal error: free_all: "
		    "rules_list is not empty");
		return (-1);
	}
	if (!TAILQ_EMPTY_HEAD_VALID(&rules_list)) {
		logmsgx(IPA_LOG_ERR, "internal error: free_all: "
		    "empty rules_list list is in inconsistent state");
		return (-1);
	}
	if (!rules_hash_is_empty()) {
		logmsgx(IPA_LOG_ERR, "internal error: free_all: "
		    "rules_hash is not empty");
		return (-1);
	}
	n = mzone_nused(rule_mzone);
	if (n != 0) {
		logmsgx(IPA_LOG_ERR, "internal error: free_all: "
		    "rule_mzone is not empty: %u", n);
		return (-1);
	}
	mzone_deinit(rule_mzone);

	free_rulepats();

#ifdef WITH_LIMITS
	if (limit_mzone != NULL) {
		n = mzone_nused(limit_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: free_all: "
			    "limit_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(limit_mzone);
	}
#endif
#ifdef WITH_SUBLIMITS
	if (sublimit_mzone != NULL) {
		n = mzone_nused(sublimit_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: free_all: "
			    "sublimit_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(sublimit_mzone);
	}
#endif
#ifdef WITH_THRESHOLDS
	if (threshold_mzone != NULL) {
		n = mzone_nused(threshold_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: free_all: "
			    "threshold_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(threshold_mzone);
	}
#endif

	mzone_deinit(tevent_mzone);

	if (cmd_mzone != NULL) {
		cmds_free(&cmds_global[RC_STARTUP]);
		cmds_free(&cmds_global[RC_SHUTDOWN]);
		n = mzone_nused(cmd_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: free_all: "
			    "cmd_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(cmd_mzone);
	}

	free_worktimes();

#ifdef WITH_RULES
	if (acg_mzone != NULL) {
		n = mzone_nused(acg_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: free_all: "
			    "acg_mzone is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(acg_mzone);
	}
	if (ictl_mzone != NULL) {
		n = mzone_nused(ictl_mzone);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: free_all: "
			    "cmd_ictl is not empty: %u", n);
			return (-1);
		}
		mzone_deinit(ictl_mzone);
	}
#endif

	free_ac_lists();
	free_db_lists();

	mem_free(parser_str_buf, m_parser);
	parser_str_buf = NULL;

	memfunc_deinit_1(0);

	rv = unload_all_mods();
	if (rv < 0)
		logmsgx(IPA_LOG_ERR, "free_all: cannot correctly unload "
		    "all modules");

	memfunc_deinit_2(0);
	return (rv);
}

/*
 * Some of children exited.
 */
static int
check_child(void)
{
#ifdef WITH_ANY_LIMITS
	struct wpid *wpid;
#endif
	pid_t pid;
	int status;

	/* SUSv3 claims that pid_t is signed integer, so can use `>' here. */
	while ((pid = waitpid((pid_t)0, &status, WNOHANG)) > 0) {
#ifdef WITH_ANY_LIMITS
		/*
		 * Find appropriate wpid structure for exited child and
		 * remove it from the hash table (zeroing its pid field).
		 * Sometimes wpid structure does not exist for child's PID,
		 * since we already use it for another child for the same
		 * section or reconfiguration occurred.
		 */
		wpid = wpid_lookup_pid(pid);
		if (wpid != NULL) {
			if (wpid->debug_exec)
				log_term_wpid(wpid);
			wpid_hash_rem(wpid);
		} else
#endif
		if (global_debug_exec)
			logdbg("child %ld exited", (long)pid);

		/* The child in normal situation should return EXIT_SUCCESS. */
		if (WIFEXITED(status)) {
			if (WEXITSTATUS(status) != EXIT_SUCCESS) {
				logmsgx(IPA_LOG_ERR, "check_child: exit "
				    "status from child %ld is %d", (long)pid,
				    WEXITSTATUS(status));
				return (-1);
			}
		} else if (WIFSIGNALED(status)) {
			logmsgx(IPA_LOG_ERR, "check_child: abnormal "
			    "termination of child %ld signal %d%s", (long)pid,
			    WTERMSIG(status), WCOREDUMP(status) ?
			    " (core file generated)" : "");
			return (-1);
		} else {
			logmsgx(IPA_LOG_ERR, "check_child: termination of "
			    "child %ld unknown status", (long)pid);
			return (-1);
		}
	}

	if (pid == (pid_t)-1 && errno != ECHILD) {
		logmsg(IPA_LOG_ERR, "check_child: waitpid");
		return (-1);
	}
	return (0);
}

/*
 * This routine is called when a new day came (newday_flag != 0),
 * or when there is a reconfiguration, or when there are some
 * problems with system time or local time (newday_flag == 0).
 * The first_call flag is used only for debugging.
 */
static int
newday(int first_call)
{
	ipa_tm curdate_save;
	struct tm tm;
	struct rule *rule;
	struct tevent *tevent;

	main_check_sec = SECONDS_IN_DAY;
	curdate_save = curdate;

	/* Get time of a new day, month can be out of range. */
	tm = curdate;
	tm.tm_sec = tm.tm_min = tm.tm_hour = 0;
	tm.tm_mday++;
	tm.tm_mon--;
	tm.tm_year -= 1900;
	tm.tm_isdst = -1;

	/* mktime(3) has to fix all incorrect ranges. */
	newdaytime = mktime(&tm);
	if (newdaytime == (time_t)-1) {
		/* This should not happen for 00:00:00. */
		logmsg(IPA_LOG_ERR, "newday: mktime failed");
		return (-1);
	}

	if (newday_flag) {
		/*
		 * We simulate that it is 00:00:00 now, but this can be
		 * incorrect, nevertheless in most cases it should be almost
		 * correct, anyway this is reflected in the database only,
		 * everywhere cursec is used to compare some event's time
		 * with current time.
		 */
		curdate.tm_sec = curdate.tm_min = curdate.tm_hour = 0;
	}

	worktimes_newday(!first_call);

	/* Initialize all tevents. */
	SLIST_FOREACH(tevent, &tevents_list, link) {
		tevent->event_sec = 0;
		while ((tevent->event_sec += tevent->event_step) <= cursec)
			if (debug_time > 1)
				logdbg("newday: skipping time event %s for "
				    "time interval %s",
				    sec_str(tevent->event_sec),
				    time_str(tevent->event_step));
	}

	TAILQ_FOREACH(rule, &rules_list, list) {
		rule->check_sec = newday_flag ?
		    rule->update_tevent->event_sec : cursec;
#ifdef WITH_LIMITS
		if (limits_newday(rule) < 0)
			goto failed;
#endif
#ifdef WITH_THRESHOLDS
		if (thresholds_newday(rule) < 0)
			goto failed;
#endif
		if (WT_IS_INACTIVE(rule->worktime)) {
			if (RULE_IS_ACTIVE(rule))
				if (set_rule_inactive(rule) < 0)
					goto failed;
		} else {
			if (RULE_IS_INACTIVE(rule)) {
				if (set_rule_active(rule) < 0)
					goto failed;
			} else if (rule->newstat) {
				/*
				 * Since during startup ictl command could be
				 * called and a rule can have ac_gather_*
				 * parameters, then need to append current
				 * value of rule's counter.
				 */
				if (db_append_rule(rule, &rule->cnt, 0) < 0)
					goto failed;
			} else {
				if (db_append_rule(rule, &uint64_zero, 1) < 0)
					goto failed;
			}
			if (rule->check_sec > rule->worktime->inactive_sec)
				rule->check_sec = rule->worktime->inactive_sec;
			if (rule->check_sec > rule->append_sec)
				rule->check_sec = rule->append_sec;
		}
		if (main_check_sec > rule->check_sec)
			main_check_sec = rule->check_sec;
	}

	sort_inactive_rules();

	if (main_check_sec > rules_inactive_check_sec)
		main_check_sec = rules_inactive_check_sec;

#ifdef WITH_AUTORULES
	if (autorules_newday() < 0)
		goto failed;
	if (main_check_sec > autorules_active_check_sec)
		main_check_sec = autorules_active_check_sec;
	if (main_check_sec > autorules_inactive_check_sec)
		main_check_sec = autorules_inactive_check_sec;
#endif

	curdate = curdate_save;
	return (0);

failed:
	logbt("newday");
	return (-1);
}

/*
 * Do accounting for one rule.
 */
static int
do_ac_rule(struct rule *rule)
{
	uint64_t chunk;
	const struct ac_elem *ac;
	int addition;
#ifdef WITH_ANY_LIMITS
	unsigned int check_sec;
#endif

	/* Get statistics from all modules. */
	STAILQ_FOREACH(ac, rule->ac_list, link) {
		if (ac->ipa_ac_mod->ac_get_rule_stat(stat_generation,
		    rule->newstat, rule->no, &addition, &chunk) < 0)
			goto failed;
		if (rule->newstat) {
			/*
			 * If this is a new statistics, then ignore chunk
			 * from the module.
			 */
			if (chunk > 0)
				logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
				    "ac_get_rule_stat(1) returned non-zero "
				    "chunk %"PRIu64", it will be ignoreed",
				    ac->mod_file, rule->name, chunk);
			chunk = 0;
		} else {
			if (addition) {
				if (rule_add_chunk(rule, &chunk) < 0)
					goto failed;
			} else {
				if (rule_sub_chunk(rule, &chunk) < 0)
					goto failed;
			}
		}
	}

	/* Update statistics and schedule next checks. */
	rule->newstat = 0;
	if (db_update_rule(rule, &rule->cnt) < 0)
		goto failed;
	rule->check_sec = rule->update_tevent->event_sec;

#ifdef WITH_LIMITS
	if (!STAILQ_EMPTY(&rule->limits)) {
		if (check_limits(rule, &check_sec) < 0)
			goto failed;
		if (rule->check_sec > check_sec)
			rule->check_sec = check_sec;
	}
#endif
#ifdef WITH_THRESHOLDS
	if (!STAILQ_EMPTY(&rule->thresholds)) {
		if (check_thresholds(rule, &check_sec) < 0)
			goto failed;
		if (rule->check_sec > check_sec)
			rule->check_sec = check_sec;
	}
#endif

	if (rule->append_sec <= cursec && !newday_flag) {
		/*
		 * Append a new record, but do not append it
		 * at the end of the day.
		 */
		if (db_append_rule(rule, &uint64_zero, 1) < 0)
			goto failed;
	}
	if (WT_IS_INACTIVE(rule->worktime)) {
		/* Rule became inactive. */
		if (!newday_flag)
			if (set_rule_inactive(rule) < 0)
				goto failed;
	} else {
		/* Rule is still active. */
		if (rule->check_sec > rule->worktime->inactive_sec)
			rule->check_sec = rule->worktime->inactive_sec;
		if (rule->check_sec > rule->append_sec)
			rule->check_sec = rule->append_sec;
		if (main_check_sec > rule->check_sec)
			main_check_sec = rule->check_sec;
		if (debug_time)
			logdbg("do_ac_rule: rule %s: next check_sec %s",
			    rule->name, sec_str(rule->check_sec));
	}
	return (0);

failed:
	logbt("do_ac_rule");
	return (-1);
}

/*
 * Do accounting for all rules, limits, sublimits and thresholds,
 * check limits and thresholds events here, calculate new main_check_sec.
 */
static int
do_ac(void)
{
	struct rule *rule, *rule_next;
	struct tevent *tevent;

	main_check_sec = SECONDS_IN_DAY;

	/* Schedule next tevents. */
	SLIST_FOREACH(tevent, &tevents_list, link) {
		if (tevent->event_sec > cursec)
			continue;
		/* Need to adjust time event. */
		while ((tevent->event_sec += tevent->event_step) <= cursec)
			if (debug_time > 1)
				logdbg("do_ac: skipping time event %s for "
				    "time interval %s",
				    sec_str(tevent->event_sec),
				    time_str(tevent->event_step));
	}

	/* Check worktimes events. */
	if (worktimes_check_sec <= cursec)
		worktimes_check();

	/* Check inactive rules. */
	if (rules_inactive_check_sec <= cursec)
		if (check_inactive_rules() < 0)
			goto failed;

#ifdef WITH_AUTORULES
	/* Check inactive autorules. */
	if (autorules_inactive_check_sec <= cursec)
		if (check_inactive_autorules() < 0)
			goto failed;
#endif

	TAILQ_FOREACH_SAFE(rule, &rules_active, queue, rule_next) {
		if (debug_time)
			logdbg("do_ac: rule %s: check_sec %s",
			    rule->name, sec_str(rule->check_sec));
		if (rule->check_sec > cursec) {
			/* It is not a time to check this rule. */
			if (main_check_sec > rule->check_sec)
				main_check_sec = rule->check_sec;
		} else if (do_ac_rule(rule) < 0)
			goto failed;
	}

	if (main_check_sec > rules_inactive_check_sec)
		main_check_sec = rules_inactive_check_sec;

#ifdef WITH_AUTORULES
	if (autorules_active_check_sec <= cursec)
		if (check_active_autorules() < 0)
			goto failed;
	if (main_check_sec > autorules_active_check_sec)
		main_check_sec = autorules_active_check_sec;
	if (main_check_sec > autorules_inactive_check_sec)
		main_check_sec = autorules_inactive_check_sec;
#endif

	return (0);

failed:
	logbt("do_ac");
	return (-1);
}

/*
 * Do accounting when some time related problem occurred.
 */
static int
do_ac_prev(void)
{
	logmsgx(IPA_LOG_WARNING, "saving current statistics with old "
	    "timestamps...");

	/* Force updating of all statistics with old timestamps. */
	set_rules_for_check();
	if (do_ac() < 0)
		goto failed;

	/* Set current values for time related variables. */
	cursec = cursec_new;
	curdate = curdate_new;
	curwday = curdate.tm_wday;

	/* Reinitialize limits and thresholds. */
	if (init_rules(0) < 0)
		goto failed;

	logmsgx(IPA_LOG_WARNING, "appending new records for all rules...");
	if (newday(0) < 0)
		goto failed;

	/* Wakeup immediatelly. */
	main_check_sec = cursec;
	return (0);

failed:
	logbt("do_ac_prev");
	return (-1);
}

/*
 * Do accounting at the end of day (change of time is not more than one day).
 */
static int
do_ac_newday(void)
{
	double d;

	if (curdate_new.tm_mday == curdate.tm_mday) {
		logmsgx(IPA_LOG_ERR, "internal error: do_ac_newday: "
		    "new day was expected");
		return (-1);
	}

	/* A new day came. */
	if (debug_time)
		logdbg("do_ac_newday: new day");

	d = difftime(curtime, newdaytime);
	if (d > sensitive_time)
		logmsgx(IPA_LOG_WARNING, "new day came, but woke too late: "
		    "delta %.0fs is greater than \"sensitive_time\" %us",
		    d, sensitive_time);

	/* Make fake previous_day/24:00:00 time, curwday is not changed. */
	curdate.tm_sec = curdate.tm_min = 0;
	curdate.tm_hour = HOURS_IN_DAY;
	cursec = SECONDS_IN_DAY;

	/* And update all active rules. */
	newday_flag = 1;
	set_rules_for_check();
	if (do_ac() < 0)
		goto failed;

	/* Set current values for time related variables. */
	cursec = cursec_new;
	curdate = curdate_new;
	curwday = curdate.tm_wday;
	if (newday(0) < 0)
		goto failed;
	newday_flag = 0;
	return (0);

failed:
	logbt("do_ac_newday");
	return (-1);
}

/*
 * Do accounting (usual function call at the same day).
 */
static int
do_ac_curr(void)
{
	/* Set current values for time related variables. */
	cursec = cursec_new;
	curdate = curdate_new;

	if (debug_time) {
		logdbg("do_ac_curr: cursec %s", sec_str(cursec));
		logdbg("do_ac_curr: main_check_sec %s",
		    sec_str(main_check_sec));
	}

	if (cursec > main_check_sec) {
		unsigned int d;

		d = cursec - main_check_sec;
		if (d > sensitive_time) {
			logmsgx(IPA_LOG_WARNING, "time changed too quickly "
			    "during sleeping or woke too late: delta %s is "
			    "greater than \"sensitive_time\" %us",
			    time_str(d), sensitive_time);
			logmsgx(IPA_LOG_WARNING, "timestamps in database "
			    "will be inaccurate");
		}
		if (d <= FAKE_TIME_DELTA) {
			/*
			 * Fake time, anyway timestamps are fake
			 * in most cases.
			 */
			cursec = main_check_sec;
			sec_to_time(cursec, &curdate);
		}
	}

	if (do_ac() < 0) {
		logbt("do_ac_curr");
		return (-1);
	}
	return (0);
}

static int
dump_finished(void)
{
	logmsgx(IPA_LOG_INFO, "statistics were successfully saved to database");
	if (ctl_send_answer() < 0) {
		logbt("dump_finished");
		return (-1);
	}
	if (sleep_after_dump != 0) {
		logmsgx(IPA_LOG_INFO, "sleeping after \"dump\" and ignoring "
		    "signals and commands at least %u seconds...",
		    sleep_after_dump);
		(void)sleep(sleep_after_dump);
	}
	return (0);
}

/*
 * Sleep and wait for a signal or a ctl command.  Sleep time is adjusted
 * and after each sleep local time is checked for problems.
 */
static int
suspend_work(void)
{
	fd_set rs;
	struct timeval tv;
	int rv, nready, sleepsec, errno_save;
	char ch, adj_sleep;

	do {
		rv = new_time_values();
		if (rv < 0)
			goto failed;
		adj_sleep = 0;
		if (rv == 0 && difftime(curtime, newdaytime) < 0.0) {
			/* No time related problem and not a new day. */
			cursec = cursec_new;
			curdate = curdate_new;
			if (main_check_sec > cursec + wakeup_time) {
				sleepsec = wakeup_time;
				adj_sleep = 1;
			} else {
				sleepsec = main_check_sec - cursec;
				if (sleepsec < 0) {
					if (-sleepsec > (int)sensitive_time)
						logmsgx(IPA_LOG_WARNING,
						    "can't correctly schedule "
						    "time events: delta %ds "
						    "is greater than "
						    "\"sensitive_time \" %us",
						    sleepsec, sensitive_time);
					sleepsec = 0;
				}
			}
		} else {
			/* Time related problem occurred or a new day came. */
			sleepsec = 0;
		}

		if (debug_time)
			logdbg("sleeping %s%s, main_check_sec %s",
			    time_str(sleepsec), adj_sleep ? " (adjusted)" : "",
			    sec_str(main_check_sec));

		tv.tv_sec = (time_t)sleepsec;
		tv.tv_usec = 0;

		for (;;) {
			/* Unmask all signals. */
			if (Sigprocmask(SIG_SETMASK, &zeromask,
			    (sigset_t *)NULL) < 0) {
				logmsg(IPA_LOG_ERR, "suspend_work: "
				    SIGPROCMASK_NAME "(SIG_SETMASK) for "
				    "zeromask");
				return (-1);
			}
			rs = readfdset;
			nready = select(maxfd, &rs, (fd_set *)NULL,
			    (fd_set *)NULL, &tv);
			errno_save = errno;
			/* Mask expected signals. */
			if (Sigprocmask(SIG_SETMASK, &sigmask,
			    (sigset_t *)NULL) < 0) {
				logmsg(IPA_LOG_ERR, "suspend_work: "
				    SIGPROCMASK_NAME "(SIG_SETMASK) for "
				    "sigmask");
				return (-1);
			}
			if (nready < 0) {
				if (errno_save == EINTR) {
					tv.tv_sec = 0;
					tv.tv_usec = 0;
					continue;
				}
				errno = errno_save;
				logmsg(IPA_LOG_ERR, "suspend_work: select");
				return (-1);
			}
			break;
		}

		if (nready == 0) {
			/* Timeout, that is normal sleep. */
			goto done;
		}

		/* Check stdout and stderr streams. */
		if (!debug && FD_ISSET(stderr_read, &rs)) {
			--nready;
			log_stderr();
		}
		if (FD_ISSET(stdout_read, &rs)) {
			--nready;
			log_stdout();
		}
		if (nready == 0)
			goto done;

		if (signal_flag) {
			signal_flag = 0;
			while (read(sig_pipe[0], &ch, 1) > 0)
				;
			if (sigchld_flag) {
				sigchld_flag = 0;
				if (check_child() < 0)
					goto failed;
				--nready;
			}
			if (shutdown_flag) {
				logmsgx(IPA_LOG_INFO, "caught signal %d, "
				    "shutdown...", (int)shutdown_flag);
				set_rules_for_check();
				/* Ignore possible another queued signals. */
				return (0);
			}
			if (reconf_flag) {
				logmsgx(IPA_LOG_INFO, "caught signal %d, "
				    "reconfiguring...", SIGHUP);
				set_rules_for_check();
				return (0);
			}
			if (nready == 0)
				goto done;
		}

		/*
		 * Check for a connection to ctl_listenfd only
		 * if there was not poll in select (simplifying code).
		 */
		if (FD_ISSET(ctl_listenfd, &rs) && sleepsec != 0) {
			/*
			 * Since ctl command can update statistics in
			 * database, need to update curdate and cursec.
			 */
			rv = new_time_values();
			if (rv < 0)
				goto failed;
			if (rv == 1 ||
			    difftime(curtime, newdaytime) >= 0.0) {
				/*
				 * Time related problem occurred or
				 * a new day came.
				 */
				return (0);
			}
			cursec = cursec_new;
			curdate = curdate_new;
			if (ctl_query() < 0)
				goto failed;
			if (freeze_flag) {
				freeze_flag = 0;
				if (freeze_time > 0) {
					logmsgx(IPA_LOG_INFO, "freezing, "
					    "ignoring signals and commands "
					    "at least %u seconds...",
					    freeze_time);
					sleep(freeze_time);
					logmsgx(IPA_LOG_INFO, "continue "
					    "working");
				} else
					logmsgx(IPA_LOG_WARNING, "do not "
					    "freeze, since \"freeze_time\" "
					    "is not defined");
			}
			if (main_check_sec == cursec) {
				/* It's time to wakeup immediately. */
				return (0);
			}
		}
done:		;
	} while (adj_sleep);
	return (0);

failed:
	logbt("suspend_work");
	return (-1);
}

/*
 * The main function for accounting, it is called from the main()
 * and does all of the work.  Return codes are the following:
 *   0 -- normal exit;
 *   1 -- need to reconfigure;
 *  -1 -- some error occurred.
 */
int
ipa_main(void)
{
	int rv;

	if (!debug)
		log_stderr();
	log_stdout();

	logmsgx(IPA_LOG_INFO, "start working");

	if (newday(1) < 0)
		goto failed;

	for (stat_generation = 1;; stat_generation += 2) {
		rv = new_time_values();
		if (rv < 0)
			break;

		/* Call ac_get_stat() for all accounting modules. */
		if (ac_get_stat() < 0)
			break;

		/* Perform all accounting related tasks. */
		if (rv == 0) {
			/* No time related problem. */
			if (difftime(curtime, newdaytime) < 0.0) {
				if (do_ac_curr() < 0)
					break;
			} else {
				if (do_ac_newday() < 0)
					break;
			}
		} else {
			/* There is time related problem. */
			if (do_ac_prev() < 0)
				break;
		}

		/* Check signals events. */
		if (shutdown_flag)
			return (0);
		if (reconf_flag) {
			reconf_flag = 0;
			return (1);
		}

		if (dump_flag) {
			/* There was ctl command "dump". */
			dump_flag = 0;
			if (dump_finished() < 0)
				break;
		}

		/* Sleep and wait for a signal or a ctl command. */
		if (suspend_work() < 0)
			break;
	}
failed:
	logbt("ipa_main");
	return (-1);
}

/*
 * SIGINT and/or SIGTERM handler.
 */
void
sig_term(int signo)
{
	(void)write(sig_pipe[1], "", 1);
	signal_flag = 1;
	need_check_flag = 1;
	shutdown_flag = signo;
}

/*
 * SIGHUP handler, cause reconfiguring.
 */
/* ARGSUSED */
void
sig_hup(int signo ATTR_UNUSED)
{
	(void)write(sig_pipe[1], "", 1);
	signal_flag = 1;
	reconf_flag = 1;
	need_check_flag = 1;
}

/*
 * SIGCHLD handler.
 */
/* ARGSUSED */
void
sig_chld(int signo ATTR_UNUSED)
{
	(void)write(sig_pipe[1], "", 1);
	signal_flag = 1;
	sigchld_flag = 1;
}

#if defined(WITH_THRESHOLDS) || defined(WITH_SUBLIMITS)
/*
 * Return part of val.
 */
uint64_t
uint64_per_cent(const uint64_t *val, unsigned int per_cent)
{
	uint64_t res;

	if (*val > UINT64_MAX / per_cent) {
		res = *val / 100;
		res *= per_cent;
	} else {
		res = *val * per_cent;
		res /= 100;
	}
	return (res);
}
#endif /* WITH_THRESHOLDS || WITH_SUBLIMITS */
