/*-
 * Copyright (c) 2000-2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_db.c,v 1.2 2011/01/23 18:42:34 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

unsigned int	ndb_mods;		/* Number of db_mods. */

signed char	debug_db_null;		/* debug_db_null parameter. */

const struct db_list *global_db_list;	/* global { db_list } */

/*
 * List of all used "db_list" parameters.  If some rules use the same
 * "db_list" parameter, then they share the same struct db_list{}.
 */
struct db_sets db_sets;

/* List of all database modules. */
struct db_mod_list db_mod_list;

/*
 * Find a database module by configuration prefix.
 */
struct db_mod *
db_mod_by_prefix(const char *prefix)
{
	struct db_mod *db_mod;

	SLIST_FOREACH(db_mod, &db_mod_list, link)
		if (db_mod->ipa_db_mod->conf_prefix != NULL &&
		    strcmp(db_mod->ipa_db_mod->conf_prefix, prefix) == 0)
			break;
	return (db_mod);
}

/*
 * Find a database module by a name.
 */
struct db_mod *
db_mod_by_name(const char *name)
{
	struct db_mod *db_mod;

	SLIST_FOREACH(db_mod, &db_mod_list, link)
		if (strcmp(db_mod->ipa_db_mod->db_name, name) == 0)
			break;
	return (db_mod);
}

/*
 * Release memory held by db_set, including struct db_set{}.
 */
void
free_db_set(struct db_set *set)
{
	struct db_elem *db, *db_next;

	STAILQ_FOREACH_SAFE(db, &set->list, link, db_next)
		mem_free(db, m_anon);
	mem_free(set, m_anon);
}

/*
 * Release memory held by all db_set.
 */
void
free_db_lists(void)
{
	struct db_set *set, *set_next;

	SLIST_FOREACH_SAFE(set, &db_sets, link, set_next)
		free_db_set(set);
}

#ifdef WITH_LIMITS

# ifdef WITH_RULES
/*
 * Initialize one static limit.
 */
static int
db_init_statlimit(const struct rule *rule, struct limit *limit)
{
	if (!STAILQ_EMPTY(limit->db_list)) {
		const struct db_elem *db;

		STAILQ_FOREACH(db, limit->db_list, link) {
			if (db->ipa_db_mod->db_init_statlimit == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s,"
				    "limit %s: db_init_statlimit: module does "
				    "not support static limits", db->mod_file,
				    rule->name, limit->name);
				return (-1);
			}
			if (db->ipa_db_mod->db_init_statlimit(rule->no,
			    rule->name, rule->info, limit->no, limit->name,
			    limit->info) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "limit %s: db_init_statlimit failed",
				    db->mod_file, rule->name, limit->name);
				return (-1);
			}
		}
	} else if (debug_db_null)
		logdbg("rule %s, limit %s: uses \"null\" database",
		    rule->name, limit->name);
	mem_free(limit->info, m_parser);
	limit->info = NULL;
	return (0);
}

/*
 * Initialize limits for one static rule in databases.
 */
static int
db_init_statlimits(const struct rule *rule)
{
	struct limit *limit;

	STAILQ_FOREACH(limit, &rule->limits, link)
		if (db_init_statlimit(rule, limit) < 0) {
			logbt("db_init_statlimits");
			return (-1);
		}
	return (0);
}
# endif /* WITH_RULES */

# ifdef WITH_AUTORULES
/*
 * Initialize one dynamic limit.
 */
static int
db_init_dynlimit(const struct autorule *autorule, const struct rule *rule,
   struct limit *limit)
{
	if (!STAILQ_EMPTY(limit->db_list)) {
		const struct db_elem *db;

		STAILQ_FOREACH(db, limit->db_list, link) {
			if (db->ipa_db_mod->db_init_dynlimit == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: autorule %s, "
				    "rule %s, limit %s: db_init_dynlimit: "
				    "module does not support static limits",
				    db->mod_file, autorule->name, rule->name,
				    limit->name);
				return (-1);
			}
			if (db->ipa_db_mod->db_init_dynlimit(autorule->no,
			    rule->no, rule->name, rule->info, limit->no,
			    limit->name, limit->info) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: autorule %s, "
				    "rule %s, limit %s: db_init_dynlimit "
				    "failed", db->mod_file, autorule->name,
				    rule->name, limit->name);
				return (-1);
			}
		}
	} else if (debug_db_null)
		logdbg("rule %s, limit %s: uses \"null\" database",
		    rule->name, limit->name);
	return (0);
}

/*
 * Initialize limits for one dynamic rule in databases.
 */
static int
db_init_dynlimits(const struct autorule *autorule, const struct rule *rule)
{
	struct limit *limit;

	STAILQ_FOREACH(limit, &rule->limits, link)
		if (db_init_dynlimit(autorule, rule, limit) < 0) {
			logbt("db_init_dynlimits");
			return (-1);
		}
	return (0);
}

# endif /* WITH_AUTORULES */

/*
 * Deinitialize limit in one rule in databases.
 */
int
db_deinit_limit(const struct rule *rule, const struct limit *limit)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, limit->db_list, link)
		if (db->ipa_db_mod->db_deinit_limit != NULL &&
		    db->ipa_db_mod->db_deinit_limit(rule->no, limit->no) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s: "
			    "db_deinit_limit failed", db->mod_file, rule->name,
			    limit->name);
			rv = -1;
		}
	return (rv);
}

/*
 * Update statistics for one limit.
 */
int
db_update_limit(const struct rule *rule, const struct limit *limit)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, limit->db_list, link)
		if (db->ipa_db_mod->db_update_limit(rule->no,
		    limit->no, &limit->cnt, &curdate) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s: "
			    "db_update_limit failed", db->mod_file, rule->name,
			    limit->name);
			rv = -1;
		}
	return (rv);
}

/*
 * Register limit event for one limit.
 */
int
db_limit_event(const struct rule *rule, struct limit *limit,
    unsigned int event, const ipa_tm *tm)
{
	const struct db_elem *db;
	int rv;

	if (ctl_enable) {
		limit->event_date_set |= 1 << event;
		limit->event_date[event] = *tm;
	}

	rv = 0;
	STAILQ_FOREACH(db, limit->db_list, link)
		if (db->ipa_db_mod->db_limit_event(rule->no, limit->no, event,
		    tm, &curdate) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s: "
			    "db_limit_event(EVENT_%s) failed", db->mod_file,
			    rule->name, limit->name, limit_event_msg[event]);
			rv = -1;
		}
	return (rv);
}

/*
 * Get a state of the limit: use first database which
 * supports db_get_limit_state method and save its name in
 * db_name.  If any of databases used by the limit does not
 * support db_get_limit_state method, then set db_name to NULL.
 */
int
db_get_limit_state(const struct rule *rule, const struct limit *limit,
    struct ipa_limit_state *state, const char **db_name)
{
	const struct db_elem *db;
	int rv;

	STAILQ_FOREACH(db, limit->db_list, link)
		if (db->ipa_db_mod->db_get_limit_state != NULL) {
			rv = db->ipa_db_mod->db_get_limit_state(rule->no,
			    limit->no, state);
			if (rv >= 0) {
				*db_name = db->ipa_db_mod->db_name;
				return (rv);
			}
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s: "
			    "db_get_limit_state failed", db->mod_file,
			    rule->name, limit->name);
			return (-1);
		}
	*db_name = NULL;
	return (0);
}

/*
 * Set a state of the limit in databases.
 */
int
db_set_limit_state(const struct rule *rule, struct limit *limit,
    struct ipa_limit_state *state, int new_state)
{
	const struct db_elem *db;
	int rv;

	if (ctl_enable) {
		state->event_date_set |= IPA_LIMIT_EVENT_UPDATED_SET;
		state->event_date[IPA_LIMIT_EVENT_UPDATED] = curdate;
	}

	limit->event_date_set = state->event_date_set;
	memcpy(limit->event_date, state->event_date,
	    sizeof(limit->event_date));

	rv = 0;
	STAILQ_FOREACH(db, limit->db_list, link)
		if (db->ipa_db_mod->db_set_limit_state(rule->no,
		    limit->no, state, new_state) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, limit %s"
			    "db_set_limit_state(%d) failed", db->mod_file,
			    rule->name, limit->name, new_state);
			rv = -1;
		}
	return (rv);
}

/*
 * Set a limit active/inactive in databases it uses.
 */
int
db_set_limit_active(const struct rule *rule, const struct limit *limit,
    int active)
{
	const struct db_elem *db;

	/* Limit uses own db_list. */
	STAILQ_FOREACH(db, limit->db_list, link)
		if (db->ipa_db_mod->db_set_limit_active != NULL)
			if (db->ipa_db_mod->db_set_limit_active(rule->no,
			    limit->no, active) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "limit %s: db_set_limit_active(%d) failed",
				    db->mod_file, rule->name, limit->name,
				    active);
				return (-1);
			}
	return (0);
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS

# ifdef WITH_RULES
/*
 * Initialize one static threshold.
 */
static int
db_init_statthreshold(const struct rule *rule, struct threshold *threshold)
{
	if (!STAILQ_EMPTY(threshold->db_list)) {
		const struct db_elem *db;

		STAILQ_FOREACH(db, threshold->db_list, link) {
			if (db->ipa_db_mod->db_init_statthreshold == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "threshold %s: db_init_statthreshold: "
				    "module does not support static thresholds",
				    db->mod_file, rule->name, threshold->name);
				return (-1);

			}
			if (db->ipa_db_mod->db_init_statthreshold(rule->no,
			    rule->name, rule->info, threshold->no,
			    threshold->name, threshold->info) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
				    "threshold %s: db_init_statthreshold "
				    "failed", db->mod_file, rule->name,
				    threshold->name);
				return (-1);
			}
		}
	} else if (debug_db_null)
		logdbg("rule %s, threshold %s: uses \"null\" database",
		    rule->name, threshold->name);
	mem_free(threshold->info, m_parser);
	threshold->info = NULL;
	return (0);
}

/*
 * Initialize thresholds for one static rule in databases.
 */
static int
db_init_statthresholds(const struct rule *rule)
{
	struct threshold *threshold;

	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (db_init_statthreshold(rule, threshold) < 0) {
			logbt("db_init_statthresholds");
			return (-1);
		}
	return (0);
}
# endif /* WITH_RULES */

# ifdef WITH_AUTORULES
/*
 * Initialize one dynamic threshold.
 */
static int
db_init_dynthreshold(const struct autorule *autorule, const struct rule *rule,
    struct threshold *threshold)
{
	if (!STAILQ_EMPTY(threshold->db_list)) {
		const struct db_elem *db;

		STAILQ_FOREACH(db, threshold->db_list, link) {
			if (db->ipa_db_mod->db_init_dynthreshold == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: autorule %s, "
				    "rule %s, threshold %s: "
				    "db_init_dynthreshold: module does not "
				    "support dynamic thresholds",
				    db->mod_file, autorule->name, rule->name,
				    threshold->name);
				return (-1);
			}
			if (db->ipa_db_mod->db_init_dynthreshold(autorule->no,
			    rule->no, rule->name, rule->info, threshold->no,
			    threshold->name, threshold->info) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: autorule %s, "
				    "rule %s, threshold %s: "
				    "db_init_dynthreshold failed",
				    db->mod_file, autorule->name, rule->name,
				    threshold->name);
				return (-1);
			}
		}
	} else if (debug_db_null)
		logdbg("rule %s, threshold %s: uses \"null\" database",
		    rule->name, threshold->name);
	return (0);
}

/*
 * Initialize thresholds for one dynamic rule in databases.
 */
static int
db_init_dynthresholds(const struct autorule *autorule, const struct rule *rule)
{
	struct threshold *threshold;

	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (db_init_dynthreshold(autorule, rule, threshold) < 0) {
			logbt("db_init_dynthresholds");
			return (-1);
		}
	return (0);
}
# endif /* WITH_AUTORULES */

/*
 * Deinitialize threshold in one rule in databases.
 */
int
db_deinit_threshold(const struct rule *rule, const struct threshold *threshold)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, threshold->db_list, link)
		if (db->ipa_db_mod->db_deinit_threshold != NULL &&
		    db->ipa_db_mod->db_deinit_threshold(rule->no,
		    threshold->no) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: db_deinit_threshold failed",
			    db->mod_file, rule->name, threshold->name);
			rv = -1;
		}
	return (rv);
}

/*
 * Update statistics for one threshold in databases.
 */
int
db_update_threshold(const struct rule *rule, const struct threshold *threshold)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, threshold->db_list, link)
		if (db->ipa_db_mod->db_update_threshold(rule->no,
		    threshold->no, &threshold->cnt, &threshold->tm_started,
		    &threshold->tm_updated) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: db_update_threshold failed",
			    db->mod_file, rule->name, threshold->name);
			rv = -1;
		}
	return (rv);
}

/*
 * Get a state of the threshold: use first database which
 * supports db_get_threshold_state method and save its name in
 * db_name.  If any of databases used by the threshold does not
 * support db_get_threshold_state method, then set db_name to NULL.
 */
int
db_get_threshold_state(const struct rule *rule,
    const struct threshold *threshold, struct ipa_threshold_state *state,
    const char **db_name)
{
	const struct db_elem *db;
	int rv;

	STAILQ_FOREACH(db, threshold->db_list, link)
		if (db->ipa_db_mod->db_get_threshold_state != NULL) {
			rv = db->ipa_db_mod->db_get_threshold_state(rule->no,
			    threshold->no, state);
			if (rv >= 0) {
				*db_name = db->ipa_db_mod->db_name;
				return (rv);
			}
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: db_get_threshold_state failed",
			    db->mod_file, rule->name, threshold->name);
			return (-1);
		}
	*db_name = NULL;
	return (0);
}

/*
 * Set a state of the threshold in databases.
 */
int
db_set_threshold_state(const struct rule *rule,
    const struct threshold *threshold, const struct ipa_threshold_state *state)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, threshold->db_list, link)
		if (db->ipa_db_mod->db_set_threshold_state(rule->no,
		    threshold->no, state) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: db_set_threshold_state failed",
			    db->mod_file, rule->name, threshold->name);
			rv = -1;
		}
	return (rv);
}

/*
 * Set a threshold active/inactive in databases it uses.
 */
int
db_set_threshold_active(const struct rule *rule,
    const struct threshold *threshold, int active)
{
	const struct db_elem *db;

	/* Threshold uses own db_list. */
	STAILQ_FOREACH(db, threshold->db_list, link)
		if (db->ipa_db_mod->db_set_threshold_active != NULL &&
		    db->ipa_db_mod->db_set_threshold_active(rule->no,
		    threshold->no, active) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s, "
			    "threshold %s: db_set_threshold_active(%d) failed",
			    db->mod_file, rule->name, threshold->name, active);
			return (-1);
		}
	return (0);
}
#endif /* WITH_THRESHOLDS */

#ifdef WITH_RULES
/*
 * Initialize one static rule in databases, also initialize its
 * limits and thresholds.
 */
int
db_init_statrule(struct rule *rule)
{
	if (!STAILQ_EMPTY(rule->db_list)) {
		const struct db_elem *db;

		STAILQ_FOREACH(db, rule->db_list, link) {
			if (db->ipa_db_mod->db_init_statrule == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s:"
				    "db_init_statrule: module does not support "
				    "static rules", db->mod_file, rule->name);
				return (-1);
			}
			if (db->ipa_db_mod->db_init_statrule(rule->no,
			    rule->name, rule->info) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
				    "db_init_statrule failed", db->mod_file,
				    rule->name);
				return (-1);
			}
		}
	} else if (debug_db_null)
		logdbg("rule %s: uses \"null\" database", rule->name);

	mem_free(rule->info, m_parser);
	rule->info = NULL;

#ifdef WITH_LIMITS
	if (db_init_statlimits(rule) < 0) {
		logbt("db_init_statrule");
		return (-1);
	}
#endif
#ifdef WITH_THRESHOLDS
	if (db_init_statthresholds(rule) < 0) {
		logbt("db_init_statrule");
		return (-1);
	}
#endif
	return (0);
}
#endif /* WITH_RULES */

/*
 * Pre-initialize all db_mods.
 */
int
pre_init_db_mods(void)
{
	const struct db_mod *db_mod;

	SLIST_FOREACH(db_mod, &db_mod_list, link)
		if (db_mod->ipa_db_mod->db_pre_init() < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: db_pre_init failed",
			    db_mod->mod_file);
			return (-1);
		}
	return (0);
}

/*
 * Initialize all db_mods.
 */
int
init_db_mods(void)
{
	const struct db_mod *db_mod;

	SLIST_FOREACH(db_mod, &db_mod_list, link)
		if (db_mod->ipa_db_mod->db_init() < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: db_init failed",
			    db_mod->mod_file);
			return (-1);
		}
	return (0);
}

/*
 * Deinitialize all db_mods.
 */
int
deinit_db_mods(void)
{
	const struct db_mod *db_mod;
	int rv;

	rv = 0;
	SLIST_FOREACH(db_mod, &db_mod_list, link)
		if (db_mod->ipa_db_mod->db_deinit() < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: deinit_db_mods: "
			    "db_deinit failed", db_mod->mod_file);
			rv = -1;
		}
	return (rv);
}

/*
 * Deinitialize one rule in databases.
 */
int
db_deinit_rule(const struct rule *rule)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, rule->db_list, link)
		if (db->ipa_db_mod->db_deinit_rule != NULL &&
		    db->ipa_db_mod->db_deinit_rule(rule->no) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "db_deinit_rule failed", db->mod_file, rule->name);
			rv = -1;
		}
	return (rv);
}

/*
 * Update statistics for one rule.
 */
int
db_update_rule(const struct rule *rule, const uint64_t *cnt)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, rule->db_list, link)
		if (db->ipa_db_mod->db_update_rule(rule->no, cnt,
		    &curdate) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "db_update_rule failed", db->mod_file, rule->name);
			rv = -1;
		}
	return (rv);
}

/*
 * Append an accounting record for one rule.
 */
int
db_append_rule(struct rule *rule, const uint64_t *cnt, int flush_cnt)
{
	const struct db_elem *db;
	int rv;

	rv = 0;
	STAILQ_FOREACH(db, rule->db_list, link)
		if (db->ipa_db_mod->db_append_rule(rule->no, cnt,
		    &curdate) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "db_append_rule failed", db->mod_file, rule->name);
			rv = -1;
		}

	if (rule->append_tevent != NULL)
		rule->append_sec = rule->append_tevent->event_sec;

	if (flush_cnt)
		rule->cnt = 0;

	return (rv);
}

/*
 * Set a rule active/inactive in databases it uses.
 */
int
db_set_rule_active(const struct rule *rule, int active)
{
	const struct db_elem *db;

	STAILQ_FOREACH(db, rule->db_list, link)
		if (db->ipa_db_mod->db_set_rule_active != NULL &&
		    db->ipa_db_mod->db_set_rule_active(rule->no,
		    active) < 0) {
			logmsgx(IPA_LOG_ERR, "module %s: rule %s: "
			    "db_set_rule_active(%d) failed", db->mod_file,
			    rule->name, active);
			return (-1);
		}
	return (0);
}

#ifdef WITH_AUTORULES
/*
 * Initialize one dynamic rule in databases, also initialize its
 * limits and thresholds.
 */
int
db_init_dynrule(const struct autorule *autorule, struct rule *rule)
{
	if (!STAILQ_EMPTY(rule->db_list)) {
		const struct db_elem *db;

		STAILQ_FOREACH(db, rule->db_list, link) {
			if (db->ipa_db_mod->db_init_dynrule == NULL) {
				logmsgx(IPA_LOG_ERR, "module %s: autorule %s, "
				    "rule %s: db_init_dynrule: module does not "
				    "support dynamic rules", db->mod_file,
				    autorule->name, rule->name);
				return (-1);
			}
			if (db->ipa_db_mod->db_init_dynrule(autorule->no,
			    rule->no, rule->name, rule->info) < 0) {
				logmsgx(IPA_LOG_ERR, "module %s: autorule %s, "
				    "rule %s: db_init_dynrule failed",
				    db->mod_file, autorule->name, rule->name);
				return (-1);
			}
		}
	} else if (debug_db_null)
		logdbg("rule %s: uses \"null\" database", rule->name);

	mem_free(rule->info, m_anon);
	rule->info = NULL;

#ifdef WITH_LIMITS
	if (db_init_dynlimits(autorule, rule) < 0) {
		logbt("db_init_dynrule");
		return (-1);
	}
#endif
#ifdef WITH_THRESHOLDS
	if (db_init_dynthresholds(autorule, rule) < 0) {
		logbt("db_init_dynrule");
		return (-1);
	}
#endif

	return (0);
}
#endif /* WITH_AUTORULES */
