/*-
 * Copyright (c) 2003-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_autorules.c,v 1.3.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_time.h"

#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

unsigned int	nautorules;		/* Number of autorules. */
unsigned int	ndynrules;		/* Number of dynamic rules. */

#ifdef WITH_AUTORULES

signed char	debug_autorule;		/* debug_autorule parameter. */

struct autorule	*autorules;		/* Array of autorules. */
ipa_marray	*autorules_marray;	/* Marray of autorules. */

struct autorules_list autorules_list;	/* List of all autorules. */

#define AUTORULE(x) &autorules[(x)]	/* Pointer to autorule by number. */

static struct rule **rules_ptr;		/* Array of pointers to rules. */
ipa_marray	*rules_ptr_marray;	/* Marray for array of ptrs to rules. */

#define RULE(x) rules_ptr[(x)]		/* Pointer to rule by number. */

/* Time when to check active autorules. */
unsigned int	autorules_active_check_sec;

/* Time when to check inactive autorules. */
unsigned int	autorules_inactive_check_sec;

/* Active/inactive autorules queue. */
TAILQ_HEAD(autorules_queue, autorule);
static struct autorules_queue autorules_active;
static struct autorules_queue autorules_inactive;

/*
 * Return pointer to autorule with the given name.
 */
struct autorule *
autorule_by_name(const char *name)
{
	struct autorule *autorule;
	unsigned int i;

	for (i = 0, autorule = autorules; i < nautorules; ++autorule, ++i)
		if (strcmp(name, autorule->name) == 0)
			return (autorule);
	return (NULL);
}

/*
 * Set autorule active or inactive in modules that it uses.
 */
static int
mod_set_autorule_active(struct autorule *autorule, int active)
{
	if ((autorule->arule_flags & AUTORULE_FLAG_ACTIVE) ==
	    (active ? AUTORULE_FLAG_ACTIVE : 0)) {
		logmsgx(IPA_LOG_ERR, "internal error: mod_set_autorule_active"
		    "(%s, %d): autorule is already %s", autorule->name,
		    active, active_msg[active]);
		return (-1);
	}

	if (debug_worktime)
		logdbg("autorule %s: set autorule %s",
		    autorule->name, active_msg[active]);

	if (active)
		AUTORULE_SET_ACTIVE(autorule);
	else
		AUTORULE_SET_INACTIVE(autorule);

	if (ac_set_autorule_active(autorule, active) < 0) {
		logbt("mod_set_autorule_active");
		return (-1);
	}

	return (0);
}

/*
 * Log information about inactive autorules.
 */
static void
show_inactive_autorules(void)
{
	const struct autorule *autorule;

	logdbg("inactive autorules list:");
	TAILQ_FOREACH(autorule, &autorules_inactive, queue)
		logdbg("    active_sec %s, autorule %s",
		    sec_str(autorule->worktime->active_sec), autorule->name);
}

/*
 * Move am autorule from the active autorules queue to the inactive
 * autorules queue.
 */
static int
set_autorule_inactive(struct autorule *autorule1)
{
	struct autorule *autorule2;

	/* Inform modules that autorule becomes inactive. */
	if (mod_set_autorule_active(autorule1, 0) < 0) {
		logbt("set_autorule_inactive");
		return (-1);
	}

	/* Remove the autorule from the active autorules queue. */
	TAILQ_REMOVE(&autorules_active, autorule1, queue);

	/*
	 * Add the autorule to the inactive autorules queue,
	 * keep that queue sorted.
	 */
	TAILQ_FOREACH(autorule2, &autorules_inactive, queue)
		if (autorule1->worktime->active_sec <
		    autorule2->worktime->active_sec) {
			TAILQ_INSERT_BEFORE(autorule2, autorule1, queue);
			goto done;
		}
	TAILQ_INSERT_TAIL(&autorules_inactive, autorule1, queue);
done:
	if (debug_worktime)
		show_inactive_autorules();

	autorules_inactive_check_sec =
	    TAILQ_FIRST(&autorules_inactive)->worktime->active_sec;

	return (0);
}

/*
 * Move an autorule from the inactive autorules queue to the
 * active autorules queue.
 */
static int
set_autorule_active(struct autorule *autorule)
{
	/* Inform modules that autorule becomes active. */
	if (mod_set_autorule_active(autorule, 1) < 0) {
		logbt("set_autorule_active");
		return (-1);
	}

	/* Remove the autorule from the inactive autorules queue. */
	TAILQ_REMOVE(&autorules_inactive, autorule, queue);

	/* Add the rule to the active autorules queue. */
	TAILQ_INSERT_TAIL(&autorules_active, autorule, queue);

	/* Get new value of autorules_inactive_check_sec. */
	autorules_inactive_check_sec = TAILQ_EMPTY(&autorules_inactive) ?
	    EVENT_NOT_SCHEDULED :
	    TAILQ_FIRST(&autorules_inactive)->worktime->active_sec;

	/* Check if rules_active_check_sec should be modified. */
	if (autorules_active_check_sec > autorule->update_tevent->event_sec)
		autorules_active_check_sec = autorule->update_tevent->event_sec;
	if (autorules_active_check_sec > autorule->worktime->inactive_sec)
		autorules_active_check_sec = autorule->worktime->inactive_sec;

	return (0);
}

/*
 * Sort inactive autorules.
 */
static void
sort_inactive_autorules(void)
{
	struct autorule *a1, *a1_next, *a2;

	if (TAILQ_EMPTY(&autorules_inactive))
		autorules_inactive_check_sec = EVENT_NOT_SCHEDULED;
	else {
		a1 = TAILQ_FIRST(&autorules_inactive);
		TAILQ_INIT(&autorules_inactive);
		for (; a1 != NULL; a1 = a1_next) {
			a1_next = TAILQ_NEXT(a1, queue);
			if (a1->worktime->active_sec != EVENT_NOT_SCHEDULED) {
				/* Will be active in current day. */
				TAILQ_FOREACH(a2, &autorules_inactive, queue)
					if (a1->worktime->active_sec <=
					    a2->worktime->active_sec) {
						TAILQ_INSERT_BEFORE(a2, a1,
						    queue);
						goto next;
					}
			}
			TAILQ_INSERT_TAIL(&autorules_inactive, a1, queue);
next:			;
		}
		autorules_inactive_check_sec =
		    TAILQ_FIRST(&autorules_inactive)->worktime->active_sec;
	}

	if (debug_worktime)
		show_inactive_autorules();
}

/*
 * Make initialization for all autorules.
 */
int
init_autorules(void)
{
	struct autorule *autorule;
	struct rule *rule;

	/* Initialize autorules queues. */
	TAILQ_INIT(&autorules_inactive);
	TAILQ_INIT(&autorules_active);

	/* Flush number of dynamic rules. */
	ndynrules = 0;

	if (nautorules == 0)
		return (0);

	/* Initialize each autorule in accounting modules. */
	STAILQ_FOREACH(autorule, &autorules_list, link) {
		if (ac_init_autorule(autorule) < 0) {
			logbt("init_autorules");
			return (-1);
		}
		TAILQ_INSERT_TAIL(&autorules_active, autorule, queue);
	}

	/* Create rules_ptr_marray. */
	rules_ptr_marray = marray_init(MARRAY_NAME(rules_ptr),
	    "Pointers to rules", 0, (void *)&rules_ptr, sizeof(struct rule *),
	    nstatrules + 1, RULE_NALLOC);
	if (rules_ptr_marray == NULL) {
		logmsgx(IPA_LOG_ERR, "init_autorules: marray_init failed");
		return (-1);
	}

	/* Mark entries in rules_ptr for static rules as used. */
	TAILQ_FOREACH(rule, &rules_list, list)
		if (marray_alloc(rules_ptr_marray, &rule->no, 1) < 0) {
			logmsgx(IPA_LOG_ERR, "init_autorules: "
			    "marray_alloc failed");
			return (-1);
		}

	return (0);
}

/*
 * Deinitialize one autorule.
 */
static int
deinit_autorule(struct autorule *autorule)
{
	int rv;

	/* Deinitialize in modules. */
	rv = 0;
	if (ac_deinit_autorule(autorule) < 0) {
		logbt("deinit_autorule");
		rv = -1;
	}

	/* Decrease ref_count for accounting systems. */
	if (AUTORULE_IS_ACTIVE(autorule))
		if (ac_dec_ref_count(autorule->ac_list) < 0) {
			logmsgx(IPA_LOG_ERR, "autorule %s: deinit_autorule: "
			    "ac_dec_ref_count failed", autorule->name);
			rv = -1;
		}

	/* And free all memory used by a autorule. */
	mem_free(autorule->name, m_anon);
	cmds_rule_free(&autorule->rc[RC_STARTUP]);
	cmds_rule_free(&autorule->rc[RC_SHUTDOWN]);
#ifdef WITH_LIMITS
	free_limits(RULE_FLAG_FREE_LIMITS, &autorule->limits, 0);
#endif
#ifdef WITH_THRESHOLDS
	free_thresholds(RULE_FLAG_FREE_THRESHOLDS, &autorule->thresholds, 0);
#endif

	return (rv);
}

/*
 * Make deinitialization for all autorules.
 */
int
deinit_autorules(void)
{
	struct autorule *autorule;
	int rv;

	if (nautorules == 0)
		return (0);

	rv = 0;
	STAILQ_FOREACH(autorule, &autorules_list, link)
		if (deinit_autorule(autorule) < 0) {
			logbt("deinit_autorules");
			rv = -1;
		}

	marray_deinit(autorules_marray);

	if (ndynrules != 0) {
		logmsgx(IPA_LOG_ERR, "internal error: deinit_autorules: "
		    "ndynrules is not zero: %u", ndynrules);
		rv = -1;
	}

	if (rules_ptr_marray != NULL) {
		unsigned int n;

		n = marray_nused(rules_ptr_marray);
		if (n != 0) {
			logmsgx(IPA_LOG_ERR, "internal error: "
			    "deinit_autorules: rules_ptr_marray is "
			    "not empty: %u", n);
			rv = -1;
		}
		marray_deinit(rules_ptr_marray);
	}

	return (rv);
}

/*
 * Inherit settings for dynamic rule from its autorule, then inherit
 * everythinh as for static rules.
 */
static int
dyn_rule_inherit(const struct autorule *autorule, struct rule *rule)
{
	unsigned int x;

	/* Dynamic rule inherits non-NULL update_tevent from its autorule. */
	rule->update_tevent = autorule->update_tevent;

	rule->append_tevent = autorule->append_tevent;
	rule->worktime = autorule->worktime_rule;

	/* Dynamic rule inherits non-NULL ac_list from autorule. */
	rule->ac_list = autorule->ac_list;
	ac_inc_ref_count(rule->ac_list);

	rule->db_list = autorule->db_list;

#ifdef WITH_RULES
	/* Dynamic rules cannot have ac_gather_* parameters. */
	rule->acg_add_pat = rule->acg_sub_pat = NULL;
	SLIST_INIT(&rule->acgs);
#endif

	rule->debug_exec = autorule->debug_exec;
	rule_init_cmds(rule);
#ifdef WITH_LIMITS
	rule->debug_limit = autorule->debug_limit;
	rule->debug_limit_init = autorule->debug_limit_init;
#endif
#ifdef WITH_THRESHOLDS
	rule->debug_threshold = autorule->debug_threshold;
	rule->debug_threshold_init = autorule->debug_threshold_init;
#endif

#ifdef CTL_CHECK_CREDS
	rule->ctl_rule_acl = autorule->ctl_rule_acl;
#endif

	/*
	 * At this point everything is ready in a rule for possible
	 * deinit_dyn_rule() invocation.
	 */

	for (x = 0; x < 2; ++x) {
		if (!autorule->rc[x].cmds.sect_set)
			continue;
		if (cmds_rule_copy(rule, &rule->rc[x], &autorule->rc[x]) < 0) {
			logmsgx(IPA_LOG_ERR, "rule %s: dyn_rule_inherit: "
			    "cannot copy all commands from autorule %s "
			    "{ %s {}}", rule->name, autorule->name,
			    rc_sect_name[x]);
			return (-1);
		}
	}

#ifdef WITH_LIMITS
	if (!STAILQ_EMPTY(&autorule->limits))
		if (copy_limits(rule, &autorule->limits) < 0) {
			logmsgx(IPA_LOG_ERR, "rule %s: dyn_rule_inherit: "
			    "cannot copy all limits from autorule %s",
			    rule->name, autorule->name);
			return (-1);
		}
#endif
#ifdef WITH_THRESHOLDS
	if (!STAILQ_EMPTY(&autorule->thresholds))
		if (copy_thresholds(rule, &autorule->thresholds) < 0) {
			logmsgx(IPA_LOG_ERR, "rule %s: dyn_rule_inherit: "
			    "cannot copy all thresholds from autorule %s",
			    rule->name, autorule->name);
			return (-1);
		}
#endif

	/* Inherit settings as for static rules. */
	if (rule_inherit(rule) < 0) {
		logbt("dyn_rule_inherit");
		return (-1);
	}

	return (0);
}

/*
 * Deinitialize one dynamic rule.
 */
int
deinit_dyn_rule(struct rule *rule)
{
	marray_free(rules_ptr_marray, rule->no);
	--ndynrules;
	if (deinit_rule(rule) < 0) {
		logbt("deinit_dyn_rule");
		return (-1);
	}
	return (0);
}

/*
 * Initialize one dynamic rule.
 */
static int
init_dyn_rule(const struct autorule *autorule, struct rule *rule)
{
	rule->newstat = 0;
	if (rule->append_tevent == NULL)
		rule->append_sec = EVENT_NOT_SCHEDULED;
	if (ac_init_dynrule(autorule, rule) < 0)
		goto failed;
	if (db_init_dynrule(autorule, rule) < 0)
		goto failed;
	return (0);

failed:
	logbt("init_dyn_rule");
	return (-1);
}

/*
 * Create a dynamic rule from the autorule, this function is invoked
 * from the ipa_ac_mod's ac_get_stat() function.
 */
int
create_rule(const char *mod_name, unsigned int autoruleno,
    const char *rule_name, const char *rule_info)
{
	struct rule *rule;
	struct autorule *autorule;
#ifdef WITH_LIMITS
	struct limit *limit;
#endif
#ifdef WITH_THRESHOLDS
	struct threshold *threshold;
#endif
	unsigned int ruleno;

	/* Validate autorule number. */
	if (autoruleno >= nautorules) {
		logmsgx(IPA_LOG_ERR, "create_rule: module %s tries to create "
		    "dynamic rule from autorule with number %u and such "
		    "autorule does not exist", mod_name, autoruleno);
		return (-1);
	}
	autorule = AUTORULE(autoruleno);

	/* Check if there is already rule with the given name. */
	if (validate_name(rule_name) < 0) {
		logmsgx(IPA_LOG_ERR, "autorule %s: create_rule: module %s "
		    "specified illegal rule name \"%s\"", autorule->name,
		    mod_name, rule_name);
		return (-1);
	}
	rule = rule_by_name(rule_name);
	if (rule != NULL) {
		logmsgx(IPA_LOG_WARNING, "autorule %s: create_rule: module %s "
		    "is trying to create duplicated rule %s",
		    autorule->name, mod_name, rule_name);
		return (-2);
	}

	/* Find first unused rule number. */
	if (marray_alloc(rules_ptr_marray, &ruleno, 0) < 0) {
		logmsgx(IPA_LOG_ERR, "rule %s: create_rule: "
		    "marray_alloc failed", rule_name);
		return (-1);
	}

	/* Allocate memory for a rule. */
	rule = mzone_alloc(rule_mzone);
	if (rule == NULL) {
		logmsgx(IPA_LOG_ERR, "rule %s: create_rule: "
		    "mzone_alloc failed", rule_name);
		marray_free(rules_ptr_marray, ruleno);
		return (-1);
	}

	rule->name = mem_strdup(rule_name, m_anon);
	if (rule->name == NULL) {
		logmsgx(IPA_LOG_ERR, "rule %s: create_rule: "
		    "mem_strdup for rule name failed", rule_name);
		mzone_free(rule_mzone, rule);
		marray_free(rules_ptr_marray, ruleno);
		return (-1);
	}

	RULE(ruleno) = rule;
	rules_hash_add(rule);
	TAILQ_INSERT_TAIL(&rules_list, rule, list);

	rule->no = ruleno;
	rule->cnt_neg = 0;
	++ndynrules;
#ifdef WITH_RULES
	rule->acg_add_pat = rule->acg_sub_pat = NULL;
	SLIST_INIT(&rule->acgs);
#endif
#ifdef WITH_LIMITS
	STAILQ_INIT(&rule->limits);
#endif
#ifdef WITH_THRESHOLDS
	STAILQ_INIT(&rule->thresholds);
#endif
	rule->info = NULL;

	rule->rule_flags = RULE_FLAG_ACTIVE | RULE_FLAG_QUEUED |
	    RULE_FLAG_DYNAMIC;
	queue_active_rule(rule);

	if (dyn_rule_inherit(autorule, rule) < 0) {
		logmsgx(IPA_LOG_ERR, "rule %s: create_rule: cannot inherit "
		    "all settings for dynamic rule", rule_name);
		goto failed;
	}

#ifdef WITH_RULES
	if (has_ac_gather)
		if (init_acg(rule) < 0)
			goto failed;
#endif

	if (rule_info != NULL) {
		rule->info = mem_strdup(rule_info, m_anon);
		if (rule->info == NULL) {
			logmsgx(IPA_LOG_ERR, "rule %s: create_rule: "
			    "mem_strdup for rule_info failed", rule_name);
			goto failed;
		}
	}

	if (init_dyn_rule(autorule, rule) < 0)
		goto failed;

#ifdef WITH_LIMITS
	if (init_limits(rule) < 0)
		goto failed;
#endif
#ifdef WITH_THRESHOLDS
	if (init_thresholds(rule) < 0)
		goto failed;
#endif

	if (run_cmds_rule(rule, RC_STARTUP) < 0)
		goto failed;

	/*
	 * By default any new rule is active as its limits and thresholds are.
	 * First ac_get_rule_stat() for a new dynamic rule returns statistics.
	 */

#ifdef WITH_LIMITS
	STAILQ_FOREACH(limit, &rule->limits, link)
		if (WT_IS_INACTIVE(limit->worktime))
			if (set_limit_inactive(rule, limit) < 0)
				goto failed;
#endif

#ifdef WITH_THRESHOLDS
	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (WT_IS_INACTIVE(threshold->worktime))
			if (set_threshold_inactive(rule, threshold) < 0)
				goto failed;
#endif

	if (WT_IS_INACTIVE(rule->worktime)) {
		if (set_rule_inactive(rule) < 0)
			goto failed;
	} else {
		/* If a rule is active, then check it immediately. */
		rule->check_sec = 0;
		if (db_append_rule(rule, &uint64_zero, 1) < 0)
			goto failed;
	}

	autorule->nrules++;
	rule->autoruleno = autoruleno;

	if (debug_autorule)
		logdbg("create_rule: module %s created dynamic rule %s "
		    "number %u from autorule %s", mod_name, rule_name,
		    ruleno, autorule->name);
	return (0);

failed:
	(void)deinit_dyn_rule(rule);
	free_rule(rule);
	logbt("create_rule");
	return (-1);
}

/*
 * Delete a dynamic rule previously created from an autorule,
 * this function is invoked from the ipa_ac_mod.
 */
int
delete_rule(const char *mod_name, unsigned int ruleno)
{
	struct autorule *autorule;
	struct rule *rule;
	int rv;

	/* Validate rule number. */
	if (ruleno < nstatrules) {
		logmsgx(IPA_LOG_ERR, "delete_rule: module %s is trying to "
		    "delete rule with number %u and this rule is not dynamic",
		    mod_name, ruleno);
		return (-1);
	}
	if (!marray_check_index(rules_ptr_marray, ruleno)) {
		logmsgx(IPA_LOG_ERR, "delete_rule: module %s is trying to "
		    "delete rule with number %u and this rule does not exist",
		    mod_name, ruleno);
		return (-1);
	}

	/* Rule exists and is dynamic, delete it. */
	rule = RULE(ruleno);
	rv = 0;
	if (run_cmds_rule(rule, RC_SHUTDOWN) < 0)
		rv = -1;
	if (deinit_dyn_rule(rule) < 0)
		rv = -1;
	if (rv < 0)
		logbt("delete_rule");

	autorule = AUTORULE(rule->autoruleno);
	autorule->nrules--;

	if (debug_autorule)
		logdbg("delete_rule: module %s deleted rule %s number %u, "
		    "created from autorule %s", mod_name, rule->name,
		    ruleno, autorule->name);

	free_rule(rule);
	return (rv);
}

/*
 * This function is called from newday().
 */
int
autorules_newday(void)
{
	struct autorule *autorule;

	autorules_active_check_sec = EVENT_NOT_SCHEDULED;

	STAILQ_FOREACH(autorule, &autorules_list, link) {
		if (WT_IS_INACTIVE(autorule->worktime)) {
			if (AUTORULE_IS_ACTIVE(autorule))
				if (set_autorule_inactive(autorule) < 0)
					goto failed;
		} else {
			if (AUTORULE_IS_INACTIVE(autorule))
				if (set_autorule_active(autorule) < 0)
					goto failed;
			if (autorules_active_check_sec >
			    autorule->update_tevent->event_sec)
				autorules_active_check_sec =
				    autorule->update_tevent->event_sec;
			if (autorules_active_check_sec >
			    autorule->worktime->inactive_sec)
				autorules_active_check_sec =
				    autorule->worktime->inactive_sec;
		}
	}

	sort_inactive_autorules();
	return (0);

failed:
	logbt("autorules_newday");
	return (-1);
}

/*
 * Check whether it is time to make some autorule active.
 * This function is called when autorules_inactive_check_sec <= cursec.
 */
int
check_inactive_autorules(void)
{
	struct autorule *autorule, *autorule_next;

	for (autorule = TAILQ_FIRST(&autorules_inactive); autorule != NULL;
	    autorule = autorule_next) {
		if (autorule->worktime->inactive_sec <= cursec) {
			/* It's time to make autorule active. */
			autorule_next = TAILQ_NEXT(autorule, queue);
			if (set_autorule_active(autorule) < 0) {
				logbt("check_inactive_autorules");
				return (-1);
			}
		} else {
			/* Set next time of this function invocation. */
			autorules_inactive_check_sec =
			    autorule->worktime->active_sec;
			return (0);
		}
	}

	/* No more inactive autorules. */
	autorules_inactive_check_sec = EVENT_NOT_SCHEDULED;
	return (0);
}

/*
 * Check whether it is time to make some autorule inactive.
 * This function is called when autorules_active_check_sec <= cursec.
 */
int
check_active_autorules(void)
{
	struct autorule *autorule;

	autorules_active_check_sec = EVENT_NOT_SCHEDULED;

	TAILQ_FOREACH(autorule, &autorules_active, queue) {
		if (WT_IS_INACTIVE(autorule->worktime)) {
			/* Autorule became inactive. */
			if (!newday_flag)
				if (set_autorule_inactive(autorule) < 0) {
					logbt("check_active_autorules");
					return (-1);
				}
		}  else {
			/* Autorule is still active. */
			if (autorules_active_check_sec >
			    autorule->worktime->inactive_sec)
				autorules_active_check_sec =
				    autorule->worktime->inactive_sec;
			if (autorules_active_check_sec >
			    autorule->update_tevent->event_sec)
				autorules_active_check_sec =
				    autorule->update_tevent->event_sec;
		}
	}

	return (0);
}

/*
 * Finish checks with autorules and inherit settings from global{}
 * in all autorule{} sections.
 */
int
autorules_inherit(void)
{
#ifdef WITH_LIMITS
	struct limit *limit;
#endif
#ifdef WITH_THRESHOLDS
	struct threshold *threshold;
#endif
	const struct ac_list *ac_list;
	struct autorule *autorule;
	unsigned int i;

	marray_minimize(autorules_marray);
	for (i = 0, autorule = autorules; i < nautorules; ++autorule, ++i) {
		STAILQ_INSERT_TAIL(&autorules_list, autorule, link);
		ac_list = autorule->ac_list != NULL ?
		    autorule->ac_list : global_ac_list;
		if (ac_list != NULL && !STAILQ_EMPTY(ac_list))
			if (STAILQ_NEXT(STAILQ_FIRST(ac_list), link) != NULL) {
				logconfe("autorule %s: only one accounting "
				    "system can be used", autorule->name);
				return (-1);
			}
		if (mimic_real_config) {
			if (autorule->ac_list == NULL) {
				autorule->ac_list = global_ac_list;
				ac_inc_ref_count(global_ac_list);
			}
			if (autorule->update_tevent == NULL)
				autorule->update_tevent = global_update_tevent;
			if (autorule->worktime == NULL)
				autorule->worktime = global_worktime;
			cmds_rule_set_sync(&autorule->rc[RC_STARTUP]);
			cmds_rule_set_sync(&autorule->rc[RC_SHUTDOWN]);
		}
# ifdef WITH_LIMITS
		STAILQ_FOREACH(limit, &autorule->limits, link) {
			if (check_worktime_subset(autorule->worktime_rule,
			    limit->worktime) < 0) {
				logconfe("autorule %s, limit %s: limit's "
				    "\"worktime\" must be a subset of "
				    "autorule's \"worktime_rule\"",
				    autorule->name, limit->name);
				return (-1);
			}
			if (mimic_real_config)
				limit_inherit(limit);
		}
# endif
# ifdef WITH_THRESHOLDS
		STAILQ_FOREACH(threshold, &autorule->thresholds, link) {
			if (check_worktime_subset(autorule->worktime_rule,
			    threshold->worktime) < 0) {
				logconfe("autorule %s, threshold %s: "
				    "threshold's \"worktime\" must be a "
				    "subset of autorule's worktime_rule",
				    autorule->name, threshold->name);
				return (-1);
			}
			if (mimic_real_config)
				threshold_inherit(threshold);
		}
# endif
	}
	return (0);
}

#else /* !WITH_AUTORULES */

/* ARGSUSED3 */
int
create_rule(const char *mod_name, unsigned int autoruleno,
    const char *rule_name, const char *rule_info ATTR_UNUSED)
{
	logmsgx(IPA_LOG_WARNING, "create_rule: module %s tries to create "
	    "rule %s from autorule %u", mod_name, rule_name, autoruleno);
	logmsgx(IPA_LOG_WARNING, "create_rule: autorules support "
	    "was not compiled");
	return (-1);
}

int
delete_rule(const char *mod_name, unsigned int ruleno)
{
	logmsgx(IPA_LOG_ERR, "delete_rule: module %s tries to delete "
	    "rule %u", mod_name, ruleno);
	logmsgx(IPA_LOG_ERR, "delete_rule: autorules support "
	    "was not compiled");
	return (-1);
}
#endif /* WITH_AUTORULES */
