/*-
 * Copyright (c) 2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipactl.c,v 1.5.2.2 2012/07/09 20:28:19 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/un.h>

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <regex.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "pathnames.h"

#include "ipa_mod.h"

#include "ipactl.h"

#if (SIZEOF_LONG_LONG * CHAR_BIT) < 64
# error "'long long' is less than 64 bits: too small"
#endif

#ifndef PF_LOCAL
# ifdef PF_UNIX
#  define PF_LOCAL PF_UNIX
# else
#  ifdef AF_LOCAL
#   define PF_LOCAL AF_LOCAL
#  else
#   ifdef AF_UNIX
#    define PF_LOCAL AF_UNIX
#   else
#    error "cannot define PF_LOCAL"
#   endif
#  endif
# endif
#endif

#ifndef SUN_LEN
# define SUN_LEN(su) (sizeof(struct sockaddr_un))
#endif

#define IPACTL_NAME "ipactl"

#ifndef IPACTL_ANSWER_MAX
# define IPACTL_ANSWER_MAX	(1 * 1024U * 1024U)
#endif

/*
 * All strto_uintxx() functions get strings with positive decimal
 * integers, so it is not necessary to check '-' in the first character
 * of a string inside strto_uintxx() functions (remember, that strtoul()
 * and strtoull() work with negative values).
 */

static const char *envprogname;

static char	wait_answer = 1;	/* 0, if -n. */
static unsigned int ctl_timeout = 0;	/* -w timeout */
static const char *ctl_socket_path = IPA_CTL_SOCKET; /* -s socket */

#ifdef WITH_AUTORULES
# define OPTSTRING_AUTORULES "a:"
#else
# define OPTSTRING_AUTORULES ""
#endif

#ifdef WITH_LIMITS
# define OPTSTRING_LIMITS "l:"
#else
# define OPTSTRING_LIMITS ""
#endif

#ifdef WITH_THRESHOLDS
# define OPTSTRING_THRESHOLDS "t:"
#else
# define OPTSTRING_THRESHOLDS ""
#endif

#define OPTSTRING \
	":hnr:s:vw:" OPTSTRING_AUTORULES OPTSTRING_LIMITS OPTSTRING_THRESHOLDS

static int	sockfd;			/* Socket descriptor. */

static struct ctl_cmdq cmdq;		/* Control command query. */
static struct ctl_cmdq_val cmdq_val;	/* One or two values. */
static char	*cmdq_names;		/* Array of one or two names. */
static size_t	cmdq_names_size;	/* Size of cmdq_names. */

static struct ctl_cmda cmda;		/* Control command answer. */
static void	*cmda_aux;		/* Auxiliary answer. */
static char	cmda_aux_exp;		/* Set if cmda_aux is expected. */
static size_t	cmda_aux_size;		/* Auxiliary answer size. */

static union {
	struct ctl_cmda_all_status all;
	struct ctl_cmda_rule_status rule;
#ifdef WITH_AUTORULES
	struct ctl_cmda_autorule_status autorule;
#endif
#ifdef WITH_LIMITS
	struct ctl_cmda_limit_status limit;
#endif
#ifdef WITH_THRESHOLDS
	struct ctl_cmda_threshold_status threshold;
#endif
} cmda_status;

#ifdef WITH_ANY_LIMITS
static struct ctl_cmda_set cmda_set;
#endif

#define OPTREQ_RULE		0x01
#ifdef WITH_AUTORULES
# define OPTREQ_AUTORULE	0x02
#endif
#ifdef WITH_LIMITS
# define OPTREQ_LIMIT		0x04
#endif
#ifdef WITH_THRESHOLDS
# define OPTREQ_THRESHOLD	0x08
#endif

#define OPTREQ_MASK		0x0f
#define OPTREQ_EXACT		0x10

/*
 * Describe command, its arguments and helper functions.
 */
struct cmd_descr {
	const char	*name;		/* Name of option.		*/
	unsigned int	cmd;		/* CTL_CMD_xxx code.		*/
	int		nargs;		/* Number of arguments or -1.	*/
	int		reqs;		/* ORed REQ_xxx or -1.		*/
	int	(*parse)(int, char **); /* Command arguments parser.	*/
	int	(*check)(void);		/* Return result cheker.	*/
};

#define PAT_TIME		"^([[:digit:]]+[smh] ?)+$"
#define PAT_BYTES		"^([[:digit:]]+[BKMGT] ?)+$"

#define SECONDS_IN_MINUTE	(60)
#define SECONDS_IN_HOUR		(60 * SECONDS_IN_MINUTE)
#define SECONDS_IN_DAY		(24 * SECONDS_IN_HOUR)

#define KBYTE			(UINT64_C(1024))
#define MBYTE			(UINT64_C(1024) * KBYTE)
#define GBYTE			(UINT64_C(1024) * MBYTE)
#define TBYTE			(UINT64_C(1024) * GBYTE)

static void	logmsg(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	logmsgx(const char *, ...) ATTR_FORMAT(printf, 1, 2);
static void	vlogmsg(const char *, va_list) ATTR_FORMAT(printf, 1, 0);
static void	vlogmsgx(const char *, va_list) ATTR_FORMAT(printf, 1, 0);
static void	exit_errx(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);

static void
vlogmsg(const char *format, va_list ap)
{
	int errno_save;

	errno_save = errno;
	fflush(stdout);
	fprintf(stderr, "%s: ", envprogname);
	vfprintf(stderr, format, ap);
	if (errno_save != 0)
		fprintf(stderr, ": %s", strerror(errno_save));
	fprintf(stderr, "\n");
}

static void
vlogmsgx(const char *format, va_list ap)
{
	fflush(stdout);
	fprintf(stderr, "%s: ", envprogname);
	vfprintf(stderr, format, ap);
	fprintf(stderr, "\n");
}

/*
 * Output the program name, a message and an error message.
 */
static void
logmsg(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsg(format, ap);
	va_end(ap);
}

/*
 * Output the program name and a message.
 */
static void
logmsgx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsgx(format, ap);
	va_end(ap);
}

/*
 * Output the program name, a message and exit.
 */
static void
exit_errx(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsgx(format, ap);
	va_end(ap);
	exit(1);
}

/*
 * Convert string to 'unsigned int'.
 */
static int
strto_u_int(unsigned int *result, const char *nptr)
{
	char *endptr;
	unsigned long val_ul;

	if (!isdigit((unsigned char)*nptr))
		goto failed;
	errno = 0;
	val_ul = strtoul(nptr, &endptr, 10);
	if (errno != 0) {
		logmsg("strtoul");
		return (-1);
	}
#if ULONG_MAX > UINT_MAX
	if (val_ul > UINT_MAX) {
		logmsgx("too big value for 'unsigned int'");
		return (-1);
	}
#endif
	if (nptr == endptr)
		goto failed;
	*result = (unsigned int)val_ul;
	return (0);

failed:
	logmsgx("wrong positive decimal number");
	return (-1);
}

/*
 * Any symbol in any name must be letter, digit or punctuation (except
 * double quote and both slashes) from the ASCII character set.
 */
static int
validate_name(const char *s)
{
	unsigned char c;

	if (*s == '\0')
		return (-1);
	do {
		c = (unsigned char)*s;
		if (!isascii(c) || !(isalnum(c) || ispunct(c)) ||
		    c == '\"' || c == '/' || c == '\\')
			return (-1);
	} while (*++s != '\0');
	return (0);
}

/*
 * Output version number and settings (-v and -h switches).
 */
static void
show_version(void)
{
	printf(IPACTL_NAME ", version " PACKAGE_VERSION "\nSupports: rules"
#ifdef WITH_LIMITS
	", limits"
#endif
#ifdef WITH_THRESHOLDS
	", thresholds"
#endif
#ifdef CTL_CHECK_CREDS
	"; sending messages credentials is enabled"
#else
	"; sending messages credentials is disabled"
#endif
	".\n");
}

/*
 * Output the help message (-h switch).
 */
static void
usage(void)
{
	show_version();
	printf(
"Usage: %s [-hnv] [-s socket] [-w time]"
#ifdef WITH_AUTORULES
" [-a autorule]"
#endif
"\n              [-r rule"
#if defined(WITH_LIMITS) && defined(WITH_THRESHOLDS)
" [-l limit|-t threshold]"
#else
# ifdef WITH_LIMITS
" [-l limit]"
# endif
# ifdef WITH_THRESHOLDS
" [-t threshold]"
# endif
#endif /* WITH_LIMITS && WITH_THRESHOLDS */
"] command [args...]\n"
"Control utility for ipa(8)\n", envprogname);
	printf(
"Options are:\n"
#ifdef WITH_AUTORULES
"  -a autorule\t\tSpecify an autorule name\n"
#endif
"  -r rule\t\tSpecify a rule name\n"
#ifdef WITH_LIMITS
"  -l limit\t\tSpecify a limit name\n"
#endif
#ifdef WITH_THRESHOLDS
"  -t threshold\t\tSpecify a threshold name\n"
#endif
"  -n\t\t\tDo not wait the answer from ipa\n\
  -s socket\t\tConnect to the given socket, by default connect\n\
\t\t\tto Unix domain socket %s\n\
  -w time\t\tHow many seconds to wait for the answer from ipa,\n\
\t\t\tzero means infinite timeout (this is default)\n\
  -h\t\t\tOutput this help message\n\
  -v\t\t\tShow version number and exit\n",
	    IPA_CTL_SOCKET);
	printf("\
Commands are (details in the manual page):\n");
#ifdef WITH_AUTORULES
	printf("\
  create\t\tCreate dynamic rule (-a, -r)\n\
  delete\t\tDelete dynamic rule (-r)\n");
#endif
	printf("\
  dump\t\t\tForce dumping statistics to database (no opts)\n\
  freeze\t\tFreeze work of ipa (no opts)\n\
  memory\t\tOutput information about used memory (no opts)\n\
  status\t\tOutput status information (no opts, -r"
#ifdef WITH_LIMITS
", -l"
#endif
#ifdef WITH_THRESHOLDS
", -t"
#endif
#ifdef WITH_AUTORULES
", -a"
#endif
	    ")\n");
#ifdef WITH_LIMITS
	printf("\
  restart\t\tRestart limit, if it is currently not reached (-r, -l)\n\
  expire\t\tExpire limit, if it was reached (-r, -l)\n\
  set limit [+|-]value [counter [+|-]value]\n\
\t\t\tChange value of the \"limit\" parameter for limit and\n\
\t\t\toptionally its counter (-r, -l)\n");
#endif /* WITH_LIMITS */
#ifdef WITH_THRESHOLDS
	printf("  set threshold [+|-]value [counter [+|-]value]\n\
\t\t\tChange value of the \"threshold\" parameter for\n\
\t\t\tthreshold and optionally its counter (-r, -t)\n");
#endif
#ifdef WITH_ANY_LIMITS
	printf("  set counter [+|-]value\n"
# if defined(WITH_LIMITS) && defined(WITH_THRESHOLDS)
"\t\t\tChange rule's, limit's or threshold's counter. In all\n\
\t\t\tcommands `+' means increasing and `-' means decreasing\n\
\t\t\tof current value (-r, -l, -t)\n"
# else
#  ifdef WITH_LIMITS
"\t\t\tChange rule's or limit's counter. In all commands `+'\n\
\t\t\tmeans increasing and `-' means decreasing of current\n\
\t\t\tvalue (-r, -l)\n"
#  endif
#  ifdef WITH_THRESHOLDS
"\t\t\tChange rule's or threshold's counter. In all\n\
\t\t\tcommands `+' means increasing and `-' means decreasing\n\
\t\t\tof current value (-r, -t)\n"
#  endif
# endif /* WITH_LIMITS && WITH_THRESHOLDS */
	    );
#else
	printf("  set counter [+|-]value\n"
"\t\t\t\tChange rule's counter. `+' means increasing and\n\
\t\t\t`-' means decreasing of current value (-r)\n"
	    );
#endif /* WITH_ANY_LIMITS */
}

/*
 * Simplified version of regexec(3).
 */
static int
regexec_simple(const regex_t *re, const char *str)
{
	return (regexec(re, str, 0, (regmatch_t *)NULL, 0));
}

/*
 * Form an error message in buffer according to code using regerror()
 * function and return pointer to it.
 */
static const char *
regerrbuf(int code)
{
	static char buf[128];

	regerror(code, (regex_t *)NULL, buf, sizeof(buf));
	return (buf);
}

#ifndef HAVE_STRTOULL
/*
 * On the system (on which I tried to build IPA), which does not
 * have strtoull(), ULLONG_MAX is also not defined.
 */
# ifndef ULLONG_MAX
#  define ULLONG_MAX	UINT64_MAX
# endif

/*
 * Since this function always is called for base 10, third argument
 * is not used, and since first argument always points to a string
 * with all digits checks and implementation can be simplified
 */
/* ARGSUSED2 */
static unsigned long long
strtoull(const char *nptr, char **endptr, int base ATTR_UNUSED)
{
	unsigned long long x;
	const char *ptr;
	unsigned char c;

	x = 0;
	for (ptr = nptr; *ptr != '\0'; ++ptr) {
		c = *ptr - '0';
		if (x > (ULLONG_MAX / 10) ||
		    (x == (ULLONG_MAX / 10) && c > (ULLONG_MAX % 10))) {
			errno = ERANGE;
			return (ULLONG_MAX);
		}
		x *= 10;
		x += c;
	}
	if (endptr != NULL)
		*endptr = (char *)ptr;
	return (x);
}
#endif /* !HAVE_STRTOULL */

/*
 * Convert string to uint64_t.
 */
static int
strto_uint64(uint64_t *result, const char *nptr, char **endptr_ret)
{
	unsigned long long val_ull;
	char *endptr;

	errno = 0;
	val_ull = strtoull(nptr, &endptr, 10);
	if (errno != 0) {
		logmsg("strtoull failed for value \"%s\"", nptr);
		return (-1);
	}
#if (SIZEOF_LONG_LONG * CHAR_BIT) > 64
	if (val_ull > UINT64_MAX) {
		logmsgx("too big value %llu for 'uint64_t' type", val_ull);
		return (-1);
	}
#endif
	if (nptr == endptr) {
		logmsgx("wrong number");
		return (-1);
	}
	*result = (uint64_t)val_ull;
	if (endptr_ret != NULL)
		*endptr_ret = endptr;
	return (0);
}

/*
 * Parse bytes value.
 */
static int
parse_value_bytes(uint64_t *res, const char *arg)
{
	uint64_t value, result;
	const char *ptr;
	char *endptr;
	char level, error, overflow;

	result = 0;
	ptr = arg;
	level = error = overflow = 0;

	for (;;) {
		if (strto_uint64(&value, ptr, &endptr) < 0)
			return (-1);
		ptr = endptr;
		switch (*ptr) {
		case 'T':
			if (level > 0)
				error = 1;
			else if (value > UINT64_MAX / TBYTE)
				overflow = 1;
			else {
				level = 1;
				value *= TBYTE;
			}
			break;
		case 'G':
			if (level > 1)
				error = 1;
			else if (value > UINT64_MAX / GBYTE)
				overflow = 1;
			else {
				level = 2;
				value *= GBYTE;
			}
			break;
		case 'M':
			if (level > 2)
				error = 1;
			else if (value > UINT64_MAX / MBYTE)
				overflow = 1;
			else {
				level = 3;
				value *= MBYTE;
			}
			break;
		case 'K':
			if (level > 3)
				error = 1;
			else if (value > UINT64_MAX / KBYTE)
				overflow = 1;
			else {
				level = 4;
				value *= KBYTE;
			}
			break;
		default: /* 'B' */
			if (level > 4)
				error = 1;
			else
				level = 5;
		}
		if (error) {
			logmsgx("wrong bytes format");
			return (-1);
		}
		if (overflow || result > UINT64_MAX - value) {
			logmsgx("too big value for 'uint64_t' type");
			return (-1);
		}
		result += value;
		while (*++ptr == ' ')
			;
		if (*ptr == '\0') {
			/* EOL */
			break;
		}
	}
	*res = result;
	return (0);
}

/*
 * Parse time value.
 */
static int
parse_value_time(uint64_t *res, const char *arg)
{
	uint64_t value, result;
	const char *ptr;
	char *endptr;
	char level, error, overflow;

	result = 0;
	ptr = arg;
	level = error = overflow = 0;

	for (;;) {
		if (strto_uint64(&value, ptr, &endptr) < 0)
			return (-1);
		ptr = endptr;
		switch (*ptr) {
		case 'h':
			if (level > 0)
				error = 1;
			else if (value > UINT64_MAX / SECONDS_IN_HOUR)
				overflow = 1;
			else {
				level = 1;
				value *= SECONDS_IN_HOUR;
			}
			break;
		case 'm':
			if (level > 1)
				error = 1;
			else if (value > UINT64_MAX / SECONDS_IN_MINUTE)
				overflow = 1;
			else {
				level = 2;
				value *= SECONDS_IN_MINUTE;
			}
			break;
		default: /* 's' */
			if (level > 2)
				error = 1;
			else
				level = 3;
		}
		if (error) {
			logmsgx("wrong time format");
			return (-1);
		}
		if (overflow || result > UINT64_MAX - value) {
			logmsgx("too big value for 'uint64_t' type");
			return (-1);
		}
		result += value;
		while (*++ptr == ' ')
			;
		if (*ptr == '\0') {
			/* EOL */
			break;
		}
	}
	*res = result;
	return (0);
}

/*
 * Parse value (bytes, time or number) possibly prepended
 * with '+' or '-'.
 */
static int
parse_value(uint64_t *result, const char *arg)
{
	regex_t reg;
	int error;

	error = regcomp(&reg, PAT_BYTES, REG_EXTENDED|REG_NOSUB);
	if (error != 0) {
		logmsgx("regcomp(\"%s\"): %s", PAT_BYTES, regerrbuf(error));
		return (-1);
	}
	if (regexec_simple(&reg, arg) == 0) {
		if (parse_value_bytes(result, arg) < 0)
			return (-1);
	} else {
		regfree(&reg);
		error = regcomp(&reg, PAT_TIME, REG_EXTENDED|REG_NOSUB);
		if (error != 0) {
			logmsgx("regcomp(\"%s\"): %s", PAT_TIME,
			    regerrbuf(error));
			return (-1);
		}
		if (regexec_simple(&reg, arg) == 0) {
			if (parse_value_time(result, arg) < 0)
				return (-1);
		} else {
			const char *ptr;

			for (ptr = arg; *ptr != '\0'; ++ptr)
				if (!isdigit((unsigned char)*ptr)) {
					logmsgx("value should be bytes, time "
					    "or number");
					return (-1);
				}
			if (strto_uint64(result, arg, (char **)NULL) < 0)
				return (-1);
		}
	}
	regfree(&reg);
	return (0);
}

#ifdef WITH_ANY_LIMITS

struct time_conv {
	uint64_t	div;
	uint64_t	val;
	char		ch;
};

static struct time_conv time_conv_tbl[] = {
	{ SECONDS_IN_HOUR,   0,	'h' },
	{ SECONDS_IN_MINUTE, 0,	'm' },
	{ 1,		     0, 's' }
};

#define TIME_CONV_TBL_SIZE (sizeof(time_conv_tbl) / sizeof(time_conv_tbl[0]))

static void
print_time(const uint64_t *ptr)
{
	uint64_t a;
	int i, i1, i2;

	a = *ptr;
	i1 = i2 = -1;
	for (i = 0; i < TIME_CONV_TBL_SIZE; ++i) {
		time_conv_tbl[i].val = a / time_conv_tbl[i].div;
		if (time_conv_tbl[i].val != 0) {
			if (i1 < 0)
				i1 = i;
			i2 = i;
		}
		a %= time_conv_tbl[i].div;
	}
	if (i1 >= 0) {
		for (i = i1;; ++i) {
			printf("%"PRIu64"%c", time_conv_tbl[i].val,
			    time_conv_tbl[i].ch);
			if (i == i2)
				break;
			printf(" ");
		}
	} else
		printf("0s");
}

struct byte_conv {
	uint64_t	div;
	unsigned int	val;
	char		ch;
};

static struct byte_conv byte_conv_tbl[] = {
	{ TBYTE, 0, 'T' },
	{ GBYTE, 0, 'G' },
	{ MBYTE, 0, 'M' },
	{ KBYTE, 0, 'K' },
	{ 1,	 0, 'B' }
};

#define BYTE_CONV_TBL_SIZE (sizeof(byte_conv_tbl) / sizeof(byte_conv_tbl[0]))

static void
print_bytes(const uint64_t *ptr)
{
	uint64_t a;
	int i, i1, i2;

	a = *ptr;
	i1 = i2 = -1;
	for (i = 0; i < BYTE_CONV_TBL_SIZE; ++i) {
		byte_conv_tbl[i].val = a / byte_conv_tbl[i].div;
		if (byte_conv_tbl[i].val != 0) {
			if (i1 < 0)
				i1 = i;
			i2 = i;
		}
		a %= byte_conv_tbl[i].div;
	}
	if (i1 >= 0) {
		for (i = i1;; ++i) {
			printf("%u%c", byte_conv_tbl[i].val,
			    byte_conv_tbl[i].ch);
			if (i == i2)
				break;
			printf(" ");
		}
	} else
		printf("0B");
}

static void
print_value(char sign, const uint64_t *value, unsigned int type)
{
	if (sign)
		printf("-");
	switch (type) {
	case IPA_CONF_TYPE_UINT64:
		printf("%"PRIu64, *value);
		return;
	case IPA_CONF_TYPE_BYTES:
		print_bytes(value);
		break;
	case IPA_CONF_TYPE_TIME:
		print_time(value);
		break;
	}
	printf(" / %s%"PRIu64, sign ? "-" : "", *value);
}

static void
print_ipa_tm(const ipa_tm *tm)
{
	printf("%d.%02d.%02d/%02d:%02d:%02d\n",
	    tm->tm_year, tm->tm_mon, tm->tm_mday,
	    tm->tm_hour, tm->tm_min, tm->tm_sec);
}
#endif /* WITH_ANY_LIMITS */

static void
print_line(int len)
{
	while (len--)
		printf("-");
}

static const char * const answer_msg[] = {
	"command succeeded",			/*  CTL_ANS_DONE */
	"some system error occurred",		/*  CTL_ANS_SYSTEM_ERROR */
	"wrong command or flag",		/*  CTL_ANS_UNKNOWN_COMMAND */
	"illegal name in control query",	/*  CTL_ANS_ILLEGAL_NAME */
	"command was denied",			/*  CTL_ANS_DENIED */
	"unknown autorule name",		/*  CTL_ANS_UNKNOWN_AUTORULE */
	"unknown rule name",			/*  CTL_ANS_UNKNOWN_RULE */
	"unknown limit name",			/*  CTL_ANS_UNKNOWN_LIMIT */
	"unknown threshold name",		/*  CTL_ANS_UNKNOWN_THRESHOLD */
	"cannot modify value",			/*  CTL_ANS_CANNOT_MODIFY */
	"cannot expire limit (not reached)",	/*  CTL_ANS_CANNOT_EXPIRE */
	"cannot restart limit (reached)",	/*  CTL_ANS_CANNOT_RESTART */
	"cannot create rule",			/*  CTL_ANS_CANNOT_CREATE */
	"cannot delete rule"			/*  CTL_ANS_CANNOT_DELETE */
};

static int
ctl_xxx_check(void)
{
	unsigned int result;

	result = cmda.result;
	if (result == CTL_ANS_DONE) {
		if (cmda_aux_size == 0 && cmda_aux_exp) {
			logmsgx("auxiliary answer was expected, "
			    "but was not received");
			return (-1);
		}
		printf("\nResult: %s\n\n", answer_msg[CTL_ANS_DONE]);
		return (0);
	}
	if (result <= CTL_MAX_ANS)
		logmsgx("Result: %s", answer_msg[result]);
	else
		logmsgx("Result: unknown result code %u", result);
	return (-1);
}

/*
 * Add the given string to cmdq_names.
 */
static void
ctl_put_name(const char *name)
{
	char *ptr;

	for (ptr = cmdq_names; *ptr != '\0'; ++ptr)
		;
	if (ptr != cmdq_names)
		++ptr;
	strcpy(ptr, name);
}

/*
 * Send control command query.
 */
static int
send_cmd_query(void)
{
#ifdef CTL_CHECK_CREDS
# ifdef __FreeBSD__
	struct cmsghdr *cmsghdr;
	union {
		struct cmsghdr	cmsghdr;
		char	control[CMSG_SPACE(sizeof(struct cmsgcred))];
	} cmsg;
# endif
#endif /* CTL_CHECK_CREDS */
	struct iovec iov[3];
	struct msghdr msghdr;
	struct sockaddr_un servaddr;
	size_t size;
	ssize_t	nsent;
	int fd;

	if (strlen(ctl_socket_path) + 1 > sizeof(servaddr.sun_path)) {
		logmsgx("too long path for control socket");
		return (-1);
	}

	fd = sockfd = socket(PF_LOCAL, SOCK_STREAM, 0);
	if (fd < 0) {
		logmsg("socket(PF_LOCAL, SOCK_STREAM)");
		return (-1);
	}

	if (ctl_timeout != 0) {
		struct timeval tv;

		tv.tv_sec = ctl_timeout;
		tv.tv_usec = 0;
		if (setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv,
		    sizeof(tv)) < 0 ||
		    setsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &tv,
		    sizeof(tv)) < 0) {
			logmsg("setsockopt(SO_RCVTIMEO/SO_SNDTIMEO");
			return (-1);
		}
	}

	memset(&servaddr, 0, sizeof(servaddr));
	servaddr.sun_family = PF_LOCAL;
	strncpy(servaddr.sun_path, ctl_socket_path,
	    sizeof(servaddr.sun_path) - 1);

	memset(&msghdr, 0, sizeof(msghdr));
	msghdr.msg_name = NULL;

	cmdq.ver = CTL_PROTOCOL_VERSION;

	iov[0].iov_base = (char *)&cmdq;
	iov[0].iov_len = sizeof(cmdq);
	size = sizeof(cmdq);
	if (cmdq_names != NULL) {
		if (cmdq.cmd == CTL_CMD_SET) {
			iov[1].iov_base = (char *)&cmdq_val;
			iov[1].iov_len = sizeof(cmdq_val);
			size += sizeof(cmdq_val);
			msghdr.msg_iovlen = 3;
		} else
			msghdr.msg_iovlen = 2;
		iov[msghdr.msg_iovlen - 1].iov_base = cmdq_names;
		iov[msghdr.msg_iovlen - 1].iov_len = cmdq_names_size;
		size += cmdq_names_size;
	} else
		msghdr.msg_iovlen = 1;
	msghdr.msg_iov = iov;

#ifdef CTL_CHECK_CREDS
# ifdef __FreeBSD__
	msghdr.msg_control = cmsg.control;
	msghdr.msg_controllen = sizeof(cmsg.control);
	cmsghdr = CMSG_FIRSTHDR(&msghdr);
	cmsghdr->cmsg_len = CMSG_LEN(sizeof(struct cmsgcred));
	cmsghdr->cmsg_level = SOL_SOCKET;
	cmsghdr->cmsg_type = SCM_CREDS;
# endif /* __FreeBSD__ */
#else
	msghdr.msg_control = NULL;
	msghdr.msg_controllen = 0;
#endif /* CTL_CHECK_CREDS */

	/*
	 * Timeout is not required for connect() for Unix domain
	 * sockets, since if listening socket's queue is full, then
	 * connect() returns ECONNREFUSED immediately.
	 */
	if (connect(fd, (struct sockaddr *)&servaddr, SUN_LEN(&servaddr)) < 0) {
		logmsg("connect(%s)", ctl_socket_path);
		return (-1);
	}

	printf("Sending command...");
	fflush(stdout);
	nsent = sendmsg(fd, &msghdr, 0);
	if (nsent < 0) {
		logmsg("sendmsg(%s)", ctl_socket_path);
		return (-1);
	}
	if (nsent != size) {
		logmsgx("sendmsg: short send (%ld of %lu bytes)",
		    (long)nsent, (unsigned long)size);
		return (-1);
	}

	if (!wait_answer)
		if (close(fd) < 0) {
			logmsg("close");
			return (-1);
		}

	printf(" done\n");
	fflush(stdout);
	return (0);
}

/*
 * Read from socket nbytes bytes to buf.
 */
static int
read_sock(void *buf, size_t nbytes)
{
	ssize_t	nread;

	nread = recv(sockfd, buf, nbytes, MSG_WAITALL);
	if (nread < 0) {
		logmsg("recv");
		return (-1);
	}
	if (nread != nbytes) {
		logmsgx("recv: short read (%ld of %lu bytes)",
		    (long)nread, (unsigned long)nbytes);
		return (-1);
	}
	return (0);
}

/*
 * Receive control command answer.
 */
static int
recv_cmd_answer(void)
{
	printf("Receiving answer...");
	fflush(stdout);

	/* Receive generic answer. */
	if (read_sock(&cmda, sizeof(cmda)) < 0) {
		logmsgx("cannot receive generic answer");
		return (-1);
	}

	/* Receive auxiliary answer, if needed. */
	if (cmda.size != 0) {
		if (!cmda_aux_exp) {
			logmsgx("auxiliary answer was received, "
			    "but was not expected");
			return (-1);
		}
		if (cmda_aux == NULL) {
			if (!cmda_aux_exp) {
				logmsgx("auxiliary answer is returned, "
				    "but not expected");
				return (-1);
			}
			cmda_aux_size = cmda.size;
			if (cmda_aux_size > IPACTL_ANSWER_MAX) {
				logmsgx("returned size %lu of auxiliary answer "
				    "is greater than %u",
				    (unsigned long)cmda_aux_size,
				    IPACTL_ANSWER_MAX);
				return (-1);
			}
			cmda_aux = malloc(cmda_aux_size);
			if (cmda_aux == NULL) {
				logmsg("malloc(%lu bytes): cannot allocate "
				    "memory for auxiliary answer",
				    (unsigned long)cmda_aux_size);
				return (-1);
			}
		} else if (cmda.size != cmda_aux_size) {
			logmsgx("returned size of auxiliary answer %lu is not "
			    "equal to expected size %lu",
			    (unsigned long)cmda.size,
			    (unsigned long)cmda_aux_size);
			return (-1);
		}
		if (read_sock(cmda_aux, cmda_aux_size) < 0) {
			logmsgx("cannot receive auxiliary answer");
			return (-1);
		}
	} else
		cmda_aux_size = 0;

	if (close(sockfd) < 0) {
		logmsg("close");
		return (-1);
	}

	printf(" done\n");
	fflush(stdout);
	return (0);
}

/*
 * Parser for "status" command.
 */
/* ARGSUSED */
static int
ctl_status_parse(int agrc ATTR_UNUSED, char *argv[] ATTR_UNUSED)
{
	unsigned int qflags;

	qflags = cmdq.flags;
#if defined(WITH_RULES) && defined(WITH_AUTORULES)
	if ((qflags & (CTL_CFLAG_AUTORULE|CTL_CFLAG_RULE)) ==
	    (CTL_CFLAG_AUTORULE|CTL_CFLAG_RULE)) {
		logmsgx("-a and -r options cannot be used together "
		    "for this command");
		return (-1);
	}
#endif
#ifdef WITH_AUTORULES
	if (qflags & CTL_CFLAG_AUTORULE) {
		cmda_aux = &cmda_status.autorule;
		cmda_aux_size = sizeof(cmda_status.autorule);
	} else
#endif
	if ((qflags &
	    (CTL_CFLAG_RULE|CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD)) ==
	    CTL_CFLAG_RULE) {
		cmda_aux = &cmda_status.rule;
		cmda_aux_size = sizeof(cmda_status.rule);
	} else
#ifdef WITH_LIMITS
	if (qflags & CTL_CFLAG_LIMIT) {
		cmda_aux = &cmda_status.limit;
		cmda_aux_size = sizeof(cmda_status.limit);
	} else
#endif
#ifdef WITH_THRESHOLDS
	if (qflags & CTL_CFLAG_THRESHOLD) {
		cmda_aux = &cmda_status.threshold;
		cmda_aux_size = sizeof(cmda_status.threshold);
	} else
#endif
	{
		cmda_aux = &cmda_status.all;
		cmda_aux_size = sizeof(cmda_status.all);
	}
	cmda_aux_exp = 1;
	return (0);
}

static void
ctl_status_check1(char active, char dynamic)
{
	printf("\n  Active:\t%s\n  Type:\t\t%s\n",
	    active ? "yes" : "no",
	    dynamic ? "dynamic" : "static");
}

/*
 * Check for "status" command.
 */
static int
ctl_status_check(void)
{
#ifdef WITH_ANY_LIMITS
	unsigned int value_type;
#endif
	unsigned int qflags;

	qflags = cmdq.flags;

#ifdef WITH_AUTORULES
# define status cmda_status.autorule
	if (qflags & CTL_CFLAG_AUTORULE) {
		printf("Autorule status:\n  Rules:\t%u\n  Active:\t%s\n",
		    status.nrules, status.active ? "yes" : "no");
	} else
# undef status
#endif
#define status cmda_status.rule
	if ((qflags & (CTL_CFLAG_RULE|CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD)) ==
	    CTL_CFLAG_RULE) {
		printf("Rule status:\n  Limits:\t%u\n  Sublimits:\t%u\n  "
		    "Thresholds:\t%u",
		    status.nlimits, status.nsublimits, status.nthresholds);
		ctl_status_check1(status.active, status.dynamic);
	} else
#undef status
#ifdef WITH_LIMITS
# define status cmda_status.limit
	if (qflags & CTL_CFLAG_LIMIT) {
		printf("Limit status:\n  Limit:\t");
		value_type = (unsigned int)status.value_type;
		print_value(0, &status.value1, value_type);
		printf("\n  Counter:\t");
		print_value(status.value2_sign, &status.value2, value_type);
		ctl_status_check1(status.active, status.dynamic);
		printf("  Sublimits:\t%u\n  State:\t", status.nsublimits);
		if (status.reached)
			printf("reached\n  Expire:\t");
		else
			printf("not reached\n  Restart:\t");
		if (status.tm.tm_year != 0)
			print_ipa_tm(&status.tm);
		else
			printf("never\n");
	} else
# undef status
#endif
#ifdef WITH_THRESHOLDS
# define status cmda_status.threshold
	if (qflags & CTL_CFLAG_THRESHOLD) {
		printf("Threshold status:\n  Threshold:\t");
		value_type = (unsigned int)status.value_type;
		print_value(0, &status.value1, value_type);
		printf("\n  Counter:\t");
		print_value(status.value2_sign, &status.value2, value_type);
		printf("\n  Deviation:\t");
		print_value(0, &status.deviation, value_type);
		ctl_status_check1(status.active, status.dynamic);
		printf("  Started:\t");
		print_ipa_tm(&status.tm[0]);
		printf("  Updated:\t");
		print_ipa_tm(&status.tm[1]);
	} else
# undef status
#endif
#define status cmda_status.all
	{
		printf("General status:\n  Ac_mods:\t%u\n  Db_mods:\t%u\n",
		    status.nac_mods, status.ndb_mods);
		printf("  Autorules:\t%u\n", status.nautorules);
		printf("  Rules:\t%u (%u static + %u dynamic)\n",
		    status.nstatrules + status.ndynrules,
		    status.nstatrules, status.ndynrules);
		printf("  Limits:\t%u (%u static + %u dynamic)\n",
		    status.nstatlimits + status.ndynlimits,
		    status.nstatlimits, status.ndynlimits);
		printf("  Sublimits:\t%u (%u static + %u dynamic)\n",
		    status.nstatsublimits + status.ndynsublimits,
		    status.nstatsublimits, status.ndynsublimits);
		printf("  Thresholds:\t%u (%u static + %u dynamic)\n",
		    status.nstatthresholds + status.ndynthresholds,
		    status.nstatthresholds, status.ndynthresholds);
	}
#undef status

	return (0);
}

/*
 * Parser for "memory" command.
 */
/* ARGSUSED */
static int
ctl_memory_parse(int agrc ATTR_UNUSED, char *argv[] ATTR_UNUSED)
{
	cmda_aux_exp = 1;
	return (0);
}

/*
 * Check for "memory" command.
 */
static int
ctl_memory_check(void)
{
	const struct ctl_cmda_memory *cmda_memory;
	const struct ctl_cmda_mzone *cmda_mzone;
	const struct ctl_cmda_marray *cmda_marray;
	const struct ctl_cmda_mem_type *cmda_mem_type;
	size_t len, name_width, desc_width;
	unsigned int i, n;
	int nwidth, dwidth;

	if (cmda_aux_size < sizeof(cmda_memory)) {
		logmsgx("too small auxiliary answer (%lu of %lu bytes)",
		    (unsigned long)cmda_aux_size,
		    (unsigned long)sizeof(cmda_memory));
		return (-1);
	}

	cmda_memory = (struct ctl_cmda_memory *)cmda_aux;

	if (cmda_memory->nmem_type * sizeof(struct ctl_cmda_mem_type) +
	    cmda_memory->nmzones * sizeof(struct ctl_cmda_mzone) +
	    cmda_memory->nmarrays * sizeof(struct ctl_cmda_marray) +
	    sizeof(*cmda_memory) != cmda_aux_size) {
		logmsgx("incorrect size of auxiliary answer (%lu bytes)",
		    (unsigned long)cmda_aux_size);
		return (-1);
	}

	printf("Memory status:\n  Allocated:\t%lu\n  Mem_types:\t%u\n  "
	    "Mzones:\t%u\n  Marrays:\t%u\n",
	    (unsigned long)cmda_memory->size, cmda_memory->nmem_type,
	    cmda_memory->nmzones, cmda_memory->nmarrays);

	name_width = sizeof("Mem_type") - 1;
	desc_width = sizeof("Description") - 1;

	cmda_mem_type = (const struct ctl_cmda_mem_type *)(cmda_memory + 1);
	n = cmda_memory->nmem_type;
	for (i = 0; i < n; ++i) {
		len = strlen(cmda_mem_type[i].name);
		if (name_width < len)
			name_width = len;
		len = strlen(cmda_mem_type[i].desc);
		if (desc_width < len)
			desc_width = len;
	}

	cmda_mzone = (const struct ctl_cmda_mzone *)(cmda_mem_type + n);
	n = cmda_memory->nmzones;
	for (i = 0; i < n; ++i) {
		len = strlen(cmda_mzone[i].name);
		if (name_width < len)
			name_width = len;
		len = strlen(cmda_mzone[i].desc);
		if (desc_width < len)
			desc_width = len;
	}

	cmda_marray = (const struct ctl_cmda_marray *)(cmda_mzone + n);
	n = cmda_memory->nmarrays;
	for (i = 0; i < n; ++i) {
		len = strlen(cmda_marray[i].name);
		if (name_width < len)
			name_width = len;
		len = strlen(cmda_marray[i].desc);
		if (desc_width < len)
			desc_width = len;
	}

	nwidth = (int)name_width;
	dwidth = (int)desc_width;

	n = cmda_memory->nmem_type;
	if (n != 0) {
		printf("\nMem_type status:\n  %-*s | Allocated |  Requests | "
		    "Description\n  ", nwidth, "Mem_type");
		print_line(nwidth);
		printf("-+-----------+-----------+-");
		print_line(dwidth);
		printf("\n");
		for (i = 0; i < n; ++i)
			printf("  %-*s | %9lu | %9u | %s\n", nwidth,
			    cmda_mem_type[i].name,
			    (unsigned long)cmda_mem_type[i].size,
			    cmda_mem_type[i].reqs, cmda_mem_type[i].desc);
		printf("  ");
		print_line(nwidth);
		printf("-+-----------+-----------+-");
		print_line(dwidth);
		printf("\n");
	}

	n = cmda_memory->nmzones;
	if (n != 0) {
		printf("\nMzones status:\n  %-*s | Allocated |  Requests | "
		    "Description\n  ", nwidth, "Mzone");
		print_line(nwidth);
		printf("-+-----------+-----------+-");
		print_line(dwidth);
		printf("\n");
		for (i = 0; i < n; ++i)
			printf("  %-*s | %9lu | %9u | %s\n", nwidth,
			    cmda_mzone[i].name,
			    (unsigned long)cmda_mzone[i].pools_size,
			    cmda_mzone[i].reqs, cmda_mzone[i].desc);
		printf("  ");
		print_line(nwidth);
		printf("-+-----------+-----------+-");
		print_line(dwidth);
		printf("\n");
	}

	n = cmda_memory->nmarrays;
	if (n != 0) {
		printf("\nMarrays status:\n  %-*s | Allocated |  Requests | "
		    "Description\n  ", nwidth, "Marray");
		print_line(nwidth);
		printf("-+-----------+-----------+-");
		print_line(dwidth);
		printf("\n");
		for (i = 0; i < n; ++i)
			printf("  %-*s | %9lu | %9u | %s\n", nwidth,
			    cmda_marray[i].name,
			    (unsigned long)(cmda_marray[i].arr_size +
			    cmda_marray[i].bitmap_size), cmda_marray[i].reqs,
			    cmda_marray[i].desc);
		printf("  ");
		print_line(nwidth);
		printf("-+-----------+-----------+-");
		print_line(dwidth);
		printf("\n");
	}

	n = cmda_memory->nmzones;
	if (n != 0) {
		printf("\nMzones details:\n  %-*s |  Isize |   Nused |   "
		    "Nfree\n  ", nwidth, "Mzone");
		print_line(nwidth);
		printf("-+--------+---------+--------\n");
		for (i = 0; i < n; ++i)
			printf("  %-*s | %6lu | %7u | %7u\n", nwidth,
			    cmda_mzone[i].name,
			    (unsigned long)cmda_mzone[i].isize,
			    cmda_mzone[i].nused, cmda_mzone[i].nfree);
		printf("  ");
		print_line(nwidth);
		printf("-+--------+---------+--------\n");
	}

	n = cmda_memory->nmarrays;
	if (n != 0) {
		printf("\nMarrays details:\n  %-*s |  Isize |   Nused |   "
		    "Nfree |  Arr_size |  Map_size\n  ", nwidth, "Marray");
		print_line(nwidth);
		printf("-+--------+---------+---------+-----------+"
		    "----------\n");
		for (i = 0; i < n; ++i)
			printf("  %-*s | %6lu | %7u | %7u | %9lu | %9lu\n",
			    nwidth, cmda_marray[i].name,
			    (unsigned long)cmda_marray[i].isize,
			    cmda_marray[i].nused, cmda_marray[i].nfree,
			    (unsigned long)cmda_marray[i].arr_size,
			    (unsigned long)cmda_marray[i].bitmap_size);
		printf("  ");
		print_line(nwidth);
		printf("-+--------+---------+---------+-----------+"
		    "----------\n");
	}

	return (0);
}

/*
 * Parser for "set" command.
 */
static int
ctl_set_parse(int argc, char *argv[])
{
	const char *arg;
	unsigned int qflags, qflagset;

	qflags = cmdq.flags;
	switch (argc - optind) {
	case 2:
		break;
#ifdef WITH_ANY_LIMITS
	case 4:
		if (qflags & (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD))
			break;
#endif
	default:
		logmsgx("wrong number of arguments for given command");
		return (-1);
	}

	for (; optind < argc;) {
		arg = argv[optind++];
		if (strcmp(arg, "counter") == 0)
			qflagset = CTL_CFLAG_VALUE2;
#ifdef WITH_LIMITS
		else if (strcmp(arg, "limit") == 0) {
			if (!(qflags & CTL_CFLAG_LIMIT))
				goto failed_wrong;
			qflagset = CTL_CFLAG_VALUE1;
		}
#endif
#ifdef WITH_THRESHOLDS
		else if (strcmp(arg, "threshold") == 0) {
			if (!(qflags & CTL_CFLAG_THRESHOLD))
				goto failed_wrong;
			qflagset = CTL_CFLAG_VALUE1;
		}
#endif
		else
			goto failed_wrong;

		if (qflags & qflagset)
			goto failed_duped;
		qflags |= qflagset;

		arg = argv[optind++];
		switch (*arg) {
		case '+':
			qflags |= qflagset == CTL_CFLAG_VALUE1 ?
			    CTL_CFLAG_VALUE1_INC : CTL_CFLAG_VALUE2_INC;
			++arg;
			break;
		case '-':
			qflags |= qflagset == CTL_CFLAG_VALUE1 ?
			    CTL_CFLAG_VALUE1_DEC : CTL_CFLAG_VALUE2_DEC;
			++arg;
			break;
		default:
			if (!(qflags &
			    (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD))) {
				logmsgx("it is only allowed to increase or "
				    "decrease statistics for rules");
				return (-1);
			}
		}
		if (parse_value(qflagset == CTL_CFLAG_VALUE1 ?
		    &cmdq_val.value1 : &cmdq_val.value2, arg) < 0)
			return (-1);
	}

#ifdef WITH_ANY_LIMITS
	if (qflags & (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD)) {
		cmda_aux = &cmda_set;
		cmda_aux_size = sizeof(cmda_set);
		cmda_aux_exp = 1;
	}
#endif

	cmdq.flags |= qflags;
	cmdq.size += sizeof(struct ctl_cmdq_val);
	return (0);

failed_duped:
	logmsgx("duplicated keyword \"%s\" in command", arg);
	return (-1);

failed_wrong:
	logmsgx("wrong keyword \"%s\" in command", arg);
	return (-1);
}

#ifdef WITH_ANY_LIMITS
/*
 * Check for "set" command.
 */
static int
ctl_set_check(void)
{
	unsigned int qflags;

	qflags = cmdq.flags;
	if (qflags & (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD)) {
		unsigned int value_type;

		value_type = (unsigned int)cmda_set.value_type;
# ifdef WITH_LIMITS
		if (qflags & CTL_CFLAG_LIMIT)
			printf("Limit was changed:\n  Limit:\t");
# endif
# ifdef WITH_THRESHOLDS
		if (qflags & CTL_CFLAG_THRESHOLD)
			printf("Threshold was changed:\n  Threshold:\t");
# endif
		print_value(0, &cmda_set.value1, value_type);
		printf("\n  Counter:\t");
		print_value(cmda_set.value2_sign, &cmda_set.value2, value_type);
		printf("\n");
	}
	return (0);
}
#else
# define ctl_set_check NULL
#endif /* WITH_ANY_LIMITS */

#ifdef WITH_AUTORULES
/*
 * Parser for "create" command.
 */
/* ARGSUSED */
static int
ctl_create_parse(int argc ATTR_UNUSED, char *argv[] ATTR_UNUSED)
{
	cmdq.flags &= ~CTL_CFLAG_RULE;
	return (0);
}
#endif

/*
 * Initialize signals handling.
 */
static int
sig_init(void)
{
	struct sigaction sigact;

	/* Ignore SIGPIPE. */
	sigact.sa_handler = SIG_IGN;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;
	if (sigaction(SIGPIPE, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg("sigaction(SIGPIPE)");
		return (-1);
	}
	return (0);
}

static const struct cmd_descr cmd_descr_tbl[] = {
	{ "status", CTL_CMD_STATUS, 0, -1,
	  ctl_status_parse, ctl_status_check
	},
	{ "memory", CTL_CMD_MEMORY, 0, 0,
	  ctl_memory_parse, ctl_memory_check
	},
	{ "dump", CTL_CMD_DUMP, 0, 0,
	  NULL, NULL
	},
	{ "freeze", CTL_CMD_FREEZE, 0, 0,
	  NULL, NULL
	},
#ifdef WITH_LIMITS
	{ "restart", CTL_CMD_RESTART, 0, OPTREQ_EXACT|OPTREQ_RULE|OPTREQ_LIMIT,
	  NULL, NULL
	},
	{ "expire", CTL_CMD_EXPIRE, 0, OPTREQ_EXACT|OPTREQ_RULE|OPTREQ_LIMIT,
	  NULL, NULL
	},
#endif
	{ "set", CTL_CMD_SET, -1, OPTREQ_RULE,
	  ctl_set_parse, ctl_set_check
	},
#ifdef WITH_AUTORULES
	{ "create", CTL_CMD_CREATE, 0, OPTREQ_EXACT|OPTREQ_RULE|OPTREQ_AUTORULE,
	  ctl_create_parse, NULL
	},
	{ "delete", CTL_CMD_DELETE, 0, OPTREQ_EXACT|OPTREQ_RULE,
	  NULL, NULL
	},
#endif
	{ NULL,	0, 0, 0,
	  NULL, NULL
	}
};

int
main(int argc, char *argv[])
{
#ifdef WITH_AUTORULES
	const char *autorule_name = NULL;
#endif
#ifdef WITH_LIMITS
	const char *limit_name = NULL;
#endif
#ifdef WITH_THRESHOLDS
	const char *threshold_name = NULL;
#endif
	const char *rule_name = NULL;
	const char *cmd_name;
	const struct cmd_descr *cmd_descr;
	unsigned int qflags;
	int opt, opts, reqs;

	/* Save the program name. */
	envprogname = strrchr(argv[0], '/');
	if (envprogname != NULL)
		++envprogname;
	else
		envprogname = argv[0];

	qflags = CTL_CFLAG_WAIT;
	cmdq_names_size = 0;
	opts = 0;
	opterr = 0;
	while ((opt = getopt(argc, argv, OPTSTRING)) != -1)
		switch (opt) {
		case ':':
			exit_errx("option requires an argument -- %c", optopt);
			/* NOTREACHED */
		case '?':
			exit_errx("illegal option -- %c", optopt);
			/* NOTREACHED */
#ifdef WITH_AUTORULES
		case 'a':
			if (autorule_name != NULL)
				goto failed_duped;
			autorule_name = optarg;
			if (validate_name(autorule_name) < 0)
				exit_errx("illegal %s name \"%s\"",
				    "autorule", autorule_name);
			cmdq_names_size += strlen(autorule_name) + 1;
			opts |= OPTREQ_AUTORULE;
			break;
#endif
		case 'h':
			usage();
			return (EXIT_SUCCESS);
#ifdef WITH_LIMITS
		case 'l':
			if (limit_name != NULL)
				goto failed_duped;
			limit_name = optarg;
			if (validate_name(limit_name) < 0)
				exit_errx("illegal %s name \"%s\"",
				    "limit", limit_name);
			cmdq_names_size += strlen(limit_name) + 1;
			opts |= OPTREQ_LIMIT;
			break;
#endif
		case 'n':
			qflags &= ~CTL_CFLAG_WAIT;
			wait_answer = 0;
			break;
		case 'r':
			if (rule_name != NULL)
				goto failed_duped;
			rule_name = optarg;
			if (validate_name(rule_name) < 0)
				exit_errx("illegal %s name \"%s\"",
				    "rule", rule_name);
			cmdq_names_size += strlen(rule_name) + 1;
			opts |= OPTREQ_RULE;
			break;
		case 's':
			ctl_socket_path = optarg;
			break;
#ifdef WITH_THRESHOLDS
		case 't':
			if (threshold_name != NULL)
				goto failed_duped;
			threshold_name = optarg;
			if (validate_name(threshold_name) < 0)
				exit_errx("illegal %s name \"%s\"",
				    "threshold", threshold_name);
			cmdq_names_size += strlen(threshold_name) + 1;
			opts |= OPTREQ_THRESHOLD;
			break;
#endif
		case 'v':
			show_version();
			return (EXIT_SUCCESS);
		case 'w':
			if (strto_u_int(&ctl_timeout, optarg) < 0) {
				logmsgx("wrong argument for the -w option");
				return (1);
			}
			break;
		default:
			exit_errx("unexpected option -- %c", optopt);
		}

	if (optind == argc)
		exit_errx("command is missed");

#if defined(WITH_LIMITS) && defined(WITH_THRESHOLDS)
	if (limit_name != NULL && threshold_name != NULL)
		exit_errx("-l and -t options cannot be used together");
#endif
#ifdef WITH_LIMITS
	if (limit_name != NULL && rule_name == NULL)
		exit_errx("-l option should be used with -r option");
#endif
#ifdef WITH_THRESHOLDS
	if (threshold_name != NULL && rule_name == NULL)
		exit_errx("-t option should be used with -r option");
#endif

	if (cmdq_names_size != 0) {
		cmdq_names = malloc(cmdq_names_size);
		if (cmdq_names == NULL) {
			logmsg("malloc");
			exit_errx("cannot allocate memory for control query");
		}
		*cmdq_names = '\0';
#ifdef WITH_AUTORULES
		if (autorule_name != NULL) {
			ctl_put_name(autorule_name);
			qflags |= CTL_CFLAG_AUTORULE;
		}
#endif
		if (rule_name != NULL) {
			ctl_put_name(rule_name);
			qflags |= CTL_CFLAG_RULE;
		}
#ifdef WITH_LIMITS
		if (limit_name != NULL) {
			ctl_put_name(limit_name);
			qflags |= CTL_CFLAG_LIMIT;
		}
#endif
#ifdef WITH_THRESHOLDS
		if (threshold_name != NULL) {
			ctl_put_name(threshold_name);
			qflags |= CTL_CFLAG_THRESHOLD;
		}
#endif
	} else
		cmdq_names = NULL;

	cmdq.flags = qflags;
	cmdq.size = cmdq_names_size;
	cmd_name = argv[optind++];

	for (cmd_descr = cmd_descr_tbl; cmd_descr->name != NULL; ++cmd_descr)
		if (strcmp(cmd_descr->name, cmd_name) == 0) {
			/* Validate options (not everything here). */
			if (cmd_descr->nargs >= 0 &&
			    cmd_descr->nargs != argc - optind)
				exit_errx("wrong number of arguments for "
				    "given command");
			reqs = cmd_descr->reqs;
			if (reqs >= 0 && ((reqs & OPTREQ_MASK) !=
			    ((reqs & OPTREQ_EXACT) ? opts : (reqs & opts))))
				exit_errx("wrong combination of options or "
				    "some option is absent for this command");

			/* Parse command. */
			cmda_aux = NULL;
			cmda_aux_exp = 0;
			if (cmd_descr->parse != NULL)
				if (cmd_descr->parse(argc, argv) < 0)
					exit_errx("cannot parse command line");

			if (sig_init() < 0)
				exit_errx("cannot initialize signal handlers");

			cmdq.cmd = cmd_descr->cmd;

			/* Send control command query. */
			if (send_cmd_query() < 0)
				exit_errx("cannot send control query");

			/* Receive control command answer. */
			if (wait_answer) {
				if (recv_cmd_answer() < 0)
					exit_errx("cannot receive answer");
				if (ctl_xxx_check() < 0)
					return (2);
				if (cmd_descr->check != NULL)
					if (cmd_descr->check() < 0)
						return (2);
			}

			free(cmdq_names);
			return (EXIT_SUCCESS);
		}

	exit_errx("unknown command \"%s\"", cmd_name);
	/* NOTREACHED */

failed_duped:
	exit_errx("duplicated option -- %c", optopt);
	/* NOTREACHED */
}
