/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_limits.c,v 1.2 2011/01/23 18:42:34 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

unsigned int	nstatlimits;		/* Number of static limits. */
unsigned int	ndynlimits;		/* Number of dynamic limits. */

unsigned int	nstatsublimits;		/* Number of static sublimits. */
unsigned int	ndynsublimits;		/* Number of dynamic sublimits. */

#ifdef WITH_LIMITS

signed char	global_debug_limit;	/* global { debug_limit } */
signed char	global_debug_limit_init;/* global { debug_limit_init } */
signed char	global_load_limit;	/* global { load_limit } */

ipa_mzone	*limit_mzone;		/* Mzone for all struct limit{}. */

#ifdef WITH_SUBLIMITS
ipa_mzone	*sublimit_mzone;	/* Mzone for all struct sublimit{}. */
#endif

#define set_limit_active(r, l)	mod_set_limit_active((r), (l), 1)

const char *const limit_event_msg[] = {
	"START",			/*  0 */
	"RESTART",			/*  1 */
	"RESTART_EXEC",			/*  2 */
	"REACH",			/*  3 */
	"REACH_EXEC",			/*  4 */
	"EXPIRE",			/*  5 */
	"EXPIRE_EXEC",			/*  6 */
	"UPDATED",			/*  7 */
	"STARTUP_IF_REACHED",		/*  8 */
	"STARTUP_IF_NOT_REACHED",	/*  9 */
	"SHUTDOWN_IF_REACHED",		/* 10 */
	"SHUTDOWN_IF_NOT_REACHED"	/* 11 */
};

#define event_msg limit_event_msg
#define EVENT(x) IPA_LIMIT_EVENT_ ## x

/*
 * Set limit active or inactive in modules it uses.
 */
int
mod_set_limit_active(const struct rule *rule, struct limit *limit, int active)
{
	if ((limit->lim_flags & LIMIT_FLAG_ACTIVE) ==
	    (active ? LIMIT_FLAG_ACTIVE : 0)) {
		logmsgx(IPA_LOG_ERR, "internal error: "
		    "mod_set_limit_active(%s, %s, %d): limit is already %s",
		    rule->name, limit->name, active, active_msg[active]);
		return (-1);
	}

	if (debug_worktime)
		logdbg("rule %s, limit %s: set limit %s",
		    rule->name, limit->name, active_msg[active]);

	if (active)
		LIMIT_SET_ACTIVE(limit);
	else
		LIMIT_SET_INACTIVE(limit);

	if (ac_set_limit_active(rule, limit, active) < 0)
		goto failed;
	if (db_set_limit_active(rule, limit, active) < 0)
		goto failed;

	return (0);

failed:
	logbt("mod_set_limit_active");
	return (-1);
}

/*
 * Set limit->event_sec from limit->event_tm.
 */
void
limit_set_event_sec(struct limit *limit)
{
	const ipa_tm *tm;

	tm = &limit->event_tm;
	if (cmp_ipa_tm(tm, &curdate) > 0) {
		if (tm->tm_year == curdate.tm_year &&
		    tm->tm_mon == curdate.tm_mon &&
		    tm->tm_mday == curdate.tm_mday)
			limit->event_sec = TIME_TO_SEC(tm);
		else
			limit->event_sec = EVENT_NOT_SCHEDULED;
	} else
		limit->event_sec = cursec;
}

/*
 * Add chunk to one limit, if positive counter overflows, then
 * return -1, since this means incorrect configuration.
 */
int
limit_add_chunk(const struct rule *rule, struct limit *limit,
    const uint64_t *chunk_ptr)
{
	uint64_t chunk;

	chunk = *chunk_ptr;
	if (limit->cnt_neg >= chunk)
		limit->cnt_neg -= chunk;
	else {
		chunk -= limit->cnt_neg;
		if (limit->cnt > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
			    "limit_add_chunk: positive counter overflowed",
			    rule->name, limit->name);
			logmsgx(IPA_LOG_ERR, "this means that something is "
			    "wrong in configuration");
			return (-1);
		}
		limit->cnt += chunk;
		limit->cnt_neg = 0;
	}
	return (0);
}

/*
 * Add chunk to every not reached and active rule's limit.
 */
int
limits_add_chunk(const struct rule *rule, const uint64_t *chunk_ptr)
{
	struct limit *limit;

	STAILQ_FOREACH(limit, &rule->limits, link)
		if (LIMIT_IS_NOTREACHED(limit) && LIMIT_IS_ACTIVE(limit))
			if (limit_add_chunk(rule, limit, chunk_ptr) < 0) {
				logbt("limits_add_chunk");
				return (-1);
			}
	return (0);
}

/*
 * Subtract chunk from one limit, if negative counter overflows, then
 * return -1, since this means incorrect configuration.
 */
int
limit_sub_chunk(const struct rule *rule, struct limit *limit,
    const uint64_t *chunk_ptr)
{
	uint64_t chunk;

	chunk = *chunk_ptr;
	if (limit->cnt >= chunk)
		limit->cnt -= chunk;
	else {
		chunk -= limit->cnt;
		if (limit->cnt_neg > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
			    "limit_sub_chunk: negative counter overflowed",
			    rule->name, limit->name);
			logmsgx(IPA_LOG_ERR, "this means that something is "
			    "wrong in configuration");
			return (-1);
		}
		limit->cnt_neg += chunk;
		limit->cnt = 0;
	}
	return (0);
}

/*
 * Subtract chunk from every not reached and active rule's limit.
 */
int
limits_sub_chunk(const struct rule *rule, const uint64_t *chunk_ptr)
{
	struct limit *limit;

	STAILQ_FOREACH(limit, &rule->limits, link)
		if (LIMIT_IS_NOTREACHED(limit) && LIMIT_IS_ACTIVE(limit))
			if (limit_sub_chunk(rule, limit, chunk_ptr) < 0) {
				logbt("limits_sub_chunk");
				return (-1);
			}
	return (0);
}

/*
 * Register a new state for the limit.
 */
static int
new_limit_state(const struct rule *rule, struct limit *limit)
{
	struct ipa_limit_state nstat;
	ipa_tm event_tm;
#ifdef WITH_SUBLIMITS
	struct sublimit *sublimit;
#endif

	if (rule->debug_limit_init || rule->debug_limit)
		logdbg("rule %s, limit %s: register new limit state",
		    rule->name, limit->name);

	event_tm = curdate;
	/* This function can be called at 24:00:00. */
	if (newday_flag)
		fix_240000(&event_tm);

	/* Set limit status as not reached. */
	LIMIT_SET_NOTREACHED(limit);

	nstat.lim = limit->lim;

	/* Flush limit's counter. */
	nstat.cnt = limit->cnt = 0;

	nstat.event_date_set = EVENT(START_SET);
	nstat.event_date[EVENT(START)] = curdate;

	if (limit->restart.restart.upto != TEXP_UPTO_NOTSET) {
		limit->event_tm = event_tm;
		ipa_tm_texp(&limit->event_tm, &limit->restart.restart);
		nstat.event_date_set |= EVENT(RESTART_SET);
		nstat.event_date[EVENT(RESTART)] = limit->event_tm;
		limit_set_event_sec(limit);
		if (rule->debug_limit || rule->debug_limit_init)
			logdbg("rule %s, limit %s: limit will be restarted "
			    "at %s", rule->name, limit->name,
			    tm_str(&limit->event_tm));
	} else
		limit->event_sec = EVENT_NOT_SCHEDULED;

#ifdef WITH_SUBLIMITS
	STAILQ_FOREACH(sublimit, &limit->sublimits, link)
		SUBLIMIT_SET_NOTREACHED(sublimit);
#endif

	if (db_set_limit_state(rule, limit, &nstat, 1) < 0) {
		logbt("new_limit_state");
		return (-1);
	}
	return (0);
}

/*
 * Check that a limit is not busy and make it busy.
 */
static int
make_limit_busy(const struct rule *rule, struct limit *limit)
{
	if (LIMIT_IS_BUSY(limit)) {
		logmsgx(IPA_LOG_ERR, "rule %s, limit %s: limit is busy, "
		    "wrong configuration of ictl commands",
		    rule->name, limit->name);
		return (-1);
	}
	LIMIT_SET_BUSY(limit);
	return (0);
}

/*
 * Restart a limit.
 */
int
restart_limit(const struct rule *rule, struct limit *limit)
{
	if (make_limit_busy(rule, limit) < 0)
		goto failed;

	if (rule->debug_limit)
		logdbg("rule %s, limit %s: restart_limit: restart limit",
		    rule->name, limit->name);

	if (limit->restart.cmds.has_cmd) {
		/* restart { exec } */
		if (run_cmds(rule, &limit->wpid, &limit->restart.cmds,
		    "rule %s { limit %s { restart {}}}", rule->name,
		    limit->name) < 0)
			goto failed;
		if (db_limit_event(rule, limit, EVENT(RESTART_EXEC),
		    &curdate) < 0)
			goto failed;
	}

	if (ac_limit_event(rule, limit, EVENT(RESTART)) < 0)
		goto failed;

	if (new_limit_state(rule, limit) < 0)
		goto failed;

	LIMIT_SET_UNBUSY(limit);
	return (0);

failed:
	logbt("restart_limit");
	return (-1);
}

/*
 * Expire a limit.
 */
int
expire_limit(const struct rule *rule, struct limit *limit)
{
	if (make_limit_busy(rule, limit) < 0)
		goto failed;

	if (rule->debug_limit)
		logdbg("rule %s, limit %s: limit expired",
		    rule->name, limit->name);

	if (limit->expire.cmds.has_cmd) {
		/* expire { exec } */
		if (run_cmds(rule, &limit->wpid, &limit->expire.cmds,
		    "rule %s { limit %s { expire {}}}", rule->name,
		    limit->name) < 0)
			goto failed;
		if (db_limit_event(rule, limit, EVENT(EXPIRE_EXEC),
		    &curdate) < 0)
			goto failed;
	}

	if (ac_limit_event(rule, limit, EVENT(EXPIRE)) < 0)
		goto failed;

	if (new_limit_state(rule, limit) < 0)
		goto failed;

	LIMIT_SET_UNBUSY(limit);

	if (limit->lim == 0) {
		/* Such limit is always reached. */
		if (reach_limit(rule, limit) < 0)
			goto failed;
	}

	return (0);

failed:
	logbt("expire_limit");
	return (-1);
}

/*
 * A limit is reached.
 */
int
reach_limit(const struct rule *rule, struct limit *limit)
{
	ipa_tm event_tm;

	if (make_limit_busy(rule, limit) < 0)
		goto failed;

	/* This function can be called at 24:00:00. */
	event_tm = curdate;
	if (newday_flag)
		fix_240000(&event_tm);

	if (rule->debug_limit)
		logdbg("rule %s, limit %s: limit reached, cnt %"PRIu64,
		    rule->name, limit->name, limit->cnt);

	/* Set limit status as reached. */
	LIMIT_SET_REACHED(limit);

	if (db_limit_event(rule, limit, EVENT(REACH), &curdate) < 0)
		goto failed;
	if (limit->reach.has_cmd) {
		/* reach { exec } */
		if (run_cmds(rule, &limit->wpid, &limit->reach,
		    "rule %s { limit %s { reach {}}}", rule->name,
		    limit->name) < 0)
			goto failed;
		if (db_limit_event(rule, limit, EVENT(REACH_EXEC),
		    &curdate) < 0)
			goto failed;
	}
	if (ac_limit_event(rule, limit, EVENT(REACH)) < 0)
		goto failed;

	LIMIT_SET_UNBUSY(limit);

	if (limit->expire.expire.upto != TEXP_UPTO_NOTSET) {
		/* limit { expire } */
		limit->event_tm = event_tm;
		ipa_tm_texp(&limit->event_tm, &limit->expire.expire);
		limit_set_event_sec(limit);
		if (rule->debug_limit)
			logdbg("rule %s, limit %s: limit will expire at %s",
			    rule->name, limit->name, tm_str(&limit->event_tm));
		if (db_limit_event(rule, limit, EVENT(EXPIRE),
		    &limit->event_tm) < 0)
			goto failed;
		if (limit->event_sec == cursec) {
			/* expire == 0s */
			if (expire_limit(rule, limit) < 0)
				goto failed;
		}
	} else
		limit->event_sec = EVENT_NOT_SCHEDULED;

	return (0);

failed:
	logbt("reach_limit");
	return (-1);
}

#ifdef WITH_SUBLIMITS
struct sublimit *
sublimit_by_name(const struct limit *limit, const char *name)
{
	struct sublimit *sublimit;

	STAILQ_FOREACH(sublimit, &limit->sublimits, link)
		if (strcmp(sublimit->name, name) == 0)
			break;
	return (sublimit);
}

/*
 * A sublimit is reached.
 */
int
reach_sublimit(const struct rule *rule, const struct limit *limit,
    struct sublimit *sublimit)
{
	if (rule->debug_limit)
		logdbg("rule %s, limit %s, sublimit %s: sublimit reached, "
		    "cnt %"PRIu64, rule->name, limit->name, sublimit->name,
		    limit->cnt);

	SUBLIMIT_SET_REACHED(sublimit);
	if (sublimit->reach.has_cmd) {
		/* reach { exec } */
		if (run_cmds(rule, &sublimit->wpid, &sublimit->reach,
		    "rule %s { limit %s { sublimit %s { reach {}}}}",
		    rule->name, limit->name, sublimit->name) < 0) {
			logbt("reach_sublimit");
			return (-1);
		}
	}

	return (0);
}

/*
 * Check for limit's sublimits events.
 */
static int
check_sublimits_events(const struct rule *rule, const struct limit *limit)
{
	struct sublimit	*sublimit;

	STAILQ_FOREACH(sublimit, &limit->sublimits, link) {
		if (SUBLIMIT_IS_REACHED(sublimit))
			continue;
		/* Sublimit is not reached. */
		if (limit->cnt >= sublimit->lim) {
			/* Sublimit has just been reached. */
			if (reach_sublimit(rule, limit, sublimit) < 0) {
				logbt("check_sublimits_events");
				return (-1);
			}
		}
	}
	return (0);
}
#endif /* WITH_SUBLIMITS */

/*
 * Check for rule's limits events.
 */
int
check_limits(const struct rule *rule, unsigned int *check_sec_ptr)
{
	const struct worktime *wt;
	struct limit *limit;
	unsigned int check_sec, d;

	check_sec = EVENT_NOT_SCHEDULED;

	STAILQ_FOREACH(limit, &rule->limits, link) {
		wt = limit->worktime;
		if (LIMIT_IS_INACTIVE(limit)) {
			/* Limit is inactive. */
			if (wt->active_sec <= cursec) {
				/* It's time to make limit active. */
				if (set_limit_active(rule, limit) < 0)
					goto failed;
			} else {
				if (check_sec > wt->active_sec)
					check_sec = wt->active_sec;
				continue; /* do not check any time events. */
			}
		}

		/* Here limit is active. */
		if (LIMIT_IS_REACHED(limit)) {
			/* Limit has been reached already. */
			if (limit->event_sec <= cursec) {
				/* { expire } and it's time to expire limit. */
				d = ipa_tm_diff(&curdate, &limit->event_tm);
				if (d > sensitive_time)
					logmsgx(IPA_LOG_WARNING, "rule %s, "
					    "limit %s: expire limit too late: "
					    "delta %s is greater than "
					    "\"sensitive_time\" %u seconds",
					    rule->name, limit->name,
					    time_str(d), sensitive_time);
				if (expire_limit(rule, limit) < 0)
					goto failed;
			}
		} else {
			/* Limit is not reached. */
			if (db_update_limit(rule, limit) < 0)
				goto failed;
#ifdef WITH_SUBLIMITS
			if (check_sublimits_events(rule, limit) < 0)
				goto failed;
#endif
			if (limit->cnt >= limit->lim) {
				/* Limit has just been reached. */
				if (reach_limit(rule, limit) < 0)
					goto failed;
			} else if (limit->event_sec <= cursec) {
				/* { restart } and it's time to restart. */
				d = ipa_tm_diff(&curdate, &limit->event_tm);
				if (d > sensitive_time)
					logmsgx(IPA_LOG_WARNING, "rule %s, "
					    "limit %s: restart limit too late: "
					    "delta %s is greater than "
					    "\"sensitive_time\" %u seconds",
					    rule->name, limit->name,
					    time_str(d), sensitive_time);
				if (restart_limit(rule, limit) < 0)
					goto failed;
			}
		}

		if (WT_IS_INACTIVE(wt)) {
			/* Limit became inactive. */
			if (!newday_flag) {
				if (set_limit_inactive(rule, limit) < 0)
					goto failed;
				if (check_sec > wt->active_sec)
					check_sec = wt->active_sec;
			}
		} else {
			/* Limit is still active. */
			if (check_sec > wt->inactive_sec)
				check_sec = wt->inactive_sec;
			if (check_sec > limit->event_sec)
				check_sec = limit->event_sec;
		}
	}

	*check_sec_ptr = check_sec;
	return (0);

failed:
	logbt("check_limits");
	return (-1);
}

/*
 * Initialize one limit.
 */
static int
init_limit(const struct rule *rule, struct limit *limit)
{
	struct ipa_limit_state ostat, nstat;
	ipa_tm tm;
	const char *rule_name, *limit_name;
	unsigned int bit, i;
	signed char debug_limit_init;
	char error;

	/*
	 * If init_limit() is called second, third... time,
	 * then limit can be inactive.
	 */
	if (LIMIT_IS_INACTIVE(limit))
		if (set_limit_active(rule, limit) < 0)
			goto failed;

	rule_name = rule->name;
	limit_name = limit->name;
	limit->cnt_neg = 0;
	limit->wpid.debug_exec = rule->debug_exec;
	debug_limit_init = rule->debug_limit_init;

	if (STAILQ_EMPTY(limit->db_list)) {
		/* "null" database is used. */
		if (LIMIT_IS_INITED(limit)) {
			/* Imitate that old state is known. */
			ostat.lim = limit->lim;
			ostat.cnt = limit->cnt;
			ostat.event_date_set = limit->event_date_set;
			memcpy(ostat.event_date, limit->event_date,
			    sizeof(ostat.event_date));
			if (debug_limit_init)
				logdbg("rule %s, limit %s: init_limit: "
				    "continue to use previous limit state",
				    rule_name, limit_name);
		} else {
			if (debug_limit_init)
				logdbg("rule %s, limit %s: init_limit: "
				    "create limit with no database",
				    rule_name, limit_name);
			goto new_state;
		}
	} else {
		const char *db_name;
		int rv;

		/* Get limit state from databases. */
		rv = db_get_limit_state(rule, limit, &ostat, &db_name);
		if (rv < 0)
			goto failed;
		if (db_name == NULL) {
			logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
			    "init_limit: no database used for this limit "
			    "supports db_get_limit_state function",
			    rule_name, limit_name);
			goto new_state;
		}
		if (rv > 0) {
			if (debug_limit_init)
				logdbg("rule %s, limit %s: init_limit: "
				    "got state from %s database",
				    rule_name, limit_name, db_name);
		} else {
			if (debug_limit_init)
				logdbg("rule %s, limit %s: init_limit: "
				    "database %s does not have limit state, "
				    "(a new limit)", rule_name, limit_name,
				    db_name);
			goto new_state;
		}
	}

	/* Check and initialize "limit" parameter. */
	if (debug_limit_init && limit->lim != ostat.lim) {
		logdbg("rule %s, limit %s: init_limit: configuration "
		    "limit %"PRIu64", database limit %"PRIu64" (will use %s "
		    "value)", rule_name, limit_name, limit->lim, ostat.lim,
		    limit->load_limit ? "database" : "configuration");
	}
	if (limit->load_limit) {
		if (ostat.lim == 0 &&
		    limit->expire.expire.upto == TEXP_UPTO_SIMPLE &&
		    limit->expire.expire.seconds == 0) {
 			logmsgx(IPA_LOG_ERR, "rule %s, limit %s: init_limit: "
			    "loaded \"limit\" parameter is equal to zero "
			    "and \"expire\" parameter is equal to 0s",
			    rule_name, limit_name);
			return (-1);
		}
		limit->lim = ostat.lim;
	}

	/* START and UPDATED events must be present. */
	if (!(ostat.event_date_set & EVENT(START_SET))) {
		logmsgx(IPA_LOG_ERR, "rule %s, limit %s: init_limit: "
		    "EVENT_START is absent", rule_name, limit_name);
		goto new_state_error;
	}
	if (!(ostat.event_date_set & EVENT(UPDATED_SET))) {
		logmsgx(IPA_LOG_ERR, "rule %s, limit %s: init_limit: "
		    "EVENT_UPDATED is absent", rule_name, limit_name);
		goto new_state_error;
	}

	/* RESTART_EXEC and EXPIRE_EXEC mean restarted or expired limit. */
	if (ostat.event_date_set & EVENT(RESTART_EXEC_SET)) {
		logmsgx(IPA_LOG_ERR, "rule %s, limit %s: init_limit: "
		    "limit has EVENT_RESTART_EXEC, this is wrong",
		    rule_name, limit_name);
		goto new_state_error;
	}
	if (ostat.event_date_set & EVENT(EXPIRE_EXEC_SET)) {
		logmsgx(IPA_LOG_ERR, "rule %s, limit %s: init_limit: "
		    "limit has EVENT_EXPIRE_EXEC, this is wrong",
		    rule_name, limit_name);
		goto new_state_error;
	}

	/* Set limit status as not reached. */
	LIMIT_SET_NOTREACHED(limit);

	/* Check START, REACH and UPDATED events. */
	error = 0;
	for (i = 0, bit = 1; i < IPA_LIMIT_EVENT_NUM; bit <<= 1, ++i) {
		if ((ostat.event_date_set & bit) && (i == EVENT(START) ||
		    i == EVENT(REACH) || i == EVENT(UPDATED))) {
			/* These events could happen at 24:00:00. */
			tm = ostat.event_date[i];
			fix_240000(&tm);
			if (check_ipa_tm(&tm) < 0) {
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: wrong value of EVENT_%s: %s",
				    rule_name, limit_name, event_msg[i],
				    tm_str(&tm));
				goto new_state_error;
			}
			/* Check whether time goes back. */
			if (cmp_ipa_tm(&tm, &curdate) > 0) {
				error = 1;
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: EVENT_%s %s is greater than "
				    "current time", rule_name, limit_name,
				    event_msg[i], tm_str(&tm));
			}
		}
	}

	/*
	 * If not reached limit has START event greater than
	 * current time, then change START event.
	 */
	if (error && !(ostat.event_date_set & EVENT(REACH_SET)) &&
	    cmp_ipa_tm(&ostat.event_date[EVENT(START]), &curdate) > 0) {
		const char *cp;

		cp = tm_str(&ostat.event_date[EVENT(START)]);
		ostat.event_date[EVENT(START)] = curdate;
		logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: init_limit: "
		    "start time changed %s -> %s", rule_name, limit_name, cp,
		    tm_str2(&ostat.event_date[EVENT(START)]));
	}

	/* Set start time of limit. */
	nstat.event_date_set = EVENT(START_SET);
	nstat.event_date[EVENT(START)] = ostat.event_date[EVENT(START)];

	if (ostat.cnt < ostat.lim ||
	    !(ostat.event_date_set & EVENT(REACH_SET))) {
		/*
		 * Limit was not reached or was not mark as reached
		 * with old "limit" parameter.
		 */
		nstat.lim = limit->lim;
		if (ostat.cnt >= ostat.lim)
			logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
			    "init_limit: limit was reached in old state, "
			    "but EVENT_REACH was not registered",
			    rule_name, limit_name);

		if (limit->restart.restart.upto != TEXP_UPTO_NOTSET) {
			/* The "restart" parameter is used. */
			limit->event_tm = ostat.event_date[EVENT(START)];
			fix_240000(&limit->event_tm);
			if (set_wday(&limit->event_tm) < 0) {
				logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
				    "init_limit: set_wday failed", rule_name,
				    limit_name);
				return (-1);
			}
			ipa_tm_texp(&limit->event_tm, &limit->restart.restart);
			nstat.event_date_set |= EVENT(RESTART_SET);
			nstat.event_date[EVENT(RESTART)] = limit->event_tm;
			if (!(ostat.event_date_set & EVENT(RESTART_SET)))
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: limit has not EVENT_RESTART "
				    "in database, but has \"restart\" "
				    "parameter", rule_name, limit_name);
			else if (cmp_ipa_tm(&limit->event_tm,
			    &ostat.event_date[EVENT(RESTART)]) != 0)
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: restart time changed %s -> %s",
				    rule_name, limit_name,
				    tm_str(&ostat.event_date[EVENT(RESTART)]),
				    tm_str2(&limit->event_tm));
			if (rule->debug_limit || debug_limit_init)
				logdbg("rule %s, limit %s: limit will be "
				    "restarted at %s", rule_name, limit_name,
				    tm_str(&limit->event_tm));
		} else {
			if (ostat.event_date_set & EVENT(RESTART_SET))
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: limit has EVENT_RESTART in "
				    "database, but limit has not \"restart\" "
				    "parameter: forgot about EVENT_RESTART",
				    rule_name, limit_name);
			limit->event_sec = EVENT_NOT_SCHEDULED;
		}
	} else {
		/*
		 * Limit was reached and marked as reached
		 * with old "limit" parameter.
		 */
		if (ostat.event_date_set & EVENT(RESTART_SET)) {
			/*
			 * If limit had restart time in previous state,
			 * then simply copy it.
			 */
			nstat.event_date_set |= EVENT(RESTART_SET);
			nstat.event_date[EVENT(RESTART)] =
			    ostat.event_date[EVENT(RESTART)];
		}
		nstat.lim = ostat.lim;
		LIMIT_SET_REACHED(limit);
		nstat.event_date_set |= EVENT(REACH_SET);
		nstat.event_date[EVENT(REACH)] = ostat.event_date[EVENT(REACH)];
		if (ostat.event_date_set & EVENT(REACH_EXEC_SET)) {
			/*
			 * If limit run any commands, then save this
			 * in the new state.
			 */
			nstat.event_date_set |= EVENT(REACH_EXEC_SET);
			nstat.event_date[EVENT(REACH_EXEC)] =
			    ostat.event_date[EVENT(REACH_EXEC)];
		}
		if (limit->expire.expire.upto != TEXP_UPTO_NOTSET) {
			nstat.event_date_set |= EVENT(EXPIRE_SET);
			limit->event_tm = nstat.event_date[EVENT(REACH)];
			fix_240000(&limit->event_tm);
			if (set_wday(&limit->event_tm) < 0) {
				logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
				    "init_limit: set_wday failed", rule_name,
				    limit_name);
				return (-1);
			}
			ipa_tm_texp(&limit->event_tm, &limit->expire.expire);
			nstat.event_date[EVENT(EXPIRE)] = limit->event_tm;
			if (!(ostat.event_date_set & EVENT(EXPIRE_SET)))
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: limit has not EVENT_EXPIRE in "
				    "database, but has \"expire\" parameter",
				    rule_name, limit_name);
			else if (cmp_ipa_tm(&limit->event_tm,
			    &ostat.event_date[EVENT(EXPIRE])) != 0)
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: expire time is changed "
				    "%s -> %s", rule_name, limit_name,
				    tm_str(&ostat.event_date[EVENT(EXPIRE)]),
				    tm_str2(&limit->event_tm));
			if (rule->debug_limit || debug_limit_init)
				logdbg("rule %s, limit %s: reached limit "
				    "will expire at %s", rule_name, limit_name,
				    tm_str(&limit->event_tm));
		} else {
			if (ostat.event_date_set & EVENT(EXPIRE_SET))
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: "
				    "init_limit: limit has EVENT_EXPIRE in "
				    "database, but limit has not \"expire\" "
				    "section in configuration: forgot about "
				    "EVENT_EXPIRE", rule_name, limit_name);
			limit->event_sec = EVENT_NOT_SCHEDULED;
		}
	}

	/* Copy counter value from the old state. */
	limit->cnt = nstat.cnt = ostat.cnt;

	if (debug_limit_init) {
		logdbg("rule %s, limit %s: init_limit: lim %"PRIu64", "
		    "cnt %"PRIu64, rule_name, limit_name, limit->lim,
		    limit->cnt);
		logdbg("rule %s, limit %s: init_limit: set limit state",
		    rule_name, limit_name);
	}

	if (db_set_limit_state(rule, limit, &nstat, 0) < 0)
		goto failed;

	LIMIT_SET_INITED(limit);
	return (0);

new_state_error:
	logmsgx(IPA_LOG_WARNING, "rule %s, limit %s: init_limit: set new "
	    "limit state, correcting errors found in database",
	    rule_name, limit_name);

new_state:
	if (debug_limit_init)
		logdbg("rule %s, limit %s: init_limit: set new limit state",
		    rule_name, limit_name);
	if (new_limit_state(rule, limit) < 0)
		goto failed;
	LIMIT_SET_INITED(limit);
	return (0);

failed:
	logbt("init_limit");
	return (-1);
}

#ifdef WITH_SUBLIMITS
/*
 * Initialize all sublimits for one limit.
 */
static void
init_sublimits(const struct rule *rule, const struct limit *limit)
{
	struct sublimit *sublimit;

	STAILQ_FOREACH(sublimit, &limit->sublimits, link) {
		sublimit->wpid.debug_exec = rule->debug_exec;
		if (sublimit->lim_pc != 0)
			sublimit->lim = uint64_per_cent(&limit->lim,
			    sublimit->lim_pc);
		if (sublimit->lim > limit->lim)
			logmsgx(IPA_LOG_WARNING, "rule %s, limit %s, "
			    "sublimit %s: init_sublimits: sublimit is greater "
			    "than limit %"PRIu64, rule->name, limit->name,
			    sublimit->name, limit->lim);
		/*
		 * If limit is reached, then "limit" parameter is
		 * not used and sublimit is always reached.
		 */
		if (LIMIT_IS_REACHED(limit) || limit->cnt >= sublimit->lim)
			SUBLIMIT_SET_REACHED(sublimit);
		else
			SUBLIMIT_SET_NOTREACHED(sublimit);
	}
}

void
sublimit_init_cmds(struct sublimit *sublimit)
{
	cmds_init(&sublimit->reach);
	cmds_limit_init(&sublimit->rc[RC_STARTUP]);
	cmds_limit_init(&sublimit->rc[RC_SHUTDOWN]);
}
#endif /* WITH_SUBLIMITS */

/*
 * Initialize all limits in one rule.
 */
int
init_limits(const struct rule *rule)
{
	struct limit *limit;

	STAILQ_FOREACH(limit, &rule->limits, link) {
		if (init_limit(rule, limit) < 0) {
			logbt("init_limits");
			return (-1);
		}
#ifdef WITH_SUBLIMITS
		init_sublimits(rule, limit);
#endif
	}
	return (0);
}

/*
 * Return pointer to limit with the given name.
 */
struct limit *
limit_by_name(const struct rule *rule, const char *name)
{
	struct limit *limit;

	STAILQ_FOREACH(limit, &rule->limits, link)
		if (strcmp(name, limit->name) == 0)
			break;
	return (limit);
}

/*
 * This function is called from newday().
 */
int
limits_newday(struct rule *rule)
{
	struct limit *limit;
	unsigned int check_sec;

	check_sec = EVENT_NOT_SCHEDULED;

	STAILQ_FOREACH(limit, &rule->limits, link) {
		if (LIMIT_IS_REACHED(limit)) {
			/* Is reached. */
			if (limit->expire.expire.upto != TEXP_UPTO_NOTSET)
				/* limit { expire } */
				limit_set_event_sec(limit);
		} else {
			/* Is not reached. */
			if (limit->restart.restart.upto != TEXP_UPTO_NOTSET)
				/* limit { restart } */
				limit_set_event_sec(limit);
		}
		if (WT_IS_INACTIVE(limit->worktime)) {
			if (LIMIT_IS_ACTIVE(limit))
				if (set_limit_inactive(rule, limit) < 0)
					goto failed;
			if (check_sec > limit->worktime->active_sec)
				check_sec = limit->worktime->active_sec;
		} else {
			if (LIMIT_IS_INACTIVE(limit))
				if (set_limit_active(rule, limit) < 0)
					goto failed;
			if (check_sec > limit->worktime->inactive_sec)
				check_sec = limit->worktime->inactive_sec;
			if (check_sec > limit->event_sec)
				check_sec = limit->event_sec;
		}
		if (rule->check_sec > check_sec)
			rule->check_sec = check_sec;
	}
	return (0);

failed:
	logbt("limits_newday");
	return (-1);
}

#ifdef WITH_SUBLIMITS
/*
 * Copy sublimits from limit2 to limit1.
 */
static int
copy_sublimits(struct rule *rule, struct limit *ldst, const struct limit *lsrc)
{
	const struct sublimit *ssrc;
	struct sublimit *sdst;
	unsigned int count;
	int rv;

	count = 0;
	STAILQ_FOREACH(ssrc, &lsrc->sublimits, link) {
		sdst = mzone_alloc(sublimit_mzone);
		if (sdst == NULL) {
			xlogmsgx(IPA_LOG_ERR, "copy_sublimits: "
			    "mzone_alloc failed");
			return (-1);
		}

		/* Copy settings from source sublimit. */
		*sdst = *ssrc;

		/*
		 * Initialize fields which are pointers to memory,
		 * which cannot be shared.
		 */
		sublimit_init_cmds(sdst);
		sdst->limit = ldst;
		sdst->wpid.u.sublimit = sdst;

		/* Link just allocated sublimit to limit. */
		STAILQ_INSERT_TAIL(&ldst->sublimits, sdst, link);

		/*
		 * If something goes wrong with memory allocation,
		 * previous settings will allow to deinitialize current rule.
		 */
		rv =
		    cmds_copy(rule, &sdst->reach, &ssrc->reach) +
		    cmds_limit_copy(rule, &sdst->rc[0], &sdst->rc[0]) +
		    cmds_limit_copy(rule, &sdst->rc[1], &ssrc->rc[1]);
		if (rv < 0)
			return (-1);
		++count;
	}
	if (RULE_IS_DYNAMIC(rule))
		ndynsublimits += count;
	else
		nstatsublimits += count;
	return (0);
}
#endif

/*
 * Copy all limits from the given limits list to rule.
 */
int
copy_limits(struct rule *rule, const struct limits_list *list)
{
	const struct limit *lsrc;
	struct limit *ldst;
	unsigned int count;
	int rv;

	count = 0;
	STAILQ_FOREACH(lsrc, list, link) {
		ldst = mzone_alloc(limit_mzone);
		if (ldst == NULL) {
			xlogmsgx(IPA_LOG_ERR, "copy_limits: "
			    "mzone_alloc failed");
			return (-1);
		}

		/* Copy settings from source limit. */
		*ldst = *lsrc;

		/*
		 * Initialize fields, which are pointers to memory,
		 * which cannot be shared.
		 */
		limit_init_cmds(ldst);
		ldst->rule = rule;
		ldst->wpid.u.limit = ldst;
#ifdef WITH_SUBLIMITS
		STAILQ_INIT(&ldst->sublimits);
#endif

		/* Link just allocated limit to rule. */
		STAILQ_INSERT_TAIL(&rule->limits, ldst, link);

		/*
		 * If something goes wrong with memory allocation,
		 * previous settings will allow to deinitialize current rule.
		 */
		rv =
		    cmds_copy(rule, &ldst->restart.cmds, &lsrc->restart.cmds) +
		    cmds_copy(rule, &ldst->reach, &lsrc->reach) +
		    cmds_copy(rule, &ldst->expire.cmds, &lsrc->expire.cmds) +
		    cmds_limit_copy(rule, &ldst->rc[0], &lsrc->rc[0]) +
		    cmds_limit_copy(rule, &ldst->rc[1], &lsrc->rc[1]);
		if (rv < 0)
			return (-1);
#ifdef WITH_SUBLIMITS
		if (copy_sublimits(rule, ldst, lsrc) < 0)
			return (-1);
#endif
		++count;
	}
	if (RULE_IS_DYNAMIC(rule))
		ndynlimits += count;
	else
		nstatlimits += count;
	return (0);
}

#ifdef WITH_SUBLIMITS
static void
free_sublimits(unsigned int rule_flags, struct sublimits_list *list,
    int dyn_flag)
{
	struct sublimit *sublimit, *sublimit_next;
	unsigned int count;

	count = 0;
	STAILQ_FOREACH_SAFE(sublimit, list, link, sublimit_next) {
		if (rule_flags & RULE_FLAG_FREE_LIMITS)
			mem_free(sublimit->name, m_anon);
		cmds_free(&sublimit->reach);
		cmds_limit_free(&sublimit->rc[RC_STARTUP]);
		cmds_limit_free(&sublimit->rc[RC_SHUTDOWN]);
		mzone_free(sublimit_mzone, sublimit);
		++count;
	}
	if (dyn_flag)
		ndynsublimits -= count;
}
#endif

/*
 * Release memory used by a list of limits, rule_flags determines
 * which part of limit{} structure to free.
 */
void
free_limits(unsigned int rule_flags, struct limits_list *limits,
    int dyn_flag)
{
	struct limit *limit, *limit_next;
	unsigned int count;

	count = 0;
	STAILQ_FOREACH_SAFE(limit, limits, link, limit_next) {
		if (rule_flags & RULE_FLAG_FREE_LIMITS) {
			mem_free(limit->name, m_anon);
			mem_free(limit->info, m_parser);
		}
		cmds_free(&limit->restart.cmds);
		cmds_free(&limit->reach);
		cmds_free(&limit->expire.cmds);
		cmds_limit_free(&limit->rc[RC_STARTUP]);
		cmds_limit_free(&limit->rc[RC_SHUTDOWN]);
#ifdef WITH_SUBLIMITS
		free_sublimits(rule_flags, &limit->sublimits, dyn_flag);
#endif
		mzone_free(limit_mzone, limit);
		++count;
	}
	if (dyn_flag)
		ndynlimits -= count;
}

void
limit_init_cmds(struct limit *limit)
{
	cmds_init(&limit->restart.cmds);
	cmds_init(&limit->reach);
	cmds_init(&limit->expire.cmds);
	cmds_limit_init(&limit->rc[RC_STARTUP]);
	cmds_limit_init(&limit->rc[RC_SHUTDOWN]);
}

/*
 * Set default settings and inherit settings from global{} in limit{}.
 */
void
limit_inherit(struct limit *limit)
{
#ifdef WITH_SUBLIMITS
	struct sublimit *sublimit;
#endif

	if (limit->load_limit < 0)
		limit->load_limit = global_load_limit;
	if (limit->restart.cmds.sync < 0)
		limit->restart.cmds.sync = 0;
	if (limit->reach.sync < 0)
		limit->reach.sync = 0;
	if (limit->expire.cmds.sync < 0)
		limit->expire.cmds.sync = 0;
	cmds_limit_set_sync(&limit->rc[RC_STARTUP]);
	cmds_limit_set_sync(&limit->rc[RC_SHUTDOWN]);
#ifdef WITH_SUBLIMITS
	STAILQ_FOREACH(sublimit, &limit->sublimits, link) {
		if (sublimit->reach.sync < 0)
			sublimit->reach.sync = 0;
		cmds_limit_set_sync(&sublimit->rc[RC_STARTUP]);
		cmds_limit_set_sync(&sublimit->rc[RC_SHUTDOWN]);
	}
#endif
}
#endif /* WITH_LIMITS */
