//Copyright (C) 2000 - 2003 Thomas Schank 
// 
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. 

#include "garmingps.h"
#include "packet.h"
#include "constants.h"
#include "records_type.h"
#include "d210_rte_link_type.h"

#include <iomanip>
#include <iostream>



namespace gpspoint2 {
using namespace std;
using namespace gpspoint2;

extern bool quiet;
extern bool want_to_die;


void GarminGPS::uploadRoutes(Routelist &rtl)
{
   if(want_to_die) return;
   if(!routeHeaderTransfer || !waypointTransfer) return;

   int n=0; // n = number of packets to be send
   int i=0; // i packets already sent

   n+=rtl.sizeRoutes(); // n Routes
   for(int i=0;i<rtl.sizeRoutes();i++)
      if(sendRouteLinkData) n+=rtl.sizeRoutepoints(i)*2; 
      else n+=rtl.sizeRoutepoints(i);

   Records_Type records; // send records
   records.n = n ;
   Packet recordsPacket = records.makePacket();
   sendPacket(recordsPacket);
   if(!quiet)cerr << "uploading " << n << " packets routedata: " ;
   printFortschritt(0,n);

   for(int r=0;r<rtl.sizeRoutes();r++) {
      garmin_routeheader->clear();
      *garmin_routeheader << rtl.header(r);
      Packet th = garmin_routeheader->makePacket();
      sendPacket(th);i++; printFortschritt(i,n); 
      for(int p=0; p<rtl.sizeRoutepoints(r);p++) {
         garmin_waypoint->clear();
         *garmin_waypoint << rtl.routepoint(r,p);
         Packet tp = garmin_waypoint->makePacket();
         sendPacket(tp);i++; printFortschritt(i,n); 
         if(sendRouteLinkData){
         	D210_Rte_Link_Type routelink;
				Packet rl = routelink.makePacket();
	         sendPacket(rl);i++; printFortschritt(i,n); } } }

   Packet transComplete;   // send track transfer complete
   transComplete.pid = Pid_Xfer_Cmplt;
   transComplete.data[0]=Cmnd_Transfer_Trk;
   transComplete.length=2;
   sendPacket(transComplete);
}


void GarminGPS::uploadTracks(Tracklist &tkl)
{
   int n=0; // n = number of packets to be send
   int i=0; // i packets already sent

   if(want_to_die) return;
   if(!trackTransfer)return;

   if(trackHeader) n+=tkl.sizeTracks(); // n tracks
   for(int i=0;i<tkl.sizeTracks();i++) n+=tkl.sizeTrackpoints(i); 

   Records_Type records; // send records
   records.n = n ;
   Packet recordsPacket = records.makePacket();
   sendPacket(recordsPacket);
   if(!quiet)cerr << "uploading " << n << " packets trackdata: " ;
   printFortschritt(0,n);

   for(int t=0;t<tkl.sizeTracks();t++) {
      if(trackHeader) {
         garmin_trackheader->clear();
         *garmin_trackheader << tkl.header(t);
         Packet th = garmin_trackheader->makePacket();
         sendPacket(th);i++; printFortschritt(i,n); }
      for(int p=0; p<tkl.sizeTrackpoints(t);p++) {
         garmin_trackpoint->clear();
         *garmin_trackpoint << tkl.trackpoint(t,p);
         Packet tp = garmin_trackpoint->makePacket();
         sendPacket(tp);i++; printFortschritt(i,n); } }

   Packet transComplete;   // send track transfer complete
   transComplete.pid = Pid_Xfer_Cmplt;
   transComplete.data[0]=Cmnd_Transfer_Trk;
   transComplete.length=2;
   sendPacket(transComplete);
}

void GarminGPS::uploadWaypoints(Waypointlist &wpl)
{
   if(want_to_die) return;
   if(!waypointTransfer) return;

   int n = wpl.size();   // send number of packets (waypoints)
   Records_Type records;
   records.n = n ;
   Packet recordsPacket = records.makePacket();
   sendPacket(recordsPacket);
   if(!quiet)cerr << "uploading " << n << " waypoints: " ;
   printFortschritt(0,n);

   for(int i=0;i<n && !want_to_die;i++) {  // send waypoints
      garmin_waypoint->clear();
      *garmin_waypoint << wpl[i];
      Packet waypointPacket  = garmin_waypoint->makePacket();
      sendPacket(waypointPacket);
      printFortschritt(i+1,n); }

   if(want_to_die) { 
      abortTransfer();
      return; }

   Packet transComplete;   // send waypoint transfer complete
   transComplete.pid = Pid_Xfer_Cmplt;
   transComplete.data[0]=Cmnd_Transfer_Wpt;
   transComplete.length=2;
   sendPacket(transComplete);
}


void GarminGPS::downloadDateTime(Date_Time_Type &dt)
{
	if(want_to_die) return;
   if(!dateTimeTransfer) return;

   Packet request; // request date time;
   request.pid=Pid_Command_Data;
   request.data[0]=Cmnd_Transfer_Time;
   sendPacket(request);

   Packet p;   
   getPacket(p);

	garmin_datetime->clear();
	garmin_datetime->set(p);

	dt << garmin_datetime->os();

}

void GarminGPS::downloadRoutes(Routelist &rtl)
{
   if(want_to_die) return;
   if(!routeHeaderTransfer && !waypointTransfer) return;
   Packet request; // request tracks;
   request.pid=Pid_Command_Data;
   request.data[0]=Cmnd_Transfer_Rte;
   sendPacket(request);
   Packet records;
   getPacket(records);
   Records_Type  rt(records);
   int n = rt.n;
	if(n>0){
	   if(!quiet) cerr << "downloading " << n << " packets routedata: " ;
	   printFortschritt(0,n);}
	else cerr << "no routes to download " << endl;

   for(int i=0; i<n && !want_to_die; i++) {
      Packet p;   
      getPacket(p);
      if(p.pid==Pid_Rte_Hdr && garmin_routeheader!=NULL) {
         garmin_routeheader->clear();
         garmin_routeheader->set(p);
         rtl << garmin_routeheader->os(); }
      if(p.pid==Pid_Rte_Wpt_Data && garmin_waypoint!=NULL) {
         garmin_waypoint->clear();   
         garmin_waypoint->set(p);
         rtl << garmin_waypoint->os(); }
      printFortschritt(i+1,n);   }

   if(want_to_die) { 
      abortTransfer();
      return; }

   Packet tc;  // get transfer complete
   getPacket(tc);
}


void GarminGPS::downloadTracks(Tracklist &tkl)
{
   if(want_to_die) return;
   if(!trackTransfer)return;
   Packet request; // request tracks;
   request.pid=Pid_Command_Data;
   request.data[0]=Cmnd_Transfer_Trk;
   sendPacket(request);
   Packet records;
   getPacket(records);
   Records_Type  rt(records);
   int n = rt.n;
   if(!quiet) cerr << "downloading " << n << " packets trackdata: " ;
   printFortschritt(0,n);

   for(int i=0; i<n && !want_to_die; i++) {
      Packet p;   
      getPacket(p);
      if(p.pid==Pid_Trk_Hdr && garmin_trackheader!=NULL) {
         garmin_trackheader->clear();
         garmin_trackheader->set(p);
         tkl << garmin_trackheader->os(); }
      if(p.pid==Pid_Trk_Data && garmin_trackpoint!=NULL) {
         garmin_trackpoint->clear();   
         garmin_trackpoint->set(p);
         tkl << garmin_trackpoint->os(); }
      printFortschritt(i+1,n); }

   if(want_to_die) { 
      abortTransfer();
      return; }

   Packet tc;  // get transfer complete
   getPacket(tc);
}

void GarminGPS::downloadWaypoints(Waypointlist &wpl)
{
   if(want_to_die) return;
   if(!waypointTransfer) return;
   Packet request;   // request waypoints
   request.pid=Pid_Command_Data;
   request.data[0]=Cmnd_Transfer_Wpt;
   sendPacket(request);
   Packet records;
   getPacket(records);
   Records_Type  rt(records);
   int n = rt.n;
   if(!quiet) cerr << "downloading " << n << " waypoints: " ;
   printFortschritt(0,n);

   for(int i=0; i<n && !want_to_die; i++) { // download waypoints
      Packet wpp;   
      if(getPacket(wpp)>0)
		{
	      garmin_waypoint->clear();
	      garmin_waypoint->set(wpp);
      	Wpt_Type wpt;
      	wpt << garmin_waypoint->os();
      	wpl.add(wpt); 
      	printFortschritt(i+1,n); 
		}
		else
		{
			cerr << endl << "!!! download failed"  << endl;
			want_to_die=true;
		}}

   if(want_to_die) { 
      abortTransfer();
      return; }

   Packet tc;   // get transfer complete
   getPacket(tc);
}


void GarminGPS::printFortschritt(int i,int n)
{
   if(quiet) return;

   const int zmax = 50;

	int percent=0;
	if(i==n) percent=100;
	else percent=int(((float(i)/float(n))*float(100))) ;

   if(i!=0) for(int d=0;d<zmax+9;d++) cerr << char(8);
   else cerr << endl;
   cerr << "";

   for(int j=0; j<zmax;j++) {
      if(float(j)<(float(i)/float(n)*float(zmax))) cerr << "=";
      else cerr <<" "; }

   cerr <<" ";
   if(i%4==0) cerr << "|";
   if(i%4==1) cerr << "/";
   if(i%4==2) cerr << "-";
   if(i%4==3) cerr << "\\";
   cerr << setw(4)<< percent  << "%" << flush;

   if(n==i){ for(int j=0;j<6;j++) cerr<<char(8); cerr << "done !" << endl; }
}



void GarminGPS::abortTransfer(void)
{
   Packet abort;
   abort.pid = Pid_Command_Data;
   abort.data[0]=Cmnd_Abort_Transfer;
   sendPacket(abort);
   Packet dummy; // get a last packet that might lie around
   getPacket(dummy);
}


void GarminGPS::turnOff(void)
{
   Packet p;
   p.pid =  Pid_Command_Data; 
   p.data[0]=Cmnd_Turn_Off_Pwr;
   p.length = 2;
   sendPacket(p);
}

}

