/*
 * Gearsystem - Sega Master System / Game Gear Emulator
 * Copyright (C) 2013  Ignacio Sanchez

 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/
 *
 */

#import "MasterViewController.h"
#import "DetailViewController.h"

@interface MasterViewController ()

@property NSMutableArray *objects;
@end

@implementation MasterViewController

- (void)awakeFromNib {
    [super awakeFromNib];
    self.title = @"Games";
    if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPad) {
        self.clearsSelectionOnViewWillAppear = NO;
        self.preferredContentSize = CGSizeMake(320.0, 600.0);
    }
    
    self.theGLViewController = [[GLViewController alloc] initWithNibName:@"GLViewController" bundle:nil];
    self.theGLViewController.preferredFramesPerSecond = 60;
    self.theGLViewController.resumeOnDidBecomeActive = NO;
    self.theGLViewController.pauseOnWillResignActive = NO;
}

- (void)viewDidLoad {
    [super viewDidLoad];
    self.detailViewController = (DetailViewController *)[[self.splitViewController.viewControllers lastObject] topViewController];
    [self reloadTableView];
}

- (BOOL)shouldAutorotate
{
    UIInterfaceOrientation orientation = [[UIApplication sharedApplication] statusBarOrientation];
    
    return (orientation == UIInterfaceOrientationPortrait) || (orientation == UIInterfaceOrientationPortraitUpsideDown);;
}

- (void)reloadTableView
{
    NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
    
    NSArray *files = nil;
    
    if ([paths count] > 0)
    {
        NSFileManager *fileManager = [[NSFileManager alloc] init];
        NSArray *dirContents = [fileManager contentsOfDirectoryAtPath:[paths objectAtIndex:0] error:nil];
        
        if ([dirContents count] > 0)
        {
            NSArray *extensions = [NSArray arrayWithObjects:@"zip", @"sms", @"gg", @"rom", @"bin", @"ZIP", @"SMS", @"GG", @"ROM", @"BIN", nil];
            files = [dirContents filteredArrayUsingPredicate:[NSPredicate predicateWithFormat:@"pathExtension IN %@", extensions]];
        }
    }
    
    self.sections = [[NSMutableDictionary alloc] init];
    
    BOOL found;
    
    for (NSString* rom in files)
    {
        NSString* c = [[rom substringToIndex:1] uppercaseString];
        
        found = NO;
        
        for (NSString* str in [self.sections allKeys])
        {
            if ([str isEqualToString:c])
            {
                found = YES;
            }
        }
        
        if (!found)
        {
            [self.sections setValue:[[NSMutableArray alloc] init] forKey:c];
        }
    }
    
    for (NSString* rom in files)
    {
        [[self.sections objectForKey:[[rom substringToIndex:1] uppercaseString]] addObject:rom];
    }
    
    [self.tableView reloadData];
    
}

- (void)loadWithROM:(NSString *)rom
{
    _openedFromOtherApp = YES;
    _openedFromOtherAppRom = rom;
    
    if (self.splitViewController.collapsed)
    {
        id vc = [[self.splitViewController.viewControllers lastObject] topViewController];
        if ([vc isMemberOfClass:[UINavigationController class]])
        {
            _openedFromOtherApp = NO;
            [self.theGLViewController loadRomWithName:_openedFromOtherAppRom];
            return;
        }
    }
    
    [self performSegueWithIdentifier: @"showDetail" sender: self];
}

#pragma mark - Segues

- (void)prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender {
    if ([[segue identifier] isEqualToString:@"showDetail"]) {
        
        NSString* rom;
        
        if (_openedFromOtherApp)
        {
            _openedFromOtherApp = NO;
            rom = _openedFromOtherAppRom;
        }
        else
        {
            NSIndexPath *indexPath = [self.tableView indexPathForSelectedRow];
            rom = [[self.sections valueForKey:[[[self.sections allKeys] sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)] objectAtIndex:indexPath.section]] objectAtIndex:indexPath.row];
        }

        DetailViewController *controller = (DetailViewController *)[[segue destinationViewController] topViewController];
        [controller setDetailItem:rom];
        [controller setTheGLViewController:self.theGLViewController];
        controller.navigationItem.leftBarButtonItem = self.splitViewController.displayModeButtonItem;
        controller.navigationItem.leftItemsSupplementBackButton = YES;
        self.splitViewController.preferredDisplayMode = UISplitViewControllerDisplayModePrimaryHidden;
    }
}

#pragma mark - Table View

- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView {
    return [[self.sections allKeys] count];
}

- (NSString *)tableView:(UITableView *)tableView titleForHeaderInSection:(NSInteger)section
{
    return [[[self.sections allKeys] sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)] objectAtIndex:section];
}

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section {
    return [[self.sections valueForKey:[[[self.sections allKeys] sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)] objectAtIndex:section]] count];
}

- (NSArray *)sectionIndexTitlesForTableView:(UITableView *)tableView {
    return [[self.sections allKeys] sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)];
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath {
    static NSString *CellIdentifier = @"Cell";
    
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier];
    if (cell == nil) {
        cell = [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier];
        if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
            cell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
        }
    }
    
    NSString* rom = [[self.sections valueForKey:[[[self.sections allKeys] sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)] objectAtIndex:indexPath.section]] objectAtIndex:indexPath.row];
    
    cell.textLabel.text = [rom stringByDeletingPathExtension];
    cell.textLabel.adjustsFontSizeToFitWidth = YES;
    cell.selectionStyle = UITableViewCellSelectionStyleGray;
    return cell;
}

- (BOOL)tableView:(UITableView *)tableView canEditRowAtIndexPath:(NSIndexPath *)indexPath {
    return YES;
}

- (void)tableView:(UITableView *)tableView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowAtIndexPath:(NSIndexPath *)indexPath {
    if (editingStyle == UITableViewCellEditingStyleDelete)
    {
        NSString* rom = [[self.sections valueForKey:[[[self.sections allKeys] sortedArrayUsingSelector:@selector(localizedCaseInsensitiveCompare:)] objectAtIndex:indexPath.section]] objectAtIndex:indexPath.row];
        
        NSString *documentsDirectory = [NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES) objectAtIndex:0];
        NSError* error;
        NSFileManager *fileManager = [[NSFileManager alloc] init];
        
        NSString* deletePath = [NSString stringWithFormat:@"%@/%@", documentsDirectory, rom];
        
        if ([fileManager removeItemAtPath:deletePath error:&error])
        {
            [tableView beginUpdates];
            
            [[self.sections objectForKey:[[rom substringToIndex:1] uppercaseString]] removeObject:rom];
            
            if ([[self.sections objectForKey:[[rom substringToIndex:1] uppercaseString]] count] > 0)
            {
                [tableView deleteRowsAtIndexPaths:[NSArray arrayWithObject:indexPath] withRowAnimation:UITableViewRowAnimationFade];
            }
            else
            {
                [self.sections removeObjectForKey:[[rom substringToIndex:1] uppercaseString]];
                
                [tableView deleteRowsAtIndexPaths:[NSArray arrayWithObject:indexPath] withRowAnimation:UITableViewRowAnimationFade];
                [tableView deleteSections:[NSIndexSet indexSetWithIndex:indexPath.section]withRowAnimation:UITableViewRowAnimationFade];
            }
            
            [tableView endUpdates];
        }
        else
        {
            NSLog(@"ERROR %@", [error localizedDescription]);
        }
        
        [tableView reloadData];
    }
}

- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    [self.theGLViewController.theEmulator pause];
    // disable screen sleep when in-game
    [[UIApplication sharedApplication] setIdleTimerDisabled: NO];
}

- (void)viewWillDisappear:(BOOL)animated
{
    [super viewWillDisappear:animated];
    [self.theGLViewController.theEmulator resume];
    // disable screen sleep when in-game
    [[UIApplication sharedApplication] setIdleTimerDisabled: YES];
}

@end
