/*
  galaxa.c
  
  GalaXa
  
  A space shoot-up for X-Window
  
  by Bill Kendrick
  kendrick@zippy.sonoma.edu
  http://zippy.sonoma.edu/kendrick/
  
  April 6, 1998 - April 22, 1998
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/types.h>
#include <sys/time.h>
#include <math.h>
#include "hints.h"
#include "randnum.h"
#include "setup.h"
#include "text.h"
#include "soundmgr.h"


/* #define ATTACH_SELF */

#define EVENTLOOP 100

#define KEY_LEFT 0
#define KEY_RIGHT 1

#define MAX_SHOTS_AT_A_TIME 3

#define DIR_UP 0
#define DIR_UPRIGHT 1
#define DIR_RIGHT 2
#define DIR_DOWNRIGHT 3
#define DIR_DOWN 4
#define DIR_DOWNLEFT 5
#define DIR_LEFT 6
#define DIR_UPLEFT 7

#define MAX_ALIENS 22
#define MAX_BULLETS 16
#define MAX_EXPLOSIONBITS 32
#define MAX_EXPLOSIONS 1
#define MAX_STARS 10

#define BULLET_SIZE 8

#define MODE_ENTER 0
#define MODE_WAIT 1
#define MODE_ATTACK 2
#define MODE_RETREAT 3
#define MODE_SUCK 4

#define GAMEMODE_WARP 0
#define GAMEMODE_ENTER 1
#define GAMEMODE_NORMAL 2
#define GAMEMODE_SUCK 3

#define TYPE_MASTER_1 0
#define TYPE_GUARD 1
#define TYPE_GUARDA 2
#define TYPE_PAWNA 3
#define TYPE_PAWN 4
#define TYPE_MASTER_0 5

int scores[6] = {250, 100, 100, 50, 50, 250};

#define OWNER_PLAYER 0
#define OWNER_ALIEN 1

#define SUCK_HEIGHT (HEIGHT - 96)

#define HELPSHAPE_LEFT 0
#define HELPSHAPE_CENTER 1
#define HELPSHAPE_RIGHT 2

#define TITLEMODE_TITLE 0
#define TITLEMODE_HELP_1 1
#define TITLEMODE_HELP_2 2
#define TITLEMODE_HELP_3 3
#define TITLEMODE_HIGHS 4
#define NUM_TITLEMODES 5

#define MAX_FIREWORKS 5
#define MAX_FIREWORKSBITS 20

#define MAX_ASTEROIDS 6


/* Typedefs: */

typedef struct player_type {
  int alive;
  int x, y;
  int size;
  int shots_out;
} player_type;

typedef struct alien_type {
  int alive, type, mode, goaway;
  int hasship, isship;
  int x, y;
  int homex, homey;
  int gotox, gotoy;
  int dir;
  int sucktime;
  int hitpoints;
} alien_type;

typedef struct bullet_type {
  int alive;
  int owner;
  int x, y;
  int xm, ym;
} bullet_type;

typedef struct explosionbit_type {
  int alive, time;
  int x, y;
  int xm, ym;
} explosionbit_type;

typedef struct explosion_type {
  int alive, frame;
  int x, y;
} explosion_type;

typedef struct planet_type {
  int alive;
  int x, y;
  int shape;
} planet_type;

typedef struct star_type {
  int x, y;
  int speed;
} star_type;

typedef struct scoretable_type {
  int score;
  char name[5];
} scoretable_type;

typedef struct fireworksbit_type {
  float x, y;
  float xm, ym;
} fireworksbit_type;

typedef struct fireworks_type {
  int alive;
  int time;
  float x, y;
  float xm, ym;
  struct fireworksbit_type bit[MAX_FIREWORKSBITS];
} fireworks_type;

typedef struct asteroid_type {
  int alive, size;
  int x, y;
  int xm, ym;
  int anim, animm, animspeed;
} asteroid_type;


/* Globals: */

player_type player, extraship;
alien_type aliens[MAX_ALIENS];
star_type stars[MAX_STARS];
asteroid_type asteroids[MAX_ASTEROIDS];
planet_type planet;
bullet_type bullets[MAX_BULLETS];
explosionbit_type explosionbits[MAX_EXPLOSIONBITS];
scoretable_type highscores[10], highlevels[10];
explosion_type explosions[MAX_EXPLOSIONS];
float highhitratio;
char highhitrationame[5];
int arrow_key[2];
int starspeed, starspeedm, starspeedtime, wantstarspeed, faststarspeed,
  changestarspeed, alien_responsible;
int gamemode, level, alien_speed, giant_bugs, flashscreen, maxlevel, score,
  lives, gameover;
float num_shots, num_hits;
int highscoreslot, highlevelslot, highhitratioslot;


/* Prototypes: */

int checkforhigh(void);
void getname(char * buf, char * title);
int title(int titlemode);
int game(void);
void initlevel(void);
void initgame(void);
void initstars(void);
int dirtowards(int wantdir, int curdir);
void addbullet(int x, int y, int xm, int ym, int owner);
void addexplosion(int x, int y, int xm, int ym);
void addexplosionbit(int x, int y, int xm, int ym);
void killplayer();
void turnoffplayer(void);
void helpshape(int pos, int y, int obj);
void helptext(int y, char * text);
void killbullet(int which);
void loadhighscores(void);
void savehighscores(void);
void addanimatedexplosion(int x, int y);
void addasteroid(int x, int y, int xm, int ym, int size);


/* -- MAIN -- */

int main(int argc, char * argv[])
{
  int quit, titlemode;
  
  
  /* Setup: */
  
  maxlevel = 20;
  
  level = setup(argc, argv);
  Xsetup();
  randinit();
  loadhighscores();
  
  titlemode = TITLEMODE_TITLE;
  
  highscoreslot = -1;
  highlevelslot = -1;
  highhitratioslot = -1;
  
  do
    {
      quit = title(titlemode);

      highscoreslot = -1;
      highlevelslot = -1;
      highhitratioslot = -1;
      
      if (quit == 0)
	titlemode = game();
    }
  while (quit == 0);
  
  savehighscores();
  shutdown();
  return(0);
}


/* Title mode: */

int title(int titlemode)
{
  char string[128], key[128], temp[128];
  XEvent event;
  KeySym keysym;
  XComposeStatus composestatus;
  struct timeval now, then;
  int done, quit, i, j, titlex, titley, levelx, levely, creditsx, creditsy,
    titlemode_section, counter, nextmode, spin, title_size,
    checkmode, flashscreen, titlemode_section_fast;
  long time_padding;
  
  
  silence(sound_fs);
  
  
  /* Quick refs. for positions: */
  
  titlex = (WIDTH - object_width[OBJ_TITLE]) / 2;
  titley = (WIDTH - object_height[OBJ_TITLE]) / 2;
  creditsx = (WIDTH - object_width[OBJ_CREDITS]) / 2;
  creditsy = titley + object_height[OBJ_TITLE] + 8;
  levelx = (WIDTH - 48) / 2;
  levely = creditsy + object_height[OBJ_CREDITS] + 16;
  
  
  /* Reset everything: */
  
  initstars();
  
  quit = 0;
  done = 0;
  
  titlemode_section = 0;
  titlemode_section_fast = 0;
  title_size = 0;
  flashscreen = 0;
  
  
  /* I do this to put the clipmask in the right place: */
  
  drawshape(titlex, titley, OBJ_TITLE, backbuffer);
  
  
  /* Title screen main loop: */
  
  do
    {
      /* Handle counters/mode switchers/etc.: */
      
      counter = counter + 1;
      
      if (counter % 50 == 0)
	titlemode_section++;
      
      if (counter % 15 == 0)
	titlemode_section_fast++;
      
      
      spin = (counter / 6) % 8;
      

      gettimeofday(&then, NULL);
      
      
      /* Handle keyboard input: */
      
      strcpy(key, "");
      
      while (XPending(display))
	{
	  XNextEvent(display, &event);
	  
	  if (event.type == KeyPress || event.type == KeyRelease)
	    {
	      /* Get the key's name: */
	      
	      XLookupString(&event.xkey, temp, 1, &keysym,
			    &composestatus);
	      
	      if (XKeysymToString(keysym) != NULL)
		{
		  strcpy(temp, XKeysymToString(keysym));
		  
		  if (strcmp(temp, "Left") == 0 ||
		      strcmp(temp, "KP Left") == 0)
		    arrow_key[KEY_LEFT] = event.type;
		  else if (strcmp(temp, "Right") == 0 ||
			   strcmp(temp, "KP Right") == 0)
		    arrow_key[KEY_RIGHT] = event.type;
		  else if (event.type == KeyPress)
		    strcpy(key, temp);
		}
	    }
	}

      nextmode = 0;
      checkmode = 0;
      
      if (strcasecmp(key, "Q") == 0)
	{
	  quit = 1;
	  done = 1;
	}
      else if (strcasecmp(key, "Space") == 0)
	{
	  done = 1;
	}
      else if (strcasecmp(key, "1") == 0)
	{
	  nextmode = 1;
	}
      else if (strcasecmp(key, "2") == 0)
	{
	  nextmode = 1;
	  titlemode = TITLEMODE_HIGHS - 1;
	}
      else if (strcasecmp(key, "L") == 0)
	{
	  level++;
	  if (level > maxlevel)
	    level = 1;
	  checkmode = 1;
	  
	  playsound(sound_fs, SND_SHOT);
	}
      else if (strcasecmp(key, "K") == 0)
	{
	  level--;
	  if (level < 1)
	    level = maxlevel;
	  checkmode = 1;

	  playsound(sound_fs, SND_SHOT);
	}
      
      
      if (checkmode == 1 && titlemode != TITLEMODE_TITLE)
	{
	  titlemode = TITLEMODE_TITLE - 1;
	  nextmode = 1;
	}
      
      
      /* Redraw screen: */
      
      if (flashscreen == 0 || (counter % 2) == 0)
	XFillRectangle(display, backbuffer, blackgc, 0, 0, WIDTH, HEIGHT);
      else
	{
	  flashscreen--;
	  XFillRectangle(display, backbuffer, whitegc, 0, 0, WIDTH, HEIGHT);
	}
      
      
      /* Draw and move stars: */
      
      for (i = 0; i < MAX_STARS; i++)
	{
	  XDrawPoint(display, backbuffer, greygc, stars[i].x, stars[i].y);
	  
	  stars[i].y = stars[i].y + stars[i].speed + 3;
	  
	  if (stars[i].y > HEIGHT)
	    {
	      stars[i].x = randnum(WIDTH);
	      stars[i].y = -randnum(20);
	      stars[i].speed = randnum(5);
	    }
	}
      
      
      if (titlemode == TITLEMODE_TITLE)
	{
	  /* Draw title: */
	  
	  if (title_size < object_height[OBJ_TITLE])
	    {
	      title_size++;
	      
	      XCopyArea(display, object_pixmap[OBJ_TITLE], backbuffer,
			object_gc[OBJ_TITLE],
			0, (object_height[OBJ_TITLE] - title_size) / 2,
			object_width[OBJ_TITLE], title_size,
			titlex, titley);
	      
	      if (title_size == object_height[OBJ_TITLE])
		{
		  flashscreen = 5;
		  
		  playsound(sound_fs, SND_FLASH);
		}
	    }
	  else
	    drawshape(titlex, titley, OBJ_TITLE, backbuffer);
	  
	  if (title_size >= object_height[OBJ_TITLE])
	    {
	      drawshape(creditsx, creditsy, OBJ_CREDITS, backbuffer);
	    }
	  
	  drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH , levely,
			   "LEVEL:", font);
	  drawshape(levelx, levely, level / 10, backbuffer);
	  drawshape(levelx + 24, levely, level % 10, backbuffer);
	  drawcenteredtext(display, backbuffer, greygc, 
			   0, WIDTH, levely + 40 + fh,
			   "Use [L] and [K] to select", font);
	  drawcenteredtext(display, backbuffer, greygc, 
			   0, WIDTH, levely + 40 + fh * 2,
			   "Press [Space] to begin", font);

	  if (titlemode_section >= 20)
	    nextmode = 1;
	}
      else if (titlemode == TITLEMODE_HELP_1)
	{
	  /* Draw help, page 1: */
	  
	  helpshape(HELPSHAPE_LEFT, 16,
		    OBJ_ALIEN_0 + TYPE_PAWN * 9 + spin);
	  helpshape(HELPSHAPE_RIGHT, 16,
		    OBJ_ALIEN_0 + TYPE_PAWNA * 9 + 7 - spin);
	  helptext(16, "Drones - 50 pts");
	  
	  if (titlemode_section >= 2)
	    {
	      helpshape(HELPSHAPE_LEFT, 96,
			OBJ_ALIEN_0 + TYPE_GUARD * 9 + 7 - spin);
	      helpshape(HELPSHAPE_RIGHT, 96,
			OBJ_ALIEN_1 + TYPE_GUARD * 9 + spin);
	      helptext(96, "Guards - 100 pts");
	    }
	  
	  if (titlemode_section >= 3)
	    {
	      helpshape(HELPSHAPE_LEFT, 172,
			OBJ_ALIEN_0 + TYPE_MASTER_0 * 9 +
			8 * ((counter / 8) % 2));
	      helpshape(HELPSHAPE_RIGHT, 172,
			OBJ_ALIEN_0 + TYPE_MASTER_1 * 9 +
			8 * ((counter / 8) % 2));
	      helptext(172, "Masters - 250 pts");
	      helptext(172 + fh, "(Requires two shots)");
	    }
	  
	  if (titlemode_section >= 5)
	    {
	      helpshape(HELPSHAPE_CENTER, 248, OBJ_SHIP_0);
	      helptext(248, "Your Fighter");
	    }
	  
	  if (titlemode_section >= 10)
	    nextmode = 1;
	}
      else if (titlemode == TITLEMODE_HELP_2)
	{
	  /* Draw help page 2: */
	  
	  helpshape(HELPSHAPE_CENTER, 32, OBJ_ALIEN_0 +
		    TYPE_MASTER_0 * 9 + DIR_DOWN);
	  helpshape(HELPSHAPE_CENTER, 64, OBJ_TRACTORBEAM);
	  
	  helptext(112, "Masters fire a beam to capture your fighter");

	  if (titlemode_section >= 2)
	    {
	      helpshape(HELPSHAPE_CENTER, 192, OBJ_SHIP_1 + spin);
	      helpshape(HELPSHAPE_CENTER, 224, OBJ_ALIEN_0 +
			TYPE_MASTER_0 * 9 + spin);
	      helptext(224, "Shoot the Master to free your ship");
	      helptext(224 + fh, "(Do this only while it's diving!)");
	    }
	  
	  if (titlemode_section >= 4)
	    {
	      drawshape(WIDTH / 2 - 32, 300, OBJ_SHIP_0, backbuffer);
	      drawshape(WIDTH / 2, 300, OBJ_SHIP_0, backbuffer);
	      helptext(300, "Dock with the freed ship for double shots!");
	    }
	  
	  if (titlemode_section >= 10)
	    nextmode = 1;
	}
      else if (titlemode == TITLEMODE_HELP_3)
	{
	  /* Draw help page 3: */
	  
	  helpshape(HELPSHAPE_CENTER, 32, OBJ_GIANT_0 +
		    (counter % 2));
	  
	  helptext(32 + object_height[OBJ_GIANT_0] - fh,
		   "Every 10th level has Super-Masters!");
	  helptext(32 + object_height[OBJ_GIANT_0],
		   "??? pts");
	  helptext(32 + object_height[OBJ_GIANT_0] + fh,
		   "(Requires many shots)");
	  
	  
	  helptext(300, "Watch for other mysteries!");
	  
	  if (titlemode_section >= 10)
	    nextmode = 1;
	}
      else if (titlemode == TITLEMODE_HIGHS)
	{
	  /* Draw high scores: */
	  	 
	  drawcenteredtext(display, backbuffer,
			   explosiongcs[counter % NUM_EXPLOSION_COLORS],
			   0, WIDTH, 80 - fh * 3, "HIGH SCORES",
			   font);
	  
	  drawtext(display, backbuffer, whitegc,
		   20, 80 - fh, "High Scores:");
	  
	  drawtext(display, backbuffer, whitegc,
		   WIDTH / 2 + 20, 80 - fh, "High Levels:");
	  
	  for (i = 0; i < titlemode_section_fast && i < 10; i++)
	    {
	      for (j = 0; j <= i; j++)
		{
		  sprintf(temp, "%2d %.6d %s", 10 - j,
			  highscores[9 - j].score,
			  highscores[9 - j].name);
		  
		  if (j != 9 && (9 - j) != highscoreslot)
		    drawtext(display, backbuffer, greygc,
			     20, fh * (i - j) + 80 + fh, temp);
		  else
		    drawtext(display, backbuffer,
			     explosiongcs[(counter / 10) %
					 NUM_EXPLOSION_COLORS],
			     20, fh * (i - j) + 80 + fh, temp);


		  sprintf(temp, "%2d %6d %s", 10 - j,
			  highlevels[9 - j].score,
			  highlevels[9 - j].name);
		  
		  
		  if (j != 9 && (9 - j) != highlevelslot)
		    drawtext(display, backbuffer, greygc,
			     WIDTH / 2 + 20, fh * (i - j) + 80 + fh, temp);
		  else
		    drawtext(display, backbuffer,
			     explosiongcs[(counter / 10) %
					 NUM_EXPLOSION_COLORS],
			     WIDTH / 2 + 20, fh * (i - j) + 80 + fh, temp);
		}
	    }
	  
	  if (titlemode_section_fast >= 11)
	    {
	      drawcenteredtext(display, backbuffer, whitegc,
			       0, WIDTH, 80 + fh * 12, "Highest Hit Ratio:",
			       font);
	      
	      sprintf(temp, "%f %s", highhitratio, highhitrationame);
	      
	      if (highhitratioslot != 0)
		drawcenteredtext(display, backbuffer, greygc,
				 0, WIDTH, 80 + fh * 14, temp,
				 font);
	      else
		drawcenteredtext(display, backbuffer,
				 explosiongcs[(counter / 10) %
					     NUM_EXPLOSION_COLORS],
				 0, WIDTH, 80 + fh * 14, temp,
				 font);
	    }
	  
	  if (titlemode_section >= 10)
	    nextmode = 1;
	}
      

      /* Switch mode: */
      
      if (nextmode == 1)
	{
	  titlemode++;
	  if (titlemode >= NUM_TITLEMODES)
	    titlemode = 0;
	  
	  titlemode_section = 0;
	  titlemode_section_fast = 0;
	  title_size = 0;
	}
      
      
      /* Refresh: */
      
      updatebackbuffer();
      XSync(display, 0);
      
      
      /* Pause: */
      
      gettimeofday(&now, NULL);
      
      time_padding = FRAMERATE - ((now.tv_sec - then.tv_sec) * 1000000 +
                                  (now.tv_usec - then.tv_usec));
      
      if (time_padding > 0)
	usleep(time_padding);
    }
  while (done == 0);
  
  return(quit);
}



/* Game mode: */

int game(void)
{  
  int i, j, x, y, counter, done, z, eventloop, w, h;
  char string[128], key[128], temp[128], text[128];
  XEvent event;
  KeySym keysym;
  XComposeStatus composestatus;
  struct timeval now, then;
  long time_padding;
  int wantleft, wantright, wantup, wantdown, all_in_place, num_aliens,
    wingflap, wave_position, wave_position_m, num_bullets,
    someone_hasship, show_text, paused;
  
  
  /* Init stuff: */
  
  initgame();
  initlevel();
  
  playsound(sound_fs, SND_INTRO);
  
  counter = 0;
  wingflap = 0;
  wave_position = 0;
  wave_position_m = 1;
  show_text = 0;
  
  done = 0;
  paused = 0;
  
  strcpy(key, "");
  
  flashscreen = 0;

  
  /* MAIN LOOP: */
  
  do
    {
      gettimeofday(&then, NULL);
      
      
      /* Handle loopers, counters, togglers, etc. */
      
      counter = counter + 1;
      
      if (counter % 10 == 0)
	wingflap = 1 - wingflap;
      
      if (counter % 3 == 0)
	{
	  wave_position = wave_position + wave_position_m;
	  
	  if (wave_position <= -5)
	    wave_position_m = 1;
	  else if (wave_position >= 5)
	    wave_position_m = -1;
	}
      
      
      /* Handle Events! */
      
      strcpy(key, "");
      
      for (eventloop = 0; eventloop < EVENTLOOP; eventloop++)
	{
	  while (XPending(display))
	    {
	      XNextEvent(display, &event);
	      
	      if (event.type == KeyPress || event.type == KeyRelease)
		{
		  /* Get the key's name: */
		  
		  XLookupString(&event.xkey, temp, 1, &keysym,
				&composestatus);
		  
		  if (XKeysymToString(keysym) != NULL)
		    {
		      strcpy(temp, XKeysymToString(keysym));
		      
		      if (strcmp(temp, "Left") == 0 ||
			  strcmp(temp, "KP Left") == 0)
			arrow_key[KEY_LEFT] = event.type;
		      else if (strcmp(temp, "Right") == 0 ||
			       strcmp(temp, "KP Right") == 0)
			arrow_key[KEY_RIGHT] = event.type;
		      else if (event.type == KeyPress)
			strcpy(key, temp);
		    }
		}
	    }
	}
      
      
      /* Handle keyboard: */
      
      if (player.alive == 1 && paused == 0)
	{
	  /* Arrow keys move: */
	  
	  if (arrow_key[KEY_LEFT] == KeyPress && player.x > 0)
	    player.x = player.x - 6;
	  else if (arrow_key[KEY_RIGHT] == KeyPress &&
		   player.x < WIDTH - (32 * player.size) - 1)
	    player.x = player.x + 6;
	  
	  
	  if (strcasecmp(key, "Space") == 0)
	    {
	      /* Space fires */
	      
	      if (player.shots_out < MAX_SHOTS_AT_A_TIME * player.size)
		playsound(sound_fs, SND_SHOT);
	      
	      for (i = 0; i < player.size; i++)
		{
		  if (player.shots_out < MAX_SHOTS_AT_A_TIME * player.size)
		    {
		      addbullet(player.x + 16 - BULLET_SIZE / 2 + (i * 32),
				HEIGHT - 32 - BULLET_SIZE / 2,
				0, -10, OWNER_PLAYER);
		      
		      num_shots++;
		      
		      player.shots_out++;
		    }
		}
	    }
	}


      if (strcasecmp(key, "Q") == 0)
	{
	  /* Q quits */
	  
	  done = 1;
	}
      if (strcasecmp(key, "P") == 0)
	{
	  /* P (un)pauses: */
	  
	  paused = 1 - paused;
	}
      
      
      /* Count aliens: */
      
      num_aliens = 0;
      someone_hasship = 0;
      
      for (i = 0; i < MAX_ALIENS; i++)
	{
	  if (aliens[i].alive)
	    {
	      num_aliens++;
	      
	      if (aliens[i].hasship)
		someone_hasship = 1;
	    }
	}
      
      
      /* Advance if there are no more aliens: */
      
      if (num_aliens == 0)
	{
	  level++;
	  
	  if (level > 99)
	    level = 0;
	  
	  if (level > maxlevel)
	    maxlevel = level;
	  
	  playsound(sound_fs, SND_NEXTLEVEL);
	  initlevel();
	}
      
      
      /* Handle aliens: */
      
      for (i = 0; i < MAX_ALIENS; i++)
	{
	  if (aliens[i].alive == 1 && gamemode != GAMEMODE_WARP &&
	      paused == 0)
	    {
	      /* if (player.alive == 0)
		aliens[i].mode = MODE_ENTER; */
	      
	      
	      /* Rotate: */
	      
	      if (aliens[i].dir < 0)
		aliens[i].dir = aliens[i].dir + 8;
	      else if (aliens[i].dir > 7)
		aliens[i].dir = aliens[i].dir - 8;
	      
	      
	      /* Move: */
	      
	      if (aliens[i].mode != MODE_WAIT &&
		  aliens[i].mode != MODE_SUCK)
		{
		  /* Move them forward: */
		  
		  aliens[i].x = aliens[i].x + (xms[aliens[i].dir] *
					       alien_speed);
		  aliens[i].y = aliens[i].y + (yms[aliens[i].dir] *
					       alien_speed);
		  
		  
		  /* Make them spin: */
		  
		  if (counter % 4 == 0)
		    {
		      /* Find out which way they want to go: */
		      
		      wantleft = 0;
		      wantright = 0;
		      wantup = 0;
		      wantdown = 0;
		      
		      if (aliens[i].x < aliens[i].gotox)
			wantright = 1;
		      else if (aliens[i].x > aliens[i].gotox)
			wantleft = 1;
		      
		      if (aliens[i].y < aliens[i].gotoy)
			wantdown = 1;
		      else if (aliens[i].y > aliens[i].gotoy)
			wantup = 1;
		      
		      
		      /* Make them spin to face that way: */
		      
		      if (wantleft)
			{
			  if (wantup)
			    {
			      aliens[i].dir = dirtowards(DIR_UPLEFT,
							 aliens[i].dir);
			    }
			  else if (wantdown)
			    {
			      aliens[i].dir = dirtowards(DIR_DOWNLEFT,
							 aliens[i].dir);
			    }
			  else
			    {
			      aliens[i].dir = dirtowards(DIR_LEFT,
							 aliens[i].dir);
			    }
			}
		      else if (wantright)
			{
			  if (wantup)
			    {
			      aliens[i].dir = dirtowards(DIR_UPRIGHT,
							 aliens[i].dir);
			    }
			  else if (wantdown)
			    {
			      aliens[i].dir = dirtowards(DIR_DOWNRIGHT,
							 aliens[i].dir);
			    }
			  else
			    {
			      aliens[i].dir = dirtowards(DIR_RIGHT,
							 aliens[i].dir);
			    }
			}
		      else
			{
			  if (wantup)
			    aliens[i].dir = dirtowards(DIR_UP,
						       aliens[i].dir);
			  else
			    aliens[i].dir = dirtowards(DIR_DOWN,
						       aliens[i].dir);
			}
		    }
		  
		  
		  /* If they're attacking, take care of making them
		     wrap around the screen: */
		  
		  if (aliens[i].mode == MODE_ATTACK)
		    {
		      /* Some aliens go off the bottom: */
		      
		      /* if (aliens[i].y < -32)
			{
			  aliens[i].y = aliens[i].y + HEIGHT + 32;
			  aliens[i].x = randnum(WIDTH - 32);
			} */
		      
		      if (aliens[i].y > HEIGHT)
			{
			  aliens[i].y = aliens[i].y - HEIGHT - 32;
			  aliens[i].x = randnum(WIDTH - 32);
			  
			  aliens[i].gotox = player.x;
			}
		      
		      
		      /* Masters switch into "suck" mode: */
		      
		      if (someone_hasship == 0 &&
			  player.alive && player.size == 1 &&
			  aliens[i].hasship == 0 &&
			  aliens[i].mode == MODE_ATTACK &&
			  aliens[i].y >= SUCK_HEIGHT &&
			  (aliens[i].type == TYPE_MASTER_0 ||
			   aliens[i].type == TYPE_MASTER_1))
			{
			  aliens[i].mode = MODE_SUCK;
			  aliens[i].sucktime = 50;
			}
		    }
		}
	      else if (aliens[i].mode == MODE_WAIT)
		{
		  /* WAIT mode: */
		  
		  if (aliens[i].dir != DIR_UP)
		    {
		      /* Rotate up if not facing up: */
		      
		      if (counter % 4 == 0)
			aliens[i].dir = dirtowards(DIR_UP, aliens[i].dir);
		    }
		  else
		    {
		      /* Float back and forth */
		      
		      if (level % 3 == 0)
			{
			  if (i % 2 == 0)
			    aliens[i].x = aliens[i].homex + wave_position;
			  else
			    aliens[i].x = aliens[i].homex - wave_position;
			}
		      else if (level % 3 == 1)
			{
			  aliens[i].x = aliens[i].homex + wave_position;
			}
		      else if (level % 3 == 2)
			{
			  j = aliens[i].homex - WIDTH / 2;
			  j = j / 32;
			  
			  aliens[i].x = aliens[i].homex + wave_position * j;
			  aliens[i].y = aliens[i].homey + wave_position;
			}
		    }
		}
	      else if (aliens[i].mode == MODE_SUCK) 
		{
		  /* "suck" mode: */
		  
		  /* Unswitch from "suck" mode: */
		  
		  aliens[i].sucktime--;
		  
		  if (aliens[i].sucktime <= 0)
		    aliens[i].mode = MODE_ENTER;
		  
		  if (aliens[i].dir != DIR_DOWN)
		    {
		      /* Rotate down if not facing down: */
		      
		      if (counter % 4 == 0)
			aliens[i].dir = dirtowards(DIR_DOWN, aliens[i].dir);
		    }
		}
	      
	      
	      /* Handle mode: */
	      
	      if (aliens[i].mode == MODE_ENTER)
		{
		  /* We want to go back home: */
		  
		  aliens[i].gotox = aliens[i].homex;
		  aliens[i].gotoy = aliens[i].homey;
		  
		  
		  /* If we are home, snap into position and switch to
		     wait mode: */
		  
		  if ((aliens[i].x >= aliens[i].homex - 8 &&
		       aliens[i].x <= aliens[i].homex + 8 &&
		       aliens[i].y >= aliens[i].homey - 32 &&
		       aliens[i].y <= aliens[i].homey + 32) ||
		      (aliens[i].x >= aliens[i].homex - 32 &&
		       aliens[i].x <= aliens[i].homex + 32 &&
		       aliens[i].y >= aliens[i].homey - 8 &&
		       aliens[i].y <= aliens[i].homey + 8))
		    {
		      aliens[i].x = aliens[i].homex;
		      aliens[i].y = aliens[i].homey;
		      
		      aliens[i].mode = MODE_WAIT;
		    }
		  
		  
		  /* If we're a "go away" bug, then head for the bottom: */
		  
		  if (aliens[i].x >= 0 && aliens[i].x <= WIDTH - 32 &&
		      aliens[i].goaway == 1)
		    {
		      aliens[i].mode = MODE_ATTACK;
		      aliens[i].gotoy = HEIGHT + 32;
		    }
		  
		}
	      else if (aliens[i].mode == MODE_ATTACK)
		{
		  /* Randomly shoot: */
		  
		  if (randnum(500) <= level / 2 &&
		      player.alive)
		    addbullet(aliens[i].x + 16 - BULLET_SIZE / 2,
			      aliens[i].y + 32,
			      xms[aliens[i].dir], 3, OWNER_ALIEN);
		  
		  
		  /* Switch back to "go home" mode: */
		  
		  if ((((aliens[i].type == TYPE_PAWN ||
			 aliens[i].type == TYPE_PAWNA) &&
			aliens[i].y > HEIGHT - 64) ||
		       ((aliens[i].type == TYPE_GUARD ||
			 aliens[i].type == TYPE_GUARDA) &&
			aliens[i].y < 0)) && aliens[i].goaway == 0)
		    {
		      aliens[i].mode = MODE_ENTER;
		    }
		  
		  
		  /* Face down: */
		  
		  if (counter % 8 == 0)
		    aliens[i].dir = dirtowards(DIR_DOWN, aliens[i].dir);
		  
		  
		  /* If we're a go-away and we've reached the bottom,
		     then go away! */
		  
		  if (aliens[i].y >= HEIGHT - 16 && aliens[i].goaway == 1)
		    {
		      aliens[i].alive = 0;
		    }
		}
	      
	      
	      /* Randomly switch to attack mode: 
		 (or do it if there are few bugs around) */
	      
	      if (player.alive &&
		  aliens[i].mode == MODE_WAIT && 
		  (randnum(1000) < level || num_aliens < 5) &&
		  gamemode == GAMEMODE_NORMAL &&
		  (aliens[i].hasship == 0 || randnum(10) < 1))
		{
		  aliens[i].mode = MODE_ATTACK;
		  aliens[i].gotox = player.x + 16 * player.size;
		  aliens[i].gotoy = HEIGHT;
		  
		  /* See if we make a convoy: */
		  
		  if (aliens[i].type == TYPE_MASTER_0 ||
		      aliens[i].type == TYPE_MASTER_1)
		    {
		      for (j = 0; j < MAX_ALIENS; j++)
			{
			  if (aliens[j].alive == 1 &&
			      aliens[j].mode == MODE_WAIT &&
			      aliens[j].homex == aliens[i].homex &&
			      aliens[j].homey < aliens[i].homey + 64)
			    {
			      aliens[j].mode = MODE_ATTACK;
			      aliens[j].gotox = aliens[i].gotox;
			      aliens[j].gotoy = aliens[i].gotoy + 32;
			    }
			}
		    }
		}
	      
	      
	      /* Detect collisions with bullets: */
	      
	      for (j = 0; j < MAX_BULLETS; j++)
		{
		  if (bullets[j].alive == 1 &&
		      bullets[j].owner == OWNER_PLAYER &&
		      bullets[j].x >= aliens[i].x - BULLET_SIZE &&
		      bullets[j].x <= aliens[i].x + 32 &&
		      bullets[j].y >= (aliens[i].y - BULLET_SIZE -
				       (32 * aliens[i].hasship)) &&
		      bullets[j].y <= aliens[i].y + 32)
		    {
		      killbullet(j);
		      num_hits++;
		      
		      
		      /* Masters take damage, all else die: */
		      
		      if (aliens[i].type == TYPE_MASTER_0)
			{
			  if (giant_bugs == 0 || aliens[i].hitpoints == 0)
			    {
			      aliens[i].type = TYPE_MASTER_1;
			      
			      playsound(sound_fs, SND_HURT);
			    }
			  else
			    {
			      aliens[i].hitpoints--;
			      addexplosion(bullets[j].x, bullets[j].y,
					   0, -2);
			    }
			}
		      else
			{
			  if (aliens[i].type == TYPE_MASTER_1 &&
			      giant_bugs)
			    {
			      flashscreen = 1;
			      score = score + level * 100;
			      
			      addanimatedexplosion(aliens[i].x, aliens[i].y);
			    }
			  else
			    {
			      score = score + scores[aliens[i].type];
			    }
			  
			  aliens[i].alive = 0;
			  
			  playsound(sound_fs, SND_KILL);

			  
			  if (aliens[i].isship == 1)
			    {
			      strcpy(text, "FIGHTER DESTROYED!");
			      show_text = 100;
			      flashscreen = 3;
			    }
			  
			  
			  /* Release the ship: */
			  
			  if (aliens[i].hasship)
			    {
			      /* Make aliens go home: */
			      
			      if (bullets[j].y + 16 >= aliens[i].y)
				{
				  /* If they shot the bug: */
				  
				  if (aliens[i].mode != MODE_WAIT)
				    {
				      /* If it was attacking or something: */
				      
				      gamemode = GAMEMODE_ENTER;
				      
				      for (j = 0; j < MAX_ALIENS; j++)
					aliens[j].mode = MODE_ENTER;
				      
				      extraship.alive = 1;
				      extraship.x = aliens[i].x;
				      extraship.y = aliens[i].y - 32;
				      
				      strcpy(text, "FIGHTER FREED!");
				      show_text = 75;
				    }
				  else
				    {
				      /* Oops, they shot the bug and now
					 the ship is stuck! */
				      
				      aliens[i].alive = 1;
				      aliens[i].hasship = 0;
				      aliens[i].isship = 1;
				    }
				}
			      else
				{
				  /* If they shot the ship: */
				  
				  flashscreen = 3;
				  aliens[i].hasship = 0;
				  aliens[i].alive = 1;
				  
				  strcpy(text, "FIGHTER DESTROYED!");
				  show_text = 100;
				}
			    }
			}
		      
		      
		      /* Add explosion: */
		      
		      addexplosion(bullets[j].x, bullets[j].y, 0, 0);
		    }
		}
	      
	      
	      /* Detect collisions with player: */
	      
	      if (aliens[i].x + 32 >= player.x &&
		  aliens[i].x <= player.x + (32 * player.size) &&
		  aliens[i].y + 32 >= HEIGHT - 32 &&
		  player.alive)
		{
		  if (aliens[i].type != TYPE_MASTER_0 &&
		      aliens[i].type != TYPE_MASTER_1 &&
		      giant_bugs == 0)
		    {
		      aliens[i].alive = 0;
		      score = score + scores[aliens[i].type];
		    }
		  
		  killplayer(aliens[i].x);
		}
	      
	      
	      /* Detect alien zap: */
	      
	      if (player.alive &&
		  aliens[i].mode == MODE_SUCK &&
		  aliens[i].x + 32 >= player.x &&
		  aliens[i].x <= (player.x + 32 * player.size))
		{
		  player.alive = 0;
		  gamemode = GAMEMODE_SUCK;
		  player.y = HEIGHT - 32;
		  alien_responsible = i;
		  aliens[i].sucktime = 1000;
		}
	    }
	}
      
      
      /* Find out if they are all at home yet or not: */
      
      all_in_place = 1;
      
      for (i = 0; i < MAX_ALIENS; i++)
	{
	  if (aliens[i].alive && aliens[i].mode != MODE_WAIT)
	    /* aliens[i].x != aliens[i].homex &&
	      aliens[i].y != aliens[i].homey) */
	    all_in_place = 0;
	}
      
      
      /* Count number of bullets: */
      
      num_bullets = 0;
      for (i = 0; i < MAX_BULLETS; i++)
	{
	  if (bullets[i].alive)
	    num_bullets++;
	}
      
      
      /* Switch to normal game mode if they are all in place */

      if (all_in_place == 1 && gamemode == GAMEMODE_ENTER)
	{
	  gamemode = GAMEMODE_NORMAL;
	}
      
      
      /* If you're gone, make you come back: */
      
      if (player.alive == 0 && gamemode != GAMEMODE_SUCK)
	{
	  if (num_bullets == 0 && all_in_place == 1 &&
	      gameover == 0)
	    player.alive = 1;
	}


      /* Handle extra ship: */
      
      if (extraship.alive && paused == 0)
	{
	  if (extraship.x > player.x)
	    extraship.x--;
	  else if (extraship.x < player.x)
	    extraship.x++;
	  
	  if (extraship.y < HEIGHT - 32)
	    {
	      extraship.y = extraship.y + 4;
	    }
	  
	  if (extraship.y >= HEIGHT - 32 - 32 &&
	      extraship.x + 32 >= player.x &&
	      extraship.x <= player.x + 32)
	    {
	      extraship.alive = 0;
	      
	      if (player.alive == 1)
		{
		  if (extraship.x < player.x)
		    player.x = player.x - 32;
		  
		  player.size = 2;
		}
	      else
		{
		  player.size = 1;
		  player.alive = 1;
		}
	    }
	}
      
      
      /* Handle bullets: */
      
      for (i = 0; i < MAX_BULLETS; i++)
	{
	  if (bullets[i].alive == 1 && paused == 0)
	    {
	      /* Move bullet: */
	      
	      bullets[i].x = bullets[i].x + bullets[i].xm;
	      bullets[i].y = bullets[i].y + bullets[i].ym;
	      
	      
	      /* Kill it if it goes offscreen: */
	      
	      if (bullets[i].x < -BULLET_SIZE || bullets[i].x >= WIDTH ||
		  bullets[i].y < -BULLET_SIZE || bullets[i].y >= HEIGHT)
		{
		  killbullet(i);
		}
	      
	      
	      /* Detect collision with player: */
	      
	      if (bullets[i].x + BULLET_SIZE >= player.x &&
		  bullets[i].x <= player.x + 32 * player.size &&
		  bullets[i].y - BULLET_SIZE >= HEIGHT - 32 &&
		  bullets[i].owner != OWNER_PLAYER &&
		  player.alive)
		{
		  killbullet(i);
		  
		  killplayer(bullets[i].x);
		}
	    }
	}
      
      
      /* Move explosion bits: */
      
      for (i = 0; i < MAX_EXPLOSIONBITS; i++)
	{
	  if (explosionbits[i].alive == 1)
	    {
	      explosionbits[i].x = explosionbits[i].x + explosionbits[i].xm;
	      explosionbits[i].y = explosionbits[i].y + explosionbits[i].ym;
	      
	      explosionbits[i].time--;
	      if (explosionbits[i].time <= 0)
		explosionbits[i].alive = 0;
	    }
	}
      
      
      /* Move explosions: */
      
      for (i = 0; i < MAX_EXPLOSIONS; i++)
	{
	  if (explosions[i].alive == 1)
	    {
	      if (counter % 4 == 0)
		{
		  explosions[i].frame++;
		  if (explosions[i].frame >= NUM_EXPLODE_SHAPES)
		    explosions[i].alive = 0;
		}
	    }
	}
      
      
      /* Handle asteroids: */
      
      for (i = 0; i < MAX_ASTEROIDS; i++)
	{
	  if (asteroids[i].alive == 1)
	    {
	      asteroids[i].x = asteroids[i].x + asteroids[i].xm;
	      asteroids[i].y = asteroids[i].y + asteroids[i].ym;
	      
	      if (counter % asteroids[i].animspeed == 0)
		{
		  asteroids[i].anim = (asteroids[i].anim + asteroids[i].animm);
		  if (asteroids[i].anim < 0)
		    asteroids[i].anim = 4;
		  else if (asteroids[i].anim > 4)
		    asteroids[i].anim = 0;
		}
	      
	      if (asteroids[i].x < -100 || asteroids[i].x > WIDTH ||
		  asteroids[i].y < -100 || asteroids[i].y > HEIGHT)
		asteroids[i].alive = 0;
	      
	      if (asteroids[i].size == 1)
		{
		  w = object_width[OBJ_ASTEROID_A];
		  h = object_width[OBJ_ASTEROID_A];
		}
	      else
		{
		  w = object_width[OBJ_ASTEROID_B];
		  h = object_width[OBJ_ASTEROID_B];
		}
	      
	      for (j = 0; j < MAX_BULLETS; j++)
		{
		  if (bullets[j].alive == 1)
		    {
		      if (bullets[j].x + BULLET_SIZE > asteroids[i].x && 
			  bullets[j].x < asteroids[i].x + w &&
			  bullets[j].y + BULLET_SIZE > asteroids[i].y &&
			  bullets[j].y < asteroids[i].y + h)
			{
			  killbullet(j);
			  addexplosion(bullets[j].x, bullets[j].y,
				       asteroids[i].xm, asteroids[i].ym);
			  
			  score = score + 150 - (50 * asteroids[i].size);
			  
			  asteroids[i].size--;
			  
			  if (asteroids[i].size == 0)
			    asteroids[i].alive = 0;
			  else
			    {
			      asteroids[i].xm = randnum(11) - 5;
			      asteroids[i].ym = randnum(11) - 5;
			      asteroids[i].animm = randnum(3) - 1;
			      asteroids[i].animspeed = randnum(5) + 1;
			      
			      if (randnum(10) < 5)
				addasteroid(asteroids[i].x,
					    asteroids[i].y,
					    (randnum(11) - 5),
					    (randnum(11) - 5), 1);
			    }
			}
		    }
		}
	      
	      if (player.x + 32 * player.size > asteroids[i].x &&
		  player.x < asteroids[i].x + w &&
		  asteroids[i].y + h > HEIGHT - 32 && player.alive == 1)
		{
		  killplayer(asteroids[i].x);
		}
	    }
	  else
	    {
	      if (randnum(10000) < level && level > 20)
		{
		  addasteroid(-1, -1, 0, 0, randnum(2) + 1);
		}
	    }
	}
      
      
      /* Move stars: */
      
      for (i = 0; i < MAX_STARS; i++)
	{
	  if (gamemode != GAMEMODE_SUCK)
	    {
	      stars[i].y = stars[i].y + starspeed + stars[i].speed;
	      
	      if (stars[i].y > HEIGHT)
		{
		  stars[i].x = randnum(WIDTH);
		  stars[i].speed = randnum(5);
		  stars[i].y = -randnum(20);
		}
	    }
	  else
	    {
	      stars[i].y = stars[i].y - starspeed - stars[i].speed;
	      
	      if (stars[i].y < 0)
		{
		  stars[i].x = randnum(WIDTH);
		  stars[i].speed = randnum(5);
		  stars[i].y = HEIGHT + randnum(20);
		}
	    }
	}
      
      
      /* Move planet: */
      
      if (planet.alive == 1)
	{
	  if (gamemode != GAMEMODE_SUCK)
	    {
	      planet.y = planet.y + starspeed + 5;
	      
	      if (planet.y > HEIGHT)
		planet.alive = 0;
	    }
	  else
	    {
	      planet.y = planet.y - (starspeed + 5);
	    }
	}
      else
	{
	  if (randnum(1000) < 1)
	    {
	      planet.alive = 1;
	      planet.x = randnum(WIDTH + 100) - 100;
	      planet.y = -100;
	      planet.shape = randnum(NUM_PLANET_SHAPES);
	    }
	}
      
      
      /* Handle star speed: */
      
      if (counter % 5 == 0)
	starspeed = starspeed + starspeedm;
      
      if (starspeed > faststarspeed)
	{
	  starspeed = faststarspeed;
	  starspeedm = 0;
	  starspeedtime = 50;
	}
      
      if (starspeedtime > 0)
	{
	  starspeedtime = starspeedtime - 1;
	  
	  if (starspeedtime <= 0)
	    {
	      if (starspeed == faststarspeed)
		starspeedm = -changestarspeed;
	      else
		starspeedm = changestarspeed;
	    }
	}
      
      if (starspeed <= wantstarspeed && starspeedm < 0)
	{
	  starspeed = wantstarspeed;
	  starspeedm = 0;
	  gamemode = GAMEMODE_ENTER;
	}
      
      
      /* Erase backbuffer */
      
      if (flashscreen <= 0)
	XFillRectangle(display, backbuffer, blackgc, 0, 0, WIDTH, HEIGHT);
      else
	{
	  XFillRectangle(display, backbuffer, whitegc, 0, 0, WIDTH, HEIGHT);
	  flashscreen--;
	}
      
      
      /* Draw stars: */
      
      for (i = 0; i < MAX_STARS; i++)
	{
	  if (starspeed < 4)
	    XDrawPoint(display, backbuffer, greygc, stars[i].x, stars[i].y);
	  else
	    XDrawLine(display, backbuffer, greygc, stars[i].x, stars[i].y,
		      stars[i].x, stars[i].y - starspeed);
	}
      
      
      /* Draw plent: */
      
      if (planet.alive == 1)
	{
	  drawshape(planet.x, planet.y, OBJ_PLANET_0 + planet.shape,
		    backbuffer);
	}
      
      
      /* Draw bullets: */
      
      for (i = 0; i < MAX_BULLETS; i++)
	{
	  if (bullets[i].alive == 1)
	    {
	      if (bullets[i].owner == OWNER_PLAYER)
		drawshape(bullets[i].x, bullets[i].y, OBJ_BULLET, backbuffer);
	      else
		drawshape(bullets[i].x, bullets[i].y, OBJ_ALIENBULLET,
			  backbuffer);
	    }
	}
      
      
      /* Draw explosionbits: */
      
      for (i = 0; i < MAX_EXPLOSIONBITS; i++)
	{
	  if (explosionbits[i].alive == 1)
	    {
	      XDrawLine(display, backbuffer,
			explosiongcs[randnum(NUM_EXPLOSION_COLORS)],
			explosionbits[i].x, explosionbits[i].y,
			explosionbits[i].x + explosionbits[i].xm,
			explosionbits[i].y + explosionbits[i].ym);
	    }
	}
      
      
      /* Draw explosions: */
      
      for (i = 0; i < MAX_EXPLOSIONS; i++)
	{
	  if (explosions[i].alive == 1)
	    {
	      j = OBJ_EXPLODE + explosions[i].frame;
	      
	      drawshape(explosions[i].x - object_width[j] / 2,
			explosions[i].y - object_height[j] / 2,
			j, backbuffer);
	    }
	}
      
      
      /* Draw "Level": */
      
      if (gamemode == GAMEMODE_WARP)
	{
	  drawshape(WIDTH / 2 - 12, HEIGHT / 2, level / 10, backbuffer);
	  drawshape(WIDTH / 2 + 12, HEIGHT / 2, level % 10, backbuffer);
	}
      
      
      /* Draw aliens: */
      
      for (i = 0; i < MAX_ALIENS; i++)
	{
	  if (aliens[i].alive == 1 && gamemode != GAMEMODE_WARP)
	    {
	      if (aliens[i].isship == 0)
		{
		  if (giant_bugs &&
		      (aliens[i].type == TYPE_MASTER_0 ||
		       aliens[i].type == TYPE_MASTER_1))
		    {
		      drawshape(aliens[i].x - 29, aliens[i].y - 62,
				OBJ_GIANT_0 + (counter % 2),
				backbuffer);
		    }
		  else
		    {
		      if (aliens[i].mode == MODE_WAIT && wingflap == 0)
			drawshape(aliens[i].x, aliens[i].y,
				  OBJ_ALIEN_0 + aliens[i].type * 9 + 8,
				  backbuffer);
		      else
			drawshape(aliens[i].x, aliens[i].y,
				  (OBJ_ALIEN_0 + aliens[i].type * 9 +
				   aliens[i].dir), backbuffer);
		    }
		}
	      else
		{
		  drawshape(aliens[i].x, aliens[i].y,
			    OBJ_SHIP_1 + aliens[i].dir,
			    backbuffer);
		}
	      
	      
	      /* Draw ship next to alien: */
	      
	      if (aliens[i].hasship)
		drawshape(aliens[i].x, aliens[i].y - 32,
			  OBJ_SHIP_1 + (aliens[i].dir * (1 - giant_bugs)),
					backbuffer);
	      
	      
	      /* Draw tractor beam: */
	      
	      if (aliens[i].mode == MODE_SUCK)
		{
		  drawshape(aliens[i].x - 16, aliens[i].y +32,
			    OBJ_TRACTORBEAM, backbuffer);
		  
		  if (counter % 10 == 0 && gamemode != GAMEMODE_SUCK)
		    playsound(sound_fs, SND_TRACTORBEAM);
		}
	      
#ifdef ATTACH_SELF
	      if (aliens[i].mode == MODE_ATTACK)
		XDrawLine(display, backbuffer, whitegc,
			  aliens[i].x + 16, aliens[i].y + 16,
			  player.x + 16, HEIGHT - 16);
	      else if (aliens[i].mode == MODE_ENTER)
		XDrawLine(display, backbuffer, whitegc,
			  aliens[i].x + 16, aliens[i].y + 16,
			  aliens[i].homex + 16, aliens[i].homey + 16);
#endif /* ATTACH_SELF */
	    }
	}
      

      /* Draw asteroids: */
      
      for (i = 0; i < MAX_ASTEROIDS; i++)
	{
	  if (asteroids[i].alive == 1)
	    {
	      drawshape(asteroids[i].x, asteroids[i].y,
			OBJ_ASTEROID_A + asteroids[i].anim +
			(asteroids[i].size * 5) - 5, backbuffer);
	    }
	}
      
      
      /* Draw ship: */
      
      if (player.alive)
	{
	  for (i = 0; i < player.size; i++)
	    drawshape(player.x + i * 32, HEIGHT - 32, OBJ_SHIP_0, backbuffer);
	}
      
      
      /* Draw score and level: */
      
      sprintf(temp, "Score: %.6d", score);
      drawtext(display, backbuffer, whitegc, 0, fh, temp);
      
      sprintf(temp, "Level: %.2d", level);
      drawtext(display, backbuffer, whitegc,
	       (WIDTH - (XTextWidth(font, "x", 1) * 9)) / 2, fh, temp);

      sprintf(temp, "Lives: %.2d", lives);
      drawtext(display, backbuffer, whitegc,
	       (WIDTH - (XTextWidth(font, "x", 1) * 9)), fh, temp);
      
      
      if (show_text > 0)
	{
	  drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH,
			   (HEIGHT - fh) / 2, text, font);
	  
	  show_text--;
	}
      
      
      /* Draw game over: */
      
      if (paused == 1 && (counter % 10) < 5)
	{
	  drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH, HEIGHT / 2,
		   "PAUSED", font);
	}
      
      if (gameover == 1)
	{
	  drawtext(display, backbuffer, whitegc,
		   (WIDTH - (XTextWidth(font, "x", 1) * 9)) / 2, HEIGHT / 2,
		   "GAME OVER");
	  
	  if (num_shots != 0)
	    {
	      sprintf(temp, "Hit Ratio = %.2f%%",
		      (num_hits * 100.0) / num_shots);
	      drawtext(display, backbuffer, greygc,
		       (WIDTH - (XTextWidth(font, "x", 1) * strlen(temp))) / 2,
		       HEIGHT / 2 + fh, temp);
	    }
	}
      

      /* Draw extra ship: */
      
      if (extraship.alive)
	{
	  drawshape(extraship.x, extraship.y,
		    OBJ_SHIP_0 + (counter / 4) % 8, backbuffer);
	}
      
      
      /* Draw ship getting sucked up */
      
      if (gamemode == GAMEMODE_SUCK)
	{
	  if (player.y > SUCK_HEIGHT + 32)
	    drawshape(player.x, player.y, OBJ_SHIP_0 + ((counter / 4) % 8),
		      backbuffer);
	  else
	    drawshape(player.x, player.y, OBJ_SHIP_1 + ((counter / 4) % 8),
		      backbuffer);
	  
	  if (counter % 2 == 0)
	    {
	      if (aliens[alien_responsible].alive)
		{
		  player.y = player.y - 4;
		  
		  if (counter % 6 == 0)
		    playsound(sound_fs, SND_CAPTURE);
		}
	      else
		player.y = player.y + 2;
	    }
	  
	  if (player.y <= SUCK_HEIGHT - 32 ||
	      (player.y >= HEIGHT - 32 &&
	       aliens[alien_responsible].alive == 0))
	    {
	      gamemode = GAMEMODE_ENTER;
	      
	      for (i = 0; i < MAX_ALIENS; i++)
		{
		  if (aliens[i].goaway == 0)
		    aliens[i].mode = MODE_ENTER;
		}
	      
	      if (player.y <= SUCK_HEIGHT - 32)
		{
		  aliens[alien_responsible].hasship = 1;
		  turnoffplayer();
		  
		  strcpy(text, "FIGHTER CAPTURED!");
		  show_text = 100;
		}
	      else
		player.alive = 1;
	    }
	  
	  if (player.x > aliens[alien_responsible].x)
	    player.x--;
	  else if (player.x < aliens[alien_responsible].x)
	    player.x++;
	}
      
      
      updatebackbuffer();
      XSync(display, 0);
      /* XFlush(display); */
      
      
      /* Keep game speed normal: */
      
      gettimeofday(&now, NULL);
      
      time_padding = FRAMERATE - ((now.tv_sec - then.tv_sec) * 1000000 +
                                  (now.tv_usec - then.tv_usec));
      
      if (time_padding > 0)
	usleep(time_padding);
      
      if (counter % 6 == 0 && giant_bugs == 1 && paused == 0 && gameover == 0)
	playsound(sound_fs, SND_BUZZ);
    }
  while (done == 0);
  
  if (checkforhigh() == 1)
    return(TITLEMODE_HIGHS);
  else
    return(TITLEMODE_TITLE);
}


int checkforhigh(void)
{
  int found, i, gotone;
  char name[5];
  
  
  gotone = 0;
  
  
  /* Is it a high score? */
  
  strcpy(name, "---");
  
  found = -1;
  
  for (i = 0; i < 10 && found == -1; i++)
    {
      if (score > highscores[i].score)
	{
	  found = i;
	}
    }
  
  if (found != -1)
    {
      gotone = 1;
      
      getname(name, "High Score");
      
      for (i = 9; i > found; i--)
	{
	  highscores[i].score = highscores[i - 1].score;
	  strcpy(highscores[i].name, highscores[i - 1].name);
	}
      
      highscores[found].score = score;
      strcpy(highscores[found].name, name);
      
      highscoreslot = found;
    }

  
  /* Is it a high level? */
  
  found = -1;
  
  for (i = 0; i < 10 && found == -1; i++)
    {
      if (level > highlevels[i].score)
	{
	  found = i;
	}
    }
  
  if (found != -1)
    {
      gotone = 1;
      
      getname(name, "High Level");
      
      for (i = 9; i > found; i--)
	{
	  highlevels[i].score = highlevels[i - 1].score;
	  strcpy(highlevels[i].name, highlevels[i - 1].name);
	}
      
      highlevels[found].score = level;
      strcpy(highlevels[found].name, name);

      highlevelslot = found;
    }
  
  if ((num_hits * 100.0) / num_shots >= highhitratio)
    {
      gotone = 1;
      
      getname(name, "High Hit Ratio");
      
      highhitratio = (num_hits * 100.0) / num_shots;
      strcpy(highhitrationame, name);
      
      highhitratioslot = 0;
    }
  
  return(gotone);
}


/* Get initials for high score/level/hr */

void getname(char * buf, char * title)
{
  int done, x, i, j;
  char string[128], key[128], temp[128];
  XEvent event;
  KeySym keysym;
  XComposeStatus composestatus;
  struct timeval now, then;
  long time_padding;
  fireworks_type fireworks[MAX_FIREWORKS];
  
  
  done = 0;
  x = 0;

  for (i = 0; i < MAX_FIREWORKS; i++)
    fireworks[i].alive = 0;
  
  silence(sound_fs);
  
  do
    {
      gettimeofday(&then, NULL);
      
      
      /* Redraw screen: */
      
      XFillRectangle(display, backbuffer, blackgc, 0, 0, WIDTH, HEIGHT);
      
      
      /* Draw fireworks: */
      
      for (i = 0; i < MAX_FIREWORKS; i++)
	{
	  if (fireworks[i].alive)
	    {
	      fireworks[i].time--;
	      
	      if (fireworks[i].time > 0)
		{
		  /* Move and draw firework missile: */
		  
		  fireworks[i].x = fireworks[i].x + fireworks[i].xm;
		  fireworks[i].y = fireworks[i].y + fireworks[i].ym;
		  fireworks[i].ym = fireworks[i].ym + 0.125;
		  
		  XDrawLine(display, backbuffer, whitegc,
			    fireworks[i].x, fireworks[i].y,
			    fireworks[i].x + fireworks[i].xm,
			    fireworks[i].y + fireworks[i].ym);
		}
	      else if (fireworks[i].time == 0)
		{
		  /* Explode!: */
		  
		  playsound(sound_fs, SND_FIREWORKS);
		  
		  for (j = 0; j < MAX_FIREWORKSBITS; j++)
		    {
		      fireworks[i].bit[j].x = fireworks[i].x;
		      fireworks[i].bit[j].y = fireworks[i].y;
		      fireworks[i].bit[j].xm = ((randnum(70) - 30) / 10.0 +
						fireworks[i].xm);
		      fireworks[i].bit[j].ym = ((randnum(70) - 30) / 10.0 +
						fireworks[i].ym);
		    }
		}
	      else if (fireworks[i].time < 0)
		{
		  /* Draw firework bits: */
		  
		  for (j = 0; j < MAX_FIREWORKSBITS; j++)
		    {
		      fireworks[i].bit[j].x = fireworks[i].bit[j].x +
			fireworks[i].bit[j].xm;
		      fireworks[i].bit[j].y = fireworks[i].bit[j].y +
			fireworks[i].bit[j].ym;
		      fireworks[i].bit[j].ym = fireworks[i].bit[j].ym + 0.125;
		      
		      XDrawPoint(display, backbuffer,
				 explosiongcs[randnum(NUM_EXPLOSION_COLORS)],
				 fireworks[i].bit[j].x, fireworks[i].bit[j].y);
		    }
		}
	      
	      if (fireworks[i].time < -50)
		fireworks[i].alive = 0;
	    }
	  else
	    {
	      if (randnum(10) < 2)
		{
		  fireworks[i].alive = 1;
		  fireworks[i].x = randnum(WIDTH);
		  fireworks[i].y = HEIGHT;
		  fireworks[i].ym = -randnum(10) - 1;
		  fireworks[i].xm = (randnum(30)) / 10.0;
		  if (fireworks[i].x > WIDTH / 2)
		    fireworks[i].xm = -fireworks[i].xm;
		  fireworks[i].time = randnum(50) + 20;
		}
	    }
	}
      
      
      /* Draw text: */
      
      drawcenteredtext(display, backbuffer,
		       explosiongcs[randnum(NUM_EXPLOSION_COLORS)], 0, WIDTH,
		       (HEIGHT - fh) / 2 - fh, "Congratulations!", font);
      
      drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH,
		       (HEIGHT - fh) / 2, title, font);
      
      strcpy(temp, "     ");
      temp[x + 1] = '~';
      
      drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH,
		       (HEIGHT - fh) / 2 + fh * 2, buf, font);
      
      drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH,
		       (HEIGHT - fh) / 2 + fh * 3, temp, font);
      
      
      /* Get keys: */
      
      strcpy(key, "");
      
      while (XPending(display))
	{
	  XNextEvent(display, &event);
	  
	  if (event.type == KeyPress)
	    {
	      /* Get the key's name: */
	      
	      XLookupString(&event.xkey, temp, 1, &keysym,
			    &composestatus);
	      
	      if (XKeysymToString(keysym) != NULL)
		{
		  strcpy(temp, XKeysymToString(keysym));
		  
		  if (strcmp(temp, "Left") == 0 ||
		      strcmp(temp, "KP Left") == 0 ||
		      strcmp(temp, "BackSpace") == 0 ||
		      strcmp(temp, "Delete") == 0)
		    {
		      if (x > 0)
			{
			  x--;
			  buf[x] = '-';
			}
		      else
			XBell(display, 50);
		    }
		  else if (strcmp(temp, "Return") == 0 ||
			   strcmp(temp, "Enter") == 0)
		    {
		      playsound(sound_fs, SND_KILL);
		      sleep(1);
		      done = 1;
		    }
		  else if (event.type == KeyPress)
		    {
		      strcpy(key, temp);
		      
		      if (strlen(key) == 1)
			{
			  if (x < 3)
			    {
			      buf[x] = toupper(key[0]);
			      x++;
			      
			      playsound(sound_fs, SND_HURT);
			    }
			  else
			    XBell(display, 50);
			}
		    }
		}
	    }
	}
      
      
      /* Refresh: */
      
      updatebackbuffer();
      XSync(display, 0);
      
      
      /* Pause: */
      
      gettimeofday(&now, NULL);
      
      time_padding = FRAMERATE - ((now.tv_sec - then.tv_sec) * 1000000 +
                                  (now.tv_usec - then.tv_usec));
      
      if (time_padding > 0)
	usleep(time_padding);
    }
  while (done == 0);
}

  
/* Reset level (graphics, shapes, objects, etc.): */

void initlevel(void)
{
  int i, j, l;
  
  
  /* Init aliens: */

  for (i = 0; i < MAX_ALIENS; i++)
    aliens[i].alive = 0;

  
  for (i = 0; i < MAX_ALIENS; i++)
    {
      if (i < 10 || (i < 20 && level > 10))
	{
	  aliens[i].alive = 1;
	  
	  aliens[i].hasship = 0;
	  aliens[i].isship = 0;
	  
	  
	  /* Alien types: */
	  
	  if (level % 2 == 0)
	    {
	      aliens[i].type = TYPE_GUARD;
	      if (i >= 5)
		aliens[i].type = TYPE_PAWN;
	    }
	  else
	    {
	      aliens[i].type = TYPE_GUARDA;
	      if (i >= 5)
		aliens[i].type = TYPE_PAWNA;
	    }
	  
	  aliens[i].mode = MODE_ENTER;
	  
	  
	  /* Set their home positions: */

	  aliens[i].homex = 24 * (i % 5) * 2 + 80 - 32;
	  aliens[i].homey = 48;
	  if (i >= 5 && i < 10)
	    aliens[i].homey = 80;
	  else if (i >= 10 && i < 15)
	    aliens[i].homey = 112;
	  
	  
	  aliens[i].gotox = aliens[i].homex;
	  aliens[i].gotoy = aliens[i].homey;
	  
	  
	  /* Do they escape or join the crew? */
	  
	  aliens[i].goaway = 0;
	  if ((i >= 10 && level < 10) || i >= 15)
	    aliens[i].goaway = 1;
	  
	  aliens[i].dir = DIR_UP;
	  
	  
	  /* Starting positions: */
	  
	  l = level % 4;
	  
	  if (l == 0)
	    {
	      aliens[i].x = -32 - i * 20;
	      if (i % 2 == 0)
		aliens[i].x = WIDTH + i * 20;
	      
	      aliens[i].y = HEIGHT / 2 + (i % 5) * 10;
	    }
	  else if (l == 1)
	    {
	      aliens[i].x = -32;
	      if (i % 2 == 0)
		aliens[i].x = WIDTH;
	      
	      aliens[i].y = HEIGHT / 2 + (i % 5) * 10;
	    }
	  else if (l == 2)
	    {
	      aliens[i].x = -32 - i * 20;
	      if (i % 2 == 0)
		aliens[i].x = WIDTH + i * 20;
	      
	      aliens[i].y = (HEIGHT * 2) / 3;
	    }
	  else if (l == 3)
	    {
	      aliens[i].x = -32 - i * 20;
	      if (i % 2 == 0)
		aliens[i].x = WIDTH + i * 20;
	      
	      aliens[i].y = HEIGHT - 32 - (i / 2) * 16;
	    }
	}
    }
  
  
  /* Init master aliens: */
  
  for (i = 0; i < 2; i++)
    {
      j = MAX_ALIENS - 2 + i;
      
      aliens[j].alive = 1;
      aliens[j].type = TYPE_MASTER_0;
      
      aliens[j].homex = WIDTH / 2 - 48 + (i * 64);
      aliens[j].homey = 16;
      
      aliens[j].gotox = WIDTH / 2 - 16;
      aliens[j].gotoy = HEIGHT / 3;
      
      aliens[j].x = (WIDTH + 32) * i - 32;
      aliens[j].y = HEIGHT / 2;
      
      aliens[j].goaway = 0;
      
      aliens[j].dir = DIR_UP;
      aliens[j].mode = MODE_ENTER;
      
      aliens[j].hasship = 0;
      aliens[j].isship = 0;
      
      aliens[j].hitpoints = 10;
    }
  
  
  /* Init star speed: */
  
  wantstarspeed = (level / 5) + 1;
  faststarspeed = ((level / 5) + 1) * 10;
  changestarspeed = (level / 5) + 1;

  starspeed = wantstarspeed;
  starspeedm = 0;
  starspeedtime = 30;
  gamemode = GAMEMODE_WARP;
  
  
  /* Init alien speed: */
  
  alien_speed = level / 3;
  if (alien_speed < 1)
    alien_speed = 1;
  else if (alien_speed > 4)
    alien_speed = 4;
  
  
  /* Giant bug mode? */
  
  giant_bugs = 0;
  if (level % 10 == 0)
    giant_bugs = 1;
}


/* Reset game (positions, etc.): */

void initgame(void)
{
  int i;
  
  
  /* Center player: */
  
  player.alive = 1;
  player.size = 1;
  player.x = WIDTH / 2;
  
  extraship.alive = 0;
  
  lives = 3;
  score = 0;
  gameover = 0;
  num_shots = 0.0;
  num_hits = 0.0;
  
  
  /* Init bullets: */
  
  for (i = 0; i < MAX_BULLETS; i++)
    bullets[i].alive = 0;
  
  player.shots_out = 0;

  
  /* Init explosion bits: */
  
  for (i = 0; i < MAX_EXPLOSIONBITS; i++)
    explosionbits[i].alive = 0;
  
  /* Init explosions: */
  
  for (i = 0; i < MAX_EXPLOSIONS; i++)
    explosions[i].alive = 0;
  
  
  /* Init stars: */
  
  initstars();
  
  
  /* Init planet: */
  
  planet.alive = 0;


  /* Init. asteroids: */
  
  for (i = 0; i < MAX_ASTEROIDS; i++)
    asteroids[i].alive = 0;
}


/* Reset stars: */

void initstars(void)
{
  int i;
  
  
  /* Starting position for stars: */
  
  for (i = 0; i < MAX_STARS; i++)
    {
      stars[i].x = randnum(WIDTH);
      stars[i].y = randnum(HEIGHT);
      stars[i].speed = randnum(5);
    }
}


/* Determine which way to turn to change direction from curdir to wantdir  */

int dirtowards(int wantdir, int curdir)
{
  int m, newdir;
  
  
  if (wantdir < curdir)
    wantdir = wantdir + 8;
  
  if (wantdir == curdir)
    m = 0;
  else if (wantdir - curdir > 4)
    m = -1;
  else if (wantdir - curdir <= 4)
    m = 1;
  
  newdir = curdir + m;
  
  if (newdir < 0)
    newdir = 7;
  else if (newdir > 7)
    newdir = 0;
  
  return(newdir);
}


/* Adds a bullet: */

void addbullet(int x, int y, int xm, int ym, int owner)
{
  int found, i;

  
  found = -1;
  
  for (i = 0; i < MAX_BULLETS; i++)
    if (bullets[i].alive == 0)
      found = i;
  
  if (found != -1)
    {
      bullets[found].alive = 1;
      bullets[found].x = x;
      bullets[found].y = y;
      bullets[found].xm = xm;
      bullets[found].ym = ym;
      bullets[found].owner = owner;
    }
}


/* Adds a few bits of explosion: */

void addexplosion(int x, int y, int xm, int ym)
{
  int i;
  
  
  for (i = 0; i < 5; i++)
    addexplosionbit(x + randnum(5) - 2, y - randnum(5) + 2,
		    xm + randnum(5) - 2, ym + randnum(5) - 2);
}


/* Adds a bit of explosion: */

void addexplosionbit(int x, int y, int xm, int ym)
{
  int i, found;
  
  found = -1;
  
  for (i = 0; i < MAX_EXPLOSIONBITS; i++)
    {
      if (explosionbits[i].alive == 0)
	found = i;
    }
  
  if (found != -1)
    {
      explosionbits[found].alive = 1;
      explosionbits[found].time = 10 + randnum(20);
      
      explosionbits[found].x = x;
      explosionbits[found].y = y;
      explosionbits[found].xm = xm / 2;
      explosionbits[found].ym = ym / 2;
      
      if (randnum(10) < 3)
	explosionbits[found].ym = -ym;
    }
}


/* Add explosion, remove player (or one of the two ships if there are two): */

void killplayer(int x)
{
  int i;
  

  /* Make explosion: */

  addanimatedexplosion(x, HEIGHT - 16);
  
  
  /* Kill or hurt player: */
  
  if (player.size > 1)
    {
      /* Lost one ship: */
      
      player.size--;
      
      if (x < player.x + 32)
	player.x = player.x + 32;
    }
  else
    {
      /* Dead! */
      
      turnoffplayer();


      /* Make all aliens go home: */
      
      for (i = 0; i < MAX_ALIENS; i++)
	{
	  if (aliens[i].alive && aliens[i].goaway == 0)
	    aliens[i].mode = MODE_ENTER;
	}
      
      gamemode = GAMEMODE_ENTER;
    }
  
  flashscreen = 4;
}


/* Kill player, remove a life, end game if no more lives: */

void turnoffplayer(void)
{
  player.alive = 0;
  lives--;
  
  if (lives == 0)
    gameover = 1;
}   


/* Call drawshape for title screens: */

void helpshape(int pos, int y, int obj)
{
  int x;
  
  if (pos == HELPSHAPE_CENTER)
    x = (WIDTH - object_width[obj]) / 2;
  else if (pos == HELPSHAPE_LEFT)
    x = WIDTH / 4 - object_width[obj] / 2;
  else if (pos == HELPSHAPE_RIGHT)
    x = (WIDTH * 3) / 4 - object_width[obj] / 2;
  
  drawshape(x, y, obj, backbuffer);
}


/* Draw help screen's text: */

void helptext(int y, char * text)
{
  drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH, y + 32 + fh,
		   text, font);
}


/* Remove bullet. Count one less bullet out if this was the player's: */

void killbullet(int which)
{
  if (bullets[which].owner == OWNER_PLAYER)
    player.shots_out--;
  
  bullets[which].alive = 0;
}


/* Load highs from disk or get defaults: */

void loadhighscores(void)
{
  FILE * fi;
  int i, z;
  
  
  /* Open file: */
  
  fi = fopen("galaxa-highs.dat", "r");
  
  if (fi == NULL)
    {
      /* Set defaults: */
      
      perror("galaxa-highs.dat");
      
      for (i = 0; i < 10; i++)
	{
	  highscores[i].score = 10000 - i * 750;
	  
	  z = randnum(3);
	  
	  if (z == 0)
	    strcpy(highscores[i].name, "NBS");
	  else if (z == 1)
	    strcpy(highscores[i].name, "BJK");
	  else if (z == 2)
	    strcpy(highscores[i].name, "GAL");
	  
	  
	  highlevels[i].score = 5;
	  strcpy(highlevels[i].name, "BJK");
	}
      
      highhitratio = 50.0;
      strcpy(highhitrationame, "FJI");
    }
  else
    {
      /* Load from file: */
      
      for (i = 0; i < 10; i++)
	{
	  fscanf(fi, "%d %s", &(highscores[i].score), highscores[i].name);
	}
      
      for (i = 0; i < 10; i++)
	{
	  fscanf(fi, "%d %s", &(highlevels[i].score), highlevels[i].name);
	}
      
      fscanf(fi, "%f %s", &highhitratio, highhitrationame);
    }
}


/* Dump scores: */

void savehighscores(void)
{
  FILE * fi;
  int i;
  
  fi = fopen("galaxa-highs.dat", "w");
  
  if (fi == NULL)
    {
      perror("galaxa-highs.dat");
    }
  else
    {
      for (i = 0; i < 10; i++)
	{
	  fprintf(fi, "%d %s\n", highscores[i].score, highscores[i].name);
	}
      for (i = 0; i < 10; i++)
	{
	  fprintf(fi, "%d %s\n", highlevels[i].score, highlevels[i].name);
	}
      
      fprintf(fi, "%f %s\n", highhitratio, highhitrationame);
      
      fclose(fi);
    }
}


/* Add an animated explosion graphic: */

void addanimatedexplosion(int x, int y)
{
  int i, found;
  
  found = -1;
  
  for (i = 0; i < MAX_EXPLOSIONS; i++)
    {
      if (explosions[i].alive == 0)
	found = i;
    }
  
  if (found == -1)
    found = 0;
  
  explosions[found].alive = 1;
  explosions[found].frame = 0;
  explosions[found].x = x;
  explosions[found].y = y;
  
  playsound(sound_fs, SND_FLASH);
}


/* Add an asteroid to the game.
   Send -1, -1, 0, 0 to have arbitrary position/direction determined: */

void addasteroid(int x, int y, int xm, int ym, int size)
{
  int i, found, z;
  
  found = -1;
  
  for (i = 0; i < MAX_ASTEROIDS; i++)
    {
      if (asteroids[i].alive == 0)
	found = i;
    }
  
  if (xm == 0 && ym == 0)
    ym = 1;
	      
  if (found != -1)
    {
      if (x == -1)
	{
	  z = randnum(3);
	  
	  if (z == 0)
	    {
	      x = randnum(WIDTH);
	      xm = randnum(5);
	      
	      if (x > WIDTH / 2)
		xm = -xm;
	      
	      ym = randnum(5);
	    }
	  else if (z == 1 || z == 2)
	    {
	      if (z == 1)
		{
		  x = 0;
		  xm = randnum(5);
		}
	      else if (z == 2)
		{
		  x = WIDTH - 50;
		  xm = -randnum(5);
		}
	      
	      y = randnum((HEIGHT / 3) * 2);
	      
	      ym = randnum(5);
	    }
	}
      
      asteroids[found].alive = 1;
      asteroids[found].x = x;
      asteroids[found].y = y;
      asteroids[found].xm = xm;
      asteroids[found].ym = ym;
      asteroids[found].size = size;
      asteroids[found].anim = randnum(5);
      asteroids[found].animm = randnum(3) - 1;
      asteroids[found].animspeed = randnum(5) + 1;
    }
}
