
/* Copyright (C) 2003-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


#include <signal.h>
#include <unistd.h>
#include "runtime.h"

#if HAVE_RESUME == 0
#undef dump_signal
#define dump_signal SIG_DFL
#endif


/* handle_fpe()-- Posix FPE signal handler */

#ifdef __INTERIX
static void handle_fpe(int sig) {        
char *msg;  
int cw;

    asm("fnstsw %0" : : "m"(cw));  
 
    msg = "Unknown exception";
   
    if (cw & 32)   msg = "Inexact result"; 
    if (cw & 16)   msg = "Underflow";
    if (cw & 8 )   msg = "Overflow";
    if (cw & 4 )   msg = "Division by zero"; 
    if (cw & 2 )   msg = "Denormalized result";
    if (cw & 1 )   msg = "Invalid operation"; 

#else

static void handle_fpe(int sig, siginfo_t *info, void *v) {
char buffer[50], *msg;

    switch(info->si_code) {

#if HAVE_FPE_INTDIV
    case FPE_INTDIV:  msg = "Integer division by zero";    break;
#endif

#if HAVE_FPE_INTOVF
    case FPE_INTOVF:  msg = "Integer overflow";            break;
#endif

#if HAVE_FPE_FLTDIV
    case FPE_FLTDIV:  msg = "Division by zero";            break;
#endif

#if HAVE_FPE_FLTOVF
    case FPE_FLTOVF:  msg = "Overflow";                    break;
#endif

#if HAVE_FPE_FLTUND
    case FPE_FLTUND:  msg = "Underflow";                   break;
#endif

#if HAVE_FPE_FLTRES
    case FPE_FLTRES:  msg = "Inexact result";              break;
#endif

#if HAVE_FPE_FLTINV
    case FPE_FLTINV:  msg = "Invalid operation";           break;
#endif

#if HAVE_FPE_FLTSUB
    case FPE_FLTSUB:  msg = "Subscript out of range";      break;
#endif

    default:       
	st_sprintf(buffer, "Unknown exception 0x%x", (G95_DINT) info->si_code);
	msg = buffer;
	break;
    }

#endif
    st_printf("Floating point exception: %s\n", msg);

    /* Some operating systems clear the floating point unit during an
     * exception, so don't do the usual display even if originally
     * requested. */

    options.view_exceptions = 0;
    show_locus();
    g95_runtime_stop();
    sys_exit(1, 95);
}



/* set_signal()-- Initialize a signal */

static void set_signal(int signo, int options) {
struct sigaction action;

    switch(options) {
    case SIGDISP_ABORT:      action.sa_handler = SIG_DFL;      break;
    case SIGDISP_IGNORE:     action.sa_handler = SIG_IGN;      break;
    case SIGDISP_DUMP:
    case SIGDISP_DUMP_QUIT:  action.sa_handler = dump_signal;  break;
    }

    sigemptyset(&action.sa_mask);
    action.sa_flags = 0;

    sigaction(signo, &action, NULL);
}



/* init_fpu_trap()-- Initialize the floating point signal handler. */

void init_fpu_trap(void) {
struct sigaction action;

#ifdef __INTERIX
    action.sa_handler = handle_fpe; 
#else
    action.sa_sigaction = handle_fpe; 
#endif

    sigemptyset(&action.sa_mask);
    sigaddset(&action.sa_mask, SIGFPE);

#ifdef __INTERIX
    action.sa_flags = SA_RESTART;
#else
    action.sa_flags = SA_SIGINFO;
#endif

    sigaction(SIGFPE, &action, NULL);

    set_signal(SIGHUP,  options.sighup);
    set_signal(SIGINT,  options.sigint);
    set_signal(SIGQUIT, options.sigquit);

#if HAVE_RESUME
    if (globals.alarm_value != 0) {
	set_signal(SIGALRM, SIGDISP_DUMP);
	alarm(globals.alarm_value);
    }
#endif
}
