/* Copyright (C) 2003-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


/* Transcendental functions */

#include <math.h>
#include "runtime.h"


/* mag_z4()-- Complex magnitude */

static float mag_z4(z4 *z) {
float a, b, c;

    a = fabsf(z->r);
    b = fabsf(z->c);

    if (a < b) {
	c = a;
	a = b;
	b = c;
    }

    if ((a+b) == a)
	return a;

    c = b / a;
    return a * sqrt(1.0 + c*c);
}


/* mag_z8()-- Complex magnitude */

static double mag_z8(z8 *z) {
double a, b, c;

    a = fabs(z->r);
    b = fabs(z->c);

    if (a < b) {
	c = a;
	a = b;
	b = c;
    }

    if ((a+b) == a)
	return a;

    c = b / a;
    return a * sqrt(1.0 + c*c);
}

#define abs_z4 prefix(abs_z4)

float abs_z4(z4 *z) {
    return mag_z4(z);
}

#define abs_z8 prefix(abs_z8)

double abs_z8(z8 *z) {
    return mag_z8(z);
}


#define sin_r4 prefix(sin_r4)
float sin_r4(float *a) {

    return (float) sin(*a);
}

#define sin_r8 prefix(sin_r8)
double sin_r8(double *a) {

    return sin(*a);
}

#define sin_z4 prefix(sin_z4)
void sin_z4(z4 *rv, z4 *a) {

    rv->r = sin(a->r)*cosh(a->c);
    rv->c = cos(a->r)*sinh(a->c);    
}

#define sin_z8 prefix(sin_z8)
void sin_z8(z8 *rv, z8 *a) {

    rv->r = sin(a->r)*cosh(a->c);
    rv->c = cos(a->r)*sinh(a->c);
}

#define cos_r4 prefix(cos_r4)
float cos_r4(float *a) {

    return (float) cos(*a);
}

#define cos_r8 prefix(cos_r8)
double cos_r8(double *a) {

    return cos(*a);
}

#define cos_z4 prefix(cos_z4)
void cos_z4(z4 *rv, z4 *a) {

    rv->r = cos(a->r)*cosh(a->c);
    rv->c = -sin(a->r)*sinh(a->c);
}

#define cos_z8 prefix(cos_z8)
void cos_z8(z8 *rv, z8 *a) {

    rv->r = cos(a->r)*cosh(a->c);
    rv->c = -sin(a->r)*sinh(a->c);
}

#define tan_r4 prefix(tan_r4)
float tan_r4(float *a) {

    return (float) tan(*a);
}

#define tan_r8 prefix(tan_r8)
double tan_r8(double *a) {

    return tan(*a);
}

#define asin_r4 prefix(asin_r4)
float asin_r4(float *a) {
float r;

#if HAVE_WORKING_ASIN == 0
    if (*a > 1.0 || *a < -1.0)
	build_nan(0, 1, (char *) &r, 4);
    else
#endif
	r = (float) asin(*a);

    return r;
}

#define asin_r8 prefix(asin_r8)
double asin_r8(double *a) {
double r;

#if HAVE_WORKING_ASIN == 0
    if (*a > 1.0 || *a < -1.0)
	build_nan(0, 1, (char *) &r, 8);
    else
#endif
	r = asin(*a);
 
    return r;
}

#define acos_r4 prefix(acos_r4)
float acos_r4(float *a) {
float r;

#if HAVE_WORKING_ACOS == 0
    if (*a > 1.0 || *a < -1.0)
	build_nan(0, 1, (char *) &r, 4);
    else
#endif
	r = (float) acos(*a);

    return r;
}

#define acos_r8 prefix(acos_r8)
double acos_r8(double *a) {
double r;

#if HAVE_WORKING_ACOS == 0
    if (*a > 1.0 || *a < -1.0)
	build_nan(0, 1, (char *) &r, 8);
    else
#endif
	r = acos(*a);

    return r;
}

#define atan_r4 prefix(atan_r4)
float atan_r4(float *a) {

    return (float) atan(*a);
}

#define atan_r8 prefix(atan_r8)
double atan_r8(double *a) {

    return atan(*a);
}

#define atan2_r4 prefix(atan2_r4)
float atan2_r4(float *y, float *x) {

    return (float) atan2(*y, *x);
}

#define atan2_r8 prefix(atan2_r8)
double atan2_r8(double *y, double *x) {

    return atan2(*y, *x);
}

#define sinh_r4 prefix(sinh_r4)
float sinh_r4(float *a) {

    return (float) sinh(*a);
}

#define sinh_r8 prefix(sinh_r8)
double sinh_r8(double *a) {

    return sinh(*a);
}

#define cosh_r4 prefix(cosh_r4)
float cosh_r4(float *a) {

    return (float) cosh(*a);
}

#define cosh_r8 prefix(cosh_r8)
double cosh_r8(double *a) {

    return cosh(*a);
}

#define tanh_r4 prefix(tanh_r4)
float tanh_r4(float *a) {

    return (float) tanh(*a);
}

#define tanh_r8 prefix(tanh_r8)
double tanh_r8(double *a) {

    return tanh(*a);
}

#define sqrt_r4 prefix(sqrt_r4)
float sqrt_r4(float *a) {

    return (float) sqrt(*a);
}

#define sqrt_r8 prefix(sqrt_r8)
double sqrt_r8(double *a) {

    return sqrt(*a);
}


#define sqrt_z4 prefix(sqrt_z4)
void sqrt_z4(z4 *rv, z4 *z) {
float mag, zr, zc;

    zr = z->r;
    zc = z->c;
    mag = mag_z4(z);

    if (mag == 0.0)
	rv->r = rv->c = 0.0;

    else if (zr > 0) {
	rv->r = sqrt(0.5*(mag + zr));
	rv->c = 0.5 * zc / rv->r;

    } else {
	rv->c = sqrt(0.5*(mag - zr));
	if (get_sign(&zc, 4))
	    rv->c = -rv->c;

	rv->r = 0.5 * zc / rv->c;
    }
}

#define sqrt_z8 prefix(sqrt_z8)
void sqrt_z8(z8 *rv, z8 *z) {
double mag, zr, zc;

    zr = z->r;
    zc = z->c;
    mag = mag_z8(z);

    if (mag == 0.0)
	rv->r = rv->c = 0.0;

    else if (zr > 0) {
	rv->r = sqrt(0.5*(mag + zr));
	rv->c = 0.5 * zc / rv->r;

    } else {
	rv->c = sqrt(0.5*(mag - zr));
	if (get_sign(&zc, 8))
	    rv->c = -rv->c;

	rv->r = 0.5 * zc / rv->c;
    }
}

#define exp_r4 prefix(exp_r4)
float exp_r4(float *a) {

    return (float) exp(*a);
}

#define exp_r8 prefix(exp_r8)
double exp_r8(double *a) {

    return exp(*a);
}

#define exp_z4 prefix(exp_z4)
void exp_z4(z4 *rv, z4 *a) {
float e_real;

    e_real = exp(a->r);

    rv->r = e_real * cos(a->c);
    rv->c = e_real * sin(a->c);
}

#define exp_z8 prefix(exp_z8)
void exp_z8(z8 *rv, z8 *a) {
double e_real;

    e_real = exp(a->r);

    rv->r = e_real * cos(a->c);
    rv->c = e_real * sin(a->c);
}



#define sys_log prefix(log)

double sys_log(double a) {
double r;

#if HAVE_WORKING_LOG == 0
    if (a < 0.0)
	build_nan(0, 1, (char *) &r, 8);
    else
#endif
	r = log(a);

    return r;
}


#define sys_logf prefix(logf)

float sys_logf(float a) {
float r;

#if HAVE_WORKING_LOG == 0
    if (a < 0.0)
	build_nan(0, 1, (char *) &r, 4);
    else
#endif
	r = log(a);

    return r;
}


#define log_r4 prefix(log_r4)
float log_r4(float *a) {

    return sys_log((double) *a);
}


#define log_r8 prefix(log_r8)
double log_r8(double *a) {

    return sys_log(*a);
}


#define log_z4 prefix(log_z4)
void log_z4(z4 *rv, z4 *z) {

    rv->r = log(mag_z4(z));
    rv->c = atan2(z->c, z->r);
}

#define log_z8 prefix(log_z8)
void log_z8(z8 *rv, z8 *z) {

    rv->r = log(mag_z8(z));
    rv->c = atan2(z->c, z->r);
}


#define sys_log10 prefix(log10)

double sys_log10(double a) {
double r;

#if HAVE_WORKING_LOG10 == 0
    if (a < 0.0)
	build_nan(0, 1, (char *) &r, 8);
    else
#endif
	r = log10(a);

    return r;
}


#define sys_log10f prefix(log10f)

float sys_log10f(float a) {
float r;

#if HAVE_WORKING_LOG10 == 0
    if (a < 0.0)
	build_nan(0, 1, (char *) &r, 4);
    else
#endif
	r = log10(a);

    return r;
}


#define log10_r4 prefix(log10_r4)
float log10_r4(float *a) {

    return sys_log10((double) *a);
}



#define log10_r8 prefix(log10_r8)
double log10_r8(double *a) {

    return sys_log10(*a);
}


#define power_r4 prefix(power_r4)

float power_r4(float *a, float *b) {

    return powf(*a, *b);
}


#define power_r8 prefix(power_r8)

double power_r8(double *a, double *b) {

    return pow(*a, *b);
}



#define power_z4 prefix(power_z4)

void power_z4(z4 *result, z4 *a, z4 *b) {
float logr, logi, x, y;

    if (a->r == 0.0 && a->c == 0.0) {
	if (b->r != 0.0 || b->c != 0.0)
	    result->r = result->c = 0.0;

	else {
	    result->r = 1.0;
	    result->c = 0.0;
	}

    } else {
	logr = log(mag_z4(a));
	logi = atan2(a->c, a->r);

	x = exp(logr*b->r - logi*b->c);
	y = logr*b->c + logi*b->r;

	result->r = x * cos(y);
	result->c = x * sin(y);
    }
}


#define power_z8 prefix(power_z8)

void power_z8(z8 *result, z8 *a, z8 *b) {
double logr, logi, x, y;
 
    if (a->r == 0.0 && a->c == 0.0) {
	if (b->r != 0.0 || b->c != 0.0)
	    result->r = result->c = 0.0;
	else {
	    result->r = 1.0;
	    result->c = 0.0;
	}

    } else {
	logr = log(mag_z8(a));
	logi = atan2(a->c, a->r);

	x = exp(logr*b->r - logi*b->c);
	y = logr*b->c + logi*b->r;

	result->r = x * cos(y);
	result->c = x * sin(y);
    }
}

