
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


#include <stdlib.h>
#include <string.h>
#include <limits.h>

#include "runtime.h"


/* Subroutines related to units */


#define CACHE_SIZE 3
static iounit_t *unit_cache[CACHE_SIZE];


/* This implementation is based on Stefan Nilsson's article in the
 * July 1997 Doctor Dobb's Journal, "Treaps in Java". */

/* rotate_left()-- Rotate the treap left */

static iounit_t *rotate_left(iounit_t *t) {
iounit_t *temp;

    temp =  t->right;
    t->right = t->right->left;
    temp->left = t;

    return temp;
}


/* rotate_right()-- Rotate the treap right */

static iounit_t *rotate_right(iounit_t *t) {
iounit_t *temp;

    temp = t->left;
    t->left = t->left->right;
    temp->right = t;

    return temp;
}


/* compare()-- Compare nodes */

static int compare(G95_MINT a, G95_MINT b) {

    if (a < b) return -1;
    if (a > b) return 1;

    return 0;
}


/* insert()-- Recursive insertion function.  Returns the updated treap. */

static iounit_t *insert(iounit_t *new, iounit_t *t) {
int c;

    if (t == NULL)
	return new;

    c = compare(new->unit_number, t->unit_number);

    if (c < 0) {
	t->left = insert(new, t->left);
	if (t->priority < t->left->priority)
	    t = rotate_right(t);
    }

    if (c > 0) {
	t->right = insert(new, t->right);
	if (t->priority < t->right->priority)
	    t = rotate_left(t);
    }

    if (c == 0)
	internal_error("insert(): Duplicate key found!");

    return t;
}



/* insert_unit()-- Given a new node, insert it into the treap.  It is
 * an error to insert a key that already exists. */

void insert_unit(iounit_t *new) {

    new->priority = xorshift128();
    globals.unit_root = insert(new, globals.unit_root);
}


/* delete_root()-- Delete a root node */

static iounit_t *delete_root(iounit_t *t) {
iounit_t *temp;

    if (t->left  == NULL)
	return t->right;

    if (t->right == NULL)
	return t->left;

    if (t->left->priority > t->right->priority) {
	temp = rotate_right(t);
	temp->right = delete_root(t);
    } else {
	temp = rotate_left(t);
	temp->left = delete_root(t);
    }

    return temp;
}



/* delete_treap()-- Delete an element from a tree.  The 'old' value
 * does not necessarily have to point to the element to be deleted, it
 * must just point to a treap structure with the key to be deleted.
 * Returns the new root node of the tree. */

static iounit_t *delete_treap(iounit_t *old, iounit_t *t) {
int c;

    if (t == NULL)
	return NULL;

    c = compare(old->unit_number, t->unit_number);

    if (c < 0)
	t->left = delete_treap(old, t->left);

    else if (c > 0)
	t->right = delete_treap(old, t->right);

    else
	t = delete_root(t);

    return t;
}


/* delete_unit()-- Delete a unit from a tree */

static void delete_unit(iounit_t *old) {

    globals.unit_root = delete_treap(old, globals.unit_root);
}


/* find_unit()-- Given an integer, return a pointer to the unit
 * structure.  Returns NULL if the unit does not exist. */

iounit_t *find_unit(void *q, G95_DINT kind) {
G95_MINT n;
iounit_t *p;
int c;

    n = extract_mint(q, kind);

    if (n < -1) {
	generate_error(ERROR_BADUNIT, NULL);
	return NULL;
    }

    for(c=0; c<CACHE_SIZE; c++)
	if (unit_cache[c] != NULL && unit_cache[c]->unit_number == n) {
	    p = unit_cache[c];
	    return p;
	}

    p = globals.unit_root; 
    while(p != NULL) {
	c = compare(n, p->unit_number);

	if (c < 0)
	    p = p->left;

	else if (c > 0)
	    p = p->right;

	else
	    break;
    }

    if (p != NULL) {
	for(c=0; c<CACHE_SIZE-1; c++)
	    unit_cache[c] = unit_cache[c+1];

	unit_cache[CACHE_SIZE-1] = p;
    }

    return p;
}



/* init_reverse()-- Given a unit number, return the value of the
 * unit-specific reverse flag. */

static int init_reverse(void *unit, int kind) {
char *p, buffer[100];

    strcpy(buffer, "G95_UNIT_ENDIAN_");
    strcat(buffer, mint_to_a(extract_mint(unit, kind)));

    p = getenv(buffer);
    if (p != NULL) {
	if (strcasecmp(p, "big") == 0)
	    return (my_endian == ENDIAN_LITTLE);

	if (strcasecmp(p, "little") == 0)
	    return (my_endian == ENDIAN_BIG);
    }

    return (default_endian == ENDIAN_NATIVE)
	? 0
	: (default_endian != my_endian);
}



/* open_unit()-- Open a file associated with a unit.  If no filename
 * is present, the name is usually of the form "fort.n" unless
 * overridden by an environment variable.  The unit structure is
 * inserted into the tree, and the file is opened for reading and
 * writing.  Assumes that the unit is not present. */

iounit_t *open_unit(unit_flags *flags) {
char *p, buffer[100], path[PATH_LENGTH+1];
iounit_t *u;
stream *s;
int m;

    if (ioparm->file == NULL) { 
	strcpy(buffer, "G95_UNIT_");
	strcat(buffer, mint_to_a(extract_mint(ioparm->unit,
					      ioparm->unit_kind)));

	p = getenv(buffer);
	if (p == NULL) {
	    strcpy(buffer, "fort.");
	    strcat(buffer, mint_to_a(extract_mint(ioparm->unit,
						  ioparm->unit_kind)));
	    p = buffer;
	}

	ioparm->file = p;
	ioparm->file_len = strlen(p);
    }

    /* Make sure the same file isn't already open someplace else */

    u = find_file();
    if (u != NULL && !terminal_device(u)) {
	generate_error(ERROR_ALREADY_OPEN, NULL);
	return NULL;
    }

    if (flags->action == ACTION_UNSPECIFIED)
	flags->action = default_action();

    s = open_external(flags->action, flags->status, path);
    error_filename = path;

    if (s == NULL) {
	generate_error(ERROR_OS, NULL);
	return NULL;
    }

    error_filename = NULL;

    m = (flags->status == STATUS_SCRATCH)
	? strlen(path)
	: fstrlen(ioparm->file, ioparm->file_len);

    u = get_mem(sizeof(iounit_t) + m);
    u->flags = *flags;

    if (u->flags.status == STATUS_NEW || u->flags.status == STATUS_REPLACE)
	u->flags.status = STATUS_OLD;

    u->s             = s;
    u->unit_number   = extract_mint(ioparm->unit, ioparm->unit_kind);
    u->s->unbuffered = check_unbuffered(u->unit_number, u->s->unbuffered);

    u->repos    = 0;
    u->file_len = m;

    u->recl = (ioparm->recl_in != NULL)
	? extract_mint(ioparm->recl_in, ioparm->recl_in_kind)
	: DEFAULT_RECL;

    switch(u->flags.endian) {
    case ENDIAN_LITTLE: 
	u->reverse = (my_endian == ENDIAN_BIG);
	break;

    case ENDIAN_BIG:
	u->reverse = (my_endian == ENDIAN_LITTLE);
	break;

    case ENDIAN_NATIVE:
	u->reverse = 0;
	break;

    case ENDIAN_SWAP:
	u->reverse = 1;

    default:
	u->reverse = init_reverse(ioparm->unit, ioparm->unit_kind);
	break;
    }

    u->last_record = 0;
    u->read_eof    = 0;
    u->last_record = 1;

    p = (flags->status == STATUS_SCRATCH) ? path : ioparm->file;
    memcpy(u->file, p, m);

    insert_unit(u);
    return u;
}



/* get_unit()-- Return a unit, opening it implicitly if necessary */

iounit_t *get_unit(void) {
unit_flags flags;
iounit_t *u;

    if (ioparm->internal_unit != NULL || ioparm->internal_array != NULL) {
	u = get_mem(sizeof(iounit_t));
	u->s = NULL;

	/* Set flags for the internal unit */

	u->flags.access = ACCESS_SEQUENTIAL;
	u->flags.action = ACTION_READWRITE;
	u->flags.form = FORM_FORMATTED;
	u->flags.delim = DELIM_NONE;
	u->flags.decimal = options.decimal_comma
	    ? DECIMAL_COMMA
	    : DECIMAL_POINT;
	u->endfile = NO_ENDFILE;

	u->recl = u->file_len =
	    (ioparm->internal_unit != NULL)
	    ? ioparm->internal_unit_len
	    : ioparm->internal_array->element_size;

	u->reverse = 0;
	u->read_eof = 0;
	u->record = NULL;
	u->unit_number = -1;
	u->repos = 0;

	return u;
    }

    if (ioparm->unit < 0)
	runtime_error("Negative unit number");

    u = find_unit(ioparm->unit, ioparm->unit_kind);
    if (u != NULL)
	return u;    /* Already exists */

    flags.access = ACCESS_SEQUENTIAL;
    flags.action = ACTION_UNSPECIFIED;
    flags.blank = BLANK_NULL;
    flags.delim = DELIM_NONE;
    flags.decimal = options.decimal_comma ? DECIMAL_COMMA : DECIMAL_POINT;
    flags.position = POSITION_ASIS;
    flags.pad = PAD_YES;
    flags.status = STATUS_UNKNOWN;   /* Open if there, create if not */
    flags.form = (ioparm->format == NULL && !ioparm->list_format &&
		  ioparm->namelist == NULL)
	? FORM_UNFORMATTED
	: FORM_FORMATTED;

    return open_unit(&flags);
}


/* is_internal_unit()-- Determine if the current unit is internal or
 * not */

int is_internal_unit() {

    return current_unit->unit_number == -1;
}


/* init_unit()-- Initialize preconnected units */

void init_units(void) {
iounit_t *u;

    if (options.stdin_unit >= 0) {    /* STDIN */
	u = get_mem(sizeof(iounit_t));

	u->unit_number = options.stdin_unit;
	u->s = input_stream();
	u->s->unbuffered = check_unbuffered(options.stdin_unit,
					    u->s->unbuffered);

	u->recl = options.default_recl;
	u->endfile = NO_ENDFILE;
	u->read_eof = 0;
	u->flags.action = ACTION_READWRITE;

	u->flags.access = ACCESS_SEQUENTIAL;
	u->flags.form = FORM_FORMATTED;
	u->flags.status = STATUS_OLD;
	u->flags.blank = BLANK_NULL;
	u->flags.position = POSITION_ASIS;

	u->flags.decimal = options.decimal_comma
	    ? DECIMAL_COMMA
	    : DECIMAL_POINT;

	u->reverse = init_reverse(&options.stdin_unit,
				  sizeof(options.stdin_unit));

	insert_unit(u);
    }

    if (options.stdout_unit >= 0) {   /* STDOUT */
	u = get_mem(sizeof(iounit_t));

	u->unit_number = options.stdout_unit;
	u->s = output_stream();
	u->s->unbuffered = check_unbuffered(options.stdout_unit,
					    u->s->unbuffered);

	u->recl = options.default_recl;
	u->endfile = AT_ENDFILE;
	u->read_eof = 0;
	u->flags.action = ACTION_READWRITE;

	u->flags.access = ACCESS_SEQUENTIAL;
	u->flags.form = FORM_FORMATTED;
	u->flags.status = STATUS_OLD;
	u->flags.blank = BLANK_NULL;
	u->flags.position = POSITION_ASIS;
	u->flags.decimal = options.decimal_comma
	    ? DECIMAL_COMMA
	    : DECIMAL_POINT;

	u->reverse = init_reverse(&options.stdout_unit,
				  sizeof(options.stdout_unit));

	insert_unit(u);
    }

    if (options.stderr_unit >= 0) {   /* STDERR */
	u = get_mem(sizeof(iounit_t));

	u->unit_number = options.stderr_unit;
	u->s = error_stream();
	u->s->unbuffered = check_unbuffered(options.stderr_unit,
					    u->s->unbuffered);

	u->recl = options.default_recl;
	u->endfile = AT_ENDFILE;
	u->read_eof = 0;
	u->flags.action = ACTION_READWRITE;

	u->flags.access = ACCESS_SEQUENTIAL;
	u->flags.form = FORM_FORMATTED;
	u->flags.status = STATUS_OLD;
	u->flags.blank = BLANK_NULL;
	u->flags.position = POSITION_ASIS;
	u->flags.decimal = options.decimal_comma
	    ? DECIMAL_COMMA
	    : DECIMAL_POINT;

	u->reverse = init_reverse(&options.stderr_unit,
				  sizeof(options.stderr_unit));

	insert_unit(u);
    }
}


/* unbuffer_stdout()-- Unbuffer the standard output. */

void unbuffer_stdout(void) {
stream *s;

    s = find_unit(&options.stdout_unit, sizeof(options.stdout_unit))->s;
    s->unbuffered = 1;
}


/* close_unit()-- Close a unit.  The stream is closed, and any memory
 * associated with the stream is freed.  Returns nonzero on I/O error. */

int close_unit(iounit_t *u) {
int i, rc;
char *p;

    for(i=0; i<CACHE_SIZE; i++)
	if (unit_cache[i] == u)
	    unit_cache[i] = NULL;

    if (u->s == NULL)
	rc = 0;

    else {
	i = u->max_offset - u->offset;

	if (i > 0) {
	    p = salloc_w(u->s, i, 0);

	    if (p != NULL)
		memcpy(p, u->record + u->offset, i);
	    else
		generate_error(ERROR_OS, NULL);

	    sfree(u->s);
	    terminate_record(u);
	}

	rc = sclose(u->s) == FAILURE;
    }

    if ((u->flags.access == ACCESS_SEQUENTIAL ||
	 u->flags.access == ACCESS_STREAM) && u->record != NULL) {
	free_mem(u->record);
	u->record = NULL;
    }

    delete_unit(u);
    free_mem(u);

    return rc; 
}


/* close_units()-- Delete units on completion.  We just keep deleting
 * the root of the treap until there is nothing left. */

void close_units(void) {

    while(globals.unit_root != NULL) 
	close_unit(globals.unit_root);
}



/* flush0()-- Recursive work function to flush all units */

static void flush0(iounit_t *u) {

    if (u == NULL)
	return;

    flush_stream(u->s);

    flush0(u->left);
    flush0(u->right);
}



/* flush_units()-- Flush all I/O units */

void flush_units(void) {

    flush0(globals.unit_root);
}

