



#include <string.h>
#include "runtime.h"

#if HAVE_ALLOCA_H && !defined(__MINGW32__)
#include <alloca.h>
#endif


/* These subroutines are based on William Clinger's paper "How to Read
 * Floating-point Numbers Accurately", published in "Proceedings of
 * the ACM SIGPLAN 90 Conference on Programming Language Design and
 * Implementation", p92-101.  The idea is to use an extended precision
 * floating point format to read an integer and a power of ten, then
 * multiply the two to get the desired number.  Powers of ten are
 * looked up in a table, with larger numbers requiring a
 * multiplication of two table values to obtain the power of ten.  The
 * final result is rounded to the desired precision.  If inaccuracies
 * in the products could affect the final result, the product is used
 * as an approximation to a second algorithm that uses high precision
 * integer arithmetic to obtain a correctly rounded result. */

#define MAX_MANTISSA 4

typedef struct {
    int exp;
    unsigned mantissa[MAX_MANTISSA];
} unpacked_real;

static int exp, digits, right_digits;


/* Multiprecision integer.  The words are stored from least
 * significant to most, unlike the unpacked_real. */

typedef struct {
    int n;
    unsigned *d;
} bigint;


#define COPY_BIGINT(a, b) { (a)->n = (b)->n; \
    memcpy((a)->d, (b)->d, (a)->n*sizeof(unsigned)); }


/* Table of powers of ten.  The binary point is taken to be at the
 * right of the rightmost word.  For kinds that need less precise
 * values, these table entries are rounded to the intermediate
 * precision.  There do not appear to be any cases of trailing zeroes
 * that would require the round-to-even rule to be invoked.
 *
 * The first J powers of ten are enumerated.  After this, the big
 * tables go in steps of H.  By multiplying we can construct any
 * integer power of ten between -MAX_EXPONENT and MAX_EXPONENT. */

#define MAX_EXPONENT 5000
#define H_PARM 50
#define J_PARM 50


static const unpacked_real negative[] = {
{   -127, {0x80000000,0x00000000,0x00000000,0x00000000}}, /* -0 */
{   -130, {0x66666666,0x66666666,0x66666666,0x66666666}}, /* -1 */
{   -133, {0x51EB851E,0xB851EB85,0x1EB851EB,0x851EB852}}, /* -2 */
{   -136, {0x4189374B,0xC6A7EF9D,0xB22D0E56,0x04189375}}, /* -3 */
{   -140, {0x68DB8BAC,0x710CB295,0xE9E1B089,0xA0275254}}, /* -4 */
{   -143, {0x53E2D623,0x8DA3C211,0x87E7C06E,0x19B90EAA}}, /* -5 */
{   -146, {0x431BDE82,0xD7B634DA,0xD31FCD24,0xE160D888}}, /* -6 */
{   -150, {0x6B5FCA6A,0xF2BD215E,0x1E99483B,0x02348DA6}}, /* -7 */
{   -153, {0x55E63B88,0xC230E77E,0x7EE10695,0x9B5D3E1F}}, /* -8 */
{   -156, {0x44B82FA0,0x9B5A52CB,0x98B40544,0x7C4A9818}}, /* -9 */
{   -160, {0x6DF37F67,0x5EF6EADF,0x5AB9A207,0x2D44268E}}, /* -10 */
{   -163, {0x57F5FF85,0xE592557F,0x7BC7B4D2,0x8A9CEBA4}}, /* -11 */
{   -166, {0x465E6604,0xB7A84465,0xFC9FC3DB,0xA21722EA}}, /* -12 */
{   -170, {0x709709A1,0x25DA0709,0x9432D2F9,0x035837DD}}, /* -13 */
{   -173, {0x5A126E1A,0x84AE6C07,0xA9C24260,0xCF79C64A}}, /* -14 */
{   -176, {0x480EBE7B,0x9D58566C,0x87CE9B80,0xA5FB0508}}, /* -15 */
{   -180, {0x734ACA5F,0x6226F0AD,0xA6175F34,0x3CC4D4DA}}, /* -16 */
{   -183, {0x5C3BD519,0x1B525A24,0x84DF7F5C,0xFD6A43E1}}, /* -17 */
{   -186, {0x49C97747,0x490EAE83,0x9D7F9917,0x3121CFE8}}, /* -18 */
{   -190, {0x760F253E,0xDB4AB0D2,0x9598F4F1,0xE8361973}}, /* -19 */
{   -193, {0x5E728432,0x49088D75,0x447A5D8E,0x535E7AC2}}, /* -20 */
{   -196, {0x4B8ED028,0x3A6D3DF7,0x69FB7E0B,0x75E52F02}}, /* -21 */
{   -200, {0x78E48040,0x5D7B9658,0xA9926345,0x896EB19C}}, /* -22 */
{   -203, {0x60B6CD00,0x4AC94513,0xBADB829E,0x078BC14A}}, /* -23 */
{   -206, {0x4D5F0A66,0xA23A9DA9,0x6249354B,0x393C9AA1}}, /* -24 */
{   -210, {0x7BCB43D7,0x69F762A8,0x9D41EEDE,0xC1FA9102}}, /* -25 */
{   -213, {0x63090312,0xBB2C4EED,0x4A9B257F,0x019540CF}}, /* -26 */
{   -216, {0x4F3A68DB,0xC8F03F24,0x3BAF5132,0x67AA9A3F}}, /* -27 */
{   -220, {0x7EC3DAF9,0x41806506,0xC5E54EB7,0x0C4429FE}}, /* -28 */
{   -223, {0x65697BFA,0x9ACD1D9F,0x04B7722C,0x09D02198}}, /* -29 */
{   -226, {0x51212FFB,0xAF0A7E18,0xD092C1BC,0xD4A68147}}, /* -30 */
{   -229, {0x40E75996,0x25A1FE7A,0x407567CA,0x43B8676C}}, /* -31 */
{   -233, {0x67D88F56,0xA29CCA5D,0x33EF0C76,0xD2C0A579}}, /* -32 */
{   -236, {0x5313A5DE,0xE87D6EB0,0xF658D6C5,0x7566EAC7}}, /* -33 */
{   -239, {0x42761E4B,0xED31255A,0x5EAD789D,0xF785889F}}, /* -34 */
{   -243, {0x6A5696DF,0xE1E83BC3,0xCAAF2763,0x25A27432}}, /* -35 */
{   -246, {0x5512124C,0xB4B9C969,0x6EF285E8,0xEAE85CF5}}, /* -36 */
{   -249, {0x440E750A,0x2A2E3ABA,0xBF286B20,0xBBED172A}}, /* -37 */
{   -253, {0x6CE3EE76,0xA9E3912A,0xCB73DE9A,0xC6482511}}, /* -38 */
{   -256, {0x571CBEC5,0x54B60DBB,0xD5F64BAF,0x0506840D}}, /* -39 */
{   -259, {0x45B0989D,0xDD5E7163,0x1191D625,0x9D9ED00B}}, /* -40 */
{   -263, {0x6F80F42F,0xC8971BD1,0xB5B6236F,0x6297B344}}, /* -41 */
{   -266, {0x5933F68C,0xA078E30E,0x2AF81C59,0x1BAC8F6A}}, /* -42 */
{   -269, {0x475CC53D,0x4D2D8271,0xBBF9B047,0x4956D921}}, /* -43 */
{   -273, {0x722E0862,0x15159D82,0xC65C4D3E,0xDBBE2835}}, /* -44 */
{   -276, {0x5B5806B4,0xDDAAE468,0x9EB03DCB,0xE2FE8691}}, /* -45 */
{   -279, {0x49133890,0xB1558386,0xE559CB09,0x82653874}}, /* -46 */
{   -283, {0x74EB8DB4,0x4EEF38D7,0xD55C780F,0x37085A53}}, /* -47 */
{   -286, {0x5D893E29,0xD8BF60AC,0xAAB0600C,0x2C06AEA9}}, /* -48 */
{   -289, {0x4AD431BB,0x13CC4D56,0xEEF38009,0xBCD22554}}, /* -49 */
{   -293, {0x77B9E92B,0x52E07BBE,0x4B1F3342,0xC7B6A220}}, /* -50 */
};

static const unpacked_real negative_big[] = {
{   -459, {0x6FFCBB92,0x3814BF5E,0xACBC3F15,0xC9DE2B7C}}, /* -100 */
{   -625, {0x68BF9DA8,0xFE51D3D0,0x7BAD0AC3,0x16528263}}, /* -150 */
{   -791, {0x61FA4855,0x3BDEB07E,0x5F6DFE22,0x0883454E}}, /* -200 */
{   -957, {0x5BA4FD76,0x8A092E9B,0x677CC076,0x338FB33E}}, /* -250 */
{  -1123, {0x55B87F0B,0xE3CD6365,0x36DEB185,0x24557A03}}, /* -300 */
{  -1289, {0x502E06EB,0x87B70B0C,0xD43935E4,0x6AAE5D8B}}, /* -350 */
{  -1455, {0x4AFF3F03,0xE48F7D7D,0x1C98DC28,0x6F84739C}}, /* -400 */
{  -1621, {0x46263A1C,0xB5210AF7,0x34BA5D60,0xEDABA87A}}, /* -450 */
{  -1787, {0x419D6D10,0x01EDCD3F,0xC766A21C,0xED028C73}}, /* -500 */
{  -1954, {0x7ABF50E6,0x91A72466,0xCE9BC606,0x5A9EF513}}, /* -550 */
{  -2120, {0x72D02553,0x15AA9F4C,0xFBA670CC,0x6B2A654A}}, /* -600 */
{  -2286, {0x6B6444AE,0xF06E6B36,0xEC02826B,0x5AC0BE5C}}, /* -650 */
{  -2452, {0x64733266,0xC69C3EFC,0x0F15E91B,0x13E34C01}}, /* -700 */
{  -2618, {0x5DF4FE56,0x16AC4B6E,0x94F1615E,0x6C00CF0E}}, /* -750 */
{  -2784, {0x57E23BB3,0x659CD3D8,0x6BDF1310,0xACC5CA2A}}, /* -800 */
{  -2950, {0x5233F892,0xA15455F6,0x7331B243,0xD51E5653}}, /* -850 */
{  -3116, {0x4CE3B5F4,0x03137466,0x2A4EE888,0x6A609BAB}}, /* -900 */
{  -3282, {0x47EB6056,0x5D068F4B,0x866229E5,0x0D407530}}, /* -950 */
{  -3448, {0x434548C4,0x544F0A33,0x880989F0,0x1BB071BC}}, /* -1000 */
{  -3615, {0x7DD83CA9,0x2ADB0C15,0x1BD11D3D,0xF44CF5C6}}, /* -1050 */
{  -3781, {0x75B5D02B,0x28E32199,0xA09266D2,0x41C33F96}}, /* -1100 */
{  -3947, {0x6E19FEBB,0xCA9DA9C4,0x3981E098,0x3F48448D}}, /* -1150 */
{  -4113, {0x66FC14F5,0x578094BB,0x9B7C4EE2,0x79A0C3E2}}, /* -1200 */
{  -4279, {0x6053EF6C,0x1B75BFF2,0x05CEF031,0xFD2ABD06}}, /* -1250 */
{  -4445, {0x5A19F15F,0xF02B1654,0x1A213F5A,0x2A19A73E}}, /* -1300 */
{  -4611, {0x5446FC07,0xC65F8994,0x520CCF77,0x8F5E1527}}, /* -1350 */
{  -4777, {0x4ED4666D,0x3AD9A0DA,0xD2E2C6AF,0xC8D23B6C}}, /* -1400 */
{  -4943, {0x49BBF5CE,0xE76CF16E,0x5D11117A,0xF6889922}}, /* -1450 */
{  -5109, {0x44F7D680,0xBA86B25E,0x8CEEF74F,0x748DBBBC}}, /* -1500 */
{  -5275, {0x40829542,0x2FD6ED18,0x16DEBB3E,0xC814FDB4}}, /* -1550 */
{  -5442, {0x78AE3205,0x96F0BDC2,0xBAECD9B9,0x3F372D24}}, /* -1600 */
{  -5608, {0x70E13A1B,0x6E1E308E,0x27780E72,0xD0774E7B}}, /* -1650 */
{  -5774, {0x6995572B,0xAC8AECEF,0xC9A1E0D2,0xB47B3113}}, /* -1700 */
{  -5940, {0x62C23138,0x2CBE2915,0x3A5D8DA2,0x64CD8156}}, /* -1750 */
{  -6106, {0x5C5FFA54,0x478FDA31,0xD816CB29,0xF59C3A00}}, /* -1800 */
{  -6272, {0x566765B8,0x22122AF9,0x39024039,0x6706AFE4}}, /* -1850 */
{  -6438, {0x50D19F67,0xA9A541BA,0x1B876D17,0x228C6A84}}, /* -1900 */
{  -6604, {0x4B984463,0xB2751AAC,0xC7980C84,0xD79F320C}}, /* -1950 */
{  -6770, {0x46B55B5C,0x4A975FC0,0x8E13977B,0x4E6F6531}}, /* -2000 */
{  -6936, {0x42234DDB,0xE7BF6AD8,0x69538B25,0x6C741A7F}}, /* -2050 */
{  -7103, {0x7BB9C3C7,0x3F63EEA2,0x95AB377A,0x657A83D8}}, /* -2100 */
{  -7269, {0x73BA67E2,0x39CCFABB,0xF5DFA5C8,0x8A09A70F}}, /* -2150 */
{  -7435, {0x6C3F62CE,0xF35B2F37,0x92E32720,0x15764E22}}, /* -2200 */
{  -7601, {0x654026A9,0x3A98DB10,0xFA17DA31,0x379761F8}}, /* -2250 */
{  -7767, {0x5EB4B31A,0x408C8104,0x92A6E05B,0x8F1DBA33}}, /* -2300 */
{  -7933, {0x58958C32,0x3FFFB9DA,0xDA35C86C,0x20D0CD09}}, /* -2350 */
{  -8099, {0x52DBB1D9,0x8CEBF8EE,0x0507A14E,0xC982C091}}, /* -2400 */
{  -8265, {0x4D8097CF,0x42A2DE96,0xD9B90379,0x5152E347}}, /* -2450 */
{  -8431, {0x487E1E2C,0x693A4E7A,0x0BF9C011,0x33806F55}}, /* -2500 */
{  -8597, {0x43CE8A63,0x023F33D0,0x97D88EEF,0x620CCE6C}}, /* -2550 */
{  -8764, {0x7ED90161,0xF4C3E90B,0x0088DF17,0xD972E374}}, /* -2600 */
{  -8930, {0x76A5FBFF,0x04AE278A,0x2C8A6ECE,0xC48CBE08}}, /* -2650 */
{  -9096, {0x6EFAA44F,0x9A39F544,0x730AED19,0xD63EB13F}}, /* -2700 */
{  -9262, {0x67CE352D,0x4B961671,0xD0FEC0F7,0x22808DE0}}, /* -2750 */
{  -9428, {0x61187A91,0x77053FE1,0x67E0A3D6,0xF2421D12}}, /* -2800 */
{  -9594, {0x5AD1C833,0xCF18D5B8,0xFA5020D7,0xADB2AFBC}}, /* -2850 */
{  -9760, {0x54F2F0C4,0x211FA425,0xCBDD22E7,0x87BF7DB1}}, /* -2900 */
{  -9926, {0x4F753DB5,0x4D25CE07,0xD5DD5346,0x1CC1915E}}, /* -2950 */
{ -10092, {0x4A526790,0x0CEBDD0A,0xB88D7302,0x06F31000}}, /* -3000 */
{ -10258, {0x45848EC4,0xC28818B6,0x527E45EA,0x33F09E5B}}, /* -3050 */
{ -10424, {0x410634F4,0x18A8A686,0xEE9E0982,0xE37BDCC9}}, /* -3100 */
{ -10591, {0x79A46D4D,0x8C992236,0x2BBE6AB8,0x177DB069}}, /* -3150 */
{ -10757, {0x71C78ADA,0x8DC2207B,0x98F542F4,0x046F5D40}}, /* -3200 */
{ -10923, {0x6A6CC4C1,0x9D213862,0xC0DE3519,0x0275BB2D}}, /* -3250 */
{ -11089, {0x638BB1FE,0x6384D8C7,0xA84E15AC,0x5517E6BA}}, /* -3300 */
{ -11255, {0x5D1C74B7,0xBD9446E2,0xE0082708,0xA0C8ADEB}}, /* -3350 */
{ -11421, {0x5717B140,0xD41F0D9C,0xB5121C79,0xAA8B9036}}, /* -3400 */
{ -11587, {0x517685AF,0x0FE8ECA7,0x1DFDFE80,0xC67270A9}}, /* -3450 */
{ -11753, {0x4C3281FB,0x49B7EA3D,0xD7343679,0x4D3DB447}}, /* -3500 */
{ -11919, {0x4745A0A5,0x36A29B83,0xDB343F49,0x22C74C24}}, /* -3550 */
{ -12085, {0x42AA3FD0,0xA58A7237,0x89C0D320,0xD1E9AD8A}}, /* -3600 */
{ -12252, {0x7CB635A9,0x5BF661F4,0xC632E3E2,0xF647BADF}}, /* -3650 */
{ -12418, {0x74A6886A,0xE5E9BB45,0x6DA293A0,0xFAE9684C}}, /* -3700 */
{ -12584, {0x6D1C4003,0x2C0B45F0,0x92CA34B4,0xC4BFE44F}}, /* -3750 */
{ -12750, {0x660EBD19,0xD230DE7B,0x788512E1,0xE3EC1509}}, /* -3800 */
{ -12916, {0x5F75EF04,0xAFFC2E5B,0x453B9059,0x78A95B15}}, /* -3850 */
{ -13082, {0x594A4A8E,0xCCE6994A,0x8D1658E6,0x127414D7}}, /* -3900 */
{ -13248, {0x5384C157,0xF95FB4D9,0x8CE962DE,0x216EA3E8}}, /* -3950 */
{ -13414, {0x4E1EB9C3,0x279C02E0,0x125CCB44,0x688F20DE}}, /* -4000 */
{ -13580, {0x4912076A,0x49EEA01D,0x896D3BF8,0x17B5F0E9}}, /* -4050 */
{ -13746, {0x4458E40F,0x15385EF9,0x1C3780AB,0xE197E281}}, /* -4100 */
{ -13913, {0x7FDBD201,0x298FEA64,0xCA958361,0xC2D047FB}}, /* -4150 */
{ -14079, {0x779811DC,0x053996C9,0xFAD3C007,0x91F7B3DC}}, /* -4200 */
{ -14245, {0x6FDD143F,0xA7E2C206,0x6E67EF30,0xE4570132}}, /* -4250 */
{ -14411, {0x68A20220,0xBDA3F31C,0x89862EDE,0xDCACE386}}, /* -4300 */
{ -14577, {0x61DE96BB,0xD5AC1D35,0xFB6F154E,0xFF5D521E}}, /* -4350 */
{ -14743, {0x5B8B1620,0xCAA7A0AC,0xFF0D75A9,0x7D020FFA}}, /* -4400 */
{ -14909, {0x55A0445A,0xA21383A3,0xC09A3A3D,0x774AF9A7}}, /* -4450 */
{ -15075, {0x50175D29,0xC5612D19,0x742F8693,0x8F5F9163}}, /* -4500 */
{ -15241, {0x4AEA0C47,0x1F2A61B3,0x5C4CE6CA,0xF5F5144E}}, /* -4550 */
{ -15407, {0x46126627,0x433F5A94,0xE6BBDE23,0x19B44BB8}}, /* -4600 */
{ -15573, {0x418AE135,0x5AC48EC3,0x0A471C3F,0x6C014692}}, /* -4650 */
{ -15740, {0x7A9C9EFC,0x2D153D46,0x47CB45EC,0xD3E621AB}}, /* -4700 */
{ -15906, {0x72AFB186,0xD3CAB8EA,0x4467BD5B,0x9927693D}}, /* -4750 */
{ -16072, {0x6B45E9E4,0x903353CB,0x99365A93,0x59BA3B1C}}, /* -4800 */
{ -16238, {0x6456CDE8,0x1D1BAF40,0xAA55E657,0x2524DD49}}, /* -4850 */
{ -16404, {0x5DDA6FAB,0x5D7B14B9,0x3495512C,0x410B68C3}}, /* -4900 */
{ -16570, {0x57C9647E,0x1A01856C,0x380AA5FF,0x5498FA89}}, /* -4950 */
};


static const unpacked_real positive[] = {
{   -127, {0x80000000,0x00000000,0x00000000,0x00000000}}, /* 0 */
{   -124, {0xA0000000,0x00000000,0x00000000,0x00000000}}, /* 1 */
{   -121, {0xC8000000,0x00000000,0x00000000,0x00000000}}, /* 2 */
{   -118, {0xFA000000,0x00000000,0x00000000,0x00000000}}, /* 3 */
{   -114, {0x9C400000,0x00000000,0x00000000,0x00000000}}, /* 4 */
{   -111, {0xC3500000,0x00000000,0x00000000,0x00000000}}, /* 5 */
{   -108, {0xF4240000,0x00000000,0x00000000,0x00000000}}, /* 6 */
{   -104, {0x98968000,0x00000000,0x00000000,0x00000000}}, /* 7 */
{   -101, {0xBEBC2000,0x00000000,0x00000000,0x00000000}}, /* 8 */
{    -98, {0xEE6B2800,0x00000000,0x00000000,0x00000000}}, /* 9 */
{    -94, {0x9502F900,0x00000000,0x00000000,0x00000000}}, /* 10 */
{    -91, {0xBA43B740,0x00000000,0x00000000,0x00000000}}, /* 11 */
{    -88, {0xE8D4A510,0x00000000,0x00000000,0x00000000}}, /* 12 */
{    -84, {0x9184E72A,0x00000000,0x00000000,0x00000000}}, /* 13 */
{    -81, {0xB5E620F4,0x80000000,0x00000000,0x00000000}}, /* 14 */
{    -78, {0xE35FA931,0xA0000000,0x00000000,0x00000000}}, /* 15 */
{    -74, {0x8E1BC9BF,0x04000000,0x00000000,0x00000000}}, /* 16 */
{    -71, {0xB1A2BC2E,0xC5000000,0x00000000,0x00000000}}, /* 17 */
{    -68, {0xDE0B6B3A,0x76400000,0x00000000,0x00000000}}, /* 18 */
{    -64, {0x8AC72304,0x89E80000,0x00000000,0x00000000}}, /* 19 */
{    -61, {0xAD78EBC5,0xAC620000,0x00000000,0x00000000}}, /* 20 */
{    -58, {0xD8D726B7,0x177A8000,0x00000000,0x00000000}}, /* 21 */
{    -54, {0x87867832,0x6EAC9000,0x00000000,0x00000000}}, /* 22 */
{    -51, {0xA968163F,0x0A57B400,0x00000000,0x00000000}}, /* 23 */
{    -48, {0xD3C21BCE,0xCCEDA100,0x00000000,0x00000000}}, /* 24 */
{    -44, {0x84595161,0x401484A0,0x00000000,0x00000000}}, /* 25 */
{    -41, {0xA56FA5B9,0x9019A5C8,0x00000000,0x00000000}}, /* 26 */
{    -38, {0xCECB8F27,0xF4200F3A,0x00000000,0x00000000}}, /* 27 */
{    -34, {0x813F3978,0xF8940984,0x40000000,0x00000000}}, /* 28 */
{    -31, {0xA18F07D7,0x36B90BE5,0x50000000,0x00000000}}, /* 29 */
{    -28, {0xC9F2C9CD,0x04674EDE,0xA4000000,0x00000000}}, /* 30 */
{    -25, {0xFC6F7C40,0x45812296,0x4D000000,0x00000000}}, /* 31 */
{    -21, {0x9DC5ADA8,0x2B70B59D,0xF0200000,0x00000000}}, /* 32 */
{    -18, {0xC5371912,0x364CE305,0x6C280000,0x00000000}}, /* 33 */
{    -15, {0xF684DF56,0xC3E01BC6,0xC7320000,0x00000000}}, /* 34 */
{    -11, {0x9A130B96,0x3A6C115C,0x3C7F4000,0x00000000}}, /* 35 */
{     -8, {0xC097CE7B,0xC90715B3,0x4B9F1000,0x00000000}}, /* 36 */
{     -5, {0xF0BDC21A,0xBB48DB20,0x1E86D400,0x00000000}}, /* 37 */
{     -1, {0x96769950,0xB50D88F4,0x13144480,0x00000000}}, /* 38 */
{      2, {0xBC143FA4,0xE250EB31,0x17D955A0,0x00000000}}, /* 39 */
{      5, {0xEB194F8E,0x1AE525FD,0x5DCFAB08,0x00000000}}, /* 40 */
{      9, {0x92EFD1B8,0xD0CF37BE,0x5AA1CAE5,0x00000000}}, /* 41 */
{     12, {0xB7ABC627,0x050305AD,0xF14A3D9E,0x40000000}}, /* 42 */
{     15, {0xE596B7B0,0xC643C719,0x6D9CCD05,0xD0000000}}, /* 43 */
{     19, {0x8F7E32CE,0x7BEA5C6F,0xE4820023,0xA2000000}}, /* 44 */
{     22, {0xB35DBF82,0x1AE4F38B,0xDDA2802C,0x8A800000}}, /* 45 */
{     25, {0xE0352F62,0xA19E306E,0xD50B2037,0xAD200000}}, /* 46 */
{     29, {0x8C213D9D,0xA502DE45,0x4526F422,0xCC340000}}, /* 47 */
{     32, {0xAF298D05,0x0E4395D6,0x9670B12B,0x7F410000}}, /* 48 */
{     35, {0xDAF3F046,0x51D47B4C,0x3C0CDD76,0x5F114000}}, /* 49 */
{     39, {0x88D8762B,0xF324CD0F,0xA5880A69,0xFB6AC800}}, /* 50 */
};

static const unpacked_real positive_big[] = {
{    205, {0x924D692C,0xA61BE758,0x593C2626,0x705F9C56}}, /* 100 */
{    371, {0x9C69A972,0x84B578D7,0xFF2A7604,0x14536EFC}}, /* 150 */
{    537, {0xA738C6BE,0xBB12D16C,0xB428F8AC,0x016561DB}}, /* 200 */
{    703, {0xB2C71D5B,0xCA9023F8,0x743E20E9,0xEF511012}}, /* 250 */
{    869, {0xBF21E440,0x03ACDD2C,0xE0470A63,0xE6BD56C3}}, /* 300 */
{   1035, {0xCC573C2A,0x0ECCDAA6,0xDFACEC6F,0x21E0C000}}, /* 350 */
{   1201, {0xDA763FC8,0xCB9FF9E5,0x8E67937D,0xE0BBE1C7}}, /* 400 */
{   1367, {0xE98F1501,0x00B36CFA,0xB65E9494,0xD8C2C796}}, /* 450 */
{   1533, {0xF9B2FF64,0x9B8695D7,0x71A58A83,0x9043C753}}, /* 500 */
{   1700, {0x857A39F8,0x4F74CC5B,0x0E108549,0x81A8C314}}, /* 550 */
{   1866, {0x8EB39714,0x297EFB27,0xBCAFDBA9,0x6EBCB609}}, /* 600 */
{   2032, {0x98902352,0xECCA4018,0xE1BA29F5,0x35FDDA99}}, /* 650 */
{   2198, {0xA31B259C,0xFA50498F,0x7478A3CB,0xBA44EC49}}, /* 700 */
{   2364, {0xAE60AC5B,0x4A7B7D7A,0xAA52AB4B,0xA4F26D4C}}, /* 750 */
{   2530, {0xBA6D9B40,0xD7CC9ECC,0xDF143BBE,0x46291877}}, /* 800 */
{   2696, {0xC74FBA07,0xF0C92EF9,0xE060C816,0xA43B5900}}, /* 850 */
{   2862, {0xD515C434,0x4C1E8EF2,0x915861C3,0x76F9D551}}, /* 900 */
{   3028, {0xE3CF79EB,0xE415A12A,0x115F83E3,0x8A28E88B}}, /* 950 */
{   3194, {0xF38DB1F9,0xDD3DAC05,0x78D29695,0x39BCB6D3}}, /* 1000 */
{   3361, {0x82313688,0x08840353,0x31E727F3,0x62C8F3DD}}, /* 1050 */
{   3527, {0x8B30752E,0xA0B737CA,0xE2B8B4AC,0x1CF4337D}}, /* 1100 */
{   3693, {0x94CEDEC7,0xC0263D5C,0xB5D274FE,0xF7216BED}}, /* 1150 */
{   3859, {0x9F17732D,0xFAC7617B,0x3B7DFFB2,0x04B6D932}}, /* 1200 */
{   4025, {0xAA15F4D3,0x764E5ECA,0x2BFC1D0A,0x0D6534BB}}, /* 1250 */
{   4191, {0xB5D6F634,0x76B4AE83,0x0450CDC5,0xFC6C12E8}}, /* 1300 */
{   4357, {0xC267E837,0xD141BC8D,0x7C062DED,0x3659A77C}}, /* 1350 */
{   4523, {0xCFD7298D,0xB6CB9672,0xDCE472C6,0x19AA3F63}}, /* 1400 */
{   4689, {0xDE34171E,0x68C19BE5,0xBD0BE900,0x8CC5E96C}}, /* 1450 */
{   4855, {0xED8F1D9B,0xA3850FA9,0xAB379218,0x6B8256CB}}, /* 1500 */
{   5021, {0xFDF9CC48,0xD502BD40,0x58FBF2AD,0xF8B54057}}, /* 1550 */
{   5188, {0x87C37487,0xCCF4B0BF,0x532430E7,0x002ACA8E}}, /* 1600 */
{   5354, {0x912542FB,0xBBE846F8,0x2EBC1368,0xA3384583}}, /* 1650 */
{   5520, {0x9B2D0C0C,0xB06EED56,0xDECC74B4,0xEB100890}}, /* 1700 */
{   5686, {0xA5E64814,0x9FE1786A,0xEF25F571,0xA9728F18}}, /* 1750 */
{   5852, {0xB15D3A58,0xCD9F7620,0xE79894CF,0x86A0D164}}, /* 1800 */
{   6018, {0xBD9EFF0F,0xA859DD20,0x5F2541AD,0xDB504F3E}}, /* 1850 */
{   6184, {0xCAB99A5E,0xBC0A0DC4,0x3A1DACEB,0xC6562446}}, /* 1900 */
{   6350, {0xD8BC0861,0xDD67D4CD,0x75B99F36,0x384816A7}}, /* 1950 */
{   6516, {0xE7B64E4D,0xE2FC4251,0x4A729F6E,0x4AAFABE9}}, /* 2000 */
{   6682, {0xF7B98CC2,0x842FE2B5,0x45688FC2,0x2F11E261}}, /* 2050 */
{   6849, {0x846C09B0,0x28AE0395,0x04F60997,0x4DD3FFE9}}, /* 2100 */
{   7015, {0x8D92BADC,0x95425D0E,0xD4B36F9A,0x42106D9D}}, /* 2150 */
{   7181, {0x975B50D9,0x934DC561,0xBC0D44F6,0xC6443FDC}}, /* 2200 */
{   7347, {0xA1D0FBBB,0xC263D17E,0xA5956D6B,0xAE6CE7B3}}, /* 2250 */
{   7513, {0xACFFB184,0x83090339,0x23131270,0x3E4EE392}}, /* 2300 */
{   7679, {0xB8F43BCF,0x7901A604,0x9FB5ABED,0x8D052022}}, /* 2350 */
{   7845, {0xC5BC4672,0x073224F7,0xB2C46D6D,0x298A0659}}, /* 2400 */
{   8011, {0xD3666F1D,0x7DDE7C4E,0x333017DB,0xD61B7EFE}}, /* 2450 */
{   8177, {0xE2025615,0xDBEA8DE3,0x604814D6,0xFA77EA36}}, /* 2500 */
{   8343, {0xF1A0B010,0x4002CECE,0x35A9CADE,0x53116E1F}}, /* 2550 */
{   8510, {0x8129ACA6,0xBC5AD3BC,0xB6FB3BFD,0xB16DDD38}}, /* 2600 */
{   8676, {0x8A16B503,0x4602504E,0x737C7DD7,0xC9DEF8C7}}, /* 2650 */
{   8842, {0x93A1A621,0x48B73C1E,0xE4DF06EB,0x48627AE9}}, /* 2700 */
{   9008, {0x9DD56997,0x6B5136AA,0x79A03B05,0xFADACA10}}, /* 2750 */
{   9174, {0xA8BDAA0A,0x0064FA44,0x8B231A70,0xEB5444CE}}, /* 2800 */
{   9340, {0xB466E082,0x5A4CE083,0x9278F062,0x12460D6C}}, /* 2850 */
{   9506, {0xC0DE62B2,0x2420D379,0x4B7727DB,0x27F9E109}}, /* 2900 */
{   9672, {0xCE327233,0x0F0EDC21,0xD315D061,0x6DCD6AFE}}, /* 2950 */
{   9838, {0xDC724CD4,0x4411AF53,0x37B24BF1,0xB205E08C}}, /* 3000 */
{  10004, {0xEBAE3E08,0x3E7F855A,0x09A05AFE,0x56F7B915}}, /* 3050 */
{  10170, {0xFBF7B186,0xFDB9F19A,0x3775A8C2,0x217E9155}}, /* 3100 */
{  10337, {0x86B0A39C,0xEE703E0A,0xF49A3139,0x8112247B}}, /* 3150 */
{  10503, {0x8FFF7443,0xEC2F51ED,0x36FF0AD5,0xE3A835B0}}, /* 3200 */
{  10669, {0x99F2EF8D,0x4E18E0B5,0xA201005C,0x40788F64}}, /* 3250 */
{  10835, {0xA496769B,0x32506BEE,0xE15DBB1C,0x1B7A2785}}, /* 3300 */
{  11001, {0xAFF633E0,0x43E23C12,0x12AE896C,0x63A66194}}, /* 3350 */
{  11167, {0xBC1F2909,0x355B1724,0x192A0948,0xDECD065F}}, /* 3400 */
{  11333, {0xC91F3DDC,0x59E056A1,0x3A1FC785,0x24E00D28}}, /* 3450 */
{  11499, {0xD7055020,0x5EE713EC,0xD67AEFFB,0xFCACC7B9}}, /* 3500 */
{  11665, {0xE5E1449C,0x558E558E,0xF3D592C1,0xD6B30174}}, /* 3550 */
{  11831, {0xF5C41944,0x7C50865F,0x30F45170,0x3325E355}}, /* 3600 */
{  11998, {0x835FFC54,0x48D5F38E,0x8F0EC7DB,0x25E256E8}}, /* 3650 */
{  12164, {0x8C74275C,0xF408B1C6,0xF036EE15,0x24094FE7}}, /* 3700 */
{  12330, {0x9628EF80,0x7E40DA2A,0x19767930,0xC53272F4}}, /* 3750 */
{  12496, {0xA0896E2D,0xFAC1C18C,0x4B63CA6B,0xAD08E621}}, /* 3800 */
{  12662, {0xABA18130,0x98B57A3A,0x89844A21,0xC7B6521F}}, /* 3850 */
{  12828, {0xB77DD841,0x75A2AADB,0x49E6AFE3,0xD2B682D9}}, /* 3900 */
{  12994, {0xC42C0389,0x7FA4005F,0xDCD8AF8C,0xF261999A}}, /* 3950 */
{  13160, {0xD1BA8323,0xFE558C61,0x0D5C82A2,0x86614F3F}}, /* 4000 */
{  13326, {0xE038D7B3,0x7EE9B37B,0xE031C039,0x6758BC3E}}, /* 4050 */
{  13492, {0xEFB7941C,0x199CC910,0x72B95FF0,0x041AB008}}, /* 4100 */
{  13659, {0x8024383B,0xAB19730D,0x1D76F2D1,0x5166EC20}}, /* 4150 */
{  13825, {0x88FF2F2B,0xADE74531,0xC9AC5047,0x5E25293A}}, /* 4200 */
{  13991, {0x9276CF38,0x4BED2392,0xEAD28BD8,0x71D47F31}}, /* 4250 */
{  14157, {0x9C95EBE1,0x50D54253,0x97102B69,0xD8BF80E6}}, /* 4300 */
{  14323, {0xA768182D,0x6B5D9561,0xD505369C,0x2C72BAC8}}, /* 4350 */
{  14489, {0xB2F9B3E6,0x7FFA5A3E,0x9296ECA3,0xA3C1181A}}, /* 4400 */
{  14655, {0xBF57F9C0,0x22A66440,0x894357C1,0xAB0D9FCD}}, /* 4450 */
{  14821, {0xCC910E78,0x664E5136,0x10A1BD37,0xA7340862}}, /* 4500 */
{  14987, {0xDAB41104,0x4E87E733,0x04E36B0D,0x50182E05}}, /* 4550 */
{  15153, {0xE9D12BDA,0x62535426,0x1B901F9F,0x9CE97951}}, /* 4600 */
{  15319, {0xF9F9A76F,0x25D6D6B2,0xD85DED12,0x1C4506BA}}, /* 4650 */
{  15486, {0x859FFEFC,0x4F6A53B6,0xCAC8C580,0x5344B447}}, /* 4700 */
{  15652, {0x8EDBF847,0xBECB677D,0x78789424,0x4D8A7251}}, /* 4750 */
{  15818, {0x98BB4EE3,0x09F04D45,0x5A050B21,0x5EEBC517}}, /* 4800 */
{  15984, {0xA3494CE7,0x7775662A,0xCF2ACA8E,0x1F49B13B}}, /* 4850 */
{  16150, {0xAE920427,0x5937A4C0,0xA8C91282,0xE5AF94EB}}, /* 4900 */
{  16316, {0xBAA25BFB,0x5DAEBD09,0x0CFB69EB,0x20590F67}}, /* 4950 */
};



/* round_product()-- Round a multiprecision number into a shorter
 * number.  Zero goes to zero, and otherwise we look for the top
 * one-bit in the source and copy it to the destination.  The topmost
 * uncopied bit (if it exists) is used to round the result.  Returns
 * the bit position of the topmost bit, -1 if the number was zero. */

static int round_product(unsigned *dest, int dest_len, 
			 unsigned *source, int source_len) {
int c, i, j, shift, pos;
unsigned mask;

    for(i=0; i<source_len; i++)
	if (source[i] != 0)
	    break;

    if (i == source_len) {  /* Special case, source == 0 */
	for(i=0; i<dest_len; i++)
	    dest[i] = 0;

	return -1;
    }

    pos = 32*i;
  
    if (source[i] & 0x80000000) {   /* No shifts necessary */
	for(j=0; j<dest_len; j++) {
	    if (i+j >= source_len) {
		while(j<dest_len)
		    dest[j++] = 0;

		break;
	    }

	    dest[j] = source[i+j];
	}

	c = (i+j >= source_len)
	    ? 0
	    : (source[i+j] & 0x80000000);

    } else {
	mask = 0x40000000;
	shift = 1;

	while((source[i] & mask) == 0) {
	    mask >>= 1;
	    shift++;
	}

	pos += shift;

	for(j=0; j<dest_len; j++) {
	    dest[j] = source[i+j] << shift;

	    if (i+j+1 >= source_len)
		break;

	    dest[j] |= source[i+j+1] >> (32-shift);
	}

	c = (i+j+1 >= source_len)
	    ? 0
	    : source[i+j+1] & mask;
    }

    /* Add the carry */

    if (c) {
	for(j=dest_len-1; j>=0; j--)
	    if (++dest[j] != 0)
		break;

	if (dest[0] == 0) {  /* Overflow! */
	    dest[0] = 0x80000000;
	    pos--;
	}
    }

    return pos;
}



/* prod32()-- Macro for computing a 64 bit product from 32 bit
 * multiplicands. */

#define prod32(a, b, p1, p2) { unsigned long long p;           \
    p = ((unsigned long long) a) * ((unsigned long long) b);   \
    p1 = (unsigned) (p >> 32); p2 = (unsigned) p; }

#define acc32(m1, m2, accumulator, carry, n) { unsigned a, b;  \
    prod32(m1, m2, a, b);                                      \
    accumulator[n] += b; if (accumulator[n] < b) carry[n-1]++;   \
    accumulator[n-1] += a; if (accumulator[n-1] < a) carry[n-2]++; }



/* Product subroutines.  */

static void multiply_1(unpacked_real *m1, unpacked_real *m2,
		       unpacked_real *product) {
unsigned acc[2];

    prod32(m1->mantissa[0], m2->mantissa[0], acc[0], acc[1]);

    product->exp = m1->exp + m2->exp + 32
	- round_product(product->mantissa, 1, acc, 2);
}



static void multiply_2(unpacked_real *m1, unpacked_real *m2,
		       unpacked_real *product) {
unsigned acc[4], carry[3];
int i;

    acc[0] = acc[1] = acc[2] = acc[3] = 0;
    carry[0] = carry[1] = carry[2] = 0;

    acc32(m1->mantissa[1], m2->mantissa[1], acc, carry, 3);
    acc32(m1->mantissa[0], m2->mantissa[1], acc, carry, 2);
    acc32(m1->mantissa[1], m2->mantissa[0], acc, carry, 2);
    acc32(m1->mantissa[0], m2->mantissa[0], acc, carry, 1);

    for(i=2; i>=0; i--) {
	acc[i] += carry[i];
	if (acc[i] < carry[i] && i > 0)
	    carry[i-1]++;
    }

    product->exp = m1->exp + m2->exp + 64
	- round_product(product->mantissa, 2, acc, 4);
}



static void multiply_3(unpacked_real *m1, unpacked_real *m2,
		       unpacked_real *product) {
unsigned acc[6], carry[5];
int i;

    acc[0] = acc[1] = acc[2] = acc[3] = acc[4] = acc[5] = 0;
    carry[0] = carry[1] = carry[2] = carry[3] = carry[4] = 0;

    acc32(m1->mantissa[2], m2->mantissa[2], acc, carry, 5);

    acc32(m1->mantissa[1], m2->mantissa[2], acc, carry, 4);
    acc32(m1->mantissa[2], m2->mantissa[1], acc, carry, 4);

    acc32(m1->mantissa[0], m2->mantissa[2], acc, carry, 3);
    acc32(m1->mantissa[1], m2->mantissa[1], acc, carry, 3);
    acc32(m1->mantissa[2], m2->mantissa[0], acc, carry, 3);

    acc32(m1->mantissa[0], m2->mantissa[1], acc, carry, 2);
    acc32(m1->mantissa[1], m2->mantissa[0], acc, carry, 2);

    acc32(m1->mantissa[0], m2->mantissa[0], acc, carry, 1);

    for(i=4; i>=0; i--) {
	acc[i] += carry[i];
	if (acc[i] < carry[i] && i > 0)
	    carry[i-1]++;
    }

    product->exp = m1->exp + m2->exp + 96
	- round_product(product->mantissa, 3, acc, 6);
}



static void multiply_4(unpacked_real *m1, unpacked_real *m2,
		       unpacked_real *product) {
unsigned acc[8], carry[7];
int i;

    acc[0] = acc[1] = acc[2] = acc[3] = acc[4] = acc[5] = acc[6] = acc[7] = 0;
    carry[0] = carry[1] = carry[2] = carry[3] =
	carry[4] = carry[5] = carry[6] = 0;

    acc32(m1->mantissa[3], m2->mantissa[3], acc, carry, 7);

    acc32(m1->mantissa[2], m2->mantissa[3], acc, carry, 6);
    acc32(m1->mantissa[3], m2->mantissa[2], acc, carry, 6);

    acc32(m1->mantissa[1], m2->mantissa[3], acc, carry, 5);
    acc32(m1->mantissa[2], m2->mantissa[2], acc, carry, 5);
    acc32(m1->mantissa[3], m2->mantissa[1], acc, carry, 5);

    acc32(m1->mantissa[0], m2->mantissa[3], acc, carry, 4);
    acc32(m1->mantissa[1], m2->mantissa[2], acc, carry, 4);
    acc32(m1->mantissa[2], m2->mantissa[1], acc, carry, 4);
    acc32(m1->mantissa[3], m2->mantissa[0], acc, carry, 4);

    acc32(m1->mantissa[0], m2->mantissa[2], acc, carry, 3);
    acc32(m1->mantissa[1], m2->mantissa[1], acc, carry, 3);
    acc32(m1->mantissa[2], m2->mantissa[0], acc, carry, 3);

    acc32(m1->mantissa[0], m2->mantissa[1], acc, carry, 2);
    acc32(m1->mantissa[1], m2->mantissa[0], acc, carry, 2);

    acc32(m1->mantissa[0], m2->mantissa[0], acc, carry, 1);

    for(i=6; i>=0; i--) {
	acc[i] += carry[i];
	if (acc[i] < carry[i] && i > 0)
	    carry[i-1]++;
    }

    product->exp = m1->exp + m2->exp + 128
	- round_product(product->mantissa, 4, acc, 8);
}



/* multiply()-- Multiply two extended precision floats.  We call a
 * different subroutine for each precision. */

static void multiply(int precision, unpacked_real *m1, unpacked_real *m2,
		     unpacked_real *product) {

    switch(precision) {
    case 1:  multiply_1(m1, m2, product);  break;
    case 2:  multiply_2(m1, m2, product);  break;
    case 3:  multiply_3(m1, m2, product);  break;
    case 4:  multiply_4(m1, m2, product);  break;
    }
}



/* lookup_p10()-- Look up a power of ten up in the tables, rounding it
 * to the desired precision.  The value is assumed to be valid. */

static void lookup_p10(int power, int precision, unpacked_real *result) {
const unpacked_real *table, *table_big, *p;

    if (power >= 0) {
	table = positive;
	table_big = positive_big;

    } else {
	table = negative;
	table_big = negative_big;

	power = -power;
    }

    p = (power <= J_PARM) 
	? table + power
	: table_big + ((power-J_PARM) / H_PARM) - 1;

    *result = *p;

    /* We assume that there are no carries during rounding and that there
     * are no trailing zeroes that would require invoking round to even. */

    if (precision < MAX_MANTISSA && result->mantissa[precision] & 0x80000000)
	result->mantissa[precision-1]++;

    result->exp += 32*(MAX_MANTISSA-precision);
}



/* get_p10()-- Calculate a power of ten.  Returns nonzero if we had to do
 * a multiplication to get the power of ten, zero if it was in the table. */

static int get_p10(int power, int precision, unpacked_real *result) {
unpacked_real a, b;
int x, y;

    if (-J_PARM <= power && power <= J_PARM) {
	lookup_p10(power, precision, result);
	return 0;
    }

    x = power % J_PARM;
    y = power - x;

    lookup_p10(x, precision, &a);
    lookup_p10(y, precision, &b);

    multiply(precision, &a, &b, result);
    return 1;
}



/* acc_digit()-- Given a precision+1 word accumulator, multiply it by
 * ten and add the new number.  Returns zero if everything is OK,
 * nonzero for overflow.  Overflow amounts to checking the top four
 * bits of the most significant word.  The extra word guarantees that
 * we still have enough bits to round correctly. */

static int acc_digit(unsigned *acc, int precision, unsigned new) {
unsigned a, b, carry[MAX_MANTISSA+1], new_acc[MAX_MANTISSA+1];
unsigned long long s;
int i;

    if (acc[0] & 0xF0000000) 
	return 1;

    for(i=0; i<=precision; i++) {
	new_acc[i] = 0;
	carry[i] = 0;
    }

    /* Multiply by 10 */

    for(i=precision; i>=0; i--) {
	s = 10 * (unsigned long long) acc[i];

	a = s >> 32;
	b = (unsigned) s;

	new_acc[i] += b;
	if (new_acc[i] < b)
	    carry[i]++;

	if (i > 0) {
	    new_acc[i-1] += a;
	    if (new_acc[i-1] < a)
		carry[i-1]++;
	}
    }

    new_acc[precision] += new;
    if (new_acc[precision] < new)
	carry[precision-1]++;

    for(i=precision; i>=0; i--) {
	acc[i] = new_acc[i] + carry[i];
	if (acc[i] < carry[i] && i > 0)
	    carry[i-1]++;
    }

    return 0;
}



/* parse_exponent()-- Parse an exponent.  Exponents larger than 10^6
 * are reported as 10^6, exponents smaller than 10^{-6} are reported
 * as 10^{-6}. */

static int parse_exponent(char *p) {
int c, sign, exp;

    c = *p++;
    switch(c) {
    case '+':   sign =  1;  c = *p++;  break;
    case '-':   sign = -1;  c = *p++;  break;
    default:    sign =  1;             break;
    }

    exp = 0;

    while('0' <= c && c <= '9') {
	exp = 10*exp + c - '0';

	if (exp >= 1000000) {
	    exp = 1000000;
	    break;
	}

	c = *p++;
    }

    return sign * exp;
}



/* parse_string()-- Parse the string that represents a real number.
 * After disposing of exceptional cases first, the we figure out two
 * integers a and b such that the input number is a x 10^b, where 'b'
 * is an int and 'a' is a mantissa.  Our task is to find 'a' within
 * one bit, so we use an extra word during conversion, and return an
 * additional power of two that the caller must shift the final result
 * by.  If this power of two is zero, the value of 'a' is exact.
 *
 * If this extended-extended precision overflows, we quit accumulating
 * digits in it and additional digits (before the decimal point)
 * contribute another power of ten.  The caller takes care of any sign
 * digit. */

static void parse_string(char *string, int precision,
			 unpacked_real *p, int *b) {
int i, overflow, shift, ignored_digits, seen_dp;
unsigned mask, sig[MAX_MANTISSA+1];
int seen_nonzero;
char ch;

    for(i=0; i<=precision; i++)
	sig[i] = 0;

    digits = 0;
    right_digits = 0;
    ignored_digits = 0;
    seen_dp = 0;
    seen_nonzero = 0;
    overflow = 0;
    exp = 0;

    for(;;) {
	ch = *string++;

	switch(ch) {
	case '1': case '2': case '3':
	case '4': case '5': case '6':
	case '7': case '8': case '9':
	    seen_nonzero = 1;

	    /* Fall through */

	case '0':
	    if (!seen_nonzero && !seen_dp)
		break;     /* consume leading zeroes */

	    digits++;

	    if (!overflow)
		overflow = acc_digit(sig, precision, ch - '0');

	    if (overflow)
		ignored_digits++;

	    if (seen_dp)
		right_digits++;

	    break;

	case '.':
	    seen_dp = 1;
	    break;

	case '+': case '-':
	    string--;

	    /* Fall through */

	case 'd': case 'D': case 'e': case 'E': case 'q': case 'Q':
	    exp = parse_exponent(string);
	    goto finish;

	default:
	    goto finish;
	}
    }

finish:
    *b = exp + ignored_digits + (seen_dp ? -right_digits : 0);

    /* Now see if we have to massage our mantissa.  If the top word is
     * zero, then the conversion was exact. */

    if (sig[0] == 0) {
	for(i=0; i<precision; i++)
	    p->mantissa[i] = sig[i+1];

	p->exp = 0;

    } else { /* Gotta round.  Figure out how many bits we have to shift left */
	mask = 0x80000000;

	shift = 0;
	while((sig[0] & mask) == 0) {
	    mask = mask >> 1;
	    shift++;
	}

	if (shift == 0) {
	    for(i=0; i<precision; i++)
		p->mantissa[i] = sig[i];

	} else
	    for(i=0; i<precision; i++)
		p->mantissa[i] = (sig[i] << shift) | (sig[i+1] >> (32-shift));

    /* See if we need to round up */

	if (sig[precision] & (1 << (31-shift)))
	    for(i=precision-1; i>=0; i--)
		if (++p->mantissa[i] != 0)
		    break;

	p->exp = 32 - shift;
    }
}



/* round_real()-- round an unpacked real to the number of bits really
 * in the mantissa of the machine number.  Returns the (absolute)
 * difference in bits in the last place from the cutoff value for
 * rounding up or down. */

/* round_b4()-- Kind 4 real has a 24 bit mantissa. */

static int round_b4(unpacked_real *p) {
int tail;

    if (p->mantissa[0] == 0) {
	p->exp = 0;
	return 999;
    }

    /* Tail is eight bits */

    tail = p->mantissa[0] & 0xFF;

    p->mantissa[0] >>= 8;

    if ((tail & 0x80) && ++p->mantissa[0] == 2*MAN4_MSW) {
	p->mantissa[0] = MAN4_MSW;
	p->exp++;
    }

    tail = tail - 0x80;
    if (tail < 0)
	tail = -tail;

    p->exp += 8;
    return tail;
}



/* round_b8()-- Kind 8 real has a 53 bit mantissa. */

static int round_b8(unpacked_real *p) {
int tail;

    if (p->mantissa[0] == 0) {
	p->exp = 0;
	return 999;
    }

    /* Tail is 11 bits */

    tail = p->mantissa[1] & 0x7FF;

    p->mantissa[1] = (p->mantissa[1] >> 11) | (p->mantissa[0] << 21);
    p->mantissa[0] >>= 11;

    if ((tail & 0x400) && ++p->mantissa[1] == 0 &&
	++p->mantissa[0] == 2*MAN8_MSW) {
	p->mantissa[0] = MAN8_MSW;
	p->exp++;
    }

    tail = tail - 0x400;
    if (tail < 0)
	tail = -tail;

    p->exp += 11;
    return tail;
}



/* round_b10()-- Kind 10 real has a 64 bit mantissa.  This means that
 * that the tail is on a word boundary. */

static int round_b10(unpacked_real *p) {
int tail;

    if (p->mantissa[0] == 0) {
	p->exp = 0;
	return 999;
    }

    /* Tail is 32 bits */

    tail = p->mantissa[2];

    if ((tail & 0x80000000) && ++p->mantissa[1] == 0 &&
	++p->mantissa[0] == 0) {
	p->mantissa[0] = 0x80000000;
	p->exp++;
    }

    if (tail == 0)
	tail = 999;

    else {
	tail = tail - 0x80000000;
	if (tail < 0)
	    tail = -tail;
    }

    p->exp += 32;
    return tail;
}



/* round_b16()-- Kind 16 real has a 112 bit mantissa. */

static int round_b16(unpacked_real *p) {
int tail;

    if (p->mantissa[0] == 0) {
	p->exp = 0;
	return 999;
    }

    /* Tail is 15 bits */

    tail = p->mantissa[3] & 0x7FFF;

    p->mantissa[3] = (p->mantissa[3] >> 15) | (p->mantissa[2] << 17);
    p->mantissa[2] = (p->mantissa[2] >> 15) | (p->mantissa[1] << 17);
    p->mantissa[1] = (p->mantissa[1] >> 15) | (p->mantissa[0] << 17);
    p->mantissa[0] >>= 15;
    
    if ((tail & 0x4000) && ++p->mantissa[3] == 0 && ++p->mantissa[2] == 0 &&
	++p->mantissa[1] == 0 && ++p->mantissa[0] == 2*MAN16_MSW) {
	p->mantissa[0] = MAN16_MSW;
	p->exp++;
    }

    tail = tail - 0x8000;
    if (tail < 0)
	tail = -tail;

    p->exp += 15;
    return tail;
}



/* infinity()-- Generate an infinity */

static void infinity(int kind, unpacked_real *value) {

    switch(kind) {
    case 4:
	value->mantissa[0] = 0;
	value->exp = EXP4_NAN - EXP4_BIAS - MAN4_LEN;
	break;

    case 8:
	value->mantissa[0] = 0;
	value->mantissa[1] = 0;
	value->exp = EXP8_NAN - EXP8_BIAS - MAN8_LEN;
	break;

    case 10:
	value->mantissa[0] = 2147483648U;
	value->mantissa[1] = 0;
	value->exp = EXP10_NAN - EXP10_BIAS - MAN10_LEN + 1;
	break;

    case 16:
	value->mantissa[0] = value->mantissa[1] = 0;
	value->mantissa[2] = value->mantissa[3] = 0;
	value->exp = EXP16_NAN - EXP16_BIAS - MAN16_LEN;
	break;
    }
}



#if 0

static void show_value(char *name, int kind, unpacked_real *value) {

    switch(kind) {
    case 4:
	printf("%-10s   %08X   2^%d\n", name, value->mantissa[0], value->exp);
	break;

    case 8:
	printf("%-10s   %08X %08X   2^%d\n", name, value->mantissa[0],
	       value->mantissa[1], value->exp);
	break;

    case 10:
	printf("%-10s   %08X %08X   2^%d\n", name, value->mantissa[0],
	       value->mantissa[1], value->exp);
	break;

    case 16:
	printf("%-10s   %08X %08X %08X %08X  2^%d\n", name, value->mantissa[0],
	       value->mantissa[1], value->mantissa[2], value->mantissa[3],
	       value->exp);
	break;
    }
}


static void show_bigint(char *name, bigint *b) {
int i;

    printf("%-10s (%3d) 0x", name, b->n);

    for(i=b->n-1; i>=0; i--)
	printf("%08X", b->d[i]);

    putchar('L');
    putchar('\n');
}


int main(int argc, char *argv[]) {
float x;

    if (argc < 2) {
	printf("Missing argument\n");
	exit(0);
    }

    convert_real(&x, argv[1], 16);

    return 0;
}

#endif



/* bellerophon()-- Convert a string to a real.  Returns nonzero if the
 * result is not certain to be correctly rounded. */

static int bellerophon(char *p, int kind, char *buffe, unpacked_real *final) {
int prec, bits, exponent, error;
unpacked_real mantissa, pow10;

    switch(kind) {
    case 4:   prec = 1;  break;
    case 8:   prec = 2;  break;
    case 10:  prec = 3;  break;
    case 16:  prec = 4;  break;
    default:  prec = 0;  internal_error("bellerophon(): Bad kind"); break;
    }

    parse_string(p, prec, &mantissa, &exponent);

    /* The maximum bits of error we can tolerate depends on if we read
     * the mantissa exactly or we have to do one or two multiplications.
     * Clinger gives the formula: d = 1/2 + \beta(d1+d2)
     * where \beta is the internal radix (=2 here) and the d's are
     * uncertainties in units of bits in the last place.  After
     * evaluating everything, the results are:
     *
     * Exact mantissa, 1 mult   = ceil(3/2)   = 2
     * Exact mantissa, 2 mult   = ceil(11/2)  = 6
     * Inexact mantissa, 1 mult = ceil(5/2)   = 3
     * Inexact mantissa, 2 mult = ceil(13/2)  = 7
     * 
     * So it turns out we can calculate the maximum error by taking an
     * exact mantissa at one bit and an inexact at two bits.  If the
     * power of ten was in the table, add one bit else add five.  */

    bits = (mantissa.exp == 0) ? 1 : 2;

    if (exponent < -MAX_EXPONENT)
	return 0;

    if (exponent > MAX_EXPONENT) {
	infinity(kind, final);
	return 0;
    }

    bits += get_p10(exponent, prec, &pow10) ? 5 : 1;

    /* Multiply the mantissa and the power of ten to get the result. */

    multiply(prec, &mantissa, &pow10, final);

    switch(kind) {
    case 4:   error = round_b4(final);   break;
    case 8:   error = round_b8(final);   break;
    case 10:  error = round_b10(final);  break;
    case 16:  error = round_b16(final);  break;
    default:  error = 0;                 break;
    }

    return error <= bits;
}


/* Subroutines associated with Clinger's Algorithm R, used when the
 * previous floating point approximation isn't good enough. */


/* parse_string2()-- Read the digit-string into a multiprecision
 * integer. */

static void parse_string2(char *number, bigint *f) {
unsigned long long prod;
unsigned *p;
int i, k;
char c;

    f->n = 1;
    f->d[0] = 0;

    for(;;) {
	c = *number++;
	if (c == '.')
	    continue;

	if (c < '0' || c > '9')
	    break;

	/* Multiply f by 10 */

	p = &f->d[0];
	k = 0;

	for(i=0; i<f->n; i++) {
	    prod = 10 * ((unsigned long long) *p);

	    *p = ((unsigned) (prod & 0xFFFFFFFF)) + k;
	    k = ((unsigned) (prod >> 32)) + (*p++ < k);
	}

	if (k != 0) {
	    *p = k;
	    f->n++;
	}

	/* Add the digit */

	i = c - '0';
	f->d[0] += i;

	if (f->d[0] >= i)
	    continue;

	for(i=1; i<f->n; i++)
	    if (++f->d[i] != 0)
		break;

	if (i >= f->n)
	    f->d[f->n++] = 1;
    }
}



/* scale()-- Scale a bigint by a nonnegative power of two. */

static void scale(int n, bigint *f) {
int i, m1, m2; 
unsigned h, k, *p;

    m1 = n % 32;
    m2 = n / 32;

    if (m1 == 0) {
	i = f->n-1;
	f->n += m2;
	p = f->d + f->n - 1;

	for(; i>=0; i--)
	    *p-- = f->d[i];

    } else {
	i = f->n - 1;
	f->n += m2 + 1;
	p = f->d + f->n - 1;
	h = 0;

	for(; i>=0; i--) {
	    k = f->d[i];
	    *p-- = (h << m1) | (k >> (32-m1));
	    h = k;
	}

	*p-- = h << m1;
    }

    while(p >= f->d)
	*p-- = 0;

    while(f->n > 0 && f->d[f->n - 1] == 0)
	f->n--;
}



/* product()-- Compute a product of two bigints.  The result can be
 * one of the multiplicands. */

static void product(bigint *a, bigint *b, bigint *q) {
unsigned long long partial;
int i, j, ah, al;
bigint prod;
unsigned *p;

    prod.n = a->n + b->n;
    prod.d = alloca(prod.n * sizeof(unsigned));

    for(i=0; i<prod.n; i++)
	prod.d[i] = 0;

    for(i=0; i<a->n; i++)
	for(j=0; j<b->n; j++) {
	    partial = ((unsigned long long) a->d[i]) *
		      ((unsigned long long) b->d[j]);

	    ah = (unsigned) (partial >> 32);
	    al = (unsigned) (partial & 0xFFFFFFFF);

	    p = prod.d + i+j;

	    *p += al;
	    if (*p < al)
		ah++;

	    p++;

	    *p += ah;
	    if (*p >= ah)
		continue;

	    p++;

	    while(++(*p) == 0)
		p++;
	}

    while(prod.n > 0 && prod.d[prod.n-1] == 0)
	prod.n--;

    q->n = prod.n;
    memcpy(q->d, prod.d, prod.n * sizeof(unsigned));
}



/* cmp()-- Compare two bigints */

static int cmp(bigint *a, bigint *b) {
int n;

    if (a->n < b->n) return -1;
    if (a->n > b->n) return  1;

    n = a->n - 1;

    while(n >= 0) {
	if (a->d[n] < b->d[n]) return -1;
	if (a->d[n] > b->d[n]) return  1;

	n--;
    }

    return 0;
}



/* diff()-- Calculate a difference between two bigints.  Returns the
 * sign of the difference.  Modifies the minuend. */

static int diff(bigint *x, bigint *y) {
bigint *a, *b, diff;
int i, j, c, n;
unsigned m;

    c = cmp(x, y);
    if (c == 0) {
	x->n = 1;
	x->d[0] = 0;
	return 0;
    }

    if (c < 0) {
	a = y;
	b = x;

    } else {
	a = x;
	b = y;
    }

    /* At this point, a > b */

    n = diff.n = a->n;
    diff.d = alloca(n*sizeof(unsigned));

    memcpy(diff.d, a->d, n*sizeof(unsigned));

    for(i=0; i<n; i++) {
	if (i >= b->n)
	    break;

	m = diff.d[i];
	diff.d[i] -= b->d[i];

	if (b->d[i] <= m)
	    continue;  /* No borrow */

	j=i+1;
	while(j < n && diff.d[j]-- == 0)
	    j++;
    }

    while(diff.n > 0 && diff.d[diff.n-1] == 0)
	diff.n--;

    x->n = diff.n;
    memcpy(x->d, diff.d, diff.n * sizeof(unsigned));

    return c;
}



/* power_n()-- Compute a power using the square-and-shift method. */

static void power_n(int base, int power, int digits, bigint *f) {
bigint square, prod;

    square.n = 1;
    square.d = alloca(sizeof(unsigned) * digits);
    square.d[0] = base;

    prod.d = alloca(sizeof(unsigned) * digits);

    f->n = 1;
    f->d[0] = 1;

    for(;;) {
	if (power & 1) {
	    product(f, &square, &prod);

	    f->n = prod.n;
	    memcpy(f->d, prod.d, prod.n*sizeof(unsigned));
	}

	power >>= 1;
	if (power == 0)
	    break;

	product(&square, &square, &prod);

	square.n = prod.n;
	memcpy(square.d, prod.d, prod.n*sizeof(unsigned));
    }
}



/* off_mantissa()-- Return the bottom bit of the mantissa */

static int odd_mantissa(int kind, unpacked_real *value) {
int m;

    switch(kind) {
    case 4:    m = value->mantissa[0] & 1;  break;
    case 8:    m = value->mantissa[1] & 1;  break;
    case 10:   m = value->mantissa[2] & 1;  break;
    case 16:   m = value->mantissa[3] & 1;  break;
    default:   m = -1;                      break;
    }

    return m;
}



/* inc_value()-- Increment a value, returns nonzero if we've
 * incremented to infinity. */

static int inc_value(int kind, unpacked_real *value) {
int rc;

    rc = 0;
    switch(kind) {
    case 4:
	if (++value->mantissa[0] != 2*MAN4_MSW ||
	    ++value->exp == EXP4_NAN - EXP4_BIAS)
	    break;

	value->mantissa[0] = 0;
	rc = 1;
	break;

    case 8:
	if (++value->mantissa[1] != 0 ||
	    ++value->mantissa[0] != 2*MAN8_MSW ||
	    ++value->exp == EXP8_NAN - EXP8_BIAS)
	    break;

	value->mantissa[0] = 0;
	rc = 1;
	break;

    case 10:
	if (++value->mantissa[1] != 0 ||
	    ++value->mantissa[0] != 0 ||
	    ++value->exp == EXP8_NAN - EXP8_BIAS)
	    break;

	value->mantissa[0] = 0;
	rc = 1;
	break;

    case 16:
	if (++value->mantissa[3] != 0 ||
	    ++value->mantissa[2] != 0 ||
	    ++value->mantissa[1] != 0 ||
	    ++value->mantissa[0] !=  2*MAN16_MSW ||
	    ++value->exp == EXP16_NAN - EXP16_BIAS)
	    break;

	value->mantissa[0] = 0;
	rc = 1;
	break;
    }

    return rc;
}



/* dec_value()-- Decrement a value, returns nonzero if we've
 * decremented to zero. */

static int dec_value(int kind, unpacked_real *value) {
int rc;

    rc = 0;
    switch(kind) {
    case 4:
	if (value->exp != -EXP4_BIAS) {
	    if (value->mantissa[0]-- != MAN4_MSW)
		break;

	    value->mantissa[0] = 2*MAN4_MSW-1;
	    value->exp--;
	    break;
	}

	if (value->mantissa[0]-- != 0)
	    break;

	value->mantissa[0] = 0;
	rc = 1;
	break;

    case 8:
	if (value->exp != -EXP8_BIAS) {
	    if (value->mantissa[1]-- != 0 ||
		value->mantissa[0]-- != MAN8_MSW)
		break;

	    value->mantissa[0] = 2*MAN8_MSW-1;
	    value->exp--;
	    break;
	}

	if (value->mantissa[1]-- != 0 ||
	    value->mantissa[0]-- != 0)
	    break;

	value->mantissa[0] = 0;
	rc = 1;
	break;

    case 10:
	if (value->exp != -EXP10_BIAS) {
	    if (value->mantissa[1]-- != 0 ||
		value->mantissa[0]-- != 0)
		break;

	    value->exp--;
	    break;
	}

	if (value->mantissa[1]-- != 0 ||
	    value->mantissa[0]-- != 0)
	    break;

	rc = 1;
	break;

    case 16:
	if (value->exp != -EXP16_BIAS) {
	    if (value->mantissa[3]-- != 0 ||
		value->mantissa[2]-- != 0 ||
		value->mantissa[1]-- != 0 ||
		value->mantissa[0]-- != MAN16_MSW)
		break;

	    value->mantissa[0] = 2*MAN16_MSW-1;
	    value->exp--;
	    break;
	}

	if (value->mantissa[3]-- != 0 ||
	    value->mantissa[2]-- != 0 ||
	    value->mantissa[1]-- != 0 ||
	    value->mantissa[0]-- != MAN16_MSW)
	    break;

	value->mantissa[0] = MAN16_MSW;
	rc = 1;
	break;
    }

    return rc;
}



/* min_mantissa()-- Return nonzero if the value is a normalized number
 * with the minimum mantissa. */

static int min_mantissa(int kind, unpacked_real *value) {

    switch(kind) {
    case 4:
	if (value->exp == -EXP4_BIAS)
	    break;

	if (value->mantissa[0] == MAN4_MSW)
	    return 1;

	break;

    case 8:
	if (value->exp == -EXP8_BIAS)
	    break;

	if (value->mantissa[0] == MAN8_MSW && value->mantissa[1] == 0)
	    return 1;

	break;

    case 10:
	if (value->exp == -EXP10_BIAS)
	    break;

	if (value->mantissa[0] == 0x80000000 && value->mantissa[1] == 0)
	    return 1;

	break;

    case 16:
	if (value->exp == -EXP16_BIAS)
	    break;

	if (value->mantissa[0] == MAN16_MSW && value->mantissa[1] == 0 &&
	    value->mantissa[2] == 0         && value->mantissa[3] == 0)
	    return 1;

	break;
    }

    return 0;
}



/* fixup()-- Made sure a number has been correctly read.  Returns
 * nonzero if the number has been set to zero or infinity. */

static int fixup(char *number, int kind, unpacked_real *value) {
bigint f, x, y, p, m;
int c, d, n, words;

    words = (0.103810252965 * digits) + 5;  /* 0.1038... = 1.0/log10(2^32) */

    f.d = alloca(words*sizeof(unsigned));

    words += (exp >= 0) ? exp : -exp;
    words += ((value->exp >= 0) ? value->exp : -value->exp) / 32;

    x.d = alloca(words*sizeof(unsigned));
    y.d = alloca(words*sizeof(unsigned));
    p.d = alloca((words+kind)*sizeof(unsigned));
    m.d = alloca(words*sizeof(unsigned));

    parse_string2(number, &f);

    /* Loop until the number is OK. */

    for(;;) {
	COPY_BIGINT(&x, &f);

	switch(kind) {
	case 4:
	    m.n = 1;
	    m.d[0] = value->mantissa[0];
	    break;

	case 8:
	case 10:
	    m.n = 2;
	    m.d[0] = value->mantissa[1];
	    m.d[1] = value->mantissa[0];
	    break;

	case 16:
	    m.n = 4;
	    m.d[0] = value->mantissa[3];
	    m.d[1] = value->mantissa[2];
	    m.d[2] = value->mantissa[1];
	    m.d[3] = value->mantissa[0];
	    break;
	}

	COPY_BIGINT(&y, &m);

	if (value->exp < 0)
	    scale(-value->exp, &x);
	else
	    scale(value->exp, &y);

	n = -right_digits + exp;

	if (n < 0) {
	    power_n(10, -n, words, &p);
	    product(&p, &y, &y);

	} else if (n > 0) {
	    power_n(10, n, words, &p);
	    product(&p, &x, &x);
	}

	c = diff(&x, &y);
	if (c == 0)
	    return 0;

	product(&m, &x, &x);
	scale(1, &x);

	/* If are we considering decrementing the value and it is the
	 * mininum mantissa, the spacing changes for the potentially
	 * lower number. */

	if (c < 0 && min_mantissa(kind, value))
	    scale(1, &x);

	d = cmp(&x, &y);
	if (d < 0)
	    return 0;

	if (d == 0) {   /* Exactly halfway, round to even */
	    if (!odd_mantissa(kind, value))
		return 0;

	    if (c < 0)
		dec_value(kind, value);

	    else
		inc_value(kind, value);

	    return 0;
	}

	if (c < 0 && dec_value(kind, value))
	    return 1;

	if (c >= 0 && inc_value(kind, value))
	    return 1;
    }
}



/* range_check()-- Make sure a number isn't too big or too small for
 * its kind.  If the slop variable is nonzero, we allow exponents less
 * than or equal to the EXP_NAN instead of strictly less than.
 * Returns zero if we didn't do anything to the number, one if we
 * denormalized the number, or two if we've massaged the number to
 * infinity or zero.  It's possible for this subroutine to be called twice. */

static int range_check(int kind, unpacked_real *value, int slop) {
int n, max_exp;

    switch(kind) {
    case 4:
	max_exp = EXP4_NAN - EXP4_BIAS - MAN4_LEN;

	if ((slop && value->exp > max_exp) ||
	    (!slop && value->exp >= max_exp)) {
	    infinity(kind, value);
	    return 2;
	}

	if (value->exp < -EXP4_BIAS - 2*MAN4_LEN) {
	    value->mantissa[0] = 0;
	    value->exp = 0;
	    return 2;
	}

	if (value->exp > -EXP4_BIAS - MAN4_LEN)
	    break;

	n = -value->exp - EXP4_BIAS - MAN4_LEN;

	value->mantissa[0] >>= n;
	value->exp += n;

	return 1;

    case 8:
	max_exp = EXP8_NAN - EXP8_BIAS - MAN8_LEN;

	if ((slop && value->exp > max_exp) ||
	    (!slop && value->exp >= max_exp)) {
	    infinity(kind, value);
	    return 2;
	}

	if (value->exp < -EXP8_BIAS - 2*MAN8_LEN) {
	    value->mantissa[0] = value->mantissa[1] = 0;
	    value->exp = 0;
	    return 2;
	}

	if (value->exp >= -EXP8_BIAS - MAN8_LEN)
	    break;

	while(value->exp < -EXP8_BIAS - MAN8_LEN) {
	    value->mantissa[1] = (value->mantissa[0] << 31) |
				 (value->mantissa[1] >> 1);
	    value->mantissa[0] = value->mantissa[0] >> 1;
	    value->exp++;
	}

	return 1;

    case 10:
	max_exp = EXP10_NAN - EXP10_BIAS - MAN10_LEN + 1;

	if ((slop && value->exp > max_exp) ||
	    (!slop && value->exp >= max_exp)) {
	    infinity(kind, value);
	    return 2;
	}

	if (value->exp < -EXP10_BIAS - 2*MAN10_LEN) {
	    value->mantissa[0] = value->mantissa[1] = 0;
	    value->exp = 0;
	    return 2;
	}

	if (value->exp > -EXP10_BIAS - MAN10_LEN)
	    break;

	while(value->exp <= -EXP10_BIAS - MAN10_LEN) {
	    value->mantissa[1] = (value->mantissa[0] << 31) |
				 (value->mantissa[1] >> 1);
	    value->mantissa[0] = value->mantissa[0] >> 1;
	    value->exp++;
	}

	return 1;

    case 16:
	max_exp = EXP16_NAN - EXP16_BIAS - MAN16_LEN;

	if ((slop && value->exp > max_exp) ||
	    (!slop && value->exp >= max_exp)) {
	    infinity(kind, value);
	    return 2;
	}

	if (value->exp < -EXP16_BIAS - 2*MAN16_LEN) {
	    value->mantissa[0] = value->mantissa[1] = 0;
	    value->mantissa[2] = value->mantissa[3] = 0;
	    value->exp = 0;
	    return 2;
	}

	if (value->exp >= -EXP16_BIAS - MAN16_LEN)
	    break;

	while(value->exp < -EXP16_BIAS - MAN16_LEN) {
	    value->mantissa[3] = (value->mantissa[2] << 31) |
				 (value->mantissa[3] >> 1);
	    value->mantissa[2] = (value->mantissa[1] << 31) |
				 (value->mantissa[2] >> 1);
	    value->mantissa[1] = (value->mantissa[0] << 31) |
				 (value->mantissa[1] >> 1);
	    value->mantissa[0] = value->mantissa[0] >> 1;
	    value->exp++;
	}

	return 1;
    }

    return 0;
}



void convert_real(void *p, char *number, int kind) {
unpacked_real value;
int m, n, exp, sign;

    sign = 0;
    if (number[0] == '+')
	number++;

    else if (number[0] == '-') {
	sign = 1;
	number++;
    }

    m = bellerophon(number, kind, NULL, &value);

    if (!m) {
	if (range_check(kind, &value, 0) == 1)
	    fixup(number, kind, &value);

    } else if (range_check(kind, &value, 1) != 2) {
	n = fixup(number, kind, &value);
	if (!n)
	    range_check(kind, &value, 0);
    }

    switch(kind) {
    case 4:
	exp = (value.exp == 0 && ((value.mantissa[0] & (2*MAN4_MSW-1)) == 0))
	    ? 0
	    : value.exp + EXP4_BIAS + MAN4_LEN;

	if (exp == 0)
	    value.mantissa[0] >>= 1;

	pack_real_4(p, value.mantissa, &exp, &sign);
	break;

    case 8:
	exp = (value.exp == 0 && ((value.mantissa[0] & (2*MAN8_MSW-1)) == 0) &&
	       value.mantissa[1] == 0)
	    ? 0
	    : value.exp + EXP8_BIAS + MAN8_LEN;

	if (exp == 0) {
	    value.mantissa[1] =
		(value.mantissa[1] >> 1) | ((value.mantissa[0] & 1) << 31);
	    value.mantissa[0] = value.mantissa[0] >> 1;
	}

	pack_real_8(p, value.mantissa, &exp, &sign);
	break;
  
    case 10:
	exp = (value.exp == 0 && value.mantissa[0] == 0 &&
	       value.mantissa[1] == 0)
	    ? 0
	    : value.exp + EXP10_BIAS + MAN10_LEN - 1;

	if (exp == 0) {
	    value.mantissa[1] =
		(value.mantissa[1] >> 1) | ((value.mantissa[0] & 1) << 31);
	    value.mantissa[0] = value.mantissa[0] >> 1;
	}

	pack_real_10(p, value.mantissa, &exp, &sign);
	break;

    case 16:
	exp = (value.exp == 0 &&
	       ((value.mantissa[0] & (2*MAN16_MSW-1)) == 0) &&
	       value.mantissa[1] == 0 && value.mantissa[2] == 0 &&
	       value.mantissa[3] == 0)
	    ? 0
	    : value.exp + EXP16_BIAS + MAN16_LEN;

	if (exp == 0) {
	    value.mantissa[3] =
		(value.mantissa[3] >> 1) | (value.mantissa[2] << 31);

	    value.mantissa[2] =
		(value.mantissa[2] >> 1) | (value.mantissa[1] << 31);

	    value.mantissa[1] =
		(value.mantissa[1] >> 1) | (value.mantissa[0] << 31);

	    value.mantissa[0] = value.mantissa[0] >> 1;
	}

	pack_real_16(p, value.mantissa, &exp, &sign);
	break;
    }
}
