
/* Copyright (C) 2003-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

/* Implement the eoshift() intrinsic. */

#include <string.h>
#include "runtime.h"


/* Case:          Boundary                Shift
 * ------------------------------------------------
 * eoshift1_??    Scalar or not present   Scalar
 * eoshift2_??    Scalar or not present   Array
 * eoshift3       Array                   Scalar
 * eoshift4       Array                   Array
 */


static g95_array_descriptor *eoshift1(g95_array_descriptor *array,
				      G95_DINT shift, char *boundary,
				      G95_DINT *dim_p) {
G95_AINT s, d_extent, element_size, extent, x[G95_MAX_DIMENSIONS];
int i, n, rank, zero_size, dim;
g95_array_descriptor *r;
char *p1, *p2, *q1, *q2;

    dim = (dim_p == NULL) ? 0 : *dim_p-1;    /* Zero based */

    rank = array->rank;
    element_size = array->element_size;

    section_info[0] = array->rank;
    section_info[1] = element_size;

    n = 2;
    zero_size = 0;
    d_extent = 0;

    for(i=0; i<rank; i++) {
	extent = array->info[i].ubound - array->info[i].lbound + 1;
	if (extent <= 0) {
	    extent = 0;
	    zero_size = 1;
	}

	if (i == dim)
	    d_extent = extent;

	section_info[n++] = 1;
	section_info[n++] = extent;

	x[i] = 1;
    }

    r = array_from_section(NULL);
    if (zero_size)
	goto done;

    for(;;) {
	p1 = r->offset;
	q1 = array->offset;

	for(i=0; i<rank; i++)
	    if (i != dim) {
		p1 += r->info[i].mult * x[i];
		q1 += array->info[i].mult*(array->info[i].lbound + x[i] - 1);
	    }

	/* End-off shift along DIM */

	for(i=0; i<d_extent; i++) {
	    p2 = p1 + r->info[dim].mult*(i + 1);

	    s = i + shift;
	    q2 = (s < 0 || s >= d_extent)
		? boundary
		: q1 + array->info[dim].mult * (s + array->info[dim].lbound);

	    memcpy(p2, q2, element_size);
	}

	if (bump_element_dim(r, x, dim))
	    break;
    }

done:
    return r;
}


static g95_array_descriptor *eoshift2(g95_array_descriptor *array,
				      g95_array_descriptor *shift,
				      char *boundary, G95_DINT *dim_p) {
G95_AINT s, shft, dim, extent, d_extent, element_size, x[G95_MAX_DIMENSIONS];
char *p1, *p2, *q1, *q2, *m;
int i, n, rank, zero_size;
g95_array_descriptor *r;

    dim = (dim_p == NULL) ? 0 : *dim_p - 1;   /* Zero based */

    rank = array->rank;
    element_size = array->element_size;

    section_info[0] = array->rank;
    section_info[1] = element_size;

    n = 2;
    zero_size = 0;
    d_extent = shft = 0;

    for(i=0; i<rank; i++) {
	extent = array->info[i].ubound - array->info[i].lbound + 1;
	if (extent <= 0) {
	    extent = 0;
	    zero_size = 1;
	}

	if (i == dim)
	    d_extent = extent;

	section_info[n++] = 1;
	section_info[n++] = extent;

	x[i] = 1;
    }

    r = array_from_section(NULL);
    if (zero_size)
	goto done;

    for(;;) {
	p1 = r->offset;
	q1 = array->offset;
	m = shift->offset;

	for(i=0; i<rank; i++) {
	    if (i != dim) {
		p1 += r->info[i].mult * x[i];
		q1 += array->info[i].mult*(array->info[i].lbound + x[i] - 1);
	    }

	    if (i < dim)
		m += shift->info[i].mult*(shift->info[i].lbound + x[i] - 1);

	    if (i > dim)
		m += shift->info[i-1].mult*(shift->info[i-1].lbound + x[i]-1);
	}

	shft = extract_dint(m, shift->element_size);

	/* Circular shift along DIM */

	for(i=0; i<d_extent; i++) {
	    p2 = p1 + r->info[dim].mult*(i + 1);

	    s = i + shft;
	    q2 = (s < 0 || s >= d_extent)
		? boundary
		: q1 + array->info[dim].mult * (s + array->info[dim].lbound);

	    memcpy(p2, q2, element_size);
	}

	if (bump_element_dim(r, x, dim))
	    break;
    }

done:
    return r;
}


#define eoshift3 prefix(eoshift3)

g95_array_descriptor *eoshift3(g95_array_descriptor *array, G95_DINT *shift_p,
			       g95_array_descriptor *boundary,
			       G95_DINT *dim_p) {

G95_AINT s, d_extent, element_size, dim, shift, extent, x[G95_MAX_DIMENSIONS];
char *p1, *p2, *q1, *q2, *b;
int i, n, rank, zero_size;
g95_array_descriptor *r;

    dim = (dim_p == NULL) ? 0 : *dim_p - 1; /* Zero based */
    shift = *shift_p;

    rank = array->rank;
    element_size = array->element_size;

    section_info[0] = array->rank;
    section_info[1] = element_size;

    n = 2;
    zero_size = 0;
    d_extent = 0;

    for(i=0; i<rank; i++) {
	extent = array->info[i].ubound - array->info[i].lbound + 1;
	if (extent <= 0) {
	    extent = 0;
	    zero_size = 1;
	}

	if (i == dim)
	    d_extent = extent;

	section_info[n++] = 1;
	section_info[n++] = extent;

	x[i] = 1;
    }

    r = array_from_section(NULL);
    if (zero_size)
	goto done;

    for(;;) {
	p1 = r->offset;
	q1 = array->offset;
	b = boundary->offset;

	for(i=0; i<rank; i++)
	    if (i != dim) {
		p1 += r->info[i].mult * x[i];
		q1 += array->info[i].mult*(array->info[i].lbound + x[i] - 1);

		if (i < dim)
		    b += boundary->info[i].mult *
			(boundary->info[i].lbound+x[i]-1);

		if (i > dim)
		    b += boundary->info[i-1].mult *
			(boundary->info[i-1].lbound+x[i]-1);
	    }

	/* End-off shift along DIM */

	for(i=0; i<d_extent; i++) {
	    p2 = p1 + r->info[dim].mult*(i + 1);

	    s = i + shift;
	    q2 = (s < 0 || s >= d_extent)
		? b
		: q1 + array->info[dim].mult * (s + array->info[dim].lbound);

	    memcpy(p2, q2, element_size);
	}

	if (bump_element_dim(r, x, dim))
	    break;
    }

done:
    return r;
}


#define eoshift4 prefix(eoshift4)

g95_array_descriptor *eoshift4(g95_array_descriptor *array,
			       g95_array_descriptor *shift,
			       g95_array_descriptor *boundary,
			       G95_DINT *dim_p) {

G95_AINT d_extent, element_size, s, dim, shft, extent, x[G95_MAX_DIMENSIONS];
char *p1, *p2, *q1, *q2, *m, *b;
int i, n, rank, zero_size;
g95_array_descriptor *r;

    dim = (dim_p == NULL) ? 0 : *dim_p - 1;   /* Zero based */

    rank = array->rank;
    element_size = array->element_size;

    section_info[0] = array->rank;
    section_info[1] = element_size;

    n = 2;
    zero_size = 0;
    d_extent = shft = 0;

    for(i=0; i<rank; i++) {
	extent = array->info[i].ubound - array->info[i].lbound + 1;
	if (extent <= 0) {
	    extent = 0;
	    zero_size = 1;
	}

	if (i == dim)
	    d_extent = extent;

	section_info[n++] = 1;
	section_info[n++] = extent;

	x[i] = 1;
    }

    r = array_from_section(NULL);
    if (zero_size)
	goto done;

    for(;;) {
	p1 = r->offset;
	q1 = array->offset;
	m = shift->offset;
	b = boundary->offset;

	for(i=0; i<rank; i++) {
	    if (i != dim) {
		p1 += r->info[i].mult * x[i];
		q1 += array->info[i].mult*(array->info[i].lbound + x[i] - 1);
	    }

	    if (i < dim) {
		m += shift->info[i].mult*(shift->info[i].lbound + x[i] - 1);
		b += boundary->info[i].mult*(boundary->info[i].lbound+x[i]-1);
	    }

	    if (i > dim) {
		m += shift->info[i-1].mult *
		    (shift->info[i-1].lbound + x[i] - 1);

		b += boundary->info[i-1].mult *
		    (boundary->info[i-1].lbound+x[i]-1);
	    }
	}

	shft = extract_dint(m, shift->element_size);

	/* Circular shift along DIM */

	for(i=0; i<d_extent; i++) {
	    p2 = p1 + r->info[dim].mult*(i + 1);

	    s = i + shft;
	    q2 = (s < 0 || s >= d_extent)
		? b
		: q1 + array->info[dim].mult * (s + array->info[dim].lbound);

	    memcpy(p2, q2, element_size);
	}

	if (bump_element_dim(r, x, dim))
	    break;
    }

done:
    return r;
}


#define eoshift1_i1 prefix(eoshift1_i1)

g95_array_descriptor *eoshift1_i1(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT1 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim);
}


#define eoshift2_i1 prefix(eoshift2_i1)

g95_array_descriptor *eoshift2_i1(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT1 b;

    if (boundary == NULL) {
	b = 0;
	boundary = &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_i2 prefix(eoshift1_i2)

g95_array_descriptor *eoshift1_i2(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT2 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim);
}


#define eoshift2_i2 prefix(eoshift2_i2)

g95_array_descriptor *eoshift2_i2(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT2 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_i4 prefix(eoshift1_i4)

g95_array_descriptor *eoshift1_i4(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT4 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim);
}


#define eoshift2_i4 prefix(eoshift2_i4)

g95_array_descriptor *eoshift2_i4(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT4 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#ifdef G95_INT8
#define eoshift1_i8 prefix(eoshift1_i8)

g95_array_descriptor *eoshift1_i8(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT8 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim);
}


#define eoshift2_i8 prefix(eoshift2_i8)

g95_array_descriptor *eoshift2_i8(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT8 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}

#endif


#ifdef G95_INT16
#define eoshift1_i16 prefix(eoshift1_i16)

g95_array_descriptor *eoshift1_i16(g95_array_descriptor *array, int *shift,
				   char *boundary, G95_DINT *dim) {
G95_INT16 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim);
}


#define eoshift2_i16 prefix(eoshift2_i16)

g95_array_descriptor *eoshift2_i16(g95_array_descriptor *array,
				   g95_array_descriptor *shift,
				   char *boundary, G95_DINT *dim) {
G95_INT16 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}

#endif


#define eoshift1_r4 prefix(eoshift1_r4)

g95_array_descriptor *eoshift1_r4(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
float b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_r4 prefix(eoshift2_r4)

g95_array_descriptor *eoshift2_r4(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
float b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_r8 prefix(eoshift1_r8)

g95_array_descriptor *eoshift1_r8(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
double b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_r8 prefix(eoshift2_r8)

g95_array_descriptor *eoshift2_r8(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
double b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_l1 prefix(eoshift1_l1)

g95_array_descriptor *eoshift1_l1(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT1 b;

    if (boundary == NULL) {
	b = 0;
	boundary = &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_l1 prefix(eoshift2_l1)

g95_array_descriptor *eoshift2_l1(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT1 b;

    if (boundary == NULL) {
	b = 0;
	boundary = &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_l2 prefix(eoshift1_l2)

g95_array_descriptor *eoshift1_l2(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT2 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_l2 prefix(eoshift2_l2)

g95_array_descriptor *eoshift2_l2(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT2 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_l4 prefix(eoshift1_l4)

g95_array_descriptor *eoshift1_l4(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT4 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_l4 prefix(eoshift2_l4)

g95_array_descriptor *eoshift2_l4(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT4 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#ifdef G95_INT8
#define eoshift1_l8 prefix(eoshift1_l8)

g95_array_descriptor *eoshift1_l8(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT8 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_l8 prefix(eoshift2_l8)

g95_array_descriptor *eoshift2_l8(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
G95_INT8 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}

#endif


#ifdef G95_INT16
#define eoshift1_l16 prefix(eoshift1_l16)

g95_array_descriptor *eoshift1_l16(g95_array_descriptor *array, int *shift,
				   char *boundary, G95_DINT *dim) {
G95_INT16 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_l16 prefix(eoshift2_l16)

g95_array_descriptor *eoshift2_l16(g95_array_descriptor *array,
				   g95_array_descriptor *shift,
				   char *boundary, G95_DINT *dim) {
G95_INT16 b;

    if (boundary == NULL) {
	b = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}

#endif


#define eoshift1_z4 prefix(eoshift1_z4)

g95_array_descriptor *eoshift1_z4(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
z4 b;

    if (boundary == NULL) {
	b.r = 0;
	b.c = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_z4 prefix(eoshift2_z4)

g95_array_descriptor *eoshift2_z4(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
z4 b;

    if (boundary == NULL) {
	b.r = 0;
	b.c = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_z8 prefix(eoshift1_z8)

g95_array_descriptor *eoshift1_z8(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
z8 b;

    if (boundary == NULL) {
	b.r = 0;
	b.c = 0;
	boundary = (char *) &b;
    }

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_z8 prefix(eoshift2_z8)

g95_array_descriptor *eoshift2_z8(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
z8 b;

    if (boundary == NULL) {
	b.r = 0;
	b.c = 0;
	boundary = (char *) &b;
    }

    return eoshift2(array, shift, boundary, dim);
}



#define eoshift1_d0 prefix(eoshift1_d0)

g95_array_descriptor *eoshift1_d0(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {

    return eoshift1(array, *shift, boundary, dim); 
}


#define eoshift2_d0 prefix(eoshift2_d0)

g95_array_descriptor *eoshift2_d0(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {

    return eoshift2(array, shift, boundary, dim);
}


#define eoshift1_c1 prefix(eoshift1_c1)

g95_array_descriptor *eoshift1_c1(g95_array_descriptor *array, int *shift,
				  char *boundary, G95_DINT *dim) {
g95_array_descriptor *r;
char *p;

    p = NULL;
    if (boundary == NULL) {
	p = get_mem(array->element_size);
	memset(p, ' ', array->element_size);
	boundary = p;
    }

    r = eoshift1(array, *shift, boundary, dim); 
    if (p != NULL)
	free_mem(p);

    return r;
}


#define eoshift2_c1 prefix(eoshift2_c1)

g95_array_descriptor *eoshift2_c1(g95_array_descriptor *array,
				  g95_array_descriptor *shift,
				  char *boundary, G95_DINT *dim) {
g95_array_descriptor *r;
char *p;

    p = NULL;
    if (boundary == NULL) {
	p = get_mem(array->element_size);
	memset(p, ' ', array->element_size);
	boundary = p;
    }

    r = eoshift2(array, shift, boundary, dim);
    if (p != NULL)
	free_mem(p);

    return r;
}
