// 
// Copyright (c) 2004 Dean Gaudet <dean@arctic.org>
// 
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.
// 

// $Id: local_mac.c,v 1.2 2004-11-08 04:22:52 dean Exp $

#define _BSD_SOURCE 1
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <pcap.h>
#if defined(__linux__) || defined(__FreeBSD__)
#include <net/ethernet.h>
#endif
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#if defined(__NetBSD__)
#include <net/if.h>
#include <net/if_ether.h>
#endif
#include <arpa/inet.h>
#ifdef AF_PACKET		// linux
#include <netpacket/packet.h>
#endif
#ifdef AF_LINK			// bsd
#include <net/if_dl.h>
#endif

#include "hash.h"
#include "local_mac.h"

static hash_t *local_macs;

void local_mac_register(void)
{
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_if_t *devlist;
	pcap_addr_t *addr;

	local_macs = hash_create(4);

	if (pcap_findalldevs(&devlist, errbuf) == -1) {
		fprintf(stderr, "pcap_findalldevs: %s\n", errbuf);
		exit(1);
	}
	while (devlist) {
		addr = devlist->addresses;
		while (addr) {
			struct sockaddr *sa = addr->addr;

			switch (sa->sa_family) {
#ifdef AF_PACKET
			case AF_PACKET:
				if (((struct sockaddr_ll *)sa)->sll_halen == 6) {
					hash_insert(local_macs, ((struct sockaddr_ll *)sa)->sll_addr, 6, &local_macs);
				}
				break;
#endif
#ifdef AF_LINK
			case AF_LINK:
				if (((struct sockaddr_dl *)sa)->sdl_alen == 6) {
					hash_insert(local_macs, LLADDR((struct sockaddr_dl *)sa), 6, &local_macs);
				}
				break;
#endif
#if !defined(AF_PACKET) && !defined(AF_LINK)
#error "no support for finding local MAC addresses"
#endif
			}
			addr = addr->next;
		}
		devlist = devlist->next;
	}
	pcap_freealldevs(devlist);
}


int is_local_mac(const uint8_t *mac)
{
	return hash_lookup(local_macs, mac, 6) != NULL;
}
