
#import "XMLParser.h"
#import "config.h"

@implementation XMLParser

/* Wrapper functions that are used by Expat.  These are the "real"
   Expat handlers, and they receive the id of the object instance that
   is performing the parsing as their user data argument so they know
   where to send messages.  */
static void
start_elem_handler(void *ud, const XML_Char *name, const XML_Char **atts)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj startElement:name :atts];
}

static void
end_elem_handler(void *ud, const XML_Char *name)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj endElement:name];
}

static void
char_data_handler(void *ud, const XML_Char *s, int len)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj characters:s :len];
}

static void
proc_instr_handler(void *ud, const XML_Char *target, const XML_Char *data)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj processingInstruction:target :data];
}

static void
comment_handler(void *ud, const XML_Char *data)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj comment:data];
}

static void
start_cdata_section_handler(void *ud)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj startCdata];
}

static void
end_cdata_section_handler(void *ud)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj endCdata];
}

static int
external_entity_handler(XML_Parser p,
			const XML_Char *context, 
			const XML_Char *base,
			const XML_Char *systemId,
			const XML_Char *publicId)
{
  id parserobj;

  parserobj = (id)XML_GetUserData(p);
  return([parserobj externalEntity:context :base :publicId :systemId]);
}

static void
unparsed_entity_decl_handler(void *ud, const XML_Char *name,
			     const XML_Char *base,
			     const XML_Char *publicId,
			     const XML_Char *systemId,
			     const XML_Char *notationName)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj unparsedEntityDecl:name :base :publicId :systemId
	     :notationName];
}

static void
start_prefix_mapping_handler(void *ud, const XML_Char *prefix,
			     const XML_Char *uri)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj startPrefixMapping:uri :prefix];
}

static void
end_prefix_mapping_handler(void *ud, const XML_Char *prefix)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj endPrefixMapping:prefix];
}

static void
default_xml_handler(void *ud, const XML_Char *s, int len)
{
  id parserobj;

  parserobj = (id)ud;
  [parserobj defaultHandler:s :len];
}

/* These are the default handler methods for all elements.  They all
   do absolutely NOTHING.  To get the parser to do something you have
   to make a subclass that overrides these methods.  */
- startElement:(const XML_Char *)qname:(const XML_Char **)atts
{
}

- endElement:(const XML_Char *)qname
{
}

- characters:(const XML_Char *)s:(int)len
{
}

- processingInstruction:(const XML_Char *)target:(const XML_Char *)data
{
}

- comment:(const XML_Char *)data
{
}

- startCdata
{
}

- endCdata
{
}

- (int)externalEntity:(const XML_Char *)context:(const XML_Char *)base
:(const XML_Char *)publicID:(const XML_Char *)systemID
{
}

- unparsedEntityDecl:(const XML_Char *)name:(const XML_Char *)base
:(const XML_Char *)publicID:(const XML_Char *)systemID
:(const XML_Char *)notationName
{
}

- startPrefixMapping:(const XML_Char *)uri:(const XML_Char *)localName
{
}

- endPrefixMapping:(const XML_Char *)prefix
{
}

- defaultHandler:(const XML_Char *)s:(int)len
{
}

- (int)dataRead:(char *)buff:(int)buflen
{
  return 0;
}

- errorHandler:(int)errcode
{
  printf("errorcode = %d\n", errcode);
  printf("Error string = %s\n", XML_ErrorString(errcode));
}

static void
set_handlers(XML_Parser parser, unsigned int parsemask)
{
#ifdef HAVE_XML_SETSTARTELEMENTHANDLER
  if (parsemask & (XML_PARSE_STARTELEM))
    XML_SetStartElementHandler(parser, start_elem_handler);
#endif
#ifdef HAVE_XML_SETENDELEMENTHANDLER
  if (parsemask & (XML_PARSE_ENDELEM))
    XML_SetEndElementHandler(parser, end_elem_handler);
#endif
#ifdef HAVE_XML_SETCHARACTERDATAHANDLER
  if (parsemask & (XML_PARSE_CHARACTERS))
    XML_SetCharacterDataHandler(parser, char_data_handler);
#endif
#ifdef HAVE_XML_SETPROCESSINGINSTRUCTIONHANDLER
  if (parsemask & (XML_PARSE_PROCINSTR))
    XML_SetProcessingInstructionHandler(parser, proc_instr_handler);
#endif
#ifdef HAVE_XML_SETCOMMENTHANDLER
  if (parsemask & (XML_PARSE_COMMENT))
    XML_SetCommentHandler(parser, comment_handler);
#endif
#ifdef HAVE_XML_SETCDATASECTIONHANDLER
  if (parsemask & (XML_PARSE_CDATA))
    XML_SetCdataSectionHandler(parser, start_cdata_section_handler,
			       end_cdata_section_handler);
#endif
#ifdef HAVE_XML_SETEXTERNALENTITYREFHANDLER
  if (parsemask & (XML_PARSE_EXTERNAL_ENTITY))
    XML_SetExternalEntityRefHandler(parser, external_entity_handler);
#endif
#ifdef HAVE_XML_SETUNPARSEDENTITYDECLHANDLER
  if (parsemask & (XML_PARSE_UNPARSED_ENTITY))
    XML_SetUnparsedEntityDeclHandler(parser, unparsed_entity_decl_handler);
#endif
#ifdef HAVE_XML_SET_NAMESPACEDECLHANDLER
  if (parsemask & (XML_PARSE_NAMESPACE_DECL))
    XML_SetNamespaceDeclHandler(parser, start_prefix_mapping_handler,
				end_prefix_mapping_handler);
#endif
#ifdef HAVE_XML_SETDEFAULTHANDLER
  if (parsemask & (XML_PARSE_DEFAULT))
    XML_SetDefaultHandler(parser, default_xml_handler);
#endif

}

- init:(const XML_Char *)encoding:(XML_Char)sep:(int)bs:(unsigned int)parsemask
{
  id s;

  [super init];
  bufsize = (bs <= 0) ? 8192 : bs; // defaults to 8192 if less than 0
  parser = (parsemask & XML_PARSE_DISABLE_NS) ?
    XML_ParserCreate(encoding) : XML_ParserCreateNS(encoding, sep);
  XML_SetUserData(parser, self);
  set_handlers(parser, parsemask);
  return self;
}

- start
{
  for (;;) {
    int bytes_read, done;
    void *buff = XML_GetBuffer(parser, bufsize);

    if (buff == NULL)
      [self errorHandler:XML_GetErrorCode(parser)];

    bytes_read = [self dataRead:buff :bufsize];
    done = (bytes_read < bufsize);

    if (!XML_ParseBuffer(parser, bytes_read, done))
      [self errorHandler:XML_GetErrorCode(parser)];
    if (done)
      break;
  }
}

- free
{
  XML_ParserFree(parser);
  [super free];
}

@end
