/* eLectrix - a pdf viewer
 * Copyright (C) 2011 Martin Linder <mali2297@users.sf.net>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/gpl-2.0.html>.
 */
#include "e6x-common.h"
#include "e6x-enc-document.h"

#define PAGE_WIDTH 500
#define PAGE_HEIGHT 500

/* Standard GObject methods */
G_DEFINE_TYPE (E6xEncDocument, e6x_enc_document, E6X_TYPE_DOCUMENT)
static void e6x_enc_document_finalize (GObject *object);
static void e6x_enc_document_dispose (GObject *object);

/* Implementations of E6xDocument virtual methods */
static gboolean e6x_enc_document_reload (E6xDocument *doc,
                                         GError **error);
static void e6x_enc_document_get_page_size (E6xDocument *doc,
                                            gdouble *width,
                                            gdouble *height);
static cairo_surface_t *e6x_enc_document_render_page (E6xDocument *doc);


GObject *
e6x_enc_document_new (const gchar *filename)
{
  GObject *object = g_object_new (E6X_TYPE_ENC_DOCUMENT, NULL);
  E6xDocument *doc = E6X_DOCUMENT (object);
  
  doc->filename = g_strdup (filename);
  doc->n_pages = 1;
  doc->page_no = 1;
  doc->title = g_path_get_basename (doc->filename);

  return object;
}


static void 
e6x_enc_document_class_init (E6xEncDocumentClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  E6xDocumentClass *doc_class = E6X_DOCUMENT_CLASS (klass);

  object_class->dispose = e6x_enc_document_dispose;
  object_class->finalize = e6x_enc_document_finalize;
  
  doc_class->reload = &e6x_enc_document_reload;
  doc_class->get_page_size = &e6x_enc_document_get_page_size;
  doc_class->render_page = &e6x_enc_document_render_page;
}


static void 
e6x_enc_document_init (E6xEncDocument *doc)
{
}

static void 
e6x_enc_document_finalize (GObject *object)
{
  G_OBJECT_CLASS (e6x_enc_document_parent_class)->finalize (object);
}


static void 
e6x_enc_document_dispose (GObject *object)
{
  G_OBJECT_CLASS (e6x_enc_document_parent_class)->dispose (object);
}


static gboolean 
e6x_enc_document_reload (E6xDocument *doc,
                         GError **error)
{
  return TRUE;
}


static void 
e6x_enc_document_get_page_size (E6xDocument *doc,
                                gdouble *width,
                                gdouble *height)
{
  *width = PAGE_WIDTH;
  *height = PAGE_HEIGHT;
}


const gchar* message = N_("ENCRYPTED");

static cairo_surface_t *
e6x_enc_document_render_page (E6xDocument *doc)
{
  cairo_surface_t *surface = NULL;
  gint surface_width = 0, surface_height = 0;
  cairo_text_extents_t extents;
  gdouble x = 0, y = 0;
  cairo_t *cr = NULL;

  if (doc->angle == 90 || doc->angle == 270)
  {
    surface_width = PAGE_HEIGHT * doc->scale;
    surface_height = PAGE_WIDTH * doc->scale;
  } 
  else
  {
    surface_width = PAGE_WIDTH * doc->scale;
    surface_height = PAGE_HEIGHT * doc->scale;
  }
  
  surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
                                        surface_width, surface_height);
  
  cr = cairo_create (surface);
  cairo_set_source_rgb (cr, 1.0, 1.0, 1.0);
  cairo_paint (cr);
  
  if (doc->angle != 0)
  {
    switch (doc->angle)
    {
    case 90:
      cairo_translate (cr, surface_width, 0);
      break;
    case 180:
      cairo_translate (cr, surface_width, surface_height);
      break;
    case 270:
      cairo_translate (cr, 0, surface_height);
      break;
    default:
      g_warn_if_reached ();
      break;
    }
    cairo_rotate (cr, (doc->angle / 90) * G_PI_2);
  }
  if (doc->scale != 1.0)
  {
    cairo_scale (cr, doc->scale, doc->scale);
  }
  
  cairo_save (cr);
  cairo_set_source_rgb (cr, 0.7, 0.0, 0.0);
  cairo_move_to (cr, 0.0, 0.0);
  cairo_line_to (cr, PAGE_WIDTH, PAGE_HEIGHT);
  cairo_move_to (cr, 0.0, PAGE_HEIGHT);
  cairo_line_to (cr, PAGE_WIDTH, 0.0);
  cairo_set_line_width (cr, 20.0);
  cairo_stroke (cr);
  cairo_restore (cr);
  
  cairo_select_font_face (cr, "Sans",
                          CAIRO_FONT_SLANT_NORMAL,
                          CAIRO_FONT_WEIGHT_NORMAL);
  cairo_set_font_size (cr, 52.0);
  cairo_text_extents (cr, _(message), &extents);
  
  x = PAGE_WIDTH/2 - extents.width/2;
  y = PAGE_HEIGHT/2 - extents.height/2;
  cairo_rectangle (cr, x, y, extents.width, extents.height);
  cairo_fill (cr);

  x = PAGE_WIDTH/2 - (extents.width/2 + extents.x_bearing);
  y = PAGE_HEIGHT/2 - (extents.height/2 + extents.y_bearing);
  cairo_set_source_rgb (cr, 0.7, 0.0, 0.0);
  cairo_move_to (cr, x, y);
  cairo_show_text (cr, _(message));
  
  cairo_destroy (cr);

  return surface;
}

