/* eLectrix - a pdf viewer
 * Copyright (C) 2011 Martin Linder <mali2297@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/gpl-2.0.html>.
 */
#include <glib/gstdio.h>
#include "e6x-common.h"
#include "e6x-util.h"
#include "e6x-dialogs.h"
#include "e6x-toc-view.h"

#define E6X_TOC_VIEW_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), E6X_TYPE_TOC_VIEW, E6xTocViewPrivate))

struct _E6xTocViewPrivate
{
  E6xDocument *doc;
};

G_DEFINE_TYPE (E6xTocView, e6x_toc_view, GTK_TYPE_TREE_VIEW);
static void e6x_toc_view_class_init (E6xTocViewClass *klass);
static void e6x_toc_view_init (E6xTocView *view);
static void e6x_toc_view_finalize (GObject *object);
static void e6x_toc_view_dispose (GObject *object);

static void e6x_toc_view_row_activated (GtkTreeView *view);
static void e6x_toc_view_doc_changed (E6xDocument *doc,
                                      E6xTocView *view);

static void
e6x_toc_view_class_init (E6xTocViewClass *klass)
{
  GObjectClass *object_class;

  g_type_class_add_private (klass, sizeof (E6xTocViewPrivate));

  object_class = G_OBJECT_CLASS (klass);
  object_class->dispose = e6x_toc_view_dispose;
  object_class->finalize = e6x_toc_view_finalize;
}

static void
e6x_toc_view_init (E6xTocView *view)
{
  E6xTocViewPrivate *priv = E6X_TOC_VIEW_GET_PRIVATE (view);
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column = NULL;
  GtkTreeSelection *selection = NULL;

  g_object_set (G_OBJECT (view),
                "enable-search", TRUE,
                "enable-tree-lines", TRUE,
                "headers-visible", FALSE,
                "search-column", 0,
                NULL);

  renderer = gtk_cell_renderer_text_new ();
  g_object_set (G_OBJECT (renderer),
                "ellipsize", PANGO_ELLIPSIZE_END,
                "ellipsize-set", TRUE,
                NULL);
  column = gtk_tree_view_column_new_with_attributes (_("Contents"),
                                                     renderer,
                                                     "text", 
                                                     E6X_TOC_COL_TITLE,
                                                     NULL);
  gtk_tree_view_column_set_expand (column, TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_renderer_set_alignment (renderer, 1.0, 0.5);
  column = gtk_tree_view_column_new_with_attributes (_("Page"),
                                                     renderer,
                                                     "text",
                                                     E6X_TOC_COL_PAGENO,
                                                     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

  g_signal_connect (GTK_TREE_VIEW (view), "row-activated",
                    G_CALLBACK (e6x_toc_view_row_activated), NULL);

  priv->doc = NULL;
  view->priv = priv;
}

static void
e6x_toc_view_finalize (GObject *object)
{
  G_OBJECT_CLASS (e6x_toc_view_parent_class)->finalize (object);
}

static void
e6x_toc_view_dispose (GObject *object)
{
  e6x_toc_view_set_document (E6X_TOC_VIEW (object), NULL);

  G_OBJECT_CLASS (e6x_toc_view_parent_class)->dispose (object);
}

GtkWidget *
e6x_toc_view_new (E6xDocument *doc)
{
  E6xTocView *view = NULL;

  view = g_object_new (E6X_TYPE_TOC_VIEW, NULL);
  e6x_toc_view_set_document (view, doc);

  return GTK_WIDGET (view);
}

void
e6x_toc_view_set_document (E6xTocView *view, E6xDocument *doc)
{
  E6xTocViewPrivate *priv = view->priv;

  if (priv->doc != NULL)
  {
    g_signal_handlers_disconnect_by_func (priv->doc,
                                          e6x_toc_view_doc_changed,
                                          view);
    gtk_tree_view_set_model (GTK_TREE_VIEW (view), NULL);
    g_object_unref (priv->doc);
  }

  priv->doc = doc;

  if (priv->doc != NULL)
  {
    g_object_ref (priv->doc);
    gtk_tree_view_set_model (GTK_TREE_VIEW (view), priv->doc->toc);
    g_signal_connect (priv->doc, "changed::reload",
                      G_CALLBACK (e6x_toc_view_doc_changed),
                      view);
  }
}


static void
e6x_toc_view_row_activated (GtkTreeView *view)
{
  E6xTocViewPrivate *priv = E6X_TOC_VIEW (view)->priv;
  GtkTreeSelection *selection;
  GtkTreeIter iter;

  if (G_UNLIKELY (priv->doc == NULL) || G_UNLIKELY (priv->doc->toc == NULL))
    return;
  
  selection = gtk_tree_view_get_selection (view);
  if (gtk_tree_selection_get_selected (selection, NULL, &iter)) 
    e6x_document_go_to_bookmark (priv->doc, &iter);
}


static void
e6x_toc_view_doc_changed (E6xDocument *doc, E6xTocView *view)
{
  gtk_tree_view_set_model (GTK_TREE_VIEW (view), doc->toc);
}

