use comfy_table::*;

#[derive(Debug)]
pub struct TableWrapper {
    table: Table,
    pub(crate) max_width: Option<u16>,
}

impl Default for TableWrapper {
    fn default() -> Self {
        Self {
            table: Table::new(),
            max_width: None,
        }
    }
}

impl TableWrapper {
    pub fn load_preset(mut self, preset: &'static str) -> Self {
        self.table.load_preset(preset);
        self
    }

    pub fn set_header<T>(mut self, header: impl IntoIterator<Item = T>) -> Self
    where
        T: std::fmt::Display,
    {
        self.table.set_header(
            header
                .into_iter()
                .map(|x| x.to_string())
                .collect::<Vec<_>>(),
        );
        self
    }

    pub fn add_error(mut self, error: impl Into<String>) -> Self {
        self.table.add_row(vec![
            comfy_table::Cell::new(error.into()).fg(comfy_table::Color::Red),
        ]);
        self
    }

    pub fn add_row<T>(mut self, row: impl IntoIterator<Item = T>) -> Self
    where
        T: std::fmt::Display,
    {
        let cols = row.into_iter().collect::<Vec<_>>();
        let width_per_col = self.max_width.unwrap_or(10000) as usize / cols.len();
        self.table.add_row(
            cols.into_iter()
                .map(|x| x.to_string())
                .map(|x| {
                    x.replace("```", "\n```\n")
                        .lines()
                        .map(|l| textwrap::refill(l, width_per_col))
                        .collect::<Vec<_>>()
                        .join("\n")
                })
                .collect::<Vec<_>>(),
        );
        self
    }

    pub fn add_rows(self, rows: impl IntoIterator<Item = Vec<String>>) -> Self {
        rows.into_iter()
            .fold(self, move |table, row| table.add_row(row))
    }

    pub fn show(self) -> String {
        let Self { table, .. } = self;
        format!("{table}")
    }
}
