use forgejo_api::structs::Repository;
use miette::{Context, IntoDiagnostic};

use crate::actions::GlobalArgs;
use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;
use crate::types::output::OutputMode;

use clap::Parser;

/// Display short summary of the current repository
#[derive(Parser, Debug)]
pub struct RepoInfoArgs {}

impl RepoInfoArgs {
    pub async fn run(self, global_args: GlobalArgs) -> miette::Result<()> {
        let ctx = BergContext::new(self, global_args).await?;

        let OwnerRepo { owner, repo } = ctx.owner_repo()?;
        let repo_data = spin_until_ready(ctx.client.repo_get(owner.as_str(), repo.as_str()).send())
            .await
            .into_diagnostic()
            .context("Current repo not found on this instance.")?;

        match ctx.global_args.output_mode {
            OutputMode::Pretty => {
                present_repo_info(&ctx, repo_data);
            }
            OutputMode::Json => repo_data.print_json()?,
        }

        Ok(())
    }
}

fn present_repo_info(ctx: &BergContext<RepoInfoArgs>, repo_data: Repository) {
    let table = ctx
        .make_table()
        .set_header(vec!["Repository Info"])
        .add_row(vec![
            String::from("Repository Name"),
            option_display(&repo_data.name),
        ])
        .add_row(vec![
            String::from("Repository Owner"),
            option_display(
                &repo_data
                    .owner
                    .as_ref()
                    .and_then(|user| user.login.as_ref()),
            ),
        ])
        .add_row(vec![
            String::from("Visibility"),
            option_display(&repo_data.private.map(
                |private| {
                    if private { "Private" } else { "Public" }
                },
            )),
        ])
        .add_row(vec![
            String::from("Stars"),
            format!("{}★", option_display(&repo_data.stars_count)),
        ]);

    println!("{table}", table = table.show());
}
