use forgejo_api::structs::User;
use miette::IntoDiagnostic;

use crate::actions::GlobalArgs;
use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;
use crate::types::output::OutputMode;

use clap::Parser;

/// List available assignee candidates
#[derive(Parser, Debug)]
pub struct RepoAssigneesArgs {}

impl RepoAssigneesArgs {
    pub async fn run(self, global_args: GlobalArgs) -> miette::Result<()> {
        let ctx = BergContext::new(self, global_args).await?;

        let OwnerRepo { repo, owner } = ctx.owner_repo()?;
        let (_, repo_assignees) = spin_until_ready(
            ctx.client
                .repo_get_assignees(owner.as_str(), repo.as_str())
                .send(),
        )
        .await
        .into_diagnostic()?;

        match ctx.global_args.output_mode {
            OutputMode::Pretty => {
                present_repo_assignees(&ctx, repo_assignees);
            }
            OutputMode::Json => repo_assignees.print_json()?,
        }

        Ok(())
    }
}

fn present_repo_assignees(ctx: &BergContext<RepoAssigneesArgs>, repo_assignees: Vec<User>) {
    let table = ctx
        .make_table()
        .set_header(vec![format!(
            "Repository Assignees{}",
            if repo_assignees.is_empty() {
                " (empty)"
            } else {
                Default::default()
            }
        )])
        .add_rows(
            repo_assignees
                .into_iter()
                .map(|assignee| vec![option_display(&assignee.login)]),
        );

    println!("{table}", table = table.show());
}
