{ inputs, ... }:
let
  sharedLib = inputs.self.sharedLib;
in
{
  flake.testLib =
    {
      self',
      lib,
      pkgs,
    }:
    rec {
      scopes = sharedLib.allScopes |> builtins.map (scope: "write:${scope}") |> lib.concatStringsSep ",";

      berg_run =
        {
          cmd,
          cdTestRepo ? false,
          success ? true,
          timeout ? 3,
        }:
        let
          # python linter says fstrings without any parameters are an error so we have to correct the cmd
          fstringRequired = (builtins.match ".*\\\{.*" cmd) != null;
          prefix = lib.optionalString fstringRequired "f";
          fixedCmd = lib.escape [ ''"'' ] cmd;

          optCd = lib.optionalString cdTestRepo "cd ${sharedLib.repo}; ";
        in
        [
          "forgejo."
          (if success then "wait_until_succeeds" else "fail")
          "("
          prefix
          "\""
          optCd
          "berg --non-interactive"
          " ${fixedCmd}"
          "\""
          ",timeout=${builtins.toString timeout}"
          ")"
        ]
        |> builtins.concatStringsSep "";

      berg_run_json =
        {
          cmd,
          cdTestRepo ? false,
          success ? true,
          timeout ? 3,
        }:
        ''json.loads(${
          berg_run {
            inherit cdTestRepo success timeout;
            cmd = "--output-mode json ${cmd}";
          }
        })'';

      # forgejo CLI with all the necessary arguments
      forgejo_run = "su -- forgejo forgejo admin user --work-path /var/lib/forgejo";

      forgjeoTest =
        {
          name,
          test,
          withLabels ? false,
          withChangeBranches ? false,
          withRepoSetup ? false,
          withAuth ? false,
        }:
        pkgs.testers.runNixOSTest {
          name = "berg-forgejo-integration-test-${name}";
          nodes.${sharedLib.forgejo.hostname} = sharedLib.forgejo.baseConfig { inherit self' lib pkgs; };
          testScript =
            [
              # imports
              ''
                import json
                def printj(val):
                  print(json.dumps(val, indent=2))
              ''
              # forgejo setup
              ''

                forgejo.wait_for_open_port(${builtins.toString sharedLib.forgejo.port})
                forgejo.succeed("${sharedLib.forgejo.run} create --email ${sharedLib.email} --username ${sharedLib.user} --password pass")
                forgejo.succeed("${sharedLib.forgejo.run} create --email ${sharedLib.otherEmail} --username ${sharedLib.otherUser} --password pass")
                forgejo.succeed("echo http://${sharedLib.user}:${sharedLib.pass}@${sharedLib.forgejo.hostname}:${builtins.toString sharedLib.forgejo.port} >> ~/.git-credentials")

                # generate an access token
                token = forgejo.succeed("${sharedLib.forgejo.run} generate-access-token --username ${sharedLib.user} --scopes '${scopes}' --raw")
                print(token)
              ''
              # auth login test
              (lib.optionalString withAuth ''
                auth = ${berg_run { cmd = "auth login --token {token}"; }}
                print(auth)
              '')
              # create a repo
              (lib.optionalString (withAuth && withRepoSetup) ''
                ${berg_run {
                  cmd = ''repo create --name "${sharedLib.repo}" --default-branch "${sharedLib.main}"'';
                }}
                print("Successfully created repo ${sharedLib.repo}")

                ${berg_run { cmd = ''repo clone ${sharedLib.user}/${sharedLib.repo}''; }}
                print("Successfully cloned ${sharedLib.user}/${sharedLib.repo}")
              '')
              # create labels
              (lib.optionalString (withAuth && withRepoSetup && withLabels) ''
                for label_name in ["bug","feat","test"]:
                  ${berg_run {
                    cmd = ''label create --name "{label_name}"'';
                    cdTestRepo = true;
                  }}
              '')
              # create changes
              (lib.optionalString (withAuth && withRepoSetup && withChangeBranches) ''
                forgejo.succeed("${
                  [
                    "cd ${sharedLib.repo}"
                    "git checkout -b ${sharedLib.main}"
                    "echo 'Hello, World!' > hello.txt"
                    "git add -A"
                    "git commit -m 'Hello'"
                    "git push origin ${sharedLib.main}"
                  ]
                  |> builtins.concatStringsSep ";"
                }")
                print("Successfully pushed some change to ${sharedLib.main}")

                forgejo.succeed("sleep 1")

                forgejo.succeed("${
                  [
                    "cd ${sharedLib.repo}"
                    "git checkout -b ${sharedLib.branch}"
                    "echo 'Bye, World!' > bye.txt"
                    "git add -A"
                    "git commit -m 'Bye'"
                    "git push origin ${sharedLib.branch}"
                  ]
                  |> builtins.concatStringsSep ";"
                }")
                print("Successfully pushed some change to ${sharedLib.branch}")

                forgejo.succeed("sleep 1")
              '')
              # the actual test case
              test
              # Final message
              ''
                print("Success")
              ''
            ]
            |> builtins.concatStringsSep "\n"
          # |> lib.traceVal
          ;
        };
      embeddInFlake =
        testAttrs:
        testAttrs
        |> builtins.mapAttrs (testName: testConfig: testConfig // { name = testName; })
        |> builtins.mapAttrs (_: forgjeoTest);
    };
}
