#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <X11/xpm.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "../wmgeneral/wmgeneral.h"
#include "bbrun.xpm"

#define MAXCMDLEN 256
#define MAXPATHLEN 256
#define MAXHISTLEN 10
#define VERSION "1.3"
#define DEFAULT_HIST_FILENAME "/.bbrun_history"
#define __DEBUG__ 0

int wminet_mask_width = 16;
int wminet_mask_height = 16;
char wminet_mask_bits[16 * 16];
GtkWidget *combo;

char command[MAXCMDLEN + 3];		 // +3 for ' &\0'
int withdrawnMode = 0;                   // 0 if normal launch mode, 1 if -w is specified
int historyLength = 0;                   // The number of items in history, or -1 if history file could not be created.
char history[MAXHISTLEN][MAXCMDLEN + 1]; // The history items, +1 is to allow room for '\0'
char historyFilename[MAXPATHLEN + 1];    // The path to the history file, +1 is to allow room for '\0'

void parseArguments(int, char **);
void readHistory(void);
void updateHistory(char *);
void writeHistory(void);
void callback(GtkWidget *, gpointer);
void keypress(GtkWidget *, GdkEventKey *, gpointer);
gint delete_event(GtkWidget *, GdkEvent *, gpointer);
int runwin();
void usage(void);

/********************************
 *           Main               *
 ********************************/
int main(int argc, char *argv[]) 
{
  int xfd;
  int buttonStatus = -1;
  int i;
  fd_set rfds;
  XEvent Event;

  gtk_init(&argc, &argv);
  parseArguments(argc, argv);
  readHistory();
  
  if (withdrawnMode) 
  {
    runwin();
    if (command[0] != '\0') 
    {
      strncat(command, " &", 2);
      system(command);
    }
    exit(0);
  }

  createXBMfromXPM(wminet_mask_bits, bbrun, wminet_mask_width, wminet_mask_height);
  openXwindow(argc, argv, bbrun, wminet_mask_bits, wminet_mask_width, wminet_mask_height);
  AddMouseRegion(0, 0, 0, 16, 16);   // whole area
  xfd = ConnectionNumber(display);

  while (1) 
  {
    RedrawWindow();
    FD_ZERO(&rfds);
    FD_SET(xfd, &rfds);
    select(xfd + 1, &rfds, NULL, NULL, NULL);

    // X Events
    while (XPending(display)) 
    {
      XNextEvent(display, &Event);
      switch (Event.type) 
      {
        case Expose:
          RedrawWindow();
          break;
        case DestroyNotify:
          XCloseDisplay(display);
          exit(0);
          break;
        case ButtonPress:
          buttonStatus = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
          break;
        case ButtonRelease:
          i = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
          if ((buttonStatus == i) && (buttonStatus == 0)) 
	  {
            runwin();
            if (command[0] != '\0') 
	    {
              strncat(command, " &", 2);
              system(command);
            }
            break;
          }
          buttonStatus = -1;
          RedrawWindow();
          break;
      }
    }
  }
  
  return 0;
}

void parseArguments(int argc, char *argv[]) 
{
  int curOpt;
  int optionIndex = 0;
  static struct option validOptions[] = {
    {"help", no_argument, 0, 'h'},
    {"history-file", required_argument, 0, 'i'},
    {"version", no_argument, 0, 'v'},
    {"withdraw", no_argument, 0, 'w'},
    {0, 0, 0, 0}
  };

  bzero(historyFilename, MAXPATHLEN + 1);
  bzero(command, MAXCMDLEN + 2 + 1);
  bzero(history, MAXHISTLEN * (MAXCMDLEN + 1));

  while ((curOpt = getopt_long(argc, argv, "hi:vw", validOptions, &optionIndex)) > -1) 
  {
    switch (curOpt) 
    {
      case 'h':
        usage();
        exit(0);
      case 'i':
        strncpy(historyFilename, optarg, MAXPATHLEN);
        break;
      case 'v':
        printf("BBrun Version %s\n", VERSION);
        exit(0);
      case 'w':
        withdrawnMode = 1;
        break;
      case '?': // getopt_long has already printed an error message, so just exit out.  
        exit( -1);
      default:
	// getopt_long thought the option was fine, but we haven't coded a proper handler for it yet.
        fprintf(stderr, "Invalid parameter '%c'.\n", curOpt);
        exit( -1);
    }
  }

  if (historyFilename[0] == '\0') 
  {
    strncat(historyFilename, getenv("HOME"), MAXPATHLEN - strlen(DEFAULT_HIST_FILENAME) - 1);
    strncat(historyFilename, DEFAULT_HIST_FILENAME, strlen(DEFAULT_HIST_FILENAME));
  }

  if (__DEBUG__) 
  {
    fprintf(stderr, "parseArguments() is complete.\n");
    fprintf(stderr, "withdrawnMode is %d.\n", withdrawnMode);
    fprintf(stderr, "historyFilename is '%s'.\n", historyFilename);
  }
}

void readHistory(void) 
{
  FILE *fp;
  char buf[MAXCMDLEN];
  char *curElement;

  if ((fp = fopen(historyFilename, "r")) == 0)
  {
    if ((fp = fopen(historyFilename, "w")) == 0) 	// History file does not exist, create it.
    {
      fprintf(stderr, "Error creating history file '%s'.\n", historyFilename);
      historyLength = -1;
      return;
    }

    fclose(fp);
    historyLength = 0;

    if (__DEBUG__) 
      fprintf(stderr, "Just finished creating a new history file '%s'.\n", historyFilename);

    return ;
  }

  while (fgets(buf, MAXCMDLEN, fp) != 0) 
  {
    curElement = strtok(buf, "\n");
    if (curElement != NULL) 
      strncpy(history[historyLength++], curElement, MAXCMDLEN);
    else 
    {
      // scott@furt.com, This is a NULL line, which should NEVER happen.  Stop any further processing, 
      // because chances are very good that the rest of the file is corrupt too.
      if (__DEBUG__)
	fprintf(stderr, "Null line encountered in history file.  Stopping at %d history items.\n", historyLength);

      break;
    }

    if (historyLength >= MAXHISTLEN) 
    {
      if (__DEBUG__)
	fprintf(stderr, "History full before EOF encountered.  Stopping read.\n");

      break;
    }
  }

  fclose(fp);
  if (__DEBUG__)
  {
    int histIndex;

    fprintf(stderr, "Finished reading old history file. History items are: \n");
    for (histIndex = 0; histIndex < historyLength; histIndex++)
      fprintf(stderr, "\t[%02d] '%s'\n", histIndex, history[histIndex]);
  }
}

void updateHistory(char *newHistoryItem) 
{
  int duplicate = -1;
  int historyIndex;
  char transit[MAXCMDLEN + 1]; 		// + 1 to account for guaranteed '\0'

  bzero(transit, MAXCMDLEN + 1);
  if (__DEBUG__)
    fprintf(stderr, "Adding '%s' to history... ", newHistoryItem);

  if (historyLength == -1)
    return;

  // See if the command is in history already
  for (historyIndex = 0; historyIndex < historyLength; historyIndex++) 
  {
    if (strncmp(history[historyIndex], newHistoryItem, MAXCMDLEN) == 0) 
    {
      duplicate = historyIndex;
      break;
    }
  }

  if (duplicate != -1) 
  {
    // The command IS in the history already, so just move it to the end
    if (__DEBUG__)
      fprintf(stderr, " duplicate of item [%02d].\n", duplicate);

    if (duplicate != (historyLength - 1)) // If the duplicate entry is not at the end 
    {
      strncpy(transit, history[duplicate], MAXCMDLEN); // Save the duplicate

      // Shift each entry forward
      for (historyIndex = duplicate; historyIndex + 1 < historyLength; historyIndex++) 
        strncpy(history[historyIndex], history[historyIndex + 1], MAXCMDLEN);

      strncpy(history[historyLength - 1], transit, MAXCMDLEN); // put duplicate at the end
    }
  } 
  else 
  {
    // The command is NOT in the history already, so add it
    if (__DEBUG__)
      fprintf(stderr, " new history item.\n");

    if (historyLength >= MAXHISTLEN) 
    {
      // History is full, so move everything up one spot (overwriting the history item at element 0.
      for (historyIndex = 0; historyIndex + 1 < historyLength; historyIndex++)
        strncpy(history[historyIndex], history[historyIndex + 1], MAXCMDLEN);
    } 
    else 
      historyLength++;	// History is not full yet, so just append the new command to the end

    // Set the last item of the history to be the new command
    strncpy(history[historyLength - 1], newHistoryItem, MAXCMDLEN);
  }

  if (__DEBUG__) 
  {
    int histIndex;
    fprintf(stderr, "History shuffling complete.  New history is: \n");
    for (histIndex = 0; histIndex < historyLength; histIndex++)
      fprintf(stderr, "\t[%02d] '%s'\n", histIndex, history[histIndex]);
  }

  writeHistory();
}

void writeHistory() 
{
  FILE *fp;
  int curHistIndex;

  if ((fp = fopen(historyFilename, "w")) == 0) 
  {
    fprintf(stderr, "Could not open history file '%s' for writing.  History will not be saved.\n", historyFilename);
    return;
  }

  for (curHistIndex = 0; curHistIndex < historyLength; curHistIndex++)
    fprintf(fp, "%s\n", history[curHistIndex]);

  fclose(fp);

  if (__DEBUG__)
    fprintf(stderr, "Finished writing new history file.\n");

}

// Event handler for Ok and Cancel buttons
void callback (GtkWidget * widget, gpointer data) 
{
  command[0] = '\0';
  if ((char *)data == "ok")	// else cancel 
  {
    strncpy(command, gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry)), MAXCMDLEN);
    if (command[0] != '\0')
      updateHistory(strtok(command, "\n"));
  }
  gtk_main_quit();
}

// Event handler for Enter and Escape keys
void keypress(GtkWidget *widget, GdkEventKey *event, gpointer data) 
{
  command[0] = '\0';
  if (event->keyval == GDK_Return) 
  {
    strncpy(command, gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(combo)->entry)), MAXCMDLEN);
    if (command[0] != '\0') 
      updateHistory(strtok(command, "\n"));

    gtk_main_quit();
    gtk_object_destroy(GTK_OBJECT(widget));
  } 
  else if (event->keyval == GDK_Escape) 
  {
    gtk_main_quit();
    gtk_object_destroy(GTK_OBJECT(widget));
  }
}

/* another callback */
gint delete_event(GtkWidget * widget, GdkEvent * event, gpointer data) 
{
  gtk_main_quit();
  return (FALSE);
}

int runwin() 
{
  GtkWidget *window;
  GtkWidget *button;
  GtkWidget *table;
  GList *combo_items = NULL;
  int c = 0;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW (window), "Run");

  // Here we just set a handler for delete_event that immediately exits GTK.
  gtk_signal_connect(GTK_OBJECT(window), "delete_event", GTK_SIGNAL_FUNC(delete_event), NULL);

  // Handle the enter and escape keys.
  gtk_signal_connect(GTK_OBJECT(window), "key-press-event", GTK_SIGNAL_FUNC(keypress), GTK_OBJECT(window));

  gtk_container_set_border_width(GTK_CONTAINER(window), 10); 	// Sets the border width of the window
  table = gtk_table_new(10, 2, FALSE);    			// Create a 2x2 table
  gtk_container_add(GTK_CONTAINER(window), table);  		// Put the table in the main window

  // Create a text entry area with drop down history list
  combo = gtk_combo_new();
  gtk_widget_set_usize(combo, 300, 30);
  gtk_combo_set_use_arrows_always(GTK_COMBO(combo), TRUE); 	// scroll through the list with the arrow keys.
  gtk_window_set_focus(GTK_WINDOW(window), GTK_COMBO(combo)->entry); // focus the entry area so we can type right away.
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "activate", GTK_SIGNAL_FUNC(callback), NULL);
  gtk_table_attach_defaults(GTK_TABLE(table), combo, 0, 10, 0, 1);
  gtk_widget_show(combo);

  // Instead of appending the items in reverse order, we prepend each item.
  // This fixes a problem with the arrow keys cycling through the list,
  // You would only be able to go up the list from oldest item to newest.
  for (c = 0; c < historyLength; c++)
    combo_items = g_list_prepend(combo_items, history[c]);

  combo_items = g_list_prepend(combo_items, "");

  gtk_combo_set_popdown_strings(GTK_COMBO(combo), combo_items);
  g_list_free(combo_items);

  // Create the Cancel button
  button = gtk_button_new_with_label("Cancel");
  gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(callback), (gpointer) "cancel");
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(window));
  gtk_table_attach_defaults(GTK_TABLE(table), button, 6, 8, 1, 2);
  gtk_widget_show(button);

  // Create the Ok button
  button = gtk_button_new_with_label("OK");
  gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(callback), (gpointer) "ok");
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(window));
  gtk_table_attach_defaults(GTK_TABLE(table), button, 8, 10, 1, 2);
  gtk_widget_show(button);

  gtk_widget_show(table);
  gtk_widget_show(window);

  gtk_main();	// Rest in gtk_main and wait for the fun to begin!

  return (0);
}

void usage (void) 
{
  fprintf(stderr, "bbrun v%s - Josh King <jking@dwave.net>\n", VERSION);
  fprintf(stderr, "Usage: bbrun [GENERAL OPTIONS] [-- [-display <display>]]...\n");
  fprintf(stderr, "\n");
  fprintf(stderr, " -h, --help\t\tthis help screen\n");
  fprintf(stderr, " -i, --history-file=<history file>\tset the history file to use\n");
  fprintf(stderr, "\t\t\t\t\t(default: '~%s')\n", DEFAULT_HIST_FILENAME);
  fprintf(stderr, " -v, --version\t\tprint the version number\n");
  fprintf(stderr, " -w, --withdraw\t\twithdrawn mode, will go straight to command entry box\n\n");
  fprintf(stderr, "Options that must occur after '--' if they are used:\n\n");
  fprintf(stderr, " -display <display>\tset the display that the gearbox should show up on\n\n");

  exit(0);
}
