/* $Id: spool.c,v 1.9 2004/03/07 00:35:19 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@uab.edu>
** Portions Copyright (C) 2003-2004 Sourcefire, Inc.
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>
#include "barnyard.h"
#include "input-plugins/dp_plugbase.h"
#include "spool.h"
#include "util.h"


/* Local functions ************************************************/
static int FindNextSpoolFileExtension(char *directory, char *base_filename, 
        u_int32_t timet, u_int32_t *extension);
static int SpoolFileHandle_New(char *directory, char *filename, 
        u_int32_t extension, SpoolFileHandle **p_sph);
static int SpoolFileHandle_Destroy(SpoolFileHandle *sph);
static int SpoolFileHandle_ReadMagic(SpoolFileHandle *sph);
static int SpoolFileHandle_ReadHeader(SpoolFileHandle *sph);
static int SpoolFileHandle_ReadRecord(SpoolFileHandle *sph);

static int ArchiveFile(char *filepath, char *archive_dir);
static int Move(char *source, char *destination);
static void FreeRecord(Record *record);

static FILE *waldo_fp = NULL;
int OpenWaldoFile();
int CloseWaldoFile();
int WriteWaldoFile(SpoolFileHandle *sph);


/* Find the next spool file timestamp extension with a value equal to or 
 * greater than timet.  If extension != NULL, the extension will be 
 * returned.
 *
 * @retval 0    file found
 * @retval -1   error
 * @retval 1    no file found
 *
 * Bugs:  This function presumes a 1 character delimeter between the base 
 * filename and the extension
 */
static int FindNextSpoolFileExtension(char *directory, char *base_filename, 
        u_int32_t timet, u_int32_t *extension)
{
    DIR *dir = NULL;
    struct dirent *entry;
    size_t base_filename_len;
    u_int32_t min_timet = 0;
    
    if(directory == NULL || base_filename == NULL)
    {
        /* invalid arguments */
        return -1;
    }

    base_filename_len = strlen(base_filename);
    
    /* open the directory */
    if(!(dir = opendir(directory)))
    {
        LogMessage("ERROR: Unable to open directory '%s': %s\n", directory,
                strerror(errno));
        return -1;
    }

    /* step through each entry in the directory */
    while((entry = readdir(dir)))
    {
        unsigned long file_timet;
        if(strncmp(base_filename, entry->d_name, base_filename_len) != 0)
            continue;

        /* this is a file we may want */
        if(String2ULong(entry->d_name + base_filename_len + 1, &file_timet) 
                != 0)
        {
            LogMessage("WARNING: Unable to extract timestamp file extension "
                    "from '%s'\n", entry->d_name);
            continue;
        }

        if(file_timet == timet)
        {
            /* exact match */
            min_timet = file_timet;
            break;
        }
        else if(file_timet > timet)
        {
            if(min_timet == 0 || file_timet < min_timet)
            {
                min_timet = file_timet;
            }
        }
    }
    closedir(dir);

    if(min_timet == 0)    /* no files found */
        return 1;

    if(extension)
    {
        *extension = min_timet;
    }
    return 0;
}

static int SpoolFileHandle_New(char *directory, char *filename, 
        u_int32_t extension, SpoolFileHandle **p_sph)
{
    SpoolFileHandle *sph = NULL;
    size_t path_len;
        
    
    if(!p_sph || !filename)
        return -1;  /* Invalid arguments */
    
    if(!(sph = (SpoolFileHandle *)calloc(1, sizeof(SpoolFileHandle))))
    {
        LogMessage("ERROR: Out of memory creating new SpoolFileHandle "
                "(wanted %u bytes)\n", sizeof(SpoolFileHandle));
        return -1;
    }
    sph->filedes = -1;

    /* construct the complete filepath */
    if(extension == 0)
    {
        path_len = strlen(directory) + 1 + strlen(filename) + 1;
    }
    else
    {
        path_len = strlen(directory) + 1 + strlen(filename) + 1 + 10 + 1;
    }
    
    if(!(sph->filepath = (char *)calloc(path_len, sizeof(char))))
    {
        LogMessage("ERROR: Out of memory creating new SpoolFileHandle "
                "(wanted %u bytes)\n", path_len * sizeof(char));
        SpoolFileHandle_Destroy(sph);
        return -1;
    }

    if(extension == 0)
    {
        snprintf(sph->filepath, path_len, "%s/%s", directory, filename);
    }
    else
    {
        snprintf(sph->filepath, path_len, "%s/%s.%u", directory, filename,
                extension);
    }

    sph->timet = extension;

    /* open the file non-blocking */
    if((sph->filedes = open(sph->filepath, O_RDONLY | O_NONBLOCK, 0)) == -1)
    {
        LogMessage("ERROR: Unable to open log spool file '%s': %s\n", 
                    sph->filepath, strerror(errno));
        SpoolFileHandle_Destroy(sph);
        return -1;
    }

    sph->state = STATE_OPENED;

    *p_sph = sph;
    return 0;
}

static int SpoolFileHandle_Destroy(SpoolFileHandle *sph)
{
    if(!sph)
        return -1;  /* Invalid argument */

    if(sph->filedes != -1)
        close(sph->filedes);
    sph->filedes = -1;

    if(sph->filepath)
        free(sph->filepath);
    sph->filepath = NULL;

    FreeRecord(&sph->record);

    free(sph);
    return 0;
}

/* Read the magic from the spool file */
static int SpoolFileHandle_ReadMagic(SpoolFileHandle *sph)
{
    ssize_t bytes_read;

    if(!sph)
        return -1;      /* Invalid argument */

    if(sph->state != STATE_OPENED)
    {
        LogMessage("ERROR: Invalid attempt to read spool file magic\n");
        return -1;
    }

    bytes_read = read(sph->filedes, &sph->magic + sph->offset, 
            sizeof(sph->magic) - sph->offset);

    if(bytes_read + sph->offset != sizeof(sph->magic))
    {
        if(bytes_read == -1)
        {
            /* Read error */
            LogMessage("ERROR: read error: %s\n", strerror(errno));
            return N_FILE_ERROR;
        }

        if(bytes_read == 0)
        {
            return N_READ_EOF;
        }

        return N_READ_PARTIAL;
    }
    
    sph->offset = 0;
    sph->state = STATE_MAGIC_READ;
    
    /* Based on magic, lookup the DpFunctionalNode */
    if(!(sph->dpfn = LookupFunctions(sph->magic)))
    {
        FatalError("ERROR: No input plugin found for magic: %.8x\n", 
                sph->magic);
    }

    return 0;
}

static int SpoolFileHandle_ReadHeader(SpoolFileHandle *sph)
{
    int rval;
    if(!sph)
        return -1;  /* invalid argument */

    if(sph->state != STATE_MAGIC_READ)
    {
        LogMessage("ERROR: Invalid attempt to read spool file header\n");
        return -1;
    }

    if(sph->dpfn->readFileHeaderFunc)
    {
        rval = sph->dpfn->readFileHeaderFunc(sph->dpfn, sph);
        if(rval != 0)
        {
            return rval;
        }

        sph->state = STATE_HEADER_READ;
        sph->offset = 0;
    }
    else
    {
        LogMessage("INFO: no function defined to read header\n");
    }

    /* Start the data processor */
    DataProcessorStart(sph->dpfn);

    return 0;
}

static int SpoolFileHandle_ReadRecord(SpoolFileHandle *sph)
{
    int rval;
    if(!sph)
        return -1;  /* invalid argument */

    if(sph->state != STATE_HEADER_READ && sph->state != STATE_RECORD_READ)
    {
        LogMessage("ERROR: Invalid attempt to read record\n");
        return -1;
    }

    if(!sph->dpfn->readRecordFunc)
    {
        LogMessage("INFO: no function defined to read header\n");
        return -1;
    }
    
    rval = sph->dpfn->readRecordFunc(sph);
    
    if(rval == 0)
    {
        sph->state = STATE_RECORD_READ;
        sph->current_record++;
        sph->offset = 0;
    }

    return rval;
}


void CloseSpoolFile(SpoolFileHandle *sph)
{
    if(sph == NULL)
        return;
    
    /* Close the old file */
    close(sph->filedes);

    /* free up memory */
    if(sph->header != NULL)
        free(sph->header);
    if(sph->filepath != NULL)
        free(sph->filepath);
    free(sph);

}

int DoOneShot(char *directory, char *filename)
{
    SpoolFileHandle *sph = NULL;
    int rval;
    
    pv.exit = 1;

    /* Open the spool file */
    if(SpoolFileHandle_New("", filename, 0, &sph) != 0)
    {
        FatalError("Unable to open spool file: %s\n", strerror(errno));
    }

    /* Read the file magic */
    rval = SpoolFileHandle_ReadMagic(sph);
    if(rval != 0)
    {
        FatalError("Unable to read magic from '%s': %i\n", sph->filepath, rval);
    }

    rval = SpoolFileHandle_ReadHeader(sph);
    if(rval != 0)
    {
        FatalError("Unable to read header from '%s': %i\n", sph->filepath, 
                rval);
    }
    
    while(pv.stop == 0)
    {
        rval = SpoolFileHandle_ReadRecord(sph);
        if(rval == 0)
        {
            if(sph->state == STATE_RECORD_READ)
            {
                if(sph->dpfn->processRecordFunc)
                {
                    /* XXX check result */
                    sph->dpfn->processRecordFunc(sph->record.data, 
                            sph->dpfn);
                }
            }
            FreeRecord(&sph->record);
        }
        else if(rval == N_READ_EOF)
        {
            rval = 0;
            break;
        }
        else
        {
            LogMessage("ERROR: Input file '%s' is corrupted\n", 
                    sph->filepath);
            rval = -1;
            break;
        }
    }

    if(pv.verbose >= 1)
        LogMessage("Number of records:  %u\n", sph->current_record);
    SpoolFileHandle_Destroy(sph);

    return 0;
}

int ProcessSpool(char *spool_directory, char *base_filename, 
        u_int32_t first_record, time_t timet)
{
    SpoolFileHandle *sph = NULL;
    int rval = 0;
    u_int32_t extension;
    int new_file_available = 0;
    int waiting_logged = 0;
    u_int32_t skipped = 0;
       
    if(pv.start_at_end == 1)
    {
        /* Find newest file extension */
        while(FindNextSpoolFileExtension(spool_directory, base_filename,
                    timet, &extension) == 0)
        {
            if(timet > 0 && pv.verbose)
                LogMessage("Skipping file: %s/%s.%u\n", spool_directory,
                        base_filename, timet);
            timet = extension + 1;
        }
        timet = extension;
    }

    /* Start the main process loop */
    while(pv.stop == 0)
    {
        if(!sph)    /* No current spool file handle */
        {
            /* Find the next spool file */
            rval = FindNextSpoolFileExtension(spool_directory, base_filename,
                    timet, &extension);
            if(rval == 1)
            {
                if(!waiting_logged)
                {
                    if(pv.start_at_end)
                        LogMessage("Skipped %u old records\n", skipped);
                    LogMessage("Waiting for new spool file\n");
                    waiting_logged = 1;
                    pv.start_at_end = 0;
                }
                sleep(1);
                continue;
            }
            else if(rval == -1)
            {
                FatalError("Error looking for next spool file\n");
                /* Code for when we stop using FatalError */
                pv.exit = 1;
                return -1;  
            }

            /* Create a new spool file handle */
            if(SpoolFileHandle_New(spool_directory, base_filename, extension, 
                        &sph) != 0)
            {
                /* XXX We can handle this better */
                FatalError("Unable to create new spool file handle\n");
            }
            LogMessage("Opened spool file '%s'\n", sph->filepath);
            waiting_logged = 0;

            /* Set timet so next time we will look for a newer file */
            timet = extension + 1;

            continue;
        }

        switch(sph->state)
        {
            case STATE_OPENED:
                rval = SpoolFileHandle_ReadMagic(sph);
                break;
            case STATE_MAGIC_READ:
                rval = SpoolFileHandle_ReadHeader(sph);
                break;
            case STATE_HEADER_READ:
            case STATE_RECORD_READ:
                rval = SpoolFileHandle_ReadRecord(sph);
                break;
            default:
                LogMessage("ERROR: Invalid spool file handle state '%i'.  "                            "Closing spool file '%s'\n", sph->state, 
                        sph->filepath);
                /* Archive the spool file */
                if(pv.archive_dir)
                    ArchiveFile(sph->filepath, pv.archive_dir);
                SpoolFileHandle_Destroy(sph);
                first_record = 0;
                sph = NULL;
                break;
        }

        if(!sph)
            continue;
        
        if(rval == 0)
        {
            if(sph->state == STATE_RECORD_READ)
            {
                if(first_record > 0)
                {
                    /* Skip this record */
                    first_record--;
                }
                else if(pv.start_at_end == 1)
                {
                    skipped++;
                }
                else
                {
                    if(sph->dpfn->processRecordFunc)
                    {
                        /* XXX check result */
                        sph->dpfn->processRecordFunc(sph->record.data, 
                                sph->dpfn);
                    }
                    WriteWaldoFile(sph);
                }
            }
            FreeRecord(&sph->record);
        }
        else
        {
            if(rval == N_FILE_ERROR)
            {
                FatalError("Read error\n");
                pv.exit = 1;
                return -1;
            }
            
            if(new_file_available)
            {
                switch(sph->state)
                {
                    case STATE_OPENED:
                        if(rval == N_READ_PARTIAL || rval == N_READ_EOF)
                            LogMessage("Error reading magic from '%s'\n",
                                    sph->filepath);
                        break;
                    case STATE_MAGIC_READ:
                        if(rval == N_READ_PARTIAL || rval == N_READ_EOF)
                            LogMessage("Error reading header from '%s'\n",
                                    sph->filepath);
                        break;
                    case STATE_HEADER_READ:
                    case STATE_RECORD_READ:
                        if(rval == N_READ_PARTIAL)
                            LogMessage("Truncated record in '%s'\n",
                                    sph->filepath);
                        break;
                    default:
                        if(rval == N_READ_PARTIAL)
                            LogMessage("Partial read from '%s'\n",
                                    sph->filepath);
                        break;
                }
                LogMessage("Closing spool file '%s'.  Read %d records\n",
                        sph->filepath, sph->current_record);
                /* Archive the file */
                if(pv.archive_dir)
                    ArchiveFile(sph->filepath, pv.archive_dir);
                /* Close this spool file so we can rotate */
                SpoolFileHandle_Destroy(sph);
                first_record = 0;
                sph = NULL;
                new_file_available = 0;
            }
            else
            {
                rval = FindNextSpoolFileExtension(spool_directory, 
                        base_filename, timet, NULL);
                if(rval == 0)
                {
                    new_file_available = 1;
                }
                else if(rval == -1)
                {
                    FatalError("Error looking for next spool file\n");
                    /* Code for when we stop using FatalError */
                    pv.exit = 1;
                    return -1;  
                }
                else
                {
                    if(!waiting_logged) 
                    {
                        if(pv.start_at_end)
                            LogMessage("Skipped %u old records\n", skipped);
                            
                        LogMessage("Waiting for new data\n");
                        waiting_logged = 1;
                        pv.start_at_end = 0;
                    }
                    sleep(1);
                    continue;
                }
            }
        }
    }

    if(pv.waldo_file)
        CloseWaldoFile();

    return 0;
}

int ArchiveFile(char *filepath, char *archive_dir)
{
    char *dest;
    size_t dest_len;
    if(!filepath || !archive_dir)
        return -1;  /* Invalid argument */

    /* Archive the file */
    dest_len = strlen(archive_dir) + 1 + strlen(strrchr(filepath, '/') + 1);
    dest = (char *)SafeAlloc(dest_len + 1);
    snprintf(dest, dest_len + 1, "%s/%s", archive_dir, 
            strrchr(filepath, '/') + 1);
    Move(filepath, dest);
    free(dest);

    return 0;
}

/********************* Waldo file functions ***************************/
/* XXX Change all of this to use file descriptors instead of streams */
int OpenWaldoFile()
{
    if(!pv.waldo_file)
        return 0;

    waldo_fp = fopen(pv.waldo_file, "w");

    if(waldo_fp == NULL)
    {
        FatalError("Failed to create waldo file \"%s\": %s\n", 
                pv.waldo_file, strerror(errno));
        return 1;
    }

    return 0;
}

int CloseWaldoFile()
{
    if(!waldo_fp)
        return 0;

    fclose(waldo_fp);
    waldo_fp = NULL;
    return 0;
}


int WriteWaldoFile(SpoolFileHandle *sph)
{
    if(!pv.waldo_file)
        return 0;

    if(!waldo_fp)
    {
        if(OpenWaldoFile() != 0)
            return 1;
    }

    /* XXX ftruncate would be better */
    rewind(waldo_fp);
    
    
    /* update fields */
    pv.timet = sph->timet;
    pv.record_number = sph->current_record;
    fprintf(waldo_fp, "%s\n", pv.spool_dir);
    fprintf(waldo_fp, "%s\n", pv.spool_file);
    fprintf(waldo_fp, "%lu\n", sph->timet);
    fprintf(waldo_fp, "%u\n", sph->current_record);

    fflush(waldo_fp);

    return 0;
}


/* XXX This function should be relocated to util.c */
int Move(char *source, char *dest)
{
    if(link(source, dest) != 0)
    {
        if(errno == EXDEV || errno == EPERM)
        {
            /* can't hardlink, do it the hard way */
            char *command;
            size_t command_len;
            command_len = strlen("mv") + 1 + strlen(source) + 1 + strlen(dest);
            command = (char *)SafeAlloc(command_len + 1);
            snprintf(command, command_len + 1, "mv %s %s", source, dest);
            if(system(command) != 0)
            {
                LogMessage("Failed to archive file \"%s\" to \"%s\": %s",
                        source, dest, strerror(errno));
            }
            free(command);
        }
        LogMessage("Failed to archive file \"%s\" to \"%s\": %s",
                source, dest, strerror(errno));
    }
    else
    {
        unlink(source);
    }
    return 0;
}


void FreeRecord(Record *record)
{
    if(record->dynamic)
        free(record->data);
    record->data = NULL;
    record->dynamic = 0;
}
