/* $Id: op_fast.c,v 1.9 2004/03/16 04:18:20 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <netinet/in.h>


#include "input-plugins/dp_alert.h"
#include "strlcpyu.h"
#include "plugbase.h"
#include "op_plugbase.h"
#include "mstring.h"
#include "op_fast.h"
#include "util.h"
#include "sid.h"
#include "classification.h"
#include "barnyard.h"

typedef struct _OpAlertFast_Data 
{
    char *filename;  /* file to open for output */
    char *filepath;  /* file to open for output */
    FILE *file;
} OpAlertFast_Data;


static int OpAlertFast_Setup(OutputPlugin *outputPlugin, char *args);
static int OpAlertFast_Exit(OutputPlugin *);
static int OpAlertFast_Start(OutputPlugin *, void *);
static int OpAlertFast_Stop(OutputPlugin *);
static int OpAlertFast(void *, void *);
static int OpAlertFast_LogConfig(OutputPlugin *);

void ParseFastAlertArgs(char *, OutputPlugin *);
FILE *OpenAlertFile(char *);

/* init routine makes this processor available for dataprocessor directives */
void OpAlertFast_Init()
{
    OutputPlugin *outputPlugin;

    outputPlugin = RegisterOutputPlugin("alert_fast", "alert");
    
    outputPlugin->setupFunc = OpAlertFast_Setup;
    outputPlugin->exitFunc = OpAlertFast_Exit;
    outputPlugin->startFunc = OpAlertFast_Start;
    outputPlugin->stopFunc = OpAlertFast_Stop;
    outputPlugin->outputFunc = OpAlertFast;
    outputPlugin->logConfigFunc = OpAlertFast_LogConfig;
}

static int OpAlertFast_LogConfig(OutputPlugin *outputPlugin)
{
    OpAlertFast_Data *data = NULL;

    if(!outputPlugin || !outputPlugin->data)
        return -1;

    data = (OpAlertFast_Data *)outputPlugin->data;

    LogMessage("OpAlertFast configured\n");
    LogMessage("  Filename: %s\n", data->filename);
    return 0;
}

/* link the output processor functions to an output function node */
int OpAlertFast_Setup(OutputPlugin *outputPlugin, char *args)
{
    ParseFastAlertArgs(args, outputPlugin);

    return 0;
}

int OpAlertFast_Exit(OutputPlugin *outputPlugin)
{
    OpAlertFast_Data *data = (OpAlertFast_Data *)outputPlugin->data;

    if(!data)
        return 0;
    
    if(data->filename)
        free(data->filename);
    data->filename = NULL;
    
    if(data->filepath)
        free(data->filepath);
    data->filepath = NULL;

    return 0;
}


int OpAlertFast_Start(OutputPlugin *outputPlugin, void *spool_header)
{
    OpAlertFast_Data *data = (OpAlertFast_Data *)outputPlugin->data;

    
    if(data == NULL)
        FatalError("ERROR: Unable to find context for AlertFast startup!\n");
    
    if(pv.verbose >= 2)
        OpAlertFast_LogConfig(outputPlugin);
    
    data->filepath = ProcessFileOption(data->filename);
    
    data->file = OpenAlertFile(data->filepath);

    return 0 ;
}


int OpAlertFast_Stop(OutputPlugin *outputPlugin)
{
    OpAlertFast_Data *data = (OpAlertFast_Data *)outputPlugin->data;

    if(data->file)
    {
        fflush(data->file);
        fclose(data->file);
        data->file = NULL;
    }
    if(data->filepath)
        free(data->filepath);
    data->filepath = NULL;

    return 0;
}

int OpAlertFast(void *data, void *alertdata)
{
    char timestamp[256];
    //struct in_addr in;
    UnifiedAlertRecord *ad = (UnifiedAlertRecord *)alertdata;
    OpAlertFast_Data *afd = (OpAlertFast_Data *)data;
    ClassType *ct;
    Sid *tmp = NULL;
    char sip[16];
    char dip[16];


    tmp = GetSid(ad->event.sig_generator, ad->event.sig_id);
    ct = GetClassType(ad->event.classification);

    if(RenderTimeval(&ad->ts, timestamp, 256) == -1)
    {
        /* could not render the timeval */
        LogMessage("ERROR: OpAlertFast failed to render timeval\n");
        return -1;  
    }

    snprintf(sip, 16, "%u.%u.%u.%u", (ad->sip & 0xff000000) >> 24,
            (ad->sip & 0x00ff0000) >> 16, (ad->sip & 0x0000ff00) >> 8,
            ad->sip & 0x000000ff);
    snprintf(dip, 16, "%u.%u.%u.%u", (ad->dip & 0xff000000) >> 24,
            (ad->dip & 0x00ff0000) >> 16, (ad->dip & 0x0000ff00) >> 8,
            ad->dip & 0x000000ff);


    if(ad->protocol == IPPROTO_TCP ||
            ad->protocol == IPPROTO_UDP)
    {
        fprintf(afd->file, "%s {%s} %s:%d -> %s:%d\n"
                "[**] [%d:%d:%d] %s [**]\n"
                "[Classification: %s] [Priority: %d]\n", timestamp, 
                protocol_names[ad->protocol], sip, ad->sp, dip, ad->dp,
                ad->event.sig_generator, ad->event.sig_id, ad->event.sig_rev,
                tmp != NULL?tmp->msg:"ALERT", 
                ct != NULL?ct->name:"Unknown", ad->event.priority);
    }
    else
    {
        fprintf(afd->file, "%s {%s} %s -> %s\n"
                "[**] [%d:%d:%d] %s [**]\n"
                "[Classification: %s] [Priority: %d]\n", timestamp, 
                protocol_names[ad->protocol], sip, dip, 
                ad->event.sig_generator, ad->event.sig_id, ad->event.sig_rev,
                tmp != NULL ? tmp->msg : "ALERT", 
                ct != NULL ? ct->name : "Unknown", ad->event.priority);
    }

    PrintXref(ad->event.sig_generator, ad->event.sig_id, afd->file);

    fprintf(afd->file, "-----------------------------------------------------"
            "-------------------\n");

    fflush(afd->file);
    return 0;
}




/* initialize the output processor for this particular instantiation */
void ParseFastAlertArgs(char *args, OutputPlugin *outputPlugin)
{
    char **toks;
    int num_toks;
    OpAlertFast_Data *data;

    data = (OpAlertFast_Data *)SafeAlloc(sizeof(OpAlertFast_Data));

    if(args == NULL)
    {
       data->filename = strdup("fast.alert");
       outputPlugin->data = (OpAlertFast_Data *) data;
       return;
    }

    toks = mSplit(args, " ", 2, &num_toks, 0);
    
    data->filename = strdup(toks[0]);
    
    FreeToks(toks, num_toks);
    
    outputPlugin->data = (OpAlertFast_Data *) data;

    return;
}


FILE *OpenAlertFile(char *filename)
{
     FILE *tmp;   

    if((tmp = fopen(filename, "a+")) == NULL)
    {
        FatalError("ERROR => fopen(%s) failed: %s\n", filename,
                    strerror(errno));
    }
 
    return tmp;
}
