/*
** Copyright (C) 2001-2004 Andrew R. Baker <andrewb@snort.org>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "ProgVars.h"
#include "ConfigFile.h"
#include "util.h"

#define DEFAULT_CONFIG_FILE  "/etc/snort/barnyard.conf"
#define DEFAULT_SID_MSG_FILE "sid-msg.map"
#define DEFAULT_GEN_MSG_FILE "gen-msg.map"
#define DEFAULT_CLASS_FILE   "classification.config"
#define DEFAULT_LOG_DIR      "/var/log/snort"
#define DEFAULT_SPOOL_DIR    "/var/log/snort"
#define DEFAULT_PID_FILE     "/var/run/by.pid"
#define PATH_MAX 8192


#define BY_SUCCESS  0
#define BY_EINVAL   1
#define BY_ENOMEM   2
#define BY_ENOENT   3
#define BY_EOPEN    4
#define BY_ETRUNC   5
#define BY_ECORRUPT 6

/**
 * Read the waldo file and extract spool directory, base filename, 
 * file timestamp extension, and record number
 *
 * BUGS: This function is really poorly written 
 *
 * @param filename Waldo filename
 * 
 * @retval BY_SUCCESS  success
 * @retval BY_EINVAL   invalid argument
 * @retval BY_ENOMEM   out of memory
 * @retval BY_ENOENT   no file found
 * @retval BY_EOPEN    open error  (FatalError)
 * @retval BY_ETRUNC   truncated
 * @retval BY_ECORRUPT waldo file is corrupt
 */
static int ProcessBookmarkFile(char *filename, ProgVars *prog_vars)
{
    FILE *f;
    char readbuf[STD_BUF];
    char *spool_dir = NULL;
    char *spool_file = NULL;
    unsigned long uvalue;
    time_t timet;
    u_int record_number;

    if(!filename || !prog_vars)
        return BY_EINVAL;

    if(!(f = fopen(filename, "r")))
    {
        if(errno != ENOENT)
        {
            FatalError("Failed to open waldo file \"%s\": %s\n",
                    filename, strerror(errno));
            return BY_EOPEN;
        }
        return BY_ENOENT;
    }

    if(fgets(readbuf, STD_BUF, f) == NULL)
        return BY_ETRUNC;

    strip(readbuf);
    if(!(spool_dir = strdup(readbuf)))
    {
        FatalError("Out of memory (wanted %u bytes)\n",
                strlen(readbuf) + 1);
        return BY_ENOMEM;
    }

    if(fgets(readbuf, STD_BUF, f) == NULL)
        return BY_ETRUNC;

    strip(readbuf);
    if(!(spool_file = strdup(readbuf)))
    {
        FatalError("Out of memory (wanted %u bytes)\n",
                strlen(readbuf) + 1);
        return BY_ENOMEM;
    }

    if(fgets(readbuf, STD_BUF, f) == NULL)
        return BY_ETRUNC;

    strip(readbuf);
    if(String2ULong(readbuf, &uvalue) != 0)
    {
        LogMessage("Timestamp value '%s' is not an unsigned integer\n",
                readbuf);
        return BY_ECORRUPT;
    }
    timet = uvalue;

    if(fgets(readbuf, STD_BUF, f) == NULL)
        return BY_ETRUNC;

    strip(readbuf);
    if(String2ULong(readbuf, &uvalue) != 0)
    {
        LogMessage("Timestamp value '%s' is not an unsigned integer\n",
                readbuf);
        return BY_ECORRUPT;
    }
    record_number = uvalue;

    fclose(f);

    /* Store data into the ProgVars data structure */
    prog_vars->spool_dir = spool_dir;
    prog_vars->spool_file = spool_file;
    prog_vars->timet = timet;
    prog_vars->record_number = record_number;
    return 0;
}

static int ProcessFileConfigOption(char *option_name, char *clvalue, 
        char *cfvalue, char *default_value, char *path, char **target)
{
    char *value = NULL;
    if(!path || !target || !default_value)
        return -1;

    if(clvalue)
    {
        if(cfvalue)
        {
            if(option_name)
                LogMessage("WARNING: Command line option for %s "
                        "overrides config file value\n", option_name);
        }
        value = clvalue;
    }
    else if(cfvalue)
    {
        value = cfvalue;
    }
    else
    {
        value = default_value;
    }
    return BuildFilePath(value, path, target);
}


int ProgVars_Populate(CommandLineArgs *clargs, ProgVars *prog_vars)
{
    ConfigFileVars *cfvars = NULL; 
    char cwd[PATH_MAX + 1];

    if(!clargs || !prog_vars)
        return -1;
    
    ProgVars_Free(prog_vars);

    /* Get the current working directory */
    if(!getcwd(cwd, PATH_MAX + 1))
    {
        FatalError("Unable to get current working directory: %s\n",
                strerror(errno));
    }

    /* Set the config file (using an absolute path) */
    if(clargs->config_file)
    {
        /* Determine the configuration path */
        if(clargs->config_file[0] == '/')
        {
            /* Absolute path */
            if(!(prog_vars->config_file = strdup(clargs->config_file)))
            {
                FatalError("Out of memory (wanted %u bytes)\n",
                        strlen(clargs->config_file) + 1);
            }
        }
        else 
        {
            /* relative path */
            size_t len = strlen(clargs->config_file) + 1 + strlen(cwd) + 1;
            if(!(prog_vars->config_file = (char *)calloc(len, sizeof(char))))
            {
                FatalError("Out of memory (wanted %u bytes)\n", len);
            }
            snprintf(prog_vars->config_file, len, "%s/%s", cwd, 
                    clargs->config_file);
        }
    }
    else
    {
        /* Use default */
        if(!(prog_vars->config_file = strdup(DEFAULT_CONFIG_FILE)))
        {
            FatalError("Out of memory (wanted %u bytes)\n",
                    strlen(DEFAULT_CONFIG_FILE));
        }
    }
    /* Set the config directory */
    if(!(prog_vars->config_dir = strdup(prog_vars->config_file)))
    {
        FatalError("Out of memory (wanted %u bytes)\n",
                strlen(prog_vars->config_file));
    }
    *(strrchr(prog_vars->config_dir, '/')) = '\0';
    
    prog_vars->verbose = clargs->verbosity;

    /* Parse the config file */
    if(ConfigFile_Parse(prog_vars->config_file, &cfvars) != 0)
    {
        FatalError("Failed to parse config file '%s'\n",
                prog_vars->config_file);
    }

    if(prog_vars->verbose >= 2)
        ConfigFileVars_Fprintf(cfvars, stdout);
    
    /* sid-msg-file */
    if(ProcessFileConfigOption("sid-msg-map", clargs->sid_msg_file,
                cfvars->sid_msg_file, DEFAULT_SID_MSG_FILE, 
                prog_vars->config_dir, &prog_vars->sid_msg_file) != 0)
        FatalError("Failed to process sid-msg-map value\n");
    
    /* gen-msg-file */
    if(ProcessFileConfigOption("gen-msg-map", clargs->gen_msg_file,
                cfvars->gen_msg_file, DEFAULT_GEN_MSG_FILE, 
                prog_vars->config_dir, &prog_vars->gen_msg_file) != 0)
        FatalError("Failed to process gen-msg-map value\n");
    
    /* class file */
    if(ProcessFileConfigOption("class file", clargs->class_file,
                cfvars->class_file, DEFAULT_CLASS_FILE, 
                prog_vars->config_dir, &prog_vars->class_file) != 0)
        FatalError("Failed to process class file value\n");
    
    /* hostname */
    {
        char *hostname;
        char host_buffer[256];
        if(cfvars->hostname)
            hostname = cfvars->hostname;
        else
        {
            if(gethostname(host_buffer, 256))
            {
                FatalError("Error querying hostname: %s\n", strerror(errno));
            }
            hostname = host_buffer;
        }
        if(!(prog_vars->hostname = strdup(hostname)))
        {
            FatalError("Out of memory (wanted %u bytes)", 
                    strlen(hostname) + 1);
        }
    }

    /* interface */
    {
        char *interface;
        if(cfvars->interface)
            interface = cfvars->interface;
        else
            interface = "";
        if(!(prog_vars->interface = strdup(interface)))
        {
            FatalError("Out of memory (wanted %u bytes)", 
                    strlen(interface) + 1);
        }
    }
    
    /* bpf filter */
    {
        char *bpf_filter;
        if(cfvars->bpf_filter)
            bpf_filter = cfvars->bpf_filter;
        else
            bpf_filter = "";
        if(!(prog_vars->bpf_filter = strdup(bpf_filter)))
        {
            FatalError("Out of memory (wanted %u bytes)", 
                    strlen(bpf_filter) + 1);
        }
    }

    /* Localtime flag */
    if(cfvars->localtime_flag == 1)
        prog_vars->localtime = 1;
    
    if(clargs->new_records_only_flag)
        prog_vars->start_at_end = 1;
    
    /* Set the runtime mode */
    if(clargs->batch_mode_flag)
    {   
        char *path = NULL;
        int idx = 0;
        int filelist_idx = 0;
        int num_files = clargs->extra_args_count;
        prog_vars->run_mode = MODE_BATCH;

        /* Set the log dir */
        {
            char *logdir;
            if(clargs->log_dir)
                logdir = clargs->log_dir;
            else
                logdir = cwd;
            if(!(prog_vars->log_dir = strdup(logdir)))
            {
                FatalError("Out of memory (wanted %u bytes)",
                        strlen(logdir) + 1);
            }
        }
        
        /* Create the file list */
        if(clargs->spool_dir)
            path = clargs->spool_dir;
        else
            path = cwd;
            
        if(clargs->file_base)
        {
            LogMessage("WARNING: Use of '-f' for batch mode processing is "
                    "deprecated\n");
            num_files++;
        }
        if(!(prog_vars->filelist = (char **)calloc(num_files + 1,
                        sizeof(char *))))
        {
            FatalError("Out of memory (wanted %u bytes)\n",
                    (num_files + 1) * sizeof(char *));
        }
        if(clargs->file_base)
        {
            if(BuildFilePath(clargs->file_base, path, 
                        &prog_vars->filelist[0]) != 0)
            {
                FatalError("Failed to add '%s' to filelist\n", 
                        clargs->file_base);
            }
            filelist_idx = 1;
        }
        while(clargs->extra_args[idx])
        {
            if(BuildFilePath(clargs->extra_args[idx], path,
                        &prog_vars->filelist[filelist_idx]) != 0)
            {
                FatalError("Failed to add '%s' to filelist\n", 
                        clargs->extra_args[idx]);
            }
            idx++;
            filelist_idx++;
        }
    }
    else
    {
        prog_vars->run_mode = MODE_CONTINUAL;
        if(clargs->daemon_flag == 1 || cfvars->daemon_flag == 1)
            prog_vars->daemon_flag = 1;

        /* Set the log dir */
        {
            char *logdir;
            if(clargs->log_dir)
                logdir = clargs->log_dir;
            else
                logdir = DEFAULT_LOG_DIR;
            if(!(prog_vars->log_dir = strdup(logdir)))
            {
                FatalError("Out of memory (wanted %u bytes)",
                        strlen(logdir) + 1);
            }
        }
        
        /* Bookmark file */
        if(clargs->waldo_file)
        {
            int rval = 0;
            if(!(prog_vars->waldo_file = strdup(clargs->waldo_file)))
            {
                FatalError("Out of memory (wanted %u bytes)", 
                        strlen(clargs->waldo_file));
            }

            /* Load program vars from the waldo file */
            rval = ProcessBookmarkFile(prog_vars->waldo_file, prog_vars);
            if(rval == BY_SUCCESS)
            {
                if(prog_vars->verbose >= 1)
                    LogMessage("Starting data processing using information from"
                            " bookmark file\n");
                prog_vars->start_at_end = 0;
            }
            else if(rval == BY_ENOENT)
            {
                /* XXX we should verify that we can write to the specified
                 * file
                 */
                if(clargs->new_records_only_flag)
                {
                    prog_vars->start_at_end = 1;
                    if(prog_vars->verbose >= 1)
                        LogMessage("No bookmark file found, only processing "
                                "new events\n");
                }
                else
                {
                    prog_vars->start_at_end = 0;
                    if(prog_vars->verbose >= 1)
                        LogMessage("No bookmark file found, processing all "
                                "events\n");
                }
            }
            else if(rval == BY_ETRUNC || rval == BY_ECORRUPT)
            {
                LogMessage("WARNING: Bookmark file is corrupt, only processing "
                        "new events\n");
                prog_vars->start_at_end = 1;
            }
            else if(rval == BY_ENOMEM)
            {
                FatalError("Out of memory\n");
            }
            else if(rval == BY_EOPEN)
            {
                FatalError("Unable to open bookmark file '%s': %s\n", 
                        prog_vars->waldo_file, strerror(errno));
            }
            else if(rval == BY_EINVAL)
            {
                FatalError("Unreachable");
            }
        }
        
        /* Archive dir */
        if(clargs->archive_dir)
        {
            if(!(prog_vars->archive_dir = strdup(clargs->archive_dir)))
            {
                FatalError("Out of memory (wanted %u bytes)",
                        strlen(clargs->archive_dir) + 1);
            }
        }
    
        /* Pid file */
        {
            char *pid_file;
            if(clargs->pid_file)
                pid_file = clargs->pid_file;
            else
                pid_file = DEFAULT_PID_FILE;
            if(!(prog_vars->pid_file = strdup(pid_file)))
            {
                FatalError("Out of memory (wanted %u bytes)",
                        strlen(pid_file) + 1);
            }
        }

        /* Set the spool file */
        if(prog_vars->spool_file && prog_vars->spool_dir)
        {
            /* already set from the bookmark file */
            if(clargs->file_base && 
                    strcmp(prog_vars->spool_file, clargs->file_base) != 0)
                LogMessage("WARNING: Using spool file from bookmark file\n");
            
            if(clargs->spool_dir && 
                    strcmp(prog_vars->spool_dir, clargs->spool_dir) != 0)
                LogMessage("WARNING: Using spool dir from bookmark file\n");
        }
        else if(!prog_vars->spool_file && ! prog_vars->spool_dir)
        {
            char *spool_file = NULL;
            char *spool_dir = NULL;
            u_int8_t dynamic_spool_dir = 0;
            if(!clargs->file_base)
            {
                FatalError("No spool file specified for continual processing "
                        "mode\n");
            }

            /* Process the spool file and directory */
            if(strchr(clargs->file_base, '/'))
            {
                /* file base includes path information */
                if(clargs->spool_dir)
                    FatalError("Path information included in both spool_dir "
                            "and spool_file arguments");

                spool_file = strrchr(clargs->file_base, '/') + 1;
                if(!(spool_dir = strdup(spool_file)))
                {
                    FatalError("Out of memory (wanted %u bytes)\n",
                            strlen(spool_file));
                }
                *(strrchr(spool_dir, '/')) = '\0';
            }
            else
            {
                spool_file = clargs->file_base;
            }
            
            if(!(prog_vars->spool_file = strdup(clargs->file_base)))
            {
                FatalError("Out of memory (wanted %u bytes)",
                        strlen(clargs->file_base) + 1);
            }

            if(!spool_dir)
            {
                if(clargs->spool_dir)
                {
                    spool_dir = clargs->spool_dir;
                }
                else 
                {
                    spool_dir = DEFAULT_SPOOL_DIR;
                }
            }

            if(spool_dir[0] != '/')
            {
                size_t len = strlen(spool_dir) + 1 + strlen(cwd) + 1;
                if(!(prog_vars->spool_dir = (char *)calloc(len, sizeof(char))))
                {
                    FatalError("Out of memory (wanted %u bytes)",
                            len * sizeof(char));
                }
                snprintf(prog_vars->spool_dir, len, "%s/%s", cwd, spool_dir);
            }
            else
            {
                if(!(prog_vars->spool_dir = strdup(spool_dir)))
                {
                    FatalError("Out of memory (wanted %u bytes)",
                            strlen(spool_dir) + 1);
                }
            }
            if(dynamic_spool_dir)
                free(spool_dir);
        }
        else
        {
            /* We should never get here */
            FatalError("Failure processing spool file and spool dir");
        }
    }

    return 0;
}

int ProgVars_Free(ProgVars *prog_vars)
{
    if(!prog_vars)
        return -1;

    if(prog_vars->config_file)
        free(prog_vars->config_file);
    if(prog_vars->config_dir)
        free(prog_vars->config_dir);

    if(prog_vars->class_file)
        free(prog_vars->class_file);
    if(prog_vars->sid_msg_file)
        free(prog_vars->sid_msg_file);
    if(prog_vars->gen_msg_file)
        free(prog_vars->gen_msg_file);

    if(prog_vars->interface)
        free(prog_vars->interface);
    if(prog_vars->hostname)
        free(prog_vars->hostname);
    if(prog_vars->bpf_filter)
        free(prog_vars->bpf_filter);
    
    if(prog_vars->log_dir)
        free(prog_vars->log_dir);
    
    if(prog_vars->filelist)
    {
        int idx = 0;
        while(prog_vars->filelist[idx])
        {
            free(prog_vars->filelist[idx]);
            idx++;
        }
        free(prog_vars->filelist);
    }

    if(prog_vars->spool_dir)
        free(prog_vars->spool_dir);
    if(prog_vars->spool_file)
        free(prog_vars->spool_file);
    if(prog_vars->pid_file)
        free(prog_vars->pid_file);
    if(prog_vars->archive_dir)
        free(prog_vars->archive_dir);
    if(prog_vars->waldo_file)
        free(prog_vars->waldo_file);

    memset(prog_vars, 0, sizeof(ProgVars));
    return 0;
}

int ProgVars_Fprintf(ProgVars *prog_vars, FILE *stream)
{
    if(!prog_vars)
        return -1;

    if(!stream)
        stream = stdout;
    
    fprintf(stream, "Program Variables:\n");
    switch(prog_vars->run_mode)
    {
        case MODE_CONTINUAL:
            fprintf(stream, "  Continual processing mode\n");
            break;
        case MODE_BATCH:
            fprintf(stream, "  Batch processing mode\n");
            break;
        default:
            fprintf(stream, "  Unknown processing mode: %u\n", 
                    prog_vars->run_mode);
            break;
    }
    fprintf(stream, "  Config dir:    %s\n", prog_vars->config_dir);
    fprintf(stream, "  Config file:   %s\n", prog_vars->config_file);
    fprintf(stream, "  Sid-msg file:  %s\n", prog_vars->sid_msg_file);
    fprintf(stream, "  Gen-msg file:  %s\n", prog_vars->gen_msg_file);
    fprintf(stream, "  Class file:    %s\n", prog_vars->class_file);
    fprintf(stream, "  Hostname:      %s\n", prog_vars->hostname);
    fprintf(stream, "  Interface:     %s\n", prog_vars->interface);
    fprintf(stream, "  BPF Filter:    %s\n", prog_vars->bpf_filter);
    fprintf(stream, "  Log dir:       %s\n", prog_vars->log_dir);
    fprintf(stream, "  Verbosity:     %u\n", prog_vars->verbose);
    fprintf(stream, "  Localtime:     %u\n", prog_vars->localtime);
    if(prog_vars->run_mode == MODE_CONTINUAL)
    {
        fprintf(stream, "  Spool dir:     %s\n", prog_vars->spool_dir);
        fprintf(stream, "  Spool file:    %s\n", prog_vars->spool_file);
        if(prog_vars->daemon_flag)
            fprintf(stream, "  Pid file:      %s\n", prog_vars->pid_file);
        if(prog_vars->archive_dir)
            fprintf(stream, "  Archive dir:   %s\n", prog_vars->archive_dir);
        if(prog_vars->waldo_file)
        {
            fprintf(stream, "  Bookmark file: %s\n", prog_vars->waldo_file);
            fprintf(stream, "  Record Number: %u\n", prog_vars->record_number);
            fprintf(stream, "  Timet:         %lu\n", prog_vars->timet);
        }
        fprintf(stream, "  Start at end:  %u\n", prog_vars->start_at_end);
    }
    
    
    if(prog_vars->run_mode == MODE_BATCH && prog_vars->filelist 
            && prog_vars->filelist[0])
    {
        int idx = 0;
        fprintf(stream, "  File list:\n");
        while(prog_vars->filelist[idx])
        {
            fprintf(stream, "    %s\n", prog_vars->filelist[idx]);
            idx++;
        }
    }

    return 0;
}


