#include "store.h"

#include "script.h"

static int nstore;
static int nwork;
static float *store = NULL;
static float *work = NULL;

void free_store()
{
    FREE(store, float);
    FREE(work, float);
}

static Status allocate_store_memory(String error_msg)
{
    free_store();

    sprintf(error_msg, "allocating %d words memory for store", nstore);
    MALLOC(store, float, nstore);

    if (nwork > 0)
    {
	sprintf(error_msg, "allocating %d words for row data", nwork);
	MALLOC(work, float, nwork);
    }

    return  OK;
}

Status allocate_store(Store_info *store_info, Size_info *size_info,
			int nscripts, Script *scripts, String error_msg)
{
    int i, j, b, n, ndim, size_of_block;
    int *npoints, *block_size;

    ndim = size_info->ndim;
    npoints = size_info->npoints;
    block_size = size_info->block_size;
    nstore = store_info->nstore;

/*  first check if ndim > 1 scripts need too much memory  */

    n = 1;
    for (i = 0; i < nscripts; i++)
    {
	if (scripts[i].ndim > 1)
	    n = MAX(n, scripts[i].npts_max);
    }

    if (n > nstore)
    {
	sprintf(error_msg, "need %d words of memory, have only %d", n, nstore);
	return  ERROR;
    }

/*  next check if ndim = 1 scripts need too much memory  */

    n = 1;
    VECTOR_PRODUCT(size_of_block, block_size, ndim);

    for (i = 0; i < nscripts; i++)
    {
	if (scripts[i].ndim == 1)
	{
	    j = scripts[i].dims[0];
	    n = MAX(npoints[j], scripts[i].npts[0]);
			/* MAX(npts_in, npts_out) */
	    b = 1 + (n - 1) / block_size[j];
			/* maximum number blocks in chosen dimension */

	    if ((b*size_of_block) > nstore)
	    {
		sprintf(error_msg, "need %d words of memory, have only %d",
						b*size_of_block, nstore);
		return  ERROR;
	    }
	}
    }

/*  ndim = 1 scripts also need a one-dimensional work area  */

    nwork = 0;
    for (i = 0; i < nscripts; i++)
    {
	if (scripts[i].ndim == 1)
	    nwork = MAX(nwork, scripts[i].npts_max);
    }

    nwork = MAX(nwork, size_of_block);

    if (allocate_store_memory(error_msg) == ERROR)
    {
	free_store();
	return  ERROR;
    }

    store_info->store = store;
    store_info->nwork = nwork;
    store_info->work = work;

    return  OK;
}

static Bool not_same_vectors(int n, int *v1, int *v2)
{
    int i;

    for (i = 0; i < n; i++)
	if (v1[i] != v2[i])
	    return  TRUE;

    return  FALSE;
}

/*  Following checks if the n scripts have non-consecutive dimensions.  */

static Bool non_consecutive_dims(int ndim, int n, Script *scripts)
{
    int i, j;
    Bool dim_done[MAX_NDIM];

    if ((n == 1) || (ndim == 1))
	return  FALSE;

    for (i = 0; i < ndim; i++)
	dim_done[i] = FALSE;

    for (i = 0; i < n; i++)
    {
	j = scripts[i].dims[0];
	dim_done[j] = TRUE;
    }

    if (dim_done[0] && !dim_done[1])
	return  TRUE;

    for (i = 1; i < ndim-1; i++)
	if (dim_done[i] && !(dim_done[i-1] || dim_done[i+1]))
	    return  TRUE;

    if (dim_done[ndim-1] && !dim_done[ndim-2])
	return  TRUE;

    return  FALSE;
}

int scripts_processed(int *method, Bool *have_output, int nscripts,
			Script *scripts, Size_info *size_info,
			Store_info *store_info, File_info *file_info)
{
    int i, j, ndone, avail_nblocks, total_nblocks, ndim, nstore, size_of_block;
    int npoints_max[MAX_NDIM], nblocks[MAX_NDIM];
    int *npoints, *block_size;

    ndim = size_info->ndim;
    npoints = size_info->npoints;
    block_size = size_info->block_size;
    nstore = store_info->nstore;

    COPY_VECTOR(npoints_max, npoints, ndim);

    VECTOR_PRODUCT(size_of_block, block_size, ndim);
    avail_nblocks = nstore / size_of_block;

    for (i = 0; i < ndim; i++)
	nblocks[i] = 1;

    for (i = 0; i < nscripts; i++)
    {
	if ((scripts[i].ndim > 1) || (scripts[i].interlace))
	    break;

	j = scripts[i].dims[0];
	npoints_max[j] = MAX(npoints_max[j], scripts[i].npts[0]);
	nblocks[j] = 1 + (npoints_max[j] - 1) / block_size[j];
	VECTOR_PRODUCT(total_nblocks, nblocks, ndim);

	if (total_nblocks > avail_nblocks)
	    break;

/*  Following check just ensures that blocks not overwritten  */
/*  during processing of scripts.  The condition could be made  */
/*  much weaker, but the extra hassles are not worth the effort.  */
/*  Block overwriting only occurs if number of blocks on output is  */
/*  not equal to number of blocks on input, so could check this too.  */

	if ((file_info->input_file == file_info->output_file) &&
				non_consecutive_dims(ndim, i+1, scripts))
	    break;
    }

    if (i == 0)
    {
	if (scripts[0].interlace)
	{
	    i = 1;
	    *method = 3;
	}
	else
	{
	    for (i = 1; i < nscripts; i++)
	    {
		if (scripts[i].ndim != scripts[0].ndim)
		    break;

		if (not_same_vectors(scripts[0].ndim, scripts[0].dims,
							scripts[i].dims))
		    break;
	    }

	    *method = 2;
	}
    }
    else
    {
	*method = 1;
    }

    ndone = i;
    *have_output = FALSE;
    for (i = 0; i < ndone; i++)
    {
	if (scripts[i].output)
	{
	    *have_output = TRUE;
	    break;
	}
    }

    return  ndone;
}
