#include "script.h"

#include "avance.h"
#include "command.h"
#include "maxent.h"
#include "par.h"
#include "parse.h"
#include "ref.h"

#define  MAX_NSCRIPTS  20

static int nscripts;
static Script *scripts = NULL;

static int ndim;
static int *block_size;
static int *npoints;
static int npoints_temp[MAX_NDIM];
static int data_type[MAX_NDIM];

static int dims[MAX_NDIM];
static int npts_in[MAX_NDIM];
static int npts_out[MAX_NDIM];

static Ref_info *ref;

static char *input_file;
static char *output_file = NULL;
static char *par_file = NULL;
static Bool swapped;
static Bool integer;
static Bool blocked;
static int header;
static int byte_size;

static String script_file;

static Bool input_found;

static Bool interlace_found;

static Varian_info varian_info;

static int current_dim;

static Bool excluded_dim[MAX_NDIM];

static int param_dim;
static  float *params;

static int parse_string[] = { PARSE_STRING };
static int parse_int[] = { PARSE_INT };
static int parse_int_string[] = { PARSE_INT, PARSE_STRING };
static int parse_int2_string[] = { PARSE_INT, PARSE_INT, PARSE_STRING };
static int parse_int3_string[] = { PARSE_INT, PARSE_INT, PARSE_INT,
							PARSE_STRING };
static int parse_int_file[] = { PARSE_INT, PARSE_FILE };
static int parse_int2_file[] = { PARSE_INT, PARSE_INT, PARSE_FILE };
static int parse_int3_file[] = { PARSE_INT, PARSE_INT, PARSE_INT,
							PARSE_FILE };

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				script_file, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			script_file, string1, string2);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				script_file, string);  return  ERROR;   }

#define  CHECK_NCOMMANDS \
	 {   if (scripts[nscripts].ncommands > 0) \
	     {   nscripts++;   }   }

#define  CHECK_NSCRIPTS \
	 {   if (nscripts >= MAX_NSCRIPTS) \
	     {   sprintf(error_msg, "too many scripts"); \
		 return  ERROR;   }   }

static Status input_parse(Generic_ptr *var, String error_msg)
{
    int i;
    String input_par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	FOUND_TWICE("input");

    input_found = TRUE;

    set_varian();
    CHECK_STATUS(read_par_file(input_par_file, &par_info, error_msg));
    get_varian(&varian_info);

    input_file = par_info.file;
    ndim = par_info.ndim;
    npoints = par_info.npoints;
    block_size = par_info.block_size;
    ref = par_info.ref;
    swapped = par_info.swapped;
    integer = par_info.integer;
    blocked = par_info.blocked;
    header = par_info.header;
    byte_size = par_info.byte_size;

    if (par_info.deflated)
    {
	sprintf(error_msg, "input file \"%s\" must not be deflated",
								input_file);
	return  ERROR;
    }

    COPY_VECTOR(npoints_temp, npoints, ndim);

    if (blocked)
    {
	if (varian_info.have_varian)
	{
	    sprintf(error_msg, "input file \"%s\" cannot be varian and blocked",
								input_file);
	    return  ERROR;
	}
    }
    else
    {
	scripts[nscripts].ndim = 1;
	scripts[nscripts].dims[0] = 0;
	scripts[nscripts].npts[0] = npoints[0];
	scripts[nscripts].npts_max = 0;
	scripts[nscripts].output = TRUE;
	scripts[nscripts].interlace = FALSE;
	scripts[nscripts].ncommands = 1;
	scripts[nscripts].commands[0].code = 0;
	scripts[nscripts].commands[0].do_process = (Do_process) NULL;
	strcpy(scripts[nscripts].commands[0].msg, "blocking");

	nscripts++;
    }

    for (i = 0; i < ndim; i++)
	data_type[i] = REAL_DATA;

    param_dim = par_info.param_dim;
    params = par_info.params;

    return  OK;
}

static Status output_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output", "input");

    if (output_file)
	FOUND_TWICE("output");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(output_file, char, strlen(name)+1);

    strcpy(output_file, name);

    return  OK;
}

static Status par_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("par", "input");

    if (par_file)
	FOUND_TWICE("par");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(par_file, char, strlen(name)+1);

    strcpy(par_file, name);

    return  OK;
}

static Status start_script(int *dim, int *n, String msg, String error_msg)
{
    if (!output_file)
	FOUND_BEFORE(msg, "output");

    if (*dim > ndim)
    {
	sprintf(error_msg, "in \"%s\" have '%s %d' but 'ndim' = %d",
						msg, script_file, *dim, ndim);
	return  ERROR;
    }

    if (*dim < 1)
    {
	sprintf(error_msg, "in \"%s\" have '%s %d'", script_file, msg, *dim);
	return  ERROR;
    }

    (*dim)--;
    *n = npoints_temp[*dim];

    CHECK_NSCRIPTS;

    scripts[nscripts].ndim = 1;
    scripts[nscripts].dims[0] = *dim;
    scripts[nscripts].output = TRUE;  /* this could change some day! */
    scripts[nscripts].interlace = FALSE;

    current_dim = *dim;

    return  OK;
}

static Status end_script(int dim, int n, int nmax, String error_msg)
{
    npoints_temp[dim] = n;

    scripts[nscripts].npts[0] = n;
    scripts[nscripts].npts_max = nmax;

    CHECK_NCOMMANDS;

    CHECK_STATUS(check_avance(error_msg));

    return  OK;
}

static Bool dim_excluded(int dim)
{
    if (!output_file) /* error which will be caught later */
	return  FALSE;

    if ((dim <= 0) || (dim > MAX_NDIM)) /* error which will be caught later */
	return  FALSE;

    dim--;

    return excluded_dim[dim];
}

static Status script_parse(Generic_ptr *var, String error_msg)
{
    int dim = *((int *) var[0]);
    char *file = (char *) var[1];
    int n, nmax;
    Line err_msg;

    if (dim_excluded(dim))
	return  OK;

    CHECK_STATUS(start_script(&dim, &n, "script", error_msg));

    if (read_command_file(&n, &nmax, data_type+dim, file,
				scripts+nscripts, err_msg) == ERROR)
    {
	sprintf(error_msg, "script for dimension %d, file '%s': %s",
						dim+1, file, err_msg);
	return  ERROR;
    }

    CHECK_STATUS(end_script(dim, n, nmax, error_msg));

    return  OK;
}

static Status script_com_parse(Generic_ptr *var, String error_msg)
{
    int dim = *((int *) var[0]);
    FILE *fp = (FILE *) var[1];
    int n, nmax;
    Line file, err_msg;

    if (dim_excluded(dim))
	return skip_command_subfile(fp, error_msg);

    CHECK_STATUS(start_script(&dim, &n, "script_com", error_msg));

    sprintf(file, "%s script_com", script_file);

    if (read_command_subfile(&n, &nmax, data_type+dim, file, fp,
				scripts+nscripts, err_msg) == ERROR)
    {
	sprintf(error_msg, "script_com for dimension %d: %s", dim+1, err_msg);
	return  ERROR;
    }

    CHECK_STATUS(end_script(dim, n, nmax, error_msg));

    return  OK;
}

static Status start_maxent(int n, Generic_ptr *var, int *type,
					String max_msg, String error_msg)
{
    int i;
    char *cmd_msg = scripts[nscripts].commands[0].msg;

    if (!output_file)
	FOUND_BEFORE(max_msg, "output");

    for (i = 0; i < n; i++)
    {
	dims[i] = *((int *) var[i]);

	if (dims[i] > ndim)
	{
	    sprintf(error_msg,
		"in \"%s\" have '%s' with dim #%d = %d, but 'ndim' = %d",
				script_file, max_msg, i+1, dims[i], ndim);
	    return  ERROR;
	}

	if (dims[i] < 1)
	{
	    sprintf(error_msg, "in \"%s\" have '%s' with dim #%d = %d < 1",
				script_file, max_msg, i+1, dims[i]);
	    return  ERROR;
	}

	dims[i]--;
    }

    strcpy(cmd_msg, max_msg);
    cmd_msg += strlen(cmd_msg);

    for (i = 0; i < n; i++)
    {
	sprintf(cmd_msg, " %d", dims[i]+1);
	cmd_msg += strlen(cmd_msg);
    }

    CHECK_NSCRIPTS;

    scripts[nscripts].ndim = n;
    scripts[nscripts].output = TRUE;
    scripts[nscripts].interlace = FALSE;
    scripts[nscripts].ncommands = 1;
    scripts[nscripts].commands[0].code = 0;
    scripts[nscripts].commands[0].do_process = do_maxent;

    for (i = 0; i < n; i++)
    {
	npts_in[i] = npoints_temp[dims[i]];
	scripts[nscripts].dims[i] = dims[i];
	type[i] = data_type[dims[i]];
    }

    return  OK;
}

static void end_maxent(int n, int *type)
{
    int i;

    scripts[nscripts].npts_max = 1;

    for (i = 0; i < n; i++)
    {
	data_type[dims[i]] = type[i];
    	npoints_temp[dims[i]] = scripts[nscripts].npts[i] = npts_out[i];
    	scripts[nscripts].npts_max *= MAX(npts_in[i], npts_out[i]);
    }

    CHECK_NCOMMANDS;
}

static Status maxents_parse(int n, Generic_ptr *var, String max_msg,
							String error_msg)
{
    int type[MAX_NDIM];
    char *ptr, *file = (char *) var[n];
    char *cmd_msg = scripts[nscripts].commands[0].msg;

    CHECK_STATUS(start_maxent(n, var, type, max_msg, error_msg));

    /* below is just protection against long file names */
    if (ptr = strrchr(file, DIRECTORY_SYMBOL))
	ptr++;
    else
	ptr = file;

    strcat(cmd_msg, " ");
    strcat(cmd_msg, ptr);

    CHECK_STATUS(setup_maxents(n, type, npts_in, npts_out, file, error_msg));

    end_maxent(n, type);

    return  OK;
}

static Status maxents_com_parse(int n, Generic_ptr *var, String max_msg,
							String error_msg)
{
    int type[MAX_NDIM];
    FILE *fp = (FILE *) var[n];

    CHECK_STATUS(start_maxent(n, var, type, max_msg, error_msg));

    CHECK_STATUS(setup_maxents_com(n, type, npts_in, npts_out, fp, error_msg));

    end_maxent(n, type);

    return  OK;
}

static Status maxent_parse(Generic_ptr *var, String error_msg)
{
    return  maxents_parse(1, var, "maxent", error_msg);
}

static Status maxent2_parse(Generic_ptr *var, String error_msg)
{
    return  maxents_parse(2, var, "maxent2", error_msg);
}

static Status maxent3_parse(Generic_ptr *var, String error_msg)
{
    return  maxents_parse(3, var, "maxent3", error_msg);
}

static Status maxent_com_parse(Generic_ptr *var, String error_msg)
{
    return  maxents_com_parse(1, var, "maxent_com", error_msg);
}

static Status maxent2_com_parse(Generic_ptr *var, String error_msg)
{
    return  maxents_com_parse(2, var, "maxent2_com", error_msg);
}

static Status maxent3_com_parse(Generic_ptr *var, String error_msg)
{
    return  maxents_com_parse(3, var, "maxent3_com", error_msg);
}

static Status interlace_parse(Generic_ptr *var, String error_msg)
{
    int dim = *((int *) var[0]);

    if (!output_file)
	FOUND_BEFORE("interlace", "output");

    if (interlace_found)
	FOUND_TWICE("interlace");

    interlace_found = TRUE;

    if (dim > ndim)
    {
	sprintf(error_msg,
		"in \"%s\" have 'interlace' with dim = %d, but 'ndim' = %d",
						script_file, dim, ndim);
	return  ERROR;
    }

    if (dim < 2)  /* it is implicit that dim = 1 is other interlace dim. */
    {
	sprintf(error_msg,
		"in \"%s\" have 'interlace' with dim = %d < 2",
							script_file, dim);
	return  ERROR;
    }

    dim--;

    CHECK_NSCRIPTS;

    scripts[nscripts].ndim = 2;
    scripts[nscripts].dims[0] = 0;
    scripts[nscripts].dims[1] = dim;
    scripts[nscripts].npts[0] = npoints[0]; /* probably not needed */
    scripts[nscripts].npts[1] = npoints[dim];
    scripts[nscripts].npts_max = 0;
    scripts[nscripts].output = TRUE;
    scripts[nscripts].interlace = TRUE;
    scripts[nscripts].ncommands = 1;
    scripts[nscripts].commands[0].code = 0;
    strcpy(scripts[nscripts].commands[0].msg, "interlace");
    scripts[nscripts].commands[0].do_process = (Do_process) NULL;

    CHECK_NCOMMANDS;

    return  OK;
}

static Status exclude_parse(Generic_ptr *var, String error_msg)
{
    int dim = *((int *) var[0]);
    Line msg;

    if (!output_file)
	FOUND_BEFORE("exclude", "output");

    if ((dim <= 0) || (dim > MAX_NDIM))
    {
	sprintf(error_msg,
	  "in \"%s\" have 'exclude' with dim = %d, should be in range 1 to %d",
						script_file, dim, MAX_NDIM);
	return  ERROR;
    }

    dim--;

    if (excluded_dim[dim])
    {
	sprintf(msg, "exclude %d", dim+1);
	FOUND_TWICE(msg);
    }

    excluded_dim[dim] = TRUE;

    printf("Warning: excluding scripts in dim %d\n", dim+1);

    return  OK;
}

static Parse_line script_file_table[] =
{
    { "input",		1,	parse_string,		input_parse },
    { "output",		1,	parse_string,		output_parse },
    { "par",		1,	parse_string,		par_parse },
    { "script",		2,	parse_int_string,	script_parse },
    { "maxent",		2,	parse_int_string,	maxent_parse },
    { "maxent2",	3,	parse_int2_string,	maxent2_parse },
    { "maxent3",	4,	parse_int3_string,	maxent3_parse },
    { "script_com",	2,	parse_int_file,		script_com_parse },
    { "maxent_com",	2,	parse_int_file,		maxent_com_parse },
    { "maxent2_com",	3,	parse_int2_file,	maxent2_com_parse },
    { "maxent3_com",	4,	parse_int3_file,	maxent3_com_parse },
    { "interlace",	1,	parse_int,		interlace_parse },
    { "exclude",	1,	parse_int,		exclude_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

Status read_script_file(String file, Size_info *size_info, Ref_info **p_ref,
			File_info *file_info, int *p_nscripts,
			Script **p_scripts, String error_msg)
{
    int i;

    script_file = file;

    nscripts = 0;
    FREE(output_file, char);
    FREE(par_file, char);
    input_found = FALSE;
    interlace_found = FALSE;

    for (i = 0; i < MAX_NDIM; i++)
	excluded_dim[i] = FALSE;

    FREE(scripts, Script);

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(scripts, Script, MAX_NSCRIPTS);

    CHECK_STATUS(parse_file(script_file, script_file_table, TRUE, error_msg));

    size_info->ndim = ndim;
    size_info->block_size = block_size;
    size_info->npoints = npoints;

    *p_ref = ref;

    file_info->input_file = input_file;
    file_info->output_file = output_file;
    file_info->par_file = par_file;
    file_info->swapped = swapped;
    file_info->integer = integer;
    file_info->blocked = blocked;
    file_info->header = header;

/*  Varian stuff  */
    file_info->have_varian = varian_info.have_varian;
    file_info->dim_order = varian_info.dim_order;
    file_info->file_header = varian_info.file_header;
    file_info->block_header = varian_info.block_header;
    file_info->byte_size = byte_size;

    *p_scripts = scripts;
    *p_nscripts = nscripts;

    return  OK;
}

Ref_info *get_reference()
{
    return  (ref + current_dim);
}

void get_params(int *p_param_dim, float **p_params)
{
    *p_param_dim = param_dim;
    *p_params = params;
}
