#include "phase.h"

#include "command.h"

static int ncodes = 0;
static int npoints[MAX_NCODES];
static float *cos_mult[MAX_NCODES];
static float *sin_mult[MAX_NCODES];

void do_phase(int code, float *data)
{
    int i, j, k;
    float d;
    float *c = cos_mult[code];
    float *s = sin_mult[code];

    for (i = 0, j = 0, k = 1; i < npoints[code]; i++, j += 2, k += 2)
    {
	d = c[i]*data[j] + s[i]*data[k];
	data[k] = - s[i]*data[j] + c[i]*data[k];
	data[j] = d;
    }
}

static Status check_allocation(int code)
{
    MALLOC(cos_mult[code], float, npoints[code]);
    MALLOC(sin_mult[code], float, npoints[code]);

    return  OK;
}

static void setup_phase(int code, float ph0, float ph1, float piv)
{
    int i;
    float angle;

    ph0 *= RADIAN_SCALE;
    ph1 *= RADIAN_SCALE / ((float) (npoints[code]));
    piv -= 1;

    for (i = 0; i < npoints[code]; i++)
    {
	angle = ph0 + (i-piv) * ph1;
	cos_mult[code][i] = cos(angle);
	sin_mult[code][i] = sin(angle);
    }
}

Status init_phase(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float ph0, ph1, piv = 1;
    Line msg;

    ph0 = *((float *) param[0]);
    ph1 = *((float *) param[1]);

    sprintf(msg, "phase %3.2f %3.2f", ph0, ph1);
    if (setup_command(&type, &npts, ncodes, msg, do_phase, error_msg)
								== ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'phase': must have complex data");

    npoints[ncodes] = npts / 2;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'phase': allocating memory");

    setup_phase(ncodes, ph0, ph1, piv);

    CHECK_STATUS(end_command(type, npts, "phase", error_msg));

    ncodes++;

    return  OK;
}

Status init_phase2(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float ph0, ph1, piv;
    Line msg;

    ph0 = *((float *) param[0]);
    ph1 = *((float *) param[1]);
    piv = *((float *) param[2]);

    sprintf(msg, "phase2 %3.2ef %3.2f %3.2f", ph0, ph1, piv);
    if (setup_command(&type, &npts, ncodes, msg, do_phase, error_msg)
								== ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'phase2': must have complex data");

    npoints[ncodes] = npts / 2;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'phase2': allocating memory");

    setup_phase(ncodes, ph0, ph1, piv);

    CHECK_STATUS(end_command(type, npts, "phase2", error_msg));

    ncodes++;

    return  OK;
}

Status init_phase_max(int *code, int npts, Generic_ptr *param, String error_msg)
{
    float ph0, ph1, piv = 1;

    if ((ncodes+2) > MAX_NCODES)
        RETURN_ERROR_MSG("too many 'phase' commands to be able to do 'maxent'");

    ph0 = *((float *) param[0]);
    ph1 = *((float *) param[1]);

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
        RETURN_ERROR_MSG("'maxent': allocating memory for 'phase'");

    setup_phase(ncodes, ph0, ph1, piv);

    npoints[ncodes+1] = npts;

    if (check_allocation(ncodes+1) == ERROR)
        RETURN_ERROR_MSG("'maxent': allocating memory for 'phase'");

    setup_phase(ncodes+1, -ph0, -ph1, piv);

    *code = ncodes;
    ncodes += 2;

    return  OK;
}

Status init_phase2_max(int *code, int npts, Generic_ptr *param,
							String error_msg)
{
    float ph0, ph1, piv;

    if ((ncodes+2) > MAX_NCODES)
        RETURN_ERROR_MSG("too many 'phase2' commands to be able to do 'maxent'");

    ph0 = *((float *) param[0]);
    ph1 = *((float *) param[1]);
    piv = *((float *) param[2]);

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
        RETURN_ERROR_MSG("'maxent': allocating memory for 'phase2'");

    setup_phase(ncodes, ph0, ph1, piv);

    npoints[ncodes+1] = npts;

    if (check_allocation(ncodes+1) == ERROR)
        RETURN_ERROR_MSG("'maxent': allocating memory for 'phase2'");

    setup_phase(ncodes+1, -ph0, -ph1, piv);

    *code = ncodes;
    ncodes += 2;

    return  OK;
}

Status init_phase_avance(int *code, int npts, float ph0, float ph1,
						String error_msg)
{
    float piv = 1;

    if (ncodes >= MAX_NCODES)
        RETURN_ERROR_MSG("too many 'phase' commands to be able to do 'avance'");

    npoints[ncodes] = npts / 2;

    if (check_allocation(ncodes) == ERROR)
        RETURN_ERROR_MSG("'avance': allocating memory for 'phase'");

    setup_phase(ncodes, ph0, ph1, piv);

    *code = ncodes;
    ncodes++;

    return  OK;
}
