#include "opus.h"

#include "fft.h"
#include "phase.h"

static int *npoints_in;
static int *npoints_out;
static int *sample;
static int **sample_list;
static int *ignore;
static int **ignore_list;
static float *work_space = NULL;
static float *decay;
static Bool *have_phase;
static Bool *have_decay;
static int *phase_code;
static int *fft_code;
static int dim;

static void do_expand(int n1, int n2, int *list, float *d1, float *d2)
{
    int i, j, k;

    if (n1 == n2)
    {
	if (d1 != d2)
	    COPY_VECTOR(d1, d2, n1);

	return;
    }

    for (i = n2-1; i >= 0; i--)
	d1[list[i]] = d2[i];

    for (i = 0; i < n2; i++)
    {
	j = list[i] + 1;

	if (i == (n2-1))
	    k = n2 - j;
	else
	    k = list[i+1] - j;

	ZERO_VECTOR(d1+j, k);
    }

    ZERO_VECTOR(d1, list[0]);
}

static void do_contract(int n1, int n2, int *list, int m, int ntimes,
							float *d1, float *d2)
{
    int i, j, n;

    if (n1 == n2)
    {
	if (d1 != d2)
	{
	    n = n1 * ntimes;
	    COPY_VECTOR(d1, d2, n);
	}

	return;
    }

    for (i = 0; i < ntimes; i++)
    {
	for (j = 0; j < n1; j++)
	    d1[j] = d2[j%m+m*list[j/m]];

	d1 += n1;
	d2 += n2;
    }
}

static void ignore1(float *data)
{
    do_contract(npoints_in[0], npoints_in[0]+ignore[0],
					ignore_list[0], 1, 1, data, data);
}

static void ignore2(float *data)
{
    int n;

    n = npoints_in[1] + ignore[1];
    do_contract(npoints_in[0], npoints_in[0]+ignore[0],
					ignore_list[0], 1, n, data, data);

    do_contract(npoints_in[0]*npoints_in[1], npoints_in[0]*n,
				ignore_list[1], npoints_in[0], 1, data, data);
}

static void ignore3(float *data)
{
    int m, n;

    n = npoints_in[1] + ignore[1];
    n *= npoints_in[2] + ignore[2];

    do_contract(npoints_in[0], npoints_in[0]+ignore[0],
					ignore_list[0], 1, n, data, data);

    m = npoints_in[1] + ignore[1];
    n = npoints_in[2] + ignore[2];
    do_contract(npoints_in[0]*npoints_in[1], npoints_in[0]*m,
				ignore_list[1], npoints_in[0], n, data, data);

    m = npoints_in[0] * npoints_in[1];
    do_contract(m*npoints_in[2], m*n, ignore_list[2], m, 1, data, data);
}

static void opus(float *data_out, float *data_in)
{
    int i, n;
    float d;

    for (i = npoints_out[dim]-1; i >= 0; i--)	/* real part of image */
	work_space[2*i] = data_in[i];

    for (i = 0; i < npoints_out[dim]; i++)	/* imag part of image = 0 */
	work_space[2*i+1] = 0;

    if (have_phase[dim])
	do_phase(phase_code[dim]+1, work_space);

    do_fft_process(fft_code[dim]+1, work_space);

    n = sample[dim];

    if (have_decay[dim])
    {
	d = decay[dim];

	for (i = 2; i < n; i += 2)
	{
	    work_space[i] *= d;
	    work_space[i+1] *= d;
            d *= decay[dim];
	}
    }

    do_contract(npoints_in[dim], n, sample_list[dim], 1, 1,
						data_out, work_space);
}

static void tropus(float *data_out, float *data_in)
{
    int i, n;
    float d;

    n = sample[dim];

    do_expand(n, npoints_in[dim], sample_list[dim], work_space, data_in);

    if (have_decay[dim])
    {
	d = decay[dim];

	for (i = 2; i < n; i += 2)
	{
	    work_space[i] *= d;
	    work_space[i+1] *= d;
	    d *= decay[dim];
	}
    }

    ZERO_VECTOR(work_space+n, 2*npoints_out[dim]-n);
    do_fft_process(fft_code[dim], work_space);

    if (have_phase[dim])
	do_phase(phase_code[dim], work_space);

    for (i = 0; i < npoints_out[dim]; i++)
	data_out[i] = work_space[2*i];
}

static void opus1(float *data_out, float *data_in)
{
    dim = 0;
    opus(data_out, data_in);
}

static void tropus1(float *data_out, float *data_in)
{
    dim = 0;
    tropus(data_out, data_in);
}

static void get_1d_from_2d(float *data, int n1, int n2)
{
    int i;

    for (i = 0; i < n2; i++)
	work_space[i] = data[i*n1];
}

static void put_1d_into_2d(float *data, int n1, int n2)
{
    int i;

    for (i = 0; i < n2; i++)
	data[i*n1] = work_space[i];
}

static void opus2(float *data_out, float *data_in)
{
    int i;

    dim = 0;
    for (i = 0; i < npoints_out[1]; i++)
	opus(data_out+i*npoints_in[0], data_in+i*npoints_out[0]);

    dim = 1;
    for (i = 0; i < npoints_in[0]; i++)
    {
	get_1d_from_2d(data_out+i, npoints_in[0], npoints_out[1]);
	opus(work_space, work_space);
	put_1d_into_2d(data_out+i, npoints_in[0], npoints_in[1]);
    }
}

static void tropus2(float *data_out, float *data_in)
{
    int i;

    dim = 0;
    for (i = npoints_in[1]-1; i >= 0; i--)
	tropus(data_out+i*npoints_out[0], data_in+i*npoints_in[0]);

    dim = 1;
    for (i = 0; i < npoints_out[0]; i++)
    {
	get_1d_from_2d(data_out+i, npoints_out[0], npoints_in[1]);
	tropus(work_space, work_space);
	put_1d_into_2d(data_out+i, npoints_out[0], npoints_out[1]);
    }
}

static void opus3(float *data_out, float *data_in)
{
    int i, j, kin, kout;

    dim = 0;
    for (i = 0; i < npoints_out[1]*npoints_out[2]; i++)
	opus(data_out+i*npoints_in[0], data_in+i*npoints_out[0]);

    dim = 1;
    for (i = 0; i < npoints_out[2]; i++)
    {
	kin = i * npoints_in[0] * npoints_out[1];
	kout = i * npoints_in[0] * npoints_in[1];

	for (j = 0; j < npoints_in[0]; j++)
	{
	    get_1d_from_2d(data_out+kin+j, npoints_in[0], npoints_out[1]);
	    opus(work_space, work_space);
	    put_1d_into_2d(data_out+kout+j, npoints_in[0], npoints_in[1]);
	}
    }

    dim = 2;
    for (i = 0; i < npoints_in[0]*npoints_in[1]; i++)
    {
	get_1d_from_2d(data_out+i, npoints_in[0]*npoints_in[1], npoints_out[2]);
	opus(work_space, work_space);
	put_1d_into_2d(data_out+i, npoints_in[0]*npoints_in[1], npoints_in[2]);
    }
}

static void tropus3(float *data_out, float *data_in)
{
    int i, j, kin, kout;

    dim = 0;
    for (i = npoints_in[1]*npoints_in[2]-1; i >= 0; i--)
	tropus(data_out+i*npoints_out[0], data_in+i*npoints_in[0]);

    dim = 1;
    for (i = npoints_in[2]-1; i >= 0; i--)
    {
	kin = i * npoints_out[0] * npoints_in[1];
	kout = i * npoints_out[0] * npoints_out[1];

	for (j = npoints_out[0]-1; j >= 0; j--)
	{
	    get_1d_from_2d(data_out+kin+j, npoints_out[0], npoints_in[1]);
	    tropus(work_space, work_space);
	    put_1d_into_2d(data_out+kout+j, npoints_out[0], npoints_out[1]);
	}
    }

    dim = 2;
    for (i = 0; i < npoints_out[0]*npoints_out[1]; i++)
    {
	get_1d_from_2d(data_out+i, npoints_out[0]*npoints_out[1],
							npoints_in[2]);
	tropus(work_space, work_space);
	put_1d_into_2d(data_out+i, npoints_out[0]*npoints_out[1],
							npoints_out[2]);
    }
}

static Status check_opus_alloc(int n)
{
/*  could use store->row_data for ndim = 1, but hardly a major issue  */

    MALLOC(work_space, float, n);

    return  OK;
}

Status init_opus(int ndim, int *npts_in, int *npts_out,
		int *ign, int **ign_list, Ignore_func *ign_func,
		int *smp, int **smp_list, int *fft_cd,
		Bool *have_phs, int *phs_cd, Bool *have_dcy, float *dcy,
		Transform *op, Transform *trop, String error_msg)
{
    int i, n;

    if ((ndim < 1) || (ndim > 3))  /* this should never happen */
	RETURN_ERROR_MSG("'maxent': number of dims. must be 1, 2, or 3");

    npoints_in = npts_in;
    npoints_out = npts_out;
    ignore = ign;
    ignore_list = ign_list;
    sample = smp;
    sample_list = smp_list;
    fft_code = fft_cd;
    have_phase = have_phs;
    phase_code = phs_cd;
    have_decay = have_dcy;
    decay = dcy;

    n = 1;
    for (i = 0; i < ndim; i++)
	n = MAX(n, npoints_out[i]);

    if (check_opus_alloc(2*n) == ERROR)
	RETURN_ERROR_MSG("'maxent': allocating memory");

    if (ndim == 1)
    {
	*op = opus1;
	*trop = tropus1;
	*ign_func = ignore1;
    }
    else if (ndim == 2)
    {
	*op = opus2;
	*trop = tropus2;
	*ign_func = ignore2;
    }
    else if (ndim == 3)
    {
	*op = opus3;
	*trop = tropus3;
	*ign_func = ignore3;
    }

    return  OK;
}
