#include "lp.h"

#include "lin_pred.h"
#include "svd.h"

void lp_fit(float *data, int npoints, int npred, int step, int npoles,
				float *w1, float *w2, float *w3, float *w4)
{
    int n;

    if (step == 1) /* real forward */
    {
	lin_pred(data, npoints, npred, 1, npoles, w1, w2, w3, w4);
    }
    else if (step == -1) /* real backward */
    {
	n = npoints + npred - 1;
	lin_pred(data+n, npoints, npred, -1, npoles, w1, w2, w3, w4);
    }
    else if (step == 2) /* complex forward */
    {
	lin_pred(data, npoints, npred, 2, npoles, w1, w2, w3, w4);
	lin_pred(data+1, npoints, npred, 2, npoles, w1, w2, w3, w4);
    }
    else if (step == -2) /* complex backward */
    {
	n = 2 * (npoints + npred - 1);
	lin_pred(data+n, npoints, npred, -2, npoles, w1, w2, w3, w4);
	lin_pred(data+n+1, npoints, npred, -2, npoles, w1, w2, w3, w4);
    }
}

void lp_first(float *data, int length, int number, int points,
		int step, float cutoff, float **u, float **v,
		float *w, float *x, float *d, float *t)
{
    int i, j, k, m;
    float s;
    Bool converged;

    for (i = 0; i < step; i++)
    {
	for (j = 0; j < number; j++)
	    for (k = 0; k < length; k++)
		u[j][k] = data[i + step*(j+k+points+1)];

	svd(u, v, w, t, number, length, &converged);

	if (converged)
	{
	    for (j = 0; j < number; j++)
		d[j] = data[i + step*(j+points)];

	    svd_fit(u, v, w, x, d, t, cutoff, number, length);
	}
	else /* !converged */
	{
	    ZERO_VECTOR(x, length);
	}

	for (m = points-1; m >= 0; m--)
	{
	    s = 0;
	    for (k = 0; k < length; k++)
		s += x[k] * data[i + step*(m+k+1)];

	    data[i + step*m] = s;
	}
    }
}

void lp_first2(float *data, int length, int number, int points,
		float cutoff, float **u, float **v,
		float *w, float *x, float *d, float *t)
{
    int j, k, m, p;
    float sr, si;
    Bool converged;

    for (j = 0; j < number; j++)
    {
	for (k = 0; k < length; k++)
	{
	    p = 2 * (j + k + points + 1);
	    u[j][k] = data[p];
	    u[j][length+k] = -data[p+1];
	    u[number+j][k] = data[p+1];
	    u[number+j][length+k] = data[p];
	}
    }

    svd(u, v, w, t, 2*number, 2*length, &converged);

    if (converged)
    {
	for (j = 0; j < number; j++)
	{
	    p = 2 * (j + points);
	    d[j] = data[p];
	    d[number+j] = data[p+1];
	}

	svd_fit(u, v, w, x, d, t, cutoff, 2*number, 2*length);
    }
    else /* !converged */
    {
	ZERO_VECTOR(x, 2*length);
    }

    for (m = points-1; m >= 0; m--)
    {
	sr = si = 0;
	for (k = 0; k < length; k++)
	{
	    p = 2 * (m + k + 1);
	    sr += x[k]*data[p] - x[length+k]*data[p+1];
	    si += x[length+k]*data[p] + x[k]*data[p+1];
	}

	p = 2 * m;
	data[p] = sr;
	data[p+1] = si;
    }
}

void lp_last(float *data, int length, int number, int points,
		int step, int npoints, float cutoff, float **u,
		float **v, float *w, float *x, float *d, float *t)
{
    int i, j, k, m;
    float s;
    Bool converged;

    for (i = 0; i < step; i++)
    {
	for (j = 0; j < number; j++)
	    for (k = 0; k < length; k++)
		u[j][k] = data[i + step*(npoints-(j+k+2))];

	svd(u, v, w, t, number, length, &converged);

	if (converged)
	{
	    for (j = 0; j < number; j++)
		d[j] = data[i + step*(npoints-(j+1))];

	    svd_fit(u, v, w, x, d, t, cutoff, number, length);
	}
	else /* !converged */
	{
	    ZERO_VECTOR(x, length);
	}

	for (m = 0; m < points; m++)
	{
	    s = 0;
	    for (k = 0; k < length; k++)
		s += x[k] * data[i + step*(npoints+m-(k+1))];

	    data[i + step*(npoints+m)] = s;
	}
    }
}

void lp_last2(float *data, int length, int number, int points,
		int npoints, float cutoff, float **u, float **v,
		float *w, float *x, float *d, float *t)
{
    int j, k, m, p;
    float sr, si;
    Bool converged;

    for (j = 0; j < number; j++)
    {
	for (k = 0; k < length; k++)
	{
	    p = 2 * (npoints - (j + k + 2));
	    u[j][k] = data[p];
	    u[j][length+k] = -data[p+1];
	    u[number+j][k] = data[p+1];
	    u[number+j][length+k] = data[p];
	}
    }

    svd(u, v, w, t, 2*number, 2*length, &converged);

    if (converged)
    {
	for (j = 0; j < number; j++)
	{
	    p = 2 * (npoints - (j + 1));
	    d[j] = data[p];
	    d[number+j] = data[p+1];
	}

	svd_fit(u, v, w, x, d, t, cutoff, 2*number, 2*length);
    }
    else /* !converged */
    {
	ZERO_VECTOR(x, 2*length);
    }

    for (m = 0; m < points; m++)
    {
	sr = si = 0;
	for (k = 0; k < length; k++)
	{
	    p = 2 * (npoints + m - (k + 1));
	    sr += x[k]*data[p] - x[length+k]*data[p+1];
	    si += x[length+k]*data[p] + x[k]*data[p+1];
	}

	p = 2 * (npoints + m);
	data[p] = sr;
	data[p+1] = si;
    }
}
