#include "avance.h"

#include "avance_param.h"
#include "command.h"
#include "phase.h"

#define  DEFAULT_ANGLE  180.0

static int ncodes = 0;
static int lower[MAX_NCODES];
static int npoints[MAX_NCODES];
static int phase_code[MAX_NCODES];

static Bool have_avance = FALSE;

static int lookup, shift, phase;
static float ph0, ph1;

static void do_avance(int code, float *data)
{
    int low, n;

    low = lower[code];
    n = npoints[code];

    COPY_VECTOR(data, data+low, n);
}

static void do_avance_phase(int code, float *data)
{
    int c;

    c = phase_code[code];
    do_phase(c, data);
}

static Status calculate_phase(int dspfvs, int decim, float angle,
						String error_msg)
{
    Avance_param param;

    param.dspfvs = dspfvs;
    param.decim = decim;
    param.angle = angle;

    CHECK_STATUS(find_avance_param(&param, error_msg));

    lookup = param.lookup;
    shift = param.shift;
    phase = param.phase;
    ph0 = param.ph0;
    ph1 = param.ph1;

    return  OK;
}

static Status setup_avance(int code, int *npts, String error_msg)
{
    shift *= 2; /* number of real points to shift */

    if (*npts <= shift)
	RETURN_ERROR_MSG("not enough points to shift");

    *npts -= shift;

    lower[code] = shift;
    npoints[code] = *npts;

    return  OK;
}

Status init_avance(Generic_ptr *param, String error_msg)
{
    int type, npts, dspfvs, decim;
    float angle = DEFAULT_ANGLE;
    char *err;
    Line msg;

    if (have_avance)
	RETURN_ERROR_MSG("'avance': missing previous 'avance_phase'");

    dspfvs = *((int *) param[0]);
    decim = *((int *) param[1]);

    sprintf(error_msg, "'avance': ");
    err = error_msg + strlen(error_msg);

    CHECK_STATUS(calculate_phase(dspfvs, decim, angle, err));

    sprintf(msg, "avance %d %d (lookup=%d, phase=%d, shift=%d)",
				dspfvs, decim, lookup, phase, shift);

    if (setup_command(&type, &npts, ncodes, msg, do_avance, error_msg)
								== ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'avance': must have complex data");

    sprintf(error_msg, "'avance': ");
    err = error_msg + strlen(error_msg);

    CHECK_STATUS(setup_avance(ncodes, &npts, err));

    CHECK_STATUS(end_command(type, npts, "avance", error_msg));

    have_avance = TRUE;

    ncodes++;

    return  OK;
}

Status init_avance2(Generic_ptr *param, String error_msg)
{
    int type, npts, dspfvs, decim;
    float angle;
    char *err;
    Line msg;

    if (have_avance)
	RETURN_ERROR_MSG("'avance2': missing previous 'avance_phase'");

    dspfvs = *((int *) param[0]);
    decim = *((int *) param[1]);
    angle = *((float *) param[2]);

    sprintf(error_msg, "'avance2': ");
    err = error_msg + strlen(error_msg);

    CHECK_STATUS(calculate_phase(dspfvs, decim, angle, err));

    sprintf(msg, "avance2 %d %d %2.1f (lookup=%d, phase=%d, shift=%d)",
				dspfvs, decim, angle, lookup, phase, shift);

    if (setup_command(&type, &npts, ncodes, msg, do_avance, error_msg)
								== ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'avance2': must have complex data");

    sprintf(error_msg, "'avance2': ");
    err = error_msg + strlen(error_msg);

    CHECK_STATUS(setup_avance(ncodes, &npts, err));

    CHECK_STATUS(end_command(type, npts, "avance2", error_msg));

    have_avance = TRUE;

    ncodes++;

    return  OK;
}

Status init_avance_phase(Generic_ptr *param, String error_msg)
{
    int c, type, npts;
    Line msg;

    if (!have_avance)
	RETURN_ERROR_MSG("missing 'avance' (or 'avance2')");

    sprintf(msg, "avance_phase (ph0=%1.0f, ph1=%1.0f)", ph0, ph1);

    if (setup_command(&type, &npts, ncodes, msg, do_avance_phase, error_msg)
								== ERROR)
        return  ERROR;

    if (type != COMPLEX_DATA)
	RETURN_ERROR_MSG("'avance_phase': must have complex data");

    CHECK_STATUS(init_phase_avance(&c, npts, ph0, ph1, error_msg));
    phase_code[ncodes] = c;

    CHECK_STATUS(end_command(type, npts, "avance_phase", error_msg));

    have_avance = FALSE;

    ncodes++;

    return  OK;
}

Status check_avance(String error_msg)
{
    if (have_avance)
	RETURN_ERROR_MSG("script missing 'avance_phase'");

    return  OK;
}
