#include "script.h"

#include "param.h"
#include "stack.h"
#include "utility.h"

#define  SCRIPT_COMMENT		'!'

#define  LOOP_KEY		"loop"
#define  LOOP2_KEY		"loop2"
#define  END_LOOP_KEY		"end_loop"
#define  INCLUDE_KEY		"include"

#define  VARIABLE_CHAR		'$'

#define  MAX_VARIABLE		256 /* only allow one character variables */

#define  COMMANDS_ALLOC		50 /* how many commands to allocate at once */

#define  SMALL_NUMBER		(1.0e-4)

typedef struct
{
    String key;
    String value;
    Generic_ptr data;
    Command_func func;
}   Script_command;

typedef Line Variable;

/* changed loop current/end/delta from int to float 25 Oct 2001 */
/* added loop2, ndecimals and format 31 Oct 2001 */
typedef struct
{
    Variable *var;
    String first, last, step, ndecimals;
    float current, end, delta;
    int end_loop;
    char format[6];
}   Loop;

static Stack loop_stack;

static Bool variables_initialized = FALSE;
static Variable *variable_table[MAX_VARIABLE];

static int ncommands_alloc = 0;
static int ncommands;
static Script_command **script_command;

#define  DOTS_PER_LINE	50

Status print_command_func(String value, Generic_ptr data, String error_msg)
{
    printf("%s\n", value);

    return  OK;
}

Status sleep_command_func(String value, Generic_ptr data, String error_msg)
{
    int i, j, s = atoi(value);
    Line msg;

    if (s <= 0)
	return  OK;

    if (s <= 3)
    {
	sleep(s);
	return  OK;
    }

    sprintf(msg, "sleeping %d seconds: ", s);
    printf(msg);

    for (i = 0; i < s; i += DOTS_PER_LINE)
    {
	if (i > 0)
	{
	    for (j = 0; j < strlen(msg); j++)
		printf(" ");
	}
	    
	for (j = i; j < MIN(s, i+DOTS_PER_LINE); j++)
	{
	    printf(".");
	    sleep(1);
	}

	printf("\n");
    }

    return  OK;
}

Status no_command_func(String value, Generic_ptr data, String error_msg)
{
    return  OK;
}

/*  below assumes structure of Variable is of given form  */
static void set_variable_value(Variable *var, float x, String format)
{
    sprintf((String) var, format, x);
}

/*  below assumes structure of Variable is of given form  */
static Status do_variable(String value, Generic_ptr data, String error_msg)
{
    Variable *var = (Variable *) data;

    strncpy((String) var, value, LINE_SIZE);

    return  OK;
}

static Status do_parameter(String value, Generic_ptr data, String error_msg)
{
    strncpy((String) data, value, LINE_SIZE);

    return  OK;
}

static Bool key_is_variable(String key)
{
    if (key[0] == VARIABLE_CHAR)
	return  TRUE;

    return  FALSE;
}

static Bool key_is_action(String key, Script_action *table,
							Command_func *func)
{
    Script_action *p;

    for (p = table; p->key; p++)
    {
	if (equal_strings(key, p->key))
	{
	    *func = p->func;

	    return  TRUE;
	}
    }

    return  FALSE;
}

static Bool key_is_loop(String key)
{
    return  equal_strings(key, LOOP_KEY);
}

static Bool key_is_loop2(String key)
{
    return  equal_strings(key, LOOP2_KEY);
}

static Bool key_is_end_loop(String key)
{
    return  equal_strings(key, END_LOOP_KEY);
}

static Bool key_is_include(String key)
{
    return  equal_strings(key, INCLUDE_KEY);
}

static Status init_command(String key, String value, String error_msg)
{
    int n;
    Script_command *s;

    if (ncommands >= ncommands_alloc)
    {
	if (ncommands_alloc == 0)
	{
	    n = COMMANDS_ALLOC;
	    sprintf(error_msg, "allocating commands memory");
	    MALLOC(script_command, Script_command *, n);
	}
	else
	{
	    n = ncommands_alloc + COMMANDS_ALLOC;
	    sprintf(error_msg, "reallocating commands memory");
	    REALLOC(script_command, Script_command *, n);
	}

	ncommands_alloc = n;
    }

    sprintf(error_msg, "allocating commands memory");
    MALLOC(s, Script_command, 1);

    script_command[ncommands] = s;

    CHECK_STATUS(copy_string(&(s->key), key, error_msg));
    CHECK_STATUS(copy_string(&(s->value), value, error_msg));

    return  OK;
}

static Status init_variable(String key, Variable **p_var, String error_msg)
{
    unsigned int n = (unsigned int) key[1];
    Variable *var = variable_table[n];

    if (!var)
    {
	sprintf(error_msg, "allocating variable memory");

	MALLOC(var, Variable, 1);

	variable_table[n] = var;
    }

    *p_var = var;

    return  OK;
}

static Status setup_variable(String key, String value, String error_msg)
{
    Variable *var;

    CHECK_STATUS(init_variable(key, &var, error_msg));

    CHECK_STATUS(init_command(key, value, error_msg));

    script_command[ncommands]->data = (Generic_ptr) var;
    script_command[ncommands]->func = do_variable;

    ncommands++;

    return  OK;
}

static Status setup_parameter(String key, String value, String data,
							String error_msg)
{
    CHECK_STATUS(init_command(key, value, error_msg));

    script_command[ncommands]->data = (Generic_ptr) data;
    script_command[ncommands]->func = do_parameter;

    ncommands++;

    return  OK;
}

static Status setup_action(String key, String value, Command_func func,
							String error_msg)
{
    CHECK_STATUS(init_command(key, value, error_msg));

    script_command[ncommands]->data = (Generic_ptr) NULL;
    script_command[ncommands]->func = func;

    ncommands++;

    return  OK;
}

static Status init_loop(String value, Loop **p_loop,
				Bool have_decimals, String error_msg)
{
    Variable *var;
    Loop *loop;
    Line key, first, last, step, ndecimals;

    sprintf(error_msg, "allocating loop memory");
    MALLOC(loop, Loop, 1);

    if (have_decimals)
    {
	if (sscanf(value, "%s %s %s %s %s",
				key, first, last, step, ndecimals) != 5)
	    RETURN_ERROR_MSG("scanning five loop2 arguments");
    }
    else
    {
	if (sscanf(value, "%s %s %s %s", key, first, last, step) != 4)
	    RETURN_ERROR_MSG("scanning four loop arguments");

	sprintf(ndecimals, "0");
    }

    if (!key_is_variable(key))
    {
	sprintf(error_msg, "loop key '%s' must be variable", key);
	return  ERROR;
    }

    CHECK_STATUS(init_variable(key, &var, error_msg));

    CHECK_STATUS(copy_string(&(loop->first), first, error_msg));
    CHECK_STATUS(copy_string(&(loop->last), last, error_msg));
    CHECK_STATUS(copy_string(&(loop->step), step, error_msg));
    CHECK_STATUS(copy_string(&(loop->ndecimals), ndecimals, error_msg));

    loop->var = var;

    *p_loop = loop;

    return  OK;
}

static Status setup_loop(String key, String value,
				Bool have_decimals, String error_msg)
{
    Loop *loop;

    CHECK_STATUS(init_loop(value, &loop, have_decimals, error_msg));
    
    value[0] = 0;
    CHECK_STATUS(init_command(key, value, error_msg));

    script_command[ncommands]->data = (Generic_ptr) loop;
    script_command[ncommands]->func = (Command_func) NULL;

    if (push_stack(&loop_stack, (Generic_ptr) script_command[ncommands])
								== ERROR)
	RETURN_ERROR_MSG("pushing loop stack");

    ncommands++;

    return  OK;
}

static Status setup_end_loop(String error_msg)
{
    Generic_ptr p;
    Script_command *s;
    Loop *loop;

    if (pop_stack(&loop_stack, &p) == ERROR)
	RETURN_ERROR_MSG("no 'loop' matching 'end_loop'");

    s = (Script_command *) p;
    loop = (Loop *) (s->data);
    loop->end_loop = ncommands;

    return  OK;
}

static Status script_key_value(String key, String value,
				Script_action *table, String error_msg)
{
    Command_func func;
    String ptr, data;

    if (ptr = strchr(key, SCRIPT_COMMENT))
	*ptr = 0;

    if (!key[0])  /* key was a comment */
	return  OK;

    STRIP_LEADING_SPACE(value);

    if (ptr = strchr(value, SCRIPT_COMMENT))
	*ptr = 0;

    STRIP_TRAILING_SPACE(value);

    if (key_is_variable(key))
    {
	CHECK_STATUS(setup_variable(key, value, error_msg));
    }
    else if (pattern_is_parameter(key, &data))
    {
	CHECK_STATUS(setup_parameter(key, value, data, error_msg));
    }
    else if (key_is_action(key, table, &func))
    {
	CHECK_STATUS(setup_action(key, value, func, error_msg));
    }
    else if (key_is_loop(key))
    {
	CHECK_STATUS(setup_loop(key, value, FALSE, error_msg));
    }
    else if (key_is_loop2(key))
    {
	CHECK_STATUS(setup_loop(key, value, TRUE, error_msg));
    }
    else if (key_is_end_loop(key))
    {
	CHECK_STATUS(setup_end_loop(error_msg));
    }
    else if (key_is_include(key))
    {
	CHECK_STATUS(read_script_file(value, table, error_msg));
    }
    else
    {
	sprintf(error_msg, "unknown key word '%s'", key);
	return  ERROR;
    }

    return  OK;
}

static Status read_file_script(FILE *fp, Script_action *table,
							String error_msg)
{
    int n;
    String msg;
    Line value, key;

    for (n = 0; fscanf(fp, "%s", key) == 1; n++)
    {
	if (!fgets(value, LINE_SIZE, fp))
	    value[0] = 0; /* only happens if last line has no carriage return */

	sprintf(error_msg, "line #%d: ", n+1);
	msg = error_msg + strlen(error_msg);

	CHECK_STATUS(script_key_value(key, value, table, msg));
    }

    return  OK;
}

Status read_script_file(String file, Script_action *table, String error_msg)
{
    String ptr;
    FILE *fp;

    if (OPEN_FOR_READING(fp, file))
    {
	sprintf(error_msg, "opening script file \"%s\" for reading", file);
	return  ERROR;
    }

    if (ptr = strrchr(file, DIRECTORY_SYMBOL))
	ptr++;
    else
	ptr = file;

    sprintf(error_msg, "reading \"%s\": ", ptr);
    if(read_file_script(fp, table, error_msg+strlen(error_msg)) == ERROR)
    {
	fclose(fp);
	return  ERROR;
    }

    fclose(fp);

    return  OK;
}

/*  below assumes structure of Variable is of given form  */
static Status copy_variable_value(String *p1, String end1, String *p2,
						String end2, String error_msg)
{
    int len, l1, l2;
    unsigned int n = (unsigned int) (**p2);
    Variable *var = variable_table[n];

    if (var)
    {
	l1 = strlen((String) var);
	l2 = (int) (end1 - *p1);
	len = MIN(l1, l2);

	strncpy(*p1, (String) var, len);
	*p1 += len;
    }

    (*p2)++;

    return  OK;
}

static Status substitute_variables(String value1, String value2,
							String error_msg)
{
    String p1, p2, end1, end2;

    end1 = value1 + LINE_SIZE;
    end2 = value2 + strlen(value2);

    for (p1 = value1, p2 = value2; (p1 < end1) && (p2 < end2); )
    {
	if (key_is_variable(p2))
	{
	    p2++;
	    CHECK_STATUS(copy_variable_value(&p1, end1, &p2, end2, error_msg));
	}
	else
	{
	    *p1++ = *p2++;
	}
    }

    *p1 = 0;

    return  OK;
}

static Status do_command(Script_command *s, String error_msg)
{
    Line value;

    CHECK_STATUS(substitute_variables(value, s->value, error_msg));

    CHECK_STATUS((*s->func)(value, s->data, error_msg));

    return  OK;
}

static Status begin_loop(Loop *loop, String error_msg)
{
    int n;
    float x;
    Line value;

/*  sscanf's below are very loose check on syntax, for example
    there could be non-numeric characters after the float characters  */

    CHECK_STATUS(substitute_variables(value, loop->first, error_msg));

    if (sscanf(value, "%f", &x) != 1)
	RETURN_ERROR_MSG("cannot scan 'first' value");

    loop->current = x;

    CHECK_STATUS(substitute_variables(value, loop->last, error_msg));

    if (sscanf(value, "%f", &x) != 1)
	RETURN_ERROR_MSG("cannot scan 'last' value");

    loop->end = x;

    CHECK_STATUS(substitute_variables(value, loop->step, error_msg));

    if (sscanf(value, "%f", &x) != 1)
	RETURN_ERROR_MSG("cannot scan 'step' value");

    loop->delta = x;

    CHECK_STATUS(substitute_variables(value, loop->ndecimals, error_msg));

    if (sscanf(value, "%d", &n) != 1)
	RETURN_ERROR_MSG("cannot scan 'ndecimals' value");

    n = MIN(n, 4);
    n = MAX(n, 0);

    sprintf(loop->format, "%%.%df", n);

/*  below in particular forces loop->delta != 0  */
    if (((loop->end >= loop->current) && (loop->delta <= SMALL_NUMBER)) ||
	((loop->end <= loop->current) && (loop->delta >= -SMALL_NUMBER)))
    {
	sprintf(error_msg,
		"have infinite loop (first=%1.2e, last=%1.2e, step=%1.2e)",
				loop->current, loop->end, loop->delta);
	return  ERROR;
    }

    return  OK;
}

static Bool continue_loop(Loop *loop)
{
    set_variable_value(loop->var, loop->current, loop->format);

    if (loop->delta > 0)
    {
	if (loop->current <= loop->end)
	    return  TRUE;
    }
    else if (loop->delta < 0)  /* should not have loop->delta = 0, but ... */
    {
	if (loop->current >= loop->end)
	    return  TRUE;
    }

    return  FALSE;
}

static void increment_loop(Loop *loop)
{
    loop->current += loop->delta;
}

static Status run_script(int n0, int n1, String error_msg)
{
    int i;
    String msg, key;
    Loop *loop;

    for (i = n0; i < n1; i++)
    {
	sprintf(error_msg, "command #%d ('%s'): ", i+1, script_command[i]->key);
	msg = error_msg + strlen(error_msg);

	key = script_command[i]->key;

	if (key_is_loop(key) || key_is_loop2(key))
	{
	    loop = (Loop *) (script_command[i]->data);
	    CHECK_STATUS(begin_loop(loop, msg));

	    while (continue_loop(loop))
	    {
		CHECK_STATUS(run_script(i+1, loop->end_loop, error_msg));
		increment_loop(loop);
	    }

	    i = loop->end_loop - 1;  /* jump to end of loop */
	}
	else
	{
	    CHECK_STATUS(do_command(script_command[i], msg));
	}
    }

    return  OK;
}

static Status init_script(String error_msg)
{
    int i;

    ncommands = 0;

    sprintf(error_msg, "initializing loop stack");
    CHECK_STATUS(init_stack(&loop_stack));

    if (!variables_initialized)
    {
	for (i = 0; i < MAX_VARIABLE; i++)
	    variable_table[i] = (Variable *) NULL;

	variables_initialized = TRUE;
    }
    /* else might want to de-initialize variables, decide later */

    return  OK;
}

static int loops_left;

static void count_stack(Generic_ptr p)
{
    loops_left++;
}

static Status end_script(String error_msg)
{
    loops_left = 0;
    destroy_stack(&loop_stack, count_stack);

    if (loops_left > 0)
    {
	sprintf(error_msg, "%d 'loop' commands without matching 'end_loop'",
								loops_left);
	return  ERROR;
    }

    return  OK;
}

Status run_script_file(String file, Script_action *table, String error_msg)
{
    CHECK_STATUS(init_script(error_msg));
    CHECK_STATUS(read_script_file(file, table, error_msg));
    CHECK_STATUS(end_script(error_msg));
    CHECK_STATUS(run_script(0, ncommands, error_msg));

    return  OK;
}
