#include "rowcol_func.h"

#include "data.h"
#include "object.h"
#include "param.h"
#include "ref.h"
#include "rowcol.h"
#include "slice.h"

#define  RANGE_FMT  "%4.2f %4.2f"

static int ref_type = REF_POINTS;
static int point_ruler_type = VISIBILITY_ON;
static int point_axis_type = VISIBILITY_ON;
static int value_ruler_type = VISIBILITY_ON;
static int value_axis_type = VISIBILITY_ON;
static int row_col_type = ROW_SLICE;
static int new_add_type = NEW_SLICE;

static int current_rowcol_set = -1;
static int nrowcol_sets = 0;

static Bool have_valid_rowcol = FALSE;

static int npoints = 0;
static int npoints_orthog = 0;

static void update_rowcol_data()
{
    if (*rowcol_ref)
	ref_type = atoi(rowcol_ref);

    if (*point_ruler)
	point_ruler_type = atoi(point_ruler);

    if (*point_axis)
	point_axis_type = atoi(point_axis);

    if (*value_ruler)
	value_ruler_type = atoi(value_ruler);

    if (*value_axis)
	value_axis_type = atoi(value_axis);

    if (*row_col)
    {
	row_col_type = atoi(row_col);

	if (row_col_type != COLUMN_SLICE)  /* protection against abuse */
	    row_col_type = ROW_SLICE;
    }

    if (*new_add)
    {
	new_add_type = atoi(new_add);

	if (new_add_type != ADD_SLICE)  /* protection against abuse */
	    new_add_type = NEW_SLICE;
    }
}

void remove_rowcol_set_func(int rowcol_set)
{
    if (rowcol_set < nrowcol_sets)
    {
	if (current_rowcol_set == rowcol_set)
	    current_rowcol_set = -1;

	nrowcol_sets--;
    }
}

void register_rowcol_set_func(int rowcol_set)
{
    if (rowcol_set >= nrowcol_sets)
	nrowcol_sets++;
}

Status rowcol_apply_func(Slice_info *slice_info, Data_info *data_info,
							String error_msg)
{
    int npoints_orig, first, step;
    Rowcol_param rowcol_param;

    update_rowcol_data();

    have_valid_rowcol = FALSE;

    if (current_rowcol_set == -1)
	RETURN_ERROR_MSG("no row/column data set selected");

    data_info += current_rowcol_set;

    rowcol_param.row_col = row_col_type;
    rowcol_param.new_add = new_add_type;
    rowcol_param.rows_cols = rows_cols;
    rowcol_param.npoints = data_info->npoints;
    rowcol_param.plane_data = data_info->data;
    rowcol_param.point_range = point_range;
    rowcol_param.value_range = value_range;
    rowcol_param.lower = slice_info->lower;
    rowcol_param.upper = slice_info->upper;

    CHECK_STATUS(initialize_rowcol(&rowcol_param, error_msg));

    if (row_col_type == ROW_SLICE)
    {
	npoints = data_info->npoints[0];
	npoints_orthog = data_info->npoints[1];
	npoints_orig = data_info->npoints_orig[0];
	first = data_info->first[0];
	step = data_info->step[0];
    }
    else
    {
	npoints = data_info->npoints[1];
	npoints_orthog = data_info->npoints[0];
	npoints_orig = data_info->npoints_orig[1];
	first = data_info->first[1];
	step = data_info->step[1];
    }

    slice_info->npoints = npoints;
    slice_info->npoints_orig = npoints_orig;
    slice_info->first = first;
    slice_info->step = step;
    slice_info->data = rowcol_param.rowcol_data;
    slice_info->ref_type = ref_type;
    slice_info->ref = data_info->ref + row_col_type;
    slice_info->ruler[0] = (point_ruler_type == VISIBILITY_ON) ? TRUE : FALSE;
    slice_info->ruler[1] = (value_ruler_type == VISIBILITY_ON) ? TRUE : FALSE;
    slice_info->axis[0] = (point_axis_type == VISIBILITY_ON) ? TRUE : FALSE;
    slice_info->axis[1] = (value_axis_type == VISIBILITY_ON) ? TRUE : FALSE;

    have_valid_rowcol = TRUE;

    return  OK;
}

Status rowcol_point_stats(Print_funcs *print_funcs, int *size, int *point,
							String error_msg)
{
    float c, result[LIMITS_DIM], desired_point[DISPLAY_DIM];

    if (!have_valid_rowcol)
	return  OK;

    CHECK_STATUS(find_rowcol_range(result, point_range, "point", error_msg));

    check_orientation(ref_type, 1, result, result+1);

    c = point[0] * (result[1] - result[0]) / ((float) size[0]);
    desired_point[0] = result[0] + c;

    CHECK_STATUS(find_rowcol_range(result, value_range, "value", error_msg));

    c = point[1] * (result[1] - result[0]) / ((float) size[1]);
    desired_point[1] = result[0] + c;

    print_slice_point_stats(desired_point, print_funcs);

    return  OK;
}

void rowcol_limits_func(int *size, int *begin, int *end)
{
    *begin = 0;
    *size = *end = MIN(1, atoi(slice_width));
}

Status rowcol_stats_func(Print_funcs *print_funcs, Limits_func limits_func,
							String error_msg)
{
    int size, begin, end;
    float c, result[LIMITS_DIM];

    if (!have_valid_rowcol)
	return  OK;

    update_rowcol_data();

    CHECK_STATUS(find_rowcol_range(result, point_range, "point", error_msg));

    check_orientation(ref_type, 1, result, result+1);

    (*limits_func)(&size, &begin, &end);

    c = (result[1] - result[0]) / ((float) size);
    result[1] = c * end + result[0];
    result[0] += c * begin;

    print_slice_limits_stats(result, print_funcs);

    return  OK;
}

Status change_point_rowcol_func(int size, int begin, int end, int type,
							String error_msg)
{
    float c, result[LIMITS_DIM];

    if (begin == end)
	return  OK;

    CHECK_STATUS(find_rowcol_range(result, point_range, "point", error_msg));

    check_orientation(ref_type, 1, result, result+1);

    if (type == EXPAND_ROWCOL)
    {
	c = (result[1] - result[0]) / ((float) size);
	result[1] = c * end + result[0];
	result[0] += c * begin;
    }
    else  /* type == CONTRACT_ROWCOL */
    {
	c = (result[1] - result[0]) / ((float) (end - begin));
	result[1] = c * (size - begin) + result[0];
	result[0] -= c * begin;
    }

    check_orientation(ref_type, 1, result, result+1);

    sprintf(point_range, RANGE_FMT, result[0], result[1]);

    return  OK;
}

Status change_value_rowcol_func(float multiplier, String error_msg)
{
    float result[LIMITS_DIM];

    CHECK_STATUS(find_rowcol_range(result, value_range, "value", error_msg));

    sprintf(value_range, RANGE_FMT, multiplier*result[0], multiplier*result[1]);

    return  OK;
}

static void cycle_set(int cycle)
{
    if ((nrowcol_sets < 2) || (current_rowcol_set == -1))
	return;

    current_rowcol_set = (current_rowcol_set + nrowcol_sets + cycle)
								% nrowcol_sets;
}

void rowcol_next_set_func(int *rowcol_set)
{
    cycle_set(1);
    *rowcol_set = current_rowcol_set;
}

void rowcol_previous_set_func(int *rowcol_set)
{
    cycle_set(-1);
    *rowcol_set = current_rowcol_set;
}

static void cycle_point(int cycle)
{
    int n;

    if (!have_valid_rowcol || (sscanf(rows_cols, "%d", &n) != 1))
	return;

    n = (n - 1 + npoints_orthog + cycle) % npoints_orthog + 1;
    sprintf(rows_cols, "%d", n);
}

void rowcol_next_point_func()
{
    cycle_point(1);
}

void rowcol_previous_point_func()
{
    cycle_point(-1);
}

void rowcol_select_func(int rowcol_set)
{
    if ((rowcol_set < 0) || (rowcol_set >= nrowcol_sets) ||
					(current_rowcol_set == rowcol_set))
	current_rowcol_set = -1;
    else
	current_rowcol_set = rowcol_set;
}
