#include "ps.h"

#include "ps_strings.h"

#include "color.h"
#include "output.h"
#include "style.h"

#define  FMT	"%6.2f "

#define  PS_BORDER		"PS_border"
#define  PS_PAGE_SET		"PS_page_set"
#define  PS_OUTLINE_BOX		"PS_outline_box"

#define  X_BORDER_OFFSET	90.0 /* pts */
#define  Y_BORDER_OFFSET	30.0 /* pts */

static FILE *file;
static int ps_or_eps;
static float paper_width, paper_height;
static float plot_width, plot_height;
static float lower_x, lower_y;
static float ax, bx, ay, by;
static Bool dashed_lines;
static Bool have_graphics_save;

static float x_border_offset = X_BORDER_OFFSET;
static float y_border_offset = Y_BORDER_OFFSET;

static void set_paper_size(int size)
{
    float *s = get_paper_size(size);

    paper_width = s[0];
    paper_height = s[1];
}

static void set_orientation(int direction)
{
    if (direction == LANDSCAPE)
	SWAP(paper_width, paper_height, float);
}

static void initialize_ps(int type, int direction, String fonts)
{
    /* some of the fputs might need to be separate routines in future */

    if(type == PS_OUTPUT)
    {
    	fputs(ps_header, file);
	ps_or_eps = PS_OUTPUT;
    }
    else
    {
    	fputs(eps_header, file);
	ps_or_eps = EPS_OUTPUT;
    }

    fprintf(file, ps_comments, paper_width, paper_height, fonts);
    fputs(ps_prolog, file);
    fputs(ps_setup, file);
    fputs(ps_begin, file);

    fprintf(file, "%% Start %s\n", PS_PAGE_SET);

    if (direction == LANDSCAPE)
    {
	fprintf(file, "-90 PS_rotate\n");
	fprintf(file, FMT "0 PS_translate\n", - paper_width);
    }

    fprintf(file, "%% End %s\n", PS_PAGE_SET);
}

static void set_color_type(int color_type)
{
    if (color_type == BLACK_WHITE)
    {
	fprintf(file, "PS_black_white\n");
	dashed_lines = TRUE;
    }
    else
    {
	fprintf(file, "PS_color_gray\n");
	dashed_lines = FALSE;
    }
}

static void set_plot_size(float *size)
{
    plot_width = size[0];  plot_height = size[1];

    lower_x = (paper_width - plot_width) / 2;
    lower_y = (paper_height - plot_height) / 2;

    fprintf(file, FMT FMT "PS_translate\n", lower_x, lower_y);
}

void do_ps_border(Output_border *border, float x_size, float y_size,
					float x_offset, float y_offset)
{
    float x, y, u, v;

    if (border->use)
    {
	if (border->justification == JUSTIFICATION_RIGHT)
	{
	    x = x_size - x_offset;
	    u = 1.0;
	}
	else if (border->justification == JUSTIFICATION_CENTER)
	{
	    x = x_size / 2;
	    u = 0.5;
	}
	else /* (border->justification == JUSTIFICATION_LEFT) */
	{
	    x = x_offset;
	    u = 0.0;
	}

	if (border->edge == EDGE_BOTTOM)
	{
	    y = y_offset;
	    v = 0.0;
	}
	else /* (border->edge == EDGE_TOP) */
	{
	    y = y_size - y_offset;
	    v = 1.0;
	}

	set_ps_font(border->font_name, border->font_size);
	fprintf(file, "PS_set_black\n");

	ax = ay = 1;
	ps_text(border->text, x, y, u, v);
    }
}

void ps_border(Output_border *border, float x_size, float y_size,
					float x_offset, float y_offset)
{
    if (ps_or_eps == PS_OUTPUT)
	do_ps_border(border, x_size, y_size, x_offset, y_offset);
}

static void do_outline_box()
{
    fprintf(file, "%% Start %s\n", PS_OUTLINE_BOX);

    fprintf(file, "PS_set_black\n");
    fprintf(file, "0 0 " FMT FMT "PS_draw_rectangle\n",
						plot_width, plot_height);

    fprintf(file, "%% End %s\n", PS_OUTLINE_BOX);
}

void start_ps(Generic_ptr data)
{
    Ps_data *ps_data = (Ps_data *) data;
    Output_choices *choices = ps_data->choices;

    file = ps_data->file;

    set_paper_size(choices->paper_size);
    set_orientation(choices->paper_dirn);

    initialize_ps(ps_data->type, choices->paper_dirn, ps_data->fonts);

    fprintf(file, "%% Start %s\n", PS_BORDER);

    ps_border(choices->title, paper_width, paper_height,
					x_border_offset, y_border_offset);
    ps_border(choices->time_date, paper_width, paper_height,
					x_border_offset, y_border_offset);
    ps_border(choices->file_name, paper_width, paper_height,
					x_border_offset, y_border_offset);

    fprintf(file, "%% End %s\n", PS_BORDER);

    set_color_type(choices->output_color);
    set_plot_size(choices->plot_size);

    if (choices->outline_box)
	do_outline_box();

    have_graphics_save = FALSE;
}

void end_ps()
{
    if (have_graphics_save)
	fprintf(file, "PS_graphics_restore\n");

    fputs(ps_end, file);
    fputs(ps_eof, file);
}

void new_ps_range(float x0, float y0, float x1, float y1, Bool clip)
{
/*  restore first because do not have end_ps_range routine  */
/*  and this finishes off the previous new_ps_range  */

    if (have_graphics_save)
	fprintf(file, "PS_graphics_restore\n");

    fprintf(file, "PS_graphics_save\n");
    have_graphics_save = TRUE;

    if (clip)
	fprintf(file, "0 0 " FMT FMT "PS_rectangle_clip\n",
						plot_width, plot_height);

    ax = plot_width / (x1 - x0);  bx = - ax * x0;
    ay = plot_height / (y1 - y0);  by = - ay * y0;

    fprintf(file, FMT FMT "PS_translate\n", bx, by);

/*  scaling causes problems with fonts, think again about this later
    fprintf(file, FMT FMT "PS_scale\n", ax, ay);
*/
}

void ps_line(float x0, float y0, float x1, float y1)
{
    fprintf(file, FMT FMT FMT FMT "PS_draw_line\n", ax*x0, ay*y0, ax*x1, ay*y1);
/*
    fprintf(file, FMT FMT FMT FMT "PS_draw_line\n", x0, y0, x1, y1);
*/
}

#define  PS_TEXT_ESCAPE		'\\'
#define  PS_TEXT_START		'('
#define  PS_TEXT_END		')'

static void string_c_to_ps(String dest, String src)
{
    while (*src)
    {
	if ((*src == PS_TEXT_ESCAPE) || (*src == PS_TEXT_START)
						|| (*src == PS_TEXT_END))
	    *dest++ = PS_TEXT_ESCAPE;

	*dest++ = *src++;
    }

    *dest = 0;
}

void ps_text(String text, float x, float y, float a, float b)
{
    Line line;

    string_c_to_ps(line, text);
    fprintf(file, FMT FMT FMT FMT "(%s) PS_text\n", ax*x, ay*y, a, b, line);
}

void set_ps_color(int color)
{
    float r, g, b;

    get_color_rgb(color, &r, &g, &b);

    fprintf(file, FMT FMT FMT "PS_set_color\n", r, g, b);
}

void set_ps_font(String name, int size)
{
    fprintf(file, "/%s %d PS_select_font\n", name, size);
}

void set_ps_line_style(int line_style)
{
    if (dashed_lines)
    {
	if (line_style == NORMAL_LINE_STYLE)
	    fprintf(file, "PS_nondashed\n");
	else /* line_style == DASHED_LINE_STYLE */
	    fprintf(file, "PS_dashed\n");
    }
}
