#include "fit_popup.h"

#include "fit.hlp"

#include "callback.h"
#include "color.h"
#include "display.h"
#include "fit.h"
#include "fit_func.h"
#include "info_popup.h"
#include "param.h"
#include "peak.h"
#include "window.h"

#define  MIN_WIDTH	200
#define  MIN_HEIGHT	200

#define  VISIBLE_ITEMS  20

#define  MARK_SIZE  3

static Widget fit_popup = (Widget) NULL;
static Widget fit_area;
static Widget fit_form;
static Widget peak_set_list = (Widget) NULL;

static Widget method_boxes[NFIT_METHODS];
static Radiobox_item method_items[NFIT_METHODS];
 
static int method = NO_FIT;
static int nmethods = NFIT_METHODS;

static Display *fit_display = (Display *) NULL;
static Window fit_window;
static GC fit_gc = (GC) NULL;

static int width = MIN_WIDTH;
static int height = MIN_HEIGHT;

static int npeak_sets = 0;
static Peak_set **peak_sets;
static int current_peak_set = -1;

static float x_min, x_max, y_min, y_max;
static float ax, bx, ay, by;

void update_fit_params()
{
    if (fit_popup)
    {
	get_widget_size(fit_area, &width, &height);
	method = get_radio_state(method_boxes, nmethods);

	sprintf(fit_width, "%d", width);
	sprintf(fit_height, "%d", height);
	sprintf(fit_method, "%d", method);
    }
    else
    {
	if (*fit_width)
	    width = atoi(fit_width);

	width = MAX(width, MIN_WIDTH);

	if (*fit_height)
	    height = atoi(fit_height);

	height = MAX(height, MIN_HEIGHT);

	if (*fit_method)
	    method = atoi(fit_method);
    }
}

static void draw_line(int x1, int y1, int x2, int y2)
{
    display_line(fit_display, fit_window, fit_gc, x1, y1, x2, y2);
}

static void draw_circle(int x, int y, int r)
{
    display_circle(fit_display, fit_window, fit_gc, x, y, r);
}

static void clear_fit()
{
    if (!fit_gc)
	return;

    set_gc_color(fit_display, fit_gc, WHITE);
    clear_region(fit_display, fit_window, fit_gc, 0, 0, width, height);
}

static void stretch_range(float *min, float *max)
{
    float a = *min, b = *max, m = 0.1;

    if (a > 0) 
	a = 0;

    if (b < 0) 
	b = 0;

    *min = (1+m)*a - m*b;
    *max = (1+m)*b - m*a;
}

static void find_scaling(int npeaks, Peak **peaks)
{
    int i;

    x_min = x_max = peaks[0]->param;
    y_min = y_max = peaks[0]->extremum;

    for (i = 0; i < npeaks; i++)
    {
	x_min = MIN(x_min, peaks[i]->param);
	x_max = MAX(x_max, peaks[i]->param);

	y_min = MIN(y_min, peaks[i]->extremum);
	y_min = MIN(y_min, peaks[i]->fit_extremum);
	y_max = MAX(y_max, peaks[i]->extremum);
	y_max = MAX(y_max, peaks[i]->fit_extremum);
    }

    stretch_range(&x_min, &x_max);
    stretch_range(&y_min, &y_max);

    ax = width / (x_max - x_min);  bx = - ax * x_min;
    ay = - height / (y_max - y_min);  by = - ay * y_max;
	/* y formulas different from x ones because y = 0 on top of screen */
}

#define  CONVERT_X(x)  (ax*(x) + bx)
#define  CONVERT_Y(y)  (ay*(y) + by)

static void draw_axes()
{
    int s;

    s = CONVERT_X(0.0);
    draw_line(s, 0, s, height);

    s = CONVERT_Y(0.0);
    draw_line(0, s, width, s);
}

/* draw circles */
static void draw_peaks(int npeaks, Peak **peaks)
{
    int i, xc, yc, r = MARK_SIZE;
    float x, y;

    for (i = 0; i < npeaks; i++)
    {
	x = peaks[i]->param;
	y = peaks[i]->extremum;

	xc = CONVERT_X(x);
	yc = CONVERT_Y(y);
	draw_circle(xc, yc, r);
    }
}

/* draw crosses */
static void draw_fit(int npeaks, Peak **peaks)
{
    int i, x1, y1, x2, y2, d = MARK_SIZE;
    static int nfit_pts = 50;
    float x, y;

    for (i = 0; i < npeaks; i++)
    {
	x = peaks[i]->param;
	y = peaks[i]->fit_extremum;

	x1 = CONVERT_X(x) - d;
	x2 = CONVERT_X(x) + d;
	y1 = CONVERT_Y(y) - d;
	y2 = CONVERT_Y(y) + d;
	draw_line(x1, y1, x2, y2);
	draw_line(x1, y2, x2, y1);
    }

    x1 = CONVERT_X(x_min);
    y1 = CONVERT_Y(calculate_fit_func(x_min));

    for (i = 1; i < nfit_pts+1; i++)
    /* the +1 is to make sure line hits right-hand border */
    {
	x = ((nfit_pts-i)*x_min + i*x_max) / nfit_pts;
	x2 = CONVERT_X(x);
	y2 = CONVERT_Y(calculate_fit_func(x));

	draw_line(x1, y1, x2, y2);

	x1 = x2;
	y1 = y2;
    }
}

static void update_label(Peak_set *peak_set)
{
    int n = get_method_nparams(peak_set->method);
    Line label;

    if (n == 2)
	sprintf(label, "%s: %s: %4.3e %4.3e (%3.2e)",
		peak_set->set, fit_methods[peak_set->method],
		peak_set->param1, peak_set->param2, peak_set->chisq);
    else if (n == 3)
	sprintf(label, "%s: %s: %4.3e %4.3e %4.3e (%3.2e)",
		peak_set->set, fit_methods[peak_set->method],
		peak_set->param1, peak_set->param2,
		peak_set->param3, peak_set->chisq);
    else
	sprintf(label, "");

    replace_in_list(peak_set_list, label, current_peak_set+1);
}

static void do_fit_drawing(Bool print_error)
{
    int npeaks;
    Peak **peaks;
    Peak_set *peak_set;
    Line error_msg;

    if (!fit_popup)
	return;

    update_fit_params();

    clear_fit();

    if ((npeak_sets == 0) || (current_peak_set < 0) || (method == NO_FIT))
	return;

    if (fit_peaks_func(error_msg) == ERROR)
    {
	if (print_error)
	    ERROR_AND_RETURN(error_msg);

	return;
    }

    peak_set = peak_sets[current_peak_set];
    npeaks = peak_set->npeaks;
    peaks = peak_set->peaks;

    set_gc_color(fit_display, fit_gc, BLACK);

    find_scaling(npeaks, peaks);

    draw_axes();

    draw_peaks(npeaks, peaks);
    draw_fit(npeaks, peaks);

    update_label(peak_set);
    select_from_list(peak_set_list, current_peak_set+1, FALSE);
}

static void cycle_peak_set(int peak_set)
{
    current_peak_set = peak_set;

    if (peak_set == -1)
        return;
 
    select_from_list(peak_set_list, peak_set+1, FALSE);
 
    do_fit_drawing(TRUE);
}

static void peak_set_description(Peak_set *peak_set, String descrip)
{
    int n;
    Line line;

    sprintf(descrip, "%s", peak_set->set);

    if (peak_set->method > 0)
    {
	strcat(descrip, ":");

	n = get_method_nparams(peak_set->method);

	if (n >= 1)
	{
	    sprintf(line, " %4.3f", peak_set->param1);
	    strcat(descrip, line);
	}

	if (n >= 2)
	{
	    sprintf(line, " %4.3f", peak_set->param2);
	    strcat(descrip, line);
	}

	if (n >= 3)
	{
	    sprintf(line, " %4.3f", peak_set->param3);
	    strcat(descrip, line);
	}
    }
}

static void create_peak_set_list()
{
    int i;
    Line line;

    if (!peak_set_list)
	return;

    peak_sets = get_peak_sets(&npeak_sets);

    for (i = 0; i < npeak_sets; i++)
    {
	peak_set_description(peak_sets[i], line);
	insert_in_list(peak_set_list, line, 0);
    }

    /* should do better than below .... */
    if (current_peak_set >= npeak_sets)
	current_peak_set = -1;
    else if (current_peak_set >= 0)
	select_from_list(peak_set_list, current_peak_set+1, FALSE);

    fit_set_peak_sets(current_peak_set, npeak_sets, peak_sets);

    do_fit_drawing(TRUE);
}

static void apply_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    do_fit_drawing(TRUE);
}

static void method_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    if (!toggle_being_set(cbs))
        return;

    do_fit_drawing(TRUE);
}

static void select_command(int peak_set)
{
    fit_select_set_func(&peak_set);
    current_peak_set = peak_set;
    do_fit_drawing(TRUE);
}

static void select_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    int peak = list_position(cbs);
 
    select_command(peak);
}

Status fit_select_command(String value, Generic_ptr data, String error_msg)
{
    int peak_set;
 
    if ((peak_set = find_peak_set(value)) < 0)
    {
        sprintf(error_msg, "peak set '%s' unknown", value);
        return  ERROR;
    }

    select_command(peak_set);
 
    return  OK;
}
 
static void next_command()
{
    int peak_set;
 
    fit_next_set_func(&peak_set);
    cycle_peak_set(peak_set);
}
 
static void next_callback(Widget widget, Callback_ptr data,
                                                        Callback_ptr cbs)
{
    next_command();
}
 
Status fit_next_command(String value, Generic_ptr data, String error_msg)
{
    next_command();
 
    return  OK;
}
 
static void previous_command()
{
    int peak_set;
 
    fit_previous_set_func(&peak_set);
    cycle_peak_set(peak_set);
}
 
static void previous_callback(Widget widget, Callback_ptr data,
                                                        Callback_ptr cbs)
{
    previous_command();
}

Status fit_previous_command(String value, Generic_ptr data,
                                                        String error_msg)
{
    previous_command();
 
    return  OK;
}

static void fit_expose_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    do_fit_drawing(FALSE);
}

static void fit_resize_callback(Widget widget, Callback_ptr data,
							Callback_ptr cbs)
{
    do_fit_drawing(FALSE);
}

static Widget create_peak_list_form(Widget parent)
{
    int i;
    Scrolled_list_info peak_list_info;
    Widget label, buttons, form, radio;
    static Button_item button_items[] =
    {
	{ "next",	next_callback,		NO_DATA },
	{ "previous",	previous_callback,	NO_DATA }
    };
    static int nbuttons = ARRAY_SIZE(button_items);
    Apply_dismiss_help_info adh_info;
    char apply_label[] = "apply";
 
    for (i = 0; i < nmethods; i++)
    {
	method_items[i].label = fit_methods[i];
	method_items[i].callback = method_callback;
	method_items[i].data = (Callback_ptr) NULL;
	method_items[i].box = method_boxes + i;
    }

    form = create_form(parent);
    CHECK_WIDGET(form);
 
    radio = create_radiobox(form, method_items, nmethods, method, HORIZONTAL);
    CHECK_WIDGET_DESTROY(radio, form);
    attachments(radio, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, NO_ATTACH);

    label = create_label(form, "peak sets:");
    CHECK_WIDGET_DESTROY(label, form);
    attachments(label, radio, NO_ATTACH, FORM_ATTACH, NO_ATTACH);
 
    buttons = create_vertical_buttons(form, button_items, nbuttons);
    CHECK_WIDGET_DESTROY(buttons, form);
    attachments(buttons, label, NO_ATTACH, FORM_ATTACH, NO_ATTACH);
    offsets(buttons, 2*UNIT_OFFSET, NO_OFFSET, 2*UNIT_OFFSET, NO_OFFSET);
 
    peak_list_info.selection_policy = SINGLE_SELECT;
    peak_list_info.nitems = 0;
    peak_list_info.func = NULL;
    peak_list_info.nvisible_items = VISIBLE_ITEMS;
    peak_list_info.callback = select_callback;
 
    peak_set_list = create_scrolled_list(form, &peak_list_info);
    CHECK_WIDGET_DESTROY(peak_set_list, form);
    attachments(WIDGET_PARENT(peak_set_list),
/*
                        label, FORM_ATTACH, buttons, FORM_ATTACH);
*/
                        label, NO_ATTACH, buttons, FORM_ATTACH);
    offsets(WIDGET_PARENT(peak_set_list),
                        UNIT_OFFSET, UNIT_OFFSET, 2*UNIT_OFFSET, UNIT_OFFSET);            
    adh_info.apply_label = apply_label;
    adh_info.apply_callback = apply_callback;
    adh_info.dismiss_form = fit_form;
    adh_info.help_message = fit_help;

    buttons = create_apply_dismiss_help(form, &adh_info);
    CHECK_WIDGET_DESTROY(buttons, form);
    attachments(buttons, WIDGET_PARENT(peak_set_list), FORM_ATTACH,
						FORM_ATTACH, FORM_ATTACH);
/*
    attachments(buttons, peak_set_list, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);
*/

    manage_widget(form);
 
    return  form;
}

static void create_fit_popup(Widget parent)
{
    Widget peak_form;
    Drawing_area_info drawing_info;

    update_fit_params();

    fit_popup = create_popup(parent, "Fit Peak Heights");
    CHECK_WIDGET_WARNING(fit_popup);

    fit_form = create_form(fit_popup);
    CHECK_WIDGET_DESTROY_WARNING(fit_form, fit_popup);

    peak_form = create_peak_list_form(fit_form);
    CHECK_WIDGET_DESTROY_WARNING(peak_form, fit_popup);
    attachments(peak_form, NO_ATTACH, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);

    drawing_info.width = width;
    drawing_info.height = height;
    drawing_info.expose = fit_expose_callback;
    drawing_info.resize = fit_resize_callback;

    fit_area = create_drawing(fit_form, &drawing_info);
    CHECK_WIDGET_DESTROY_WARNING(fit_area, fit_popup);
    attachments(fit_area, FORM_ATTACH, peak_form, FORM_ATTACH, FORM_ATTACH);

    manage_widget(fit_form);

    fit_display = WIDGET_DISPLAY(fit_area);
    initialize_display(fit_display);

    fit_window = WIDGET_WINDOW(fit_area);
    fit_gc = create_gc(fit_display, fit_window);

    clear_fit();
    create_peak_set_list();
}

void fit_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!fit_popup)
        create_fit_popup(parent);

    if (fit_popup)
    {
	popup(fit_form);
	do_fit_drawing(TRUE);
    }
}

Status fit_popup_command(String value, Generic_ptr data, String error_msg)
{
    fit_popup_callback(get_topshell(), (Callback_ptr) NULL,
							(Callback_ptr) NULL);

    return  OK;
}

void new_peak_set()
{
    int n;
    Line line;

    if (!peak_set_list)
	return;

    peak_sets = get_peak_sets(&n);

    if (n == npeak_sets)
	return;

    npeak_sets = n;

    peak_set_description(peak_sets[n-1], line);
    insert_in_list(peak_set_list, line, 0);

    fit_set_peak_sets(current_peak_set, npeak_sets, peak_sets);

    do_fit_drawing(TRUE);
}

void update_peak_set_list()
{
    int i, n;

    if (!peak_set_list)
	return;

    (void) get_peak_sets(&n);

    if (n == npeak_sets)
	return;

    for (i = npeak_sets-1; i >= 0; i--)
	delete_from_list(peak_set_list, i+1);

    create_peak_set_list();
}
