#include "draw.h"

#include "contour.h"
#include "fold.h"
#include "data.h"
#include "list.h"
#include "object.h"
#include "ref.h"
#include "region.h"
#include "ticks.h"

static float lower[DISPLAY_DIM];
static float upper[DISPLAY_DIM];
static int begin[DISPLAY_DIM];
static int end[DISPLAY_DIM];
static int size[DISPLAY_DIM];

static float lower_orig[DISPLAY_DIM];
/*
static float upper_orig[DISPLAY_DIM];
*/
static float lower_draw[DISPLAY_DIM];
static float upper_draw[DISPLAY_DIM];
static float delta[DISPLAY_DIM];

static int *fold_type;
static int *flip_type;
static int fold_begin[DISPLAY_DIM];
static int fold_end[DISPLAY_DIM];

static Data_info *data_info;
static Object_property *property;

static Draw_funcs *draw_funcs;
static Timer_funcs *timer_funcs;

static int nlevels;
static int nlevels_alloc = 0;
static float *levels = (float *) NULL;

static Contour_info contour_info;
static List *vertices = (List *) NULL;

static float rowcol_scale;

static Coord contour_offset = { 0, 0 };
static Coord contour_scale = { 1, 1 };

static Object_property *global_property;

void free_draw_memory()
{
    FREE(levels, float);
    FREE(vertices, List);

    nlevels_alloc = 0;
}

static Status allocate_draw_memory()
{
    if (nlevels > nlevels_alloc)
    {
	free_draw_memory();

	MALLOC(levels, float, nlevels);
	MALLOC(vertices, List, nlevels);

	nlevels_alloc = nlevels;
    }

    return  OK;
}

static void initialize_range(Data_info *d, float *l, float *u)
{
    int i;

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	if (d)
	{
	    lower[i] = d->scale[i] * l[i] + d->offset[i];
	    upper[i] = d->scale[i] * u[i] + d->offset[i];
	}
	else
	{
	    lower[i] = l[i];
	    upper[i] = u[i];
	}
    }
}

#define  SMALL_NUMBER  1.0e-4

static void do_range(int data_set)
{
    int i;

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	begin[i] = lower[i] - 1;
		/* floor, taking into account that points start at 1 */
	begin[i] = MAX(0, begin[i]);

	end[i] = 1 + upper[i] - SMALL_NUMBER;
		/* 1 + ceiling, taking into account that points start at 1 */
	end[i] = MIN(data_info[data_set].npoints[i], end[i]);

	size[i] = end[i] - begin[i];
    }
}

static void do_levels(int sign, float *lvls, float level_scale)
{
    int i;

    for (i = 0; i < nlevels; i++)
	levels[i] = sign * lvls[i] * level_scale / property->scale;
}

static void init_contours()
{
    contour_info.nlevels = nlevels;
    contour_info.levels = levels;
    contour_info.npoints = size;
    contour_info.offset = &contour_offset;
    contour_info.scale = &contour_scale;
    contour_info.vertices = vertices;
    contour_info.get_row = get_data_row;
    contour_info.timer_funcs = timer_funcs;
}

static Status do_contours(int data_set, String message, String error_msg)
{
    int i;
    float a0, b0, a1, b1, fraction;
    Line err_msg, msg;
    List vert;
    Vertex *v;
    Status status;

    sprintf(msg, "find %s", message);
    (*(timer_funcs->start_timer))(msg);

    initialize_data_row(data_set, begin);

    if ((status = construct_contours(&contour_info, err_msg)) == ERROR)
    {
    	(*(timer_funcs->stop_timer))(ERROR);

	sprintf(error_msg, "constructing contours for data set %d: %s",
						data_set+1, err_msg);
	return  ERROR;
    }
    else if (status != OK)
    {
	(*(timer_funcs->stop_timer))(status);
	return  status;
    }

    (*(timer_funcs->stop_timer))(OK);

    /* draw these contours */

    (*(draw_funcs->new_draw_range))(lower_draw[0], lower_draw[1],
					upper_draw[0], upper_draw[1], TRUE);
    (*(draw_funcs->set_draw_color))(property->color);
    (*(draw_funcs->set_line_style))(property->line_style);

    sprintf(msg, "draw %s", message);
    (*(timer_funcs->start_timer))(msg);

    for (i = 0; i < nlevels; i++)
    {
	fraction = ((float) i) / ((float) nlevels);

	if ((*(timer_funcs->update_timer))(fraction) == ABORT)
	{
	    (*(timer_funcs->stop_timer))(OTHER);
	    return  OTHER;
	}

	for (vert = vertices[i]; vert; vert = NEXT(vert))
	{
	    v = (Vertex *) DATA(vert);

	    if (v->v1)
	    {
		a0 = v->p.x;  b0 = v->p.y;

		v = v->v1;
		a1 = v->p.x;  b1 = v->p.y;

	        (*(draw_funcs->draw_line))(a0, b0, a1, b1);
	    }
	}
    }

    (*(timer_funcs->stop_timer))(OK);

    return  OK;
}

static void determine_fold(int n, int npts)
{
    if ((fold_type[n] == REGION_NO_FOLD) || (delta[n] < SMALL_NUMBER))
		/* delta condition should not happen but play safe */
    {
	fold_begin[n] = 0;
	fold_end[n] = 0;
    }
    else
    {
	/* lower and upper range from 1 to npts  */
	fold_begin[n] = - 1 - FLOOR((lower[n] - 1 - SMALL_NUMBER) / delta[n]);
	fold_end[n] = 1 + FLOOR((npts - upper[n] - SMALL_NUMBER) / delta[n]);
    }
}

static void determine_folds(int data_set)
{
    int i, npts;

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	npts = data_info[data_set].npoints[i];

	determine_fold(i, npts);
    }
}

static void determine_fold_range(int n0, int n1)
{
    int i, n;

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	if (i == 0)
	    n = n0;
	else
	    n = n1;

	lower[i] = lower_orig[i] + n * delta[i];
	upper[i] = lower[i] + delta[i];
    }
}

static int determine_sign(int n, int m)
{
    if ((fold_type[n] == REGION_NO_FOLD) || (flip_type[n] == PEAK_FLIP_OFF))
	return  1;
    else
	return  (((m % 2) == 0)  ?  1  :  -1);
}

static Bool determine_flip(int n0, int n1)
{
    int s0 = determine_sign(0, n0);
    int s1 = determine_sign(1, n1);
    int s = s0 * s1;

    if (s == 1)
	return  FALSE;
    else
	return  TRUE;
}

static void determine_draw_range(int n0, int n1)
{
    int i, n;

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	if (i == 0)
	    n = n0;
	else
	    n = n1;

	lower_draw[i] = lower[i] - 1 - begin[i];
	upper_draw[i] = upper[i] - 1 - begin[i];

	if ((fold_type[i] == REGION_REFLECT) && ((n % 2) != 0))
	    SWAP(lower_draw[i], upper_draw[i], float);
    }
}

static Status draw_contours(int n, float *lvls, String error_msg)
{
    int i0, i1, sign;
    Bool flip;
    Line message;

    COPY_VECTOR(lower_orig, lower, DISPLAY_DIM);
/*
    COPY_VECTOR(upper_orig, upper, DISPLAY_DIM);
*/
    SUBTRACT_VECTORS(delta, upper, lower, DISPLAY_DIM);

    determine_folds(n);

    for (i1 = fold_begin[1]; i1 <= fold_end[1]; i1++)
    {
	for (i0 = fold_begin[0]; i0 <= fold_end[0]; i0++)
	{
	    determine_fold_range(i0, i1);
	    do_range(n);

	    if ((size[0] < 1) || (size[1] < 1))
		continue;

	    flip = determine_flip(i0, i1);
	    determine_draw_range(i0, i1);

    	    property = data_info[n].property + POSITIVE_OBJECT;
	    if ((property->visible == VISIBILITY_ON) &&
		(global_property[POSITIVE_OBJECT].visible == VISIBILITY_ON))
	    {
		sign = 1;
		do_levels(sign, lvls, data_info[n].level_scale);

		if (flip)
    		    property = data_info[n].property + NEGATIVE_OBJECT;

		sprintf(message, "positive: data set %d", n+1);

		CHECK_OK(do_contours(n, message, error_msg));
	    }

    	    property = data_info[n].property + NEGATIVE_OBJECT;
	    if ((property->visible == VISIBILITY_ON) &&
		(global_property[NEGATIVE_OBJECT].visible == VISIBILITY_ON))
	    {
		sign = - 1;
		do_levels(sign, lvls, data_info[n].level_scale);

		if (flip)
    		    property = data_info[n].property + POSITIVE_OBJECT;

		sprintf(message, "negative: data set %d", n+1);

		CHECK_OK(do_contours(n, message, error_msg));
	    }
	}
    }

    return  OK;
}

Status do_drawing(Draw_info *draw_info, Draw_funcs *d_funcs,
				Timer_funcs *t_funcs, String error_msg)
{
    int i, *major_ticks, *minor_ticks, ref_type;
    static Bool done[] = { TRUE, TRUE };

    draw_funcs = d_funcs;
    timer_funcs = t_funcs;
    nlevels = draw_info->nlevels;

    major_ticks = draw_info->global_info.major_ticks;
    minor_ticks = draw_info->global_info.minor_ticks;
    ref_type = draw_info->ref_type;
    fold_type = draw_info->fold_type;
    flip_type = draw_info->flip_type;

    if (allocate_draw_memory() == ERROR)
    {
	sprintf(error_msg, "allocating draw memory");
	return  ERROR;
    }

    init_contours();

    initialize_range(NULL, draw_info->lower, draw_info->upper);

    (*(draw_funcs->start_draw))(draw_funcs->data);

    check_orientation(ref_type, DISPLAY_DIM, lower, upper);

    (*(draw_funcs->new_draw_range))(lower[0], lower[1],
						upper[0], upper[1], FALSE);

    property = global_property = draw_info->global_info.property;

    if (property[RULER_OBJECT].visible == VISIBILITY_ON)
    {
	(*(draw_funcs->set_draw_color))(property[RULER_OBJECT].color);

	draw_minor_ticks(minor_ticks, done, lower, upper, draw_funcs);
	draw_major_ticks(major_ticks, done, lower, upper, draw_funcs);
    }

    if (property[GRID_OBJECT].visible == VISIBILITY_ON)
    {
	(*(draw_funcs->set_draw_color))(property[GRID_OBJECT].color);
	draw_grid(major_ticks, lower, upper, draw_funcs);
    }

    data_info = draw_info->data_info;

    for (i = 0; i < draw_info->ndata_sets; i++)
    {
	if (data_info[i].on_off == VISIBILITY_OFF)
	    continue;

    	initialize_range(data_info + i, draw_info->lower, draw_info->upper);

	convert_range_to_points(ref_type, DISPLAY_DIM, data_info[i].npoints,
					data_info[i].ref, lower, upper);

	CHECK_OK(draw_contours(i, draw_info->levels, error_msg));
    }

    (*(draw_funcs->end_draw))();

    return  OK;
}

#define  DATA_POINT(i, j)  data[i+j*npoints]

static void do_cols(int data_set, int point, int width, float largest)
{
    int i, left, right, npoints;
    float scale, x, x_left, x_right, f_left, f_right, a0, b0, a1, b1;
    float *data;

    (*(draw_funcs->new_draw_range))(0.0, lower[1],
					(float) (width-1), upper[1], FALSE);
    (*(draw_funcs->set_draw_color))(property->color);

    data = data_info[data_set].data;

    scale = width * rowcol_scale * property->scale
			/ (largest * data_info[data_set].level_scale);

    npoints = data_info[data_set].npoints[0];

    x = lower[0]  - 1 + ((float) point)*(upper[0]-lower[0])/((float) (width-1));
		/* points start at 1 */

    left = x;
    right = left + 1;

    if ((left < 0) || (right >= npoints))
	return;

    f_right = x - left;  f_left = 1 - f_right;

    i = begin[1];

    x_left = DATA_POINT(left, i);
    x_right = DATA_POINT(right, i);
    a0 = point + scale * (f_left*x_left + f_right*x_right);
    b0 = i + 1;  /* points start at 1 */

    for (i = begin[1]+1; i < end[1]; i++)
    {
	x_left = DATA_POINT(left, i);
	x_right = DATA_POINT(right, i);
	a1 = point + scale * (f_left*x_left + f_right*x_right);
	b1 = i + 1;

	(*(draw_funcs->draw_line))(a0, b0, a1, b1);

	a0 = a1;  b0 = b1;
    }
}

static void do_rows(int data_set, int point, int height, float largest)
{
    int i, top, bottom, npoints;
    float scale, y, y_top, y_bottom, f_top, f_bottom, a0, b0, a1, b1;
    float *data;

    (*(draw_funcs->new_draw_range))(lower[0], 0.0,
					upper[0], (float) (height-1), FALSE);
    (*(draw_funcs->set_draw_color))(property->color);

    data = data_info[data_set].data;

    scale = height * rowcol_scale * property->scale
			/ (largest * data_info[data_set].level_scale);

    npoints = data_info[data_set].npoints[1];

    y = lower[1] -1 + ((float) point)*(upper[1]-lower[1])/((float) (height-1));
		/* points start at 1 */

    bottom = y;
    top = bottom + 1;

    if ((bottom < 0) || (top >= npoints))
	return;

    npoints = data_info[data_set].npoints[0];

    f_top = y - bottom;  f_bottom = 1 - f_top;

    i = begin[0];

    y_bottom = DATA_POINT(i, bottom);
    y_top = DATA_POINT(i, top);
    b0 = point + scale * (f_bottom*y_bottom + f_top*y_top);
    a0 = i + 1;  /* points start at 1 */

    for (i = begin[0]+1; i < end[0]; i++)
    {
	y_bottom = DATA_POINT(i, bottom);
	y_top = DATA_POINT(i, top);
	b1 = point + scale * (f_bottom*y_bottom + f_top*y_top);
	a1 = i + 1;

	(*(draw_funcs->draw_line))(a0, b0, a1, b1);

	a0 = a1;  b0 = b1;
    }
}

void draw_rows_and_cols(Draw_rowcol_info *rowcol_info, Draw_funcs *d_funcs)
{
    int i, ref_type;
    float largest;

    ref_type = rowcol_info->ref_type;
    rowcol_scale = rowcol_info->scale;

    draw_funcs = d_funcs;

    initialize_range(NULL, rowcol_info->lower, rowcol_info->upper);

    (*(draw_funcs->start_draw))(draw_funcs->data);

    property = global_property = rowcol_info->global_info.property;

    if (property[CROSSHAIR_OBJECT].visible == VISIBILITY_ON)
    {
    	(*(draw_funcs->new_draw_range))(0.0, 0.0, (float) (rowcol_info->w-1),
					(float) (rowcol_info->h-1), FALSE);

    	(*(draw_funcs->set_draw_color))(property[CROSSHAIR_OBJECT].color);

	(*(draw_funcs->draw_line))(rowcol_info->x, 0.0, rowcol_info->x,
						(float) (rowcol_info->h-1));
	(*(draw_funcs->draw_line))(0.0, rowcol_info->y,
				(float) (rowcol_info->w-1), rowcol_info->y);
    }

    data_info = rowcol_info->data_info;

    largest = largest_data_value();

    for (i = 0; i < rowcol_info->ndata_sets; i++)
    {
	if (data_info[i].on_off == VISIBILITY_OFF)
	    continue;

    	property = data_info[i].property;

    	initialize_range(data_info + i, rowcol_info->lower, rowcol_info->upper);

	convert_range_to_points(ref_type, DISPLAY_DIM, data_info[i].npoints,
					data_info[i].ref, lower, upper);

	do_range(i);

	if ((size[0] < 1) || (size[1] < 1))
	    continue;

    	property = data_info[i].property + COLUMN_OBJECT;
	if ((property->visible == VISIBILITY_ON) &&
		(global_property[COLUMN_OBJECT].visible == VISIBILITY_ON))
	{
	    do_cols(i, rowcol_info->x, rowcol_info->w, largest);
	}

    	property = data_info[i].property + ROW_OBJECT;
	if ((property->visible == VISIBILITY_ON) &&
		(global_property[ROW_OBJECT].visible == VISIBILITY_ON))
	{
	    do_rows(i, rowcol_info->y, rowcol_info->h, largest);
	}
    }

    (*(draw_funcs->end_draw))();
}
