#include "data.h"

#include "object.h"
#include "object_func.h"
#include "ref.h"

#define  DATA_SET_ALLOC  8

#define  SMALL_NUMBER  1.0e-6
#define  LARGE_NUMBER  1.0e20

static int ndata_sets = 0;
static int ndata_sets_alloc = 0;

static Data_info *data_info;

static int current_data_set;
static int current_index;
static int current_skip;

float largest_data_value()
{
    int i;
    float largest;

    largest = 0;
    for (i = 0; i < ndata_sets; i++)
    {
	largest = MAX(largest, data_info[i].largest);
    }

    return  largest;
}

Object_property *get_data_property(int data_set, int object)
{
    if ((data_set >= 0) && (data_set < ndata_sets) && (object < NDATA_OBJECTS))
	return  data_info[data_set].property + object;
    else
	return  (Object_property *) NULL;
}

void set_data_on_off(int data_set, int on_off)
{
    if ((data_set >= 0) && (data_set < ndata_sets))
	data_info[data_set].on_off = on_off;
}

int get_data_on_off(int data_set)
{
    if ((data_set >= 0) && (data_set < ndata_sets))
	return  data_info[data_set].on_off;
    else
	return  VISIBILITY_OFF;
}

void toggle_data_property(int data_set)
{
    if ((data_set >= 0) && (data_set < ndata_sets))
    {
	data_info[data_set].property_selected =
					!data_info[data_set].property_selected;
    }
}

Bool data_property_selected(int data_set)
{
    if ((data_set >= 0) && (data_set < ndata_sets))
	return  data_info[data_set].property_selected;
    else
	return  FALSE;
}

Status get_data_row(float **data, String error_msg)
{
    *data = data_info[current_data_set].data + current_index;
    current_index += current_skip;

    return  OK;
}

void initialize_data_row(int data_set, int *begin)
{
    int n = data_info[data_set].npoints[0];

    current_data_set = data_set;
    current_index = begin[0] + begin[1] * n;
    current_skip = n;
}

Status get_data_info(int *n, Data_info **info, String error_msg)
{
    *n = ndata_sets;
    *info = data_info;

    return  OK;
}

Data_info *get_data_set(int data_set)
{
    if ((data_set >= 0) && (data_set < ndata_sets))
	return  data_info + data_set;
    else
	return  (Data_info *) NULL;
}

Bool data_name_exists(String name, int *p_data_set)
{
    int i;

    for (i = 0; i < ndata_sets; i++)
    {
	if (!strcmp(name, data_info[i].name))
	{
	    *p_data_set = i;
	    return  TRUE;
	}
    }

    *p_data_set = ndata_sets;

    return  FALSE;
}

String get_data_name(int data_set)
{
    static char string[] = "";

    if (data_set < ndata_sets)
	return  data_info[data_set].name;
    else
	return  string;
}

void free_data_set(int data_set)
{
    int i;

    if (data_set < ndata_sets)
    {
	FREE(data_info[data_set].name, char);
	FREE(data_info[data_set].data, float);

	for (i = data_set; i < ndata_sets-1; i++)
	    data_info[i] = data_info[i+1];

	ndata_sets--;
    }
}

static void init_data_scale_offset(int data_set)
{
    int i;

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	data_info[data_set].scale[i] = 1;
	data_info[data_set].offset[i] = 0;
    }

    data_info[data_set].level_scale = 1;
}

Status allocate_data_set(int data_set, int nwords, String name,
							float **p_data)
{
    int n;

    if (data_set < ndata_sets)
    {
	n = data_info[data_set].nwords;

	if ((nwords <= n) && (2*nwords >= n))
	{
	    *p_data = data_info[data_set].data;
	    init_data_scale_offset(data_set);

	    return  OK;
	}

	FREE(data_info[data_set].data, float);
    }
    else
    {
    	if (data_set >= ndata_sets_alloc)
	{
	    n = ndata_sets_alloc + DATA_SET_ALLOC;

	    if (ndata_sets_alloc == 0)
	    {
	    	MALLOC(data_info, Data_info, n);
	    }
	    else
	    {
	    	REALLOC(data_info, Data_info, n);
	    }

	    ndata_sets_alloc = n;
	}

	data_info[data_set].data = (float *) NULL;

    	n = strlen(name) + 1;
    	MALLOC(data_info[data_set].name, char, n);
	strcpy(data_info[data_set].name, name);
    }

    MALLOC(data_info[data_set].data, float, nwords);
    init_data_scale_offset(data_set);

    data_info[data_set].nwords = nwords;
    *p_data = data_info[data_set].data;

    return  OK;
}

static void find_largest(int nwords, float *data, float *p_largest)
{
    int i;
    float d, largest;

    largest = 0;
    for (i = 0; i < nwords; i++)
    {
	d = ABS(data[i]);
	largest = MAX(largest, d);
    }

    *p_largest = largest;
}

void enrol_data_set(int data_set, Size_info *size_info, Ref_info *ref_info,
			Extract_info *extract_info, Param_info *param_info)
{
    int i, j, first, last, step, n, nwords, npoints;
    Ref_info *ref;

    if (data_set == ndata_sets)
    {
	data_info[data_set].on_off = VISIBILITY_ON;
	data_info[data_set].property_selected = FALSE;
	default_property(data_info[data_set].property);

    	ndata_sets++;
    }

    nwords = 1;
    for (i = 0; i < DISPLAY_DIM; i++)
    {
	j = extract_info->dim_extr[i];

	first = extract_info->first[j];
	last = extract_info->last[j];
	step = extract_info->step[j];
	npoints = size_info->npoints[j];

	n = 1 + (last - first - 1) / step;
	nwords *= n;

	data_info[data_set].npoints[i] = n;
	data_info[data_set].npoints_orig[i] = npoints;
	data_info[data_set].first[i] = first;
	data_info[data_set].step[i] = step;

	ref = data_info[data_set].ref + i;
	find_sub_ref(ref, ref_info+j, first, last, step, npoints);
    }

    data_info[data_set].have_param = param_info->have_param;
    data_info[data_set].param = param_info->param;

    find_largest(nwords, data_info[data_set].data,
					&(data_info[data_set].largest));
}

void print_point_stats(int ref_type, float *desired_point,
						Print_funcs *print_funcs)
{
    int i, j, k, n, begin[DISPLAY_DIM], end[DISPLAY_DIM];
    float point[DISPLAY_DIM];
    Line error_msg, message;

    if (ndata_sets == 0)  return;

    if ((*(print_funcs->start_print))(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    sprintf(message, "Chosen point is (%5.2f, %5.2f %s)\n",
		desired_point[0], desired_point[1],
		ref_type == REF_POINTS  ?  "\b"  :  ref_names[ref_type]);

    (*(print_funcs->print_message))(message);

    for (i = 0; i < ndata_sets; i++)
    {
	sprintf(message, "\nStatistics for data set '%s'\n",
							data_info[i].name);
    	(*(print_funcs->print_message))(message);

	for (j = 0; j < DISPLAY_DIM; j++)
	    point[j] = desired_point[j];

	if (ref_type != REF_POINTS)
	{
	    convert_to_points(ref_type, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, point);

    	    sprintf(message, "Chosen point is (%5.2f, %5.2f)\n",
							point[0], point[1]);
    	    (*(print_funcs->print_message))(message);
	}

	sprintf(message, "Closest data values:\n");
    	(*(print_funcs->print_message))(message);

	for (j = 0; j < DISPLAY_DIM; j++)
	{
	    begin[j] = point[j] - 1;  /* points start at 1 */
	    end[j] = begin[j] + 2;
	    begin[j] = MAX(0, begin[j]);
	    begin[j] = MIN(begin[j], data_info[i].npoints[j]-1);
	    end[j] = MAX(1, end[j]);
	    end[j] = MIN(end[j], data_info[i].npoints[j]);
	}

	n = data_info[i].npoints[0];

	for (k = begin[1]; k < end[1]; k++)
	{
	    for (j = begin[0]; j < end[0]; j++)
	    {
		sprintf(message, "\tPoint (%d, %d):\t%5.2f\n", j+1, k+1,
						data_info[i].data[j+k*n]);
    		(*(print_funcs->print_message))(message);
	    }
	}
    }

    (*(print_funcs->end_print))();
}

void print_region_stats(int ref_type, float *lower, float *upper,
			Print_funcs *print_funcs, FILE *fp, String file)
{
    int i, j, k, n, n0, n1, begin[DISPLAY_DIM], end[DISPLAY_DIM];
    float l_point[DISPLAY_DIM], u_point[DISPLAY_DIM];
    float d, data_min, data_max, data_sum, data_sqr, data_avg;
    Line error_msg, message;

    if (ndata_sets == 0)  return;

    if ((*(print_funcs->start_print))(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    if (fp)
    {
	sprintf(message, "Saving statistics to file '%s'\n\n", file);

	(*(print_funcs->print_message))(message);
    }

    sprintf(message, "Chosen x range is (%5.2f %5.2f %s)\n",
		lower[0], upper[0],
		ref_type == REF_POINTS  ?  "\b"  :  ref_names[ref_type]);

    (*(print_funcs->print_message))(message);

    sprintf(message, "Chosen y range is (%5.2f %5.2f %s)\n",
		lower[1], upper[1],
		ref_type == REF_POINTS  ?  "\b"  :  ref_names[ref_type]);

    (*(print_funcs->print_message))(message);

    for (i = 0; i < ndata_sets; i++)
    {
	if (fp)
	    fprintf(fp, "%s", data_info[i].name);

	sprintf(message, "\nStatistics for data set '%s'\n",
							data_info[i].name);
	(*(print_funcs->print_message))(message);

	for (j = 0; j < DISPLAY_DIM; j++)
	{
	    l_point[j] = lower[j];
	    u_point[j] = upper[j];
	}

	if (fp || (ref_type != REF_POINTS))
	{
	    convert_to_points(ref_type, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, l_point);
	    convert_to_points(ref_type, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, u_point);

	    if (fp)
	    {
		fprintf(fp, "\t%5.2f\t%5.2f", l_point[0], u_point[0]);
		fprintf(fp, "\t%5.2f\t%5.2f", l_point[1], u_point[1]);
	    }

	    sprintf(message, "Chosen x range is (%5.2f %5.2f)\n",
						l_point[0], u_point[0]);
	    (*(print_funcs->print_message))(message);

	    sprintf(message, "Chosen y range is (%5.2f %5.2f)\n",
						l_point[1], u_point[1]);
	    (*(print_funcs->print_message))(message);
	}

	for (j = 0; j < DISPLAY_DIM; j++)
	{
	    begin[j] = l_point[j] - SMALL_NUMBER;
		/* points start at 1, and want ceiling of lower */
	    begin[j] = MAX(0, begin[j]);

	    end[j] = u_point[j];
		/* points start at 1, and want floor of upper */
	    end[j] = MIN(end[j], data_info[i].npoints[j]);

	}

	if (end[0] <= begin[0])  continue;
	if (end[1] <= begin[1])  continue;

	n = data_info[i].npoints[0];

	data_min = LARGE_NUMBER;
	data_max = - LARGE_NUMBER;
	data_sum = 0;
	data_sqr = 0;

	for (k = begin[1]; k < end[1]; k++)
	{
	    for (j = begin[0]; j < end[0]; j++)
	    {
		d = data_info[i].data[j+k*n];
		data_min = MIN(d, data_min);
		data_max = MAX(d, data_max);
		data_sum += d;
		data_sqr += d*d;
	    }
	}

	n0 = end[0] - begin[0];
	n1 = end[1] - begin[1];
	n = n0 * n1;

	data_avg = data_sum / (float) n;

	data_sqr /= (float) n;
	data_sqr -= data_avg*data_avg;
	data_sqr = sqrt(data_sqr);

	if (n > 1)
	    data_sqr *= ((float) n) / ((float) (n-1));

	if (fp)
	{
	    fprintf(fp, "\t%d\t%d\t%d", n0, n1, n);
	    fprintf(fp, "\t%5.2f", data_min);
	    fprintf(fp, "\t%5.2f", data_max);
	    fprintf(fp, "\t%5.2f", data_sum);
	    fprintf(fp, "\t%5.2f", data_avg);
	    fprintf(fp, "\t%5.2f", data_sqr);
	    fprintf(fp, "\n");
	}

	sprintf(message, "\tNumber points used (%d, %d):\t%d\n", n0, n1, n);
	(*(print_funcs->print_message))(message);

	sprintf(message, "\tMinimum data value:\t%5.2f\n", data_min);
	(*(print_funcs->print_message))(message);

	sprintf(message, "\tMaximum data value:\t%5.2f\n", data_max);
	(*(print_funcs->print_message))(message);

	sprintf(message, "\tSum of data values:\t%5.2f\n", data_sum);
	(*(print_funcs->print_message))(message);

	sprintf(message, "\tAverage of data values:\t%5.2f\n", data_avg);
	(*(print_funcs->print_message))(message);

	sprintf(message, "\tStandard deviation of data values:\t%5.2f\n",
								data_sqr);
	(*(print_funcs->print_message))(message);
    }

    (*(print_funcs->end_print))();
}

Status find_data_extremum(String set, int ref_type,
		float *lower, float *upper, Data_extremum_func func,
		Generic_ptr data, String error_msg)
{
    int i, j, k, n, begin[DISPLAY_DIM], end[DISPLAY_DIM];
    int data_min_x, data_min_y, data_max_x, data_max_y, x, y;
    float l_point[DISPLAY_DIM], u_point[DISPLAY_DIM];
    float d, data_min, data_max, z_ppm[DISPLAY_DIM];

    for (i = 0; i < ndata_sets; i++)
    {
	for (j = 0; j < DISPLAY_DIM; j++)
	{
	    l_point[j] = lower[j];
	    u_point[j] = upper[j];
	}

	if (ref_type != REF_POINTS)
	{
	    convert_to_points(ref_type, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, l_point);
	    convert_to_points(ref_type, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, u_point);
	}

	for (j = 0; j < DISPLAY_DIM; j++)
	{
	    begin[j] = l_point[j] - SMALL_NUMBER;
		/* points start at 1, and want ceiling of lower */
	    begin[j] = MAX(0, begin[j]);

	    end[j] = u_point[j];
		/* points start at 1, and want floor of upper */
	    end[j] = MIN(end[j], data_info[i].npoints[j]);

	}

	if (end[0] <= begin[0])  continue;
	if (end[1] <= begin[1])  continue;

	n = data_info[i].npoints[0];

	data_min = LARGE_NUMBER;
	data_max = - LARGE_NUMBER;
	data_min_x = 0;
	data_min_y = 0;
	data_max_x = 0;
	data_max_y = 0;

	for (k = begin[1]; k < end[1]; k++)
	{
	    for (j = begin[0]; j < end[0]; j++)
	    {
		d = data_info[i].data[j+k*n];

		if (d < data_min)
		{
		    data_min = d;
		    data_min_x = j;
		    data_min_y = k;
		}

		if (d > data_max)
		{
		    data_max = d;
		    data_max_x = j;
		    data_max_y = k;
		}
	    }
	}

	if (data_max > -data_min)
	{
	    x = data_max_x;
	    y = data_max_y;
	}
	else
	{
	    x = data_min_x;
	    y = data_min_y;
	}

	d = data_info[i].data[x+y*n];
	x++; /* points need to have 1 added */
	y++;
	z_ppm[0] = x;
	z_ppm[1] = y;

	convert_from_points(REF_PPM, DISPLAY_DIM,
			data_info[i].npoints, data_info[i].ref, z_ppm);

	CHECK_STATUS((*func)(i, set, data_info[i].name, x, z_ppm[0], y, z_ppm[1], d, data, error_msg));
    }

    return  OK;
}

void reset_data_scale_offset()
{
    int i;

    for (i = 0; i < ndata_sets; i++)
	init_data_scale_offset(i);
}

void set_data_scale_offset(int data_set, float *x, float extremum)
{
    int i;
    float s;
    Data_info *d;

    if (data_set >= ndata_sets)
	return;

    d = data_info + data_set;

    COPY_VECTOR(d->center, x, DISPLAY_DIM);
    d->extremum = extremum;

    if (data_set == 0) /* base data set */
	return;

    for (i = 0; i < DISPLAY_DIM; i++)
    {
	s = data_info[0].ref[i].sf / d->ref[i].sf;

	d->scale[i] = s;
	d->offset[i] = x[i] - s*data_info[0].center[i];
    }

    if ((ABS(d->extremum) > SMALL_NUMBER) &&
				(ABS(data_info[0].extremum) > SMALL_NUMBER))
	d->level_scale = d->extremum / data_info[0].extremum;
    else
	d->level_scale = 1; /* not a great choice but protection */
}
