#include "script.h"

#include "list.h"
#include "parser.h"

#include "script_command.h"

#define  NCOMMANDS_ALLOC  50

typedef struct
{
    char msg[20];
    int nstore;
    Parser_store **store;
    Command_func func;
}   Command;

static int ncommands;
static int ncommands_alloc;
static Command *commands = NULL;

static Bool first_run;

static String script_file;

static List include_list;

static void free_command_memory()
{
    if (ncommands_alloc > 0)
    {
	FREE(commands, Command);
	ncommands_alloc = 0;
    }
}

static Status alloc_command_memory(String error_msg)
{
    int n;

    sprintf(error_msg, "allocating script memory for commands");

    n = NCOMMANDS_ALLOC;
    MALLOC(commands, Command, n);
    ncommands_alloc = n;

    return  OK;
}

static Status realloc_command_memory(String error_msg)
{
    int n;

    sprintf(error_msg, "reallocating script memory for commands");

    n = ncommands_alloc + NCOMMANDS_ALLOC;
    REALLOC(commands, Command, n);
    ncommands_alloc = n;

    return  OK;
}

Status setup_command(int nstore, Parser_store **store,
		String cmd_msg, Command_func func, String error_msg)
{
    sprintf(error_msg, "'%s': ", cmd_msg);
    error_msg += strlen(error_msg);

    if (ncommands == ncommands_alloc)
	CHECK_STATUS(realloc_command_memory(error_msg));

    sprintf(error_msg, "allocating script memory for store");
    MALLOC(commands[ncommands].store, Parser_store *, nstore);

    strcpy(commands[ncommands].msg, cmd_msg);
    commands[ncommands].nstore = nstore;
    COPY_VECTOR(commands[ncommands].store, store, nstore);
    commands[ncommands].func = func;

    check_for_button(ncommands, nstore, store);
    check_for_slider(ncommands, nstore, store);

    ncommands++;

    return  OK;
}

Status run_script(int start, String error_msg)
{
    int i, n;
    char *msg;
    Parser_store **s;

    for (i = start; i < ncommands; i++)
    {
	sprintf(error_msg, "'%s' (command %d): ", commands[i].msg, i+1);
	msg = error_msg + strlen(error_msg);
	n = commands[i].nstore;
	s = commands[i].store;

	CHECK_STATUS((*(commands[i].func))(first_run, n, s, msg));
    }

    first_run = FALSE;

    return  OK;
}

static Status check_file(Generic_ptr data)
{
    String file = (String) data;

    if (strcmp(file, script_file))
	return  OK;
    else
	return  ERROR;
}

static Status parse_script(String file, String error_msg)
{
    Generic_ptr data;

    script_file = file;
    sprintf(error_msg, "\"%s\": cannot have files nested in themselves", file);
    CHECK_STATUS(traverse_list(include_list, check_file));

    sprintf(error_msg, "inserting '%s' in include list", file);
    CHECK_STATUS(insert_list(&include_list, (Generic_ptr) file));

    CHECK_STATUS(parser_file(file, script_table, TRUE, error_msg));

    sprintf(error_msg, "deleting '%s' from include list", file);
    CHECK_STATUS(delete_list(&include_list, &data));

    return  OK;
}

#define  INCLUDE_NAME	1

static Status init_include(int nstore, Parser_store **store, String error_msg)
{
    String file = (String) (store[INCLUDE_NAME]->data);

    if (file)
	return  parse_script(file, error_msg);
    else
	RETURN_ERROR_MSG("'include': must have constant string as argument");
}

Status read_script_file (String file, String error_msg)
{
    free_command_memory();

    ncommands = ncommands_alloc = 0;
    first_run = TRUE;

    CHECK_STATUS(alloc_command_memory(error_msg));

    sprintf(error_msg, "initializing include list");
    CHECK_STATUS(init_list(&include_list));

    CHECK_STATUS(init_parser(error_msg));

    if (parse_script(file, error_msg) == ERROR)
    {
	destroy_list(&include_list, (Data_func) NULL);
	return  ERROR;
    }

    return  OK;
}
