#include "convolve.h"

#include "conv.h"
#include "parser.h"
#include "script.h"

#define  CONVOLVE_W	2
#define  CONVOLVE_FILE	2
#define  CONVOLVE_END	3

static int walloc = 0;
static float *filter;
static float *sum_filter;

static int n;
static int w;
static int s;
static float *data_in;
static float  *data_out;

static void free_convolve_memory()
{
    if (walloc > 0)
    {
	FREE(filter, float);
	FREE(sum_filter, float);

	walloc = 0;
    }
}

static Status alloc_convolve_memory(int w, String error_msg)
{
    if (w <= walloc)
	return  OK;

    free_convolve_memory();

    sprintf(error_msg, "allocating convolve memory");

    MALLOC(filter, float, w);
    MALLOC(sum_filter, float, w);

    walloc = w;

    return  OK;
}


static Status check_convolve(int convolve_w, Parser_store **store,
							String error_msg)
{
    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_COMPLEX)
    {
	s = 2;
    }
    else if (store[INPUT_X]->data_type & PARSER_REAL)
    {
	s = 1;
    }

    store_float_to_int(store[convolve_w]);
    w = *((int *) (store[convolve_w]->data));

    if ((w < 1) || ((n/2) < (w+1)))
    {
	sprintf(error_msg, "half width = %d, must be in range 1 to %d",
								w, (n - 2)/2);
	return  ERROR;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    w = 2*w + 1;
    CHECK_STATUS(alloc_convolve_memory(w, error_msg));

    return  OK;
}

static void calc_convolve()
{
    int i;

    n *= s;
    calculate_convolve(n, data_in, data_out, w, filter, sum_filter, s);

    for (i = 0; i < n; i++)
	data_out[i] = data_in[i] - data_out[i];
}

static Status do_convolve(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    CHECK_STATUS(check_convolve(CONVOLVE_W, store, error_msg));

    calculate_sine_filter(w, filter, sum_filter);

    calc_convolve();

    return  OK;
}

static Status do_conv_sine(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    CHECK_STATUS(check_convolve(CONVOLVE_W, store, error_msg));

    calculate_sine_filter(w, filter, sum_filter);

    calc_convolve();

    return  OK;
}

static Status do_conv_box(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    CHECK_STATUS(check_convolve(CONVOLVE_W, store, error_msg));

    calculate_box_filter(w, filter, sum_filter);

    calc_convolve();

    return  OK;
}

static Status do_conv_triangle(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    CHECK_STATUS(check_convolve(CONVOLVE_W, store, error_msg));

    calculate_triangle_filter(w, filter, sum_filter);

    calc_convolve();

    return  OK;
}

static Status do_conv_gaussian(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    float end;

    CHECK_STATUS(check_convolve(CONVOLVE_W, store, error_msg));

    store_int_to_float(store[CONVOLVE_END]);
    end = *((float *) (store[CONVOLVE_END]->data));

    if ((end <= 0) || (end >= 1))
    {
	sprintf(error_msg, "end value = %f, must be in range (0, 1)", end);
	return  ERROR;
    }

    calculate_gaussian_filter(w, end, filter, sum_filter);

    calc_convolve();

    return  OK;
}

static Status do_conv_file(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    String file = (String) (store[CONVOLVE_FILE]->data);

    CHECK_STATUS(check_convolve(CONVOLVE_W+1, store, error_msg)); /* note +1 */

    CHECK_STATUS(read_and_sum_filter(file, w, filter, sum_filter, error_msg));

    calc_convolve();

    return  OK;
}

static Status init_conv(int nstore, Parser_store **store, String cmd,
					Command_func func, String error_msg)
{
    CHECK_STATUS(setup_command(nstore, store, cmd, func, error_msg));

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_convolve(int nstore, Parser_store **store, String error_msg)
{
    return  init_conv(nstore, store, "convolve", do_convolve, error_msg);
}

Status init_conv_sine(int nstore, Parser_store **store, String error_msg)
{
    return  init_conv(nstore, store, "conv_sine", do_conv_sine, error_msg);
}

Status init_conv_box(int nstore, Parser_store **store, String error_msg)
{
    return  init_conv(nstore, store, "conv_box", do_conv_box, error_msg);
}

Status init_conv_triangle(int nstore, Parser_store **store, String error_msg)
{
    return  init_conv(nstore, store, "conv_triangle", do_conv_triangle,
								error_msg);
}

Status init_conv_gaussian(int nstore, Parser_store **store, String error_msg)
{
    return  init_conv(nstore, store, "conv_gaussian", do_conv_gaussian,
								error_msg);
}

Status init_conv_file(int nstore, Parser_store **store, String error_msg)
{
    return  init_conv(nstore, store, "conv_file", do_conv_file, error_msg);
}
