#include "script_find.h"

#include "par.h"
#include "parse.h"
#include "ref.h"

static int ndim;
static int *block_size;
static int *npoints;

static Ref_info *ref;

static char *input_data_file;
static char *output_peak_file = NULL;
static Bool swapped;
static Bool integer;
static Bool blocked;
static int header;
static Bool deflated;
static int dir_size;
static int byte_size;

static String script_file;

static Bool input_found;
static Bool range_found;

static int dim;
static Bool dim_found[MAX_NDIM];

static Bool have_high;
static Bool have_low;
static float high;
static float low;
static Bool nonadjacent;
static Bool parabolic;
static int buffer[MAX_NDIM];
static int first[MAX_NDIM];
static int last[MAX_NDIM];
static Bool periodic[MAX_NDIM];
static int nexclude;
static Exclude exclude[MAX_NDIM*(MAX_NDIM-1)/2];

static int parse_string[] = { PARSE_STRING };
static int parse_int[] = { PARSE_INT };
static int parse_int2[] = { PARSE_INT, PARSE_INT };
static int parse_float[] = { PARSE_FLOAT };
static int parse_int2_float[] = { PARSE_INT, PARSE_INT, PARSE_FLOAT };

#define  CHECK_DIM_NOT_THERE(string) \
	 {   if (dim >= 0) \
	     {   sprintf(error_msg, \
				"in \"%s\" '%s' should be before any 'dim'", \
				script_file, string);  return  ERROR;   }   }

#define  CHECK_DIM_THERE(string) \
	 {   if (dim < 0) \
	     {   sprintf(error_msg, "in \"%s\" '%s' found before any 'dim'", \
				script_file, string);  return  ERROR;   }   }

#define  CHECK_NPTS_THERE(string, dim) \
	 {   if (npts[dim] == 0) \
	     {   sprintf(error_msg, \
			"in \"%s\" for dim %d '%s' found before 'npts'", \
			script_file, dim+1, string);  return  ERROR;   }   }

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				script_file, string);  return  ERROR;   }

#define  FOUND_TWICE_FOR_DIM(string, dim) \
	 {   sprintf(error_msg, "in \"%s\" for dim %d '%s' found twice", \
			script_file, dim+1, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			script_file, string1, string2);  return  ERROR;   }

#define  NOT_ALLOWED(string, condition) \
	 {   sprintf(error_msg, "in \"%s\" '%s' %s", \
			script_file, string, condition);  return  ERROR;   }

#define  NOT_ALLOWED_FOR_DIM(dim, string, condition) \
	 {   sprintf(error_msg, "in \"%s\" for dim %d '%s' %s", \
		script_file, dim+1, string, condition);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				script_file, string);  return  ERROR;   }

#define  NOT_FOUND_FOR_DIM(string, dim) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found for dim %d", \
			script_file, string, dim+1);  return  ERROR;   }

static Status input_data_parse(Generic_ptr *var, String error_msg)
{
    int i;
    String input_par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	FOUND_TWICE("input_data");

    input_found = TRUE;

    CHECK_STATUS(read_par_file(input_par_file, &par_info, error_msg));

    input_data_file = par_info.file;
    ndim = par_info.ndim;
    npoints = par_info.npoints;
    block_size = par_info.block_size;
    ref = par_info.ref;
    swapped = par_info.swapped;
    integer = par_info.integer;
    blocked = par_info.blocked;
    header = par_info.header;
    deflated = par_info.deflated;
    dir_size = par_info.dir_size;
    byte_size = par_info.byte_size;

    if (par_info.deflated)
    {
/*
	sprintf(error_msg, "input data file \"%s\" must not be deflated",
							input_data_file);
	return  ERROR;
*/
	printf("Warning: input data file \"%s\" deflated\n", input_data_file);
    }

    if (!blocked)
    {
	sprintf(error_msg, "input data file \"%s\" must be blocked",
							input_data_file);
	return  ERROR;
    }

    for (i = 0; i < ndim; i++)
    {
	dim_found[i] = FALSE;

	buffer[i] = 0;
	first[i] = 0;  last[i] = npoints[i];
	periodic[i] = TRUE;
    }

    return  OK;
}

static Status output_peak_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output_peak", "input_data");

    if (output_peak_file)
	FOUND_TWICE("output_peak");

    CHECK_DIM_NOT_THERE("output_peak");

    sprintf(error_msg, "allocating memory while reading '%s'", script_file);
    MALLOC(output_peak_file, char, strlen(name)+1);

    strcpy(output_peak_file, name);

    return  OK;
}

static Status high_parse(Generic_ptr *var, String error_msg)
{
    float level = *((float *) var[0]);

    if (!input_found)
	FOUND_BEFORE("high", "input_data");

    if (have_high)
	FOUND_TWICE("high");

    CHECK_DIM_NOT_THERE("high");

    have_high = TRUE;
    high = level;

    return  OK;
}

static Status low_parse(Generic_ptr *var, String error_msg)
{
    float level = *((float *) var[0]);

    if (!input_found)
	FOUND_BEFORE("low", "input_data");

    if (have_low)
	FOUND_TWICE("low");

    CHECK_DIM_NOT_THERE("low");

    have_low = TRUE;
    low = level;

    return  OK;
}

static Status nonadjacent_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("nonadjacent", "input_data");

    if (nonadjacent)
	FOUND_TWICE("nonadjacent");

    CHECK_DIM_NOT_THERE("nonadjacent");

    nonadjacent = TRUE;

    return  OK;
}

static Status parabolic_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("parabolic", "input_data");

    if (parabolic)
	FOUND_TWICE("parabolic");

    CHECK_DIM_NOT_THERE("parabolic");

    parabolic = TRUE;

    return  OK;
}

static Status exclude_parse(Generic_ptr *var, String error_msg)
{
    int i;
    Line msg;
    int dim1 = *((int *) var[0]);
    int dim2 = *((int *) var[1]);
    float delta = *((float *) var[2]);

    if (!input_found)
	FOUND_BEFORE("exclude", "input_data");

    CHECK_DIM_NOT_THERE("exclude");

    if ((dim1 < 1) || (dim1 > ndim))
    {
	sprintf(error_msg, "in \"%s\" exclude 'dim1' = %d, out of range",
		                                        script_file, dim1);
	return  ERROR;
    }

    if ((dim2 < 1) || (dim2 > ndim))
    {
	sprintf(error_msg, "in \"%s\" exclude 'dim2' = %d, out of range",
		                                        script_file, dim2);
	return  ERROR;
    }

    if (dim1 == dim2)
    {
	sprintf(error_msg, "in \"%s\" exclude 'dim1' = 'dim2' = %d",
		                                        script_file, dim1);
	return  ERROR;
    }

    if (delta <= 0)
    {
	sprintf(error_msg, "in \"%s\" exclude 'delta' = %3.2e, must be > 0",
		                                        script_file, delta);
	return  ERROR;
    }

    dim1--;  dim2--;

    if (dim1 > dim2)
	SWAP(dim1, dim2, int);

    for (i = 0; i < nexclude; i++)
    {
	if ((exclude[i].dim1 == dim1) && (exclude[i].dim2 == dim2))
	{
	    sprintf(msg, "exclude %d %d\n", dim1+1, dim2+1);
	    FOUND_TWICE(msg);
	}
    }

    exclude[nexclude].dim1 = dim1;
    exclude[nexclude].dim2 = dim2;
    exclude[nexclude].delta = delta;
    nexclude++;

    return  OK;
}

static Status dim_parse(Generic_ptr *var, String error_msg)
{
    int d = *((int *) var[0]);
    Line msg;

    dim = d;

    if (dim > ndim)
    {
	sprintf(error_msg, "in \"%s\" 'dim' = %d > 'ndim' = %d",
	                                        script_file, dim+1, ndim);
	return  ERROR;
    }

    if (dim < 1)
	NOT_ALLOWED("dim", "< 1");

    dim--;
    if (dim_found[dim] == TRUE)
    {
	sprintf(msg, "dim %d", dim+1);
	FOUND_TWICE(msg);
    }

    dim_found[dim] = TRUE;
    range_found = FALSE;

    return  OK;
}

static Status buffer_parse(Generic_ptr *var, String error_msg)
{
    int d = *((int *) var[0]);

    CHECK_DIM_THERE("buffer");

    if (buffer[dim] > 0)
	FOUND_TWICE_FOR_DIM("buffer", dim);

    if (d <= 0)
    {
	sprintf(error_msg,
		"for dim %d 'buffer' has argument %d, must be > 0", dim+1, d);
	return  ERROR;
    }

    buffer[dim] = d;

    return  OK;
}

static Status range_parse(Generic_ptr *var, String error_msg)
{
    int f = *((int *) var[0]);
    int l = *((int *) var[1]);

    CHECK_DIM_THERE("range");

    if (range_found)
	FOUND_TWICE_FOR_DIM("range", dim);

    if ((f < 1) || (f > l) || (l > npoints[dim]))
    {
	sprintf(error_msg,
"for dim %d 'range' has illegal arguments %d, %d (should be in range 1 to %d)",
						dim+1, f, l, npoints[dim]);
	return  ERROR;
    }

    range_found = TRUE;

    first[dim] = f - 1;
    last[dim] = l;

    return  OK;
}

static Status nonperiodic_parse(Generic_ptr *var, String error_msg)
{
    CHECK_DIM_THERE("nonperiodic");

    if (!periodic[dim])
	FOUND_TWICE_FOR_DIM("nonperiodic", dim);

    periodic[dim] = FALSE;

    return  OK;
}

static Status check_find_validity(String error_msg)
{
    if (!input_data_file)
	NOT_FOUND("input_data");

    if (!output_peak_file)
	NOT_FOUND("output_peak");

    sprintf(error_msg, "in \"%s\": ", script_file);
    error_msg += strlen(error_msg);

    if ((high == 0) && (low == 0))
	RETURN_ERROR_MSG("must have 'high' and/or 'low'");

    return  OK;
}

static Parse_line find_script_table[] =
{
    { "input_data",	1,	parse_string,		input_data_parse },
    { "output_peak",	1,	parse_string,		output_peak_parse },
    { "high",		1,	parse_float,		high_parse },
    { "low",		1,	parse_float,		low_parse },
    { "nonadjacent",	0,	(int *) NULL,		nonadjacent_parse },
    { "parabolic",	0,	(int *) NULL,		parabolic_parse },
    { "exclude",	3,	parse_int2_float,	exclude_parse },
    { "dim",		1,	parse_int,		dim_parse },
    { "buffer",		1,	parse_int,		buffer_parse },
    { "range",		2,	parse_int2,		range_parse },
    { "nonperiodic",	0,	(int *) NULL,		nonperiodic_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

Status read_script_file(String file, Size_info *size_info,
			Ref_info **p_ref, File_info *file_info,
			Find_param *find_param, String error_msg)
{
    script_file = file;

    FREE(output_peak_file, char);

    input_found = FALSE;
    nonadjacent = FALSE;
    parabolic = FALSE;
    dim = -1;
    have_high = have_low = FALSE;
    nexclude = 0;

    CHECK_STATUS(parse_file(script_file, find_script_table, TRUE, error_msg));

    CHECK_STATUS(check_find_validity(error_msg));

    size_info->ndim = ndim;
    size_info->block_size = block_size;
    size_info->npoints = npoints;

    *p_ref = ref;

    file_info->input_data_file = input_data_file;
    file_info->output_peak_file = output_peak_file;
    file_info->swapped = swapped;
    file_info->integer = integer;
    file_info->blocked = blocked;
    file_info->header = header;
    file_info->deflated = deflated;
    file_info->dir_size = dir_size;
    file_info->byte_size = byte_size;

    find_param->have_high = have_high;
    find_param->have_low = have_low;
    find_param->high = high;
    find_param->low = low;
    find_param->nonadjacent = nonadjacent;
    find_param->parabolic = parabolic;
    find_param->buffer = buffer;
    find_param->first = first;
    find_param->last = last;
    find_param->periodic = periodic;
    find_param->nexclude = nexclude;
    find_param->exclude = exclude;

    return  OK;
}
