#include "lin_pred.h"

#include "complex.h"
#include "poly_roots.h"

static void fix_coeffs(float *coeffs, int npoles, Complex *w1, Complex *w2, Complex *w3)
{
    int i, j;
    float r;
    Complex t;
    Bool polish = TRUE;

    COMPLEX_NEW(COMPLEX_ELEMENT(w1, npoles), 1, 0);

    for (i = 0; i < npoles; i++)
        COMPLEX_NEW(COMPLEX_ELEMENT(w1, i), -coeffs[i], 0);

    if (!poly_roots(w1, npoles, w2, polish, w3))
        return; /* not much can do if not converged */

    for (i = 0; i < npoles; i++)
    {
        r = complex_abs(COMPLEX_ELEMENT(w2, i));

        if (r > 1)
        {
            r = 1 / (r * r);
            COMPLEX_SCALE(COMPLEX_ELEMENT(w2, i), r);
        }
    }

    COMPLEX_SET(COMPLEX_ELEMENT(w1, 0), COMPLEX_ELEMENT(w2, 0));
    COMPLEX_SCALE(COMPLEX_ELEMENT(w1, 0), -1);

    COMPLEX_NEW(COMPLEX_ELEMENT(w1, 1), 1, 0);

    for (i = 1; i < npoles; i++)
    {
        COMPLEX_NEW(COMPLEX_ELEMENT(w1, i+1), 1, 0);

        for (j = i; j >= 0; j--)
        {
            COMPLEX_MULTIPLY(t, COMPLEX_ELEMENT(w2, i), COMPLEX_ELEMENT(w1, j));

            if (j > 0)
            {
                COMPLEX_SUBTRACT(COMPLEX_ELEMENT(w1, j), COMPLEX_ELEMENT(w1, j-1), t);
            }
            else
            {
                COMPLEX_SCALE(t, -1);
                COMPLEX_SET(COMPLEX_ELEMENT(w1, j), t);
            }
        }
    }

    for (i = 0; i < npoles; i++)
        coeffs[i] = - COMPLEX_REAL(COMPLEX_ELEMENT(w1, i));
}

void lp_coeffs(float *data, int npoints, int step,
		int npoles, float *coeffs, float *w1, float *w2, float *w3)
{
    int i, j, m;
    float a, b, c;

    w1[0] = data[0];
    w2[npoints-2] = data[step * (npoints - 1)];

    for (i = 1; i < npoints-1; i++)
	w2[i-1] = w1[i] = data[step * i];

    for (j = 0; j < npoles; j++)
    {
	m = npoints - j - 1;
	INNER_PRODUCT(a, w1, w2, m);
	INNER_PRODUCT(b, w1, w1, m);
	INNER_PRODUCT(c, w2, w2, m);

	coeffs[j] = 2 * a / (b + c);

	for (i = 0; i < j; i++)
	    coeffs[i] = w3[i] - coeffs[j]*w3[j-i-1];

	if (j == (npoles-1))
	    break;

	COPY_VECTOR(w3, coeffs, j+1);

	for (i = 0; i < npoints-j-1; i++)
	{
	    w1[i] -= w3[j] * w2[i];
	    w2[i] = w2[i+1] - w3[j] * w1[i+1];
	}
    }

    /* reverse direction of coeffs */
    for (i = 0; i < npoles/2; i++)
        SWAP(coeffs[i], coeffs[npoles - i - 1], float);
}

static void do_prediction(float *data, int npoints, int npred, int step,
                                                int npoles, float *coeffs)
{
    int i, j;
    float s;

    for (i = 0; i < npred; i++)
    {
        s = 0;

        for (j = 0; j < npoles; j++)
            s += coeffs[j] * data[step * (npoints - npoles + i + j)];

        data[step * (npoints + i)] = s;
    }
}

void lin_pred(float *data, int npoints, int npred, int step,
                int npoles, float *coeffs, float *w1, float *w2, float *w3)
{
    lp_coeffs(data, npoints, step, npoles, coeffs, w1, w2, w3);
    fix_coeffs(coeffs, npoles, (Complex *) w1, (Complex *) w2, (Complex *) w3);
    do_prediction(data, npoints, npred, step, npoles, coeffs);
}
