#include "extract.h"

#include "block.h"
#include "par.h"
#include "parse.h"
#include "ref.h"
#include "utility.h"

static int ndim;
static int *block_size_in;
static int *npoints_in;

static int ndim_extr;
static int ndim_rest;
static int dim_extr[MAX_NDIM];
static int dim_rest[MAX_NDIM];
static int point_rest[MAX_NDIM];
static int block_size_out[MAX_NDIM];
static int npoints_out[MAX_NDIM];

static int nstore_in;
static int nstore_out;
static float *store_in;
static float *store_out;

static Ref_info *ref_in;
static Ref_info ref_out[MAX_NDIM];

static char *input_file;
static Line output_file;
static FILE *file_in;
static FILE *file_out;
static Bool swapped;
static Bool integer;
static Bool blocked;
static int header;
static Bool deflated;
static float level;
static int dir_size;
static int *directory;

static Bool input_found;
static Bool output_found;
static Bool par_found;
static Bool extract_found;
static Line output_par_file;
static char *extract_file;

static int parse_string[] = { PARSE_STRING };
static int parse_string_free[] = { PARSE_STRING | PARSE_FREE };

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				extract_file, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			extract_file, string1, string2);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				extract_file, string);  return  ERROR;   }

static Status allocate_memory(String error_msg)
{
    sprintf(error_msg, "allocating memory for store_in");
    MALLOC(store_in, float, nstore_in);

    sprintf(error_msg, "allocating memory for store_out");
    MALLOC(store_out, float, nstore_out);

    if (deflated)
    {
	sprintf(error_msg, "allocating memory for directory");
	MALLOC(directory, int, dir_size);
    }

    return  OK;
}

static void determine_params()
{
    int i, j, n;

    ndim_extr = ndim - ndim_rest;

    n = 0;
    for (i = 0; i < ndim; i++)
    {
        for (j = 0; j < ndim_rest; j++)
            if (i == dim_rest[j])
                break;

        if (j == ndim_rest)
            dim_extr[n++] = i;
    }

    VECTOR_PRODUCT(nstore_in, block_size_in, ndim);

    for (i = 0; i < ndim_extr; i++)
    {
	j = dim_extr[i];
	npoints_out[i] = npoints_in[j];
	block_size_out[i] = block_size_in[j];

	ref_out[i] = ref_in[j];
    }

    nstore_out = nstore_in;

    find_block_sizes(ndim_extr, nstore_out, npoints_out, block_size_out, TRUE);
}

static Status input_parse(Generic_ptr *var, String error_msg)
{
    String input_par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	FOUND_TWICE("input");

    CHECK_STATUS(read_par_file(input_par_file, &par_info, error_msg));

    input_file = par_info.file;
    ndim = par_info.ndim;
    npoints_in = par_info.npoints;
    block_size_in = par_info.block_size;
    ref_in = par_info.ref;
    swapped = par_info.swapped;
    integer = par_info.integer;
    blocked = par_info.blocked;
    header = par_info.header;
    deflated = par_info.deflated;
    level = par_info.level;
    dir_size = par_info.dir_size;

    if (!blocked)
    {
	sprintf(error_msg, "input file \"%s\" must be blocked", input_file);
	return  ERROR;
    }

    if (deflated)
    {
/*
	sprintf(error_msg, "input file \"%s\" must not be deflated",
								input_file);
	return  ERROR;
*/
	printf("Warning: input file \"%s\" deflated\n", input_file);
    }

    input_found = TRUE;

    return  OK;
}

static Status output_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output", "input");

    if (output_found)
	FOUND_TWICE("output");

    strcpy(output_file, name);

    output_found = TRUE;

    return  OK;
}

static Status par_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("par", "input");

    if (par_found)
	FOUND_TWICE("par");

    strcpy(output_par_file, name);

    par_found = TRUE;

    return  OK;
}

static Status extract_parse(Generic_ptr *var, String error_msg)
{
    int n;
    char *ptr;

    if (!output_found)
	FOUND_BEFORE("extract", "output");

    if (extract_found)
	FOUND_TWICE("extract");

    ndim_rest = *((int *) var[0]);

    if (ndim_rest > ndim)
    {
	sprintf(error_msg,
	    "in \"%s\" have 'extract' with %d arguments, but 'ndim' = %d",
					extract_file, ndim_rest, ndim);
	return  ERROR;
    }

    if (ndim_rest < 1)
    {
	sprintf(error_msg,
	    "in \"%s\" have 'extract' with no arguments", extract_file);
	return  ERROR;
    }

    for (n = 0; n < ndim_rest; n++)
    {
	ptr = (char *) var[n+1];

	if (sscanf(ptr, "%d:%d", dim_rest+n, point_rest+n) == 2)
	{
	    if (dim_rest[n] < 1)
	    {
		sprintf(error_msg, "in \"%s\" extracted dim #%d = %d",
					extract_file, n+1, dim_rest[n]);
		return  ERROR;
	    }

	    if (dim_rest[n] > ndim)
	    {
		sprintf(error_msg,
			"in \"%s\" extracted dim #%d = %d, but 'ndim' = %d",
					extract_file, n+1, dim_rest[n], ndim);
		return  ERROR;
	    }

	    dim_rest[n]--;

	    if (point_rest[n] < 1)
	    {
		sprintf(error_msg, "in \"%s\" extracted point #%d = %d",
					extract_file, n+1, point_rest[n]);
		return  ERROR;
	    }

	    if (point_rest[n] > npoints_in[dim_rest[n]])
	    {
		sprintf(error_msg,
			"in \"%s\" extracted point #%d = %d, but 'npts' = %d",
					extract_file, n+1, point_rest[n],
					npoints_in[dim_rest[n]]);
		return  ERROR;
	    }

	    point_rest[n]--;
	}
	else
	{
	    sprintf(error_msg,
			"in \"%s\" cannot read extracted (dim, point) #%d",
							extract_file, n+1);
	    return  ERROR;
	}
    }

    extract_found = TRUE;

    return  OK;
}

static Parse_line extract_table[] =
{
    { "input",		1,	parse_string,		input_parse },
    { "output",		1,	parse_string,		output_parse },
    { "par",		1,	parse_string,		par_parse },
    { "extract",	1,	parse_string_free,	extract_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

static Status read_extract_file(String error_msg)
{
    int i, j;

    input_found = FALSE;
    output_found = FALSE;
    par_found = FALSE;
    extract_found = FALSE;
    ndim_rest = 0;

    CHECK_STATUS(parse_file(extract_file, extract_table, TRUE, error_msg));

    if (!input_found)
	NOT_FOUND("input");

    if (!output_found)
	NOT_FOUND("output");

    if (ndim_rest < 1)
	NOT_FOUND("extract");

    for (i = 0; i < ndim_rest; i++)
    {
	for (j = i+1; j < ndim_rest; j++)
	{
	    if (dim_rest[j] == dim_rest[i])
	    {
		sprintf(error_msg, "in \"%s\" extracted dim #%d = dim#%d = %d",
				extract_file, i+1, j+1, dim_rest[i]+1);
		return  ERROR;
	    }
	}
    }

    return  OK;
}

static void print_extract_info()
{
    printf("Number of dimensions of input data = %d\n", ndim);
    printf("Number of dimensions of output data = %d\n", ndim_extr);

    printf("Fixed dimension%s = ", ndim_rest == 1 ? "" : "s");
    print_integer_array(ndim_rest, dim_rest);
    printf(", at the point%s = ", ndim_rest == 1 ? "" : "s");
    print_integer_array(ndim_rest, point_rest);
    printf("\n");
}

void main(int argc, char **argv)
{
    Line error_msg;
    Size_info size_in, size_out;
    Store_info store_info;
    File_info file_info;
    Extract_info extract_info;
    Par_info par_info;
    String par_file;

    printf(product);

    if (help_request(argc, argv, help_table))
	exit (0);

    if (argc != 2)
    {
        sprintf(error_msg, "correct usage: %s <extract file>", argv[0]);
        ERROR_AND_EXIT(error_msg);
    }

    extract_file = argv[1];

    if (read_extract_file(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    determine_params();

    if (allocate_memory(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    if (OPEN_FOR_BINARY_READING(file_in, input_file))
    {
	sprintf(error_msg, "opening \"%s\" for reading", input_file);
	ERROR_AND_EXIT(error_msg);
    }

    if (OPEN_FOR_BINARY_WRITING(file_out, output_file))
    {
	sprintf(error_msg, "opening \"%s\" for writing", output_file);
	ERROR_AND_EXIT(error_msg);
    }

    size_in.ndim = ndim;
    size_in.block_size = block_size_in;
    size_in.npoints = npoints_in;

    size_out.ndim = ndim_extr;
    size_out.block_size = block_size_out;
    size_out.npoints = npoints_out;

    store_info.store_in = store_in;
    store_info.store_out = store_out;
    store_info.directory = directory;

    file_info.input_file = input_file;
    file_info.output_file = output_file;
    file_info.file_in = file_in;
    file_info.file_out = file_out;
    file_info.swapped = swapped;
    file_info.integer = integer;
    file_info.blocked = blocked;
    file_info.header = header;
    file_info.deflated = deflated;
    file_info.dir_size = dir_size;

    extract_info.dim_rest = dim_rest;
    extract_info.point_rest = point_rest;

    print_extract_info();
    FLUSH;

    if (block_process(&size_in, &size_out, &store_info,
		&file_info, &extract_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    par_info.file = output_file;
    par_info.ndim = ndim_extr;
    par_info.npoints = npoints_out;
    par_info.block_size = block_size_out;
    par_info.ref = ref_out;
    par_info.blocked = TRUE;
    par_info.deflated = FALSE;
    par_info.level = level;
    par_info.param_dim = -1;

    if (par_found)
	par_file = output_par_file;
    else
	par_file = NULL;

    if (write_par_file(par_file, &par_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);
}
