#include "components.h"

#include "block.h"
#include "par.h"
#include "parse.h"
#include "ref.h"

static int ndim;
static int block_size[MAX_NDIM];
static int npoints[MAX_NDIM];

static Ref_info *ref;

static float nstore;
static float **store_in;
static float **store_out;
static float **matrix;
static float *eigenvalues;
static float *work;
static int **order;

static int ninput_files;
static int noutput_files;
static Line *input_files;
static Line *output_data_files;
static Line output_data_prefix;
static Line output_weights_file;
static FILE **file_in;
static FILE **file_out;
static FILE *file_weights;
static Bool swapped;
static Bool integer;
static Bool blocked;
static int header;
static Bool deflated;
static float level;

static Bool input_found;
static Bool output_data_found;
static Bool output_weights_found;
static Bool components_found;
static Bool fraction_found;
static Bool correlation_found;
static char *components_file;

static Bool correlation;
static int ncomponents;
static float fraction;

static int parse_int[] = { PARSE_INT };
static int parse_float[] = { PARSE_FLOAT };
static int parse_string[] = { PARSE_STRING };

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found twice", \
				components_file, string);  return  ERROR;   }

#define  FOUND_BEFORE(string1, string2) \
	 {   sprintf(error_msg, "in \"%s\" '%s' found before '%s'", \
			components_file, string1, string2);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "in \"%s\" no '%s' found", \
				components_file, string);  return  ERROR;   }

#define  CHECK_EQUAL(n1, n2, string) \
	 {   if (n1 != n2) \
	     {   sprintf(error_msg, "in \"%s\": '%s' has inconsistent value", \
				components_file, string);  return  ERROR;   }   }

#define  CHECK_EQUAL_VECTORS(n, v1, v2, string) \
	 {   int I; \
	     for (I = 0; I < n; I++) \
	     {   if (v1[I] != v2[I]) \
	         {   sprintf(error_msg, \
			"in \"%s\": '%s[%d]' has inconsistent value", \
			components_file, string, I+1);  return  ERROR;   }   }   }

static Status allocate_memory1(String error_msg)
{
    sprintf(error_msg, "allocating memory for input_files");
    MALLOC(input_files, Line, ninput_files);

    sprintf(error_msg, "allocating memory for file_in");
    MALLOC(file_in, FILE *, ninput_files);

    return  OK;
}

static Status allocate_memory2(String error_msg)
{
    int i;

    sprintf(error_msg, "allocating memory for store_in");
    MALLOC(store_in, float *, ninput_files);

    for (i = 0; i < ninput_files; i++)
    {
	sprintf(error_msg, "allocating memory for store_in[%d]", i);
	MALLOC(store_in[i], float, nstore);
    }

    sprintf(error_msg, "allocating memory for matrix");
    MALLOC(matrix, float *, ninput_files);

    for (i = 0; i < ninput_files; i++)
    {
	sprintf(error_msg, "allocating memory for matrix[%d]", i);
	MALLOC(matrix[i], float, ninput_files);
    }

    sprintf(error_msg, "allocating memory for eigenvalues");
    MALLOC(eigenvalues, float, ninput_files);

    sprintf(error_msg, "allocating memory for work");
    MALLOC(work, float, ninput_files);

    sprintf(error_msg, "allocating memory for order");
    MALLOC(order, int *, ninput_files);

    for (i = 0; i < ninput_files; i++)
    {
	sprintf(error_msg, "allocating memory for order[%d]", i);
	MALLOC(order[i], int, i);
	*order[i] = i;
    }

    return  OK;
}

static Status allocate_memory3(String error_msg)
{
    int i;

    if (output_data_found)
    {
	sprintf(error_msg, "allocating memory for output_data_files");
	MALLOC(output_data_files, Line, noutput_files);

	for (i = 0; i < noutput_files; i++)
	    sprintf(output_data_files[i], "%s%d", output_data_prefix, i+1);

	sprintf(error_msg, "allocating memory for file_out");
	MALLOC(file_out, FILE *, noutput_files);

	sprintf(error_msg, "allocating memory for store_out");
	MALLOC(store_out, float *, noutput_files);

	for (i = 0; i < noutput_files; i++)
	{
	    sprintf(error_msg, "allocating memory for store_out[%d]", i);
	    MALLOC(store_out[i], float, nstore);
	}
    }

    return  OK;
}

static void determine_params()
{
    int i;

    if (!blocked)  /* bit of a fudge, but it ought to work */
    {
	block_size[0] = npoints[0];

	for (i = 1; i < ndim; i++)
	    block_size[i] = 1;
    }

    VECTOR_PRODUCT(nstore, block_size, ndim);
}

static Status read_par_files(String error_msg)
{
    int i;
    Par_info par_info;

    for (i = 0; i < ninput_files; i++)
    {
	CHECK_STATUS(read_par_file(input_files[i], &par_info, error_msg));

	strcpy(input_files[i], par_info.file);

	if (i == 0)
	{
	    ndim = par_info.ndim;
	    COPY_VECTOR(npoints, par_info.npoints, ndim);
	    swapped = par_info.swapped;
	    integer = par_info.integer;
	    blocked = par_info.blocked;
	    header = par_info.header;
	    deflated = par_info.deflated;
	    level = par_info.level;

	    if (blocked)
		COPY_VECTOR(block_size, par_info.block_size, ndim);

	    if (deflated)
	    {
		sprintf(error_msg, "input files cannot be deflated");
		return  ERROR;
	    }
	}
	else
	{
	    CHECK_EQUAL(ndim, par_info.ndim, "ndim");
	    CHECK_EQUAL(swapped, par_info.swapped, "swap");
	    CHECK_EQUAL(integer, par_info.integer, "int");
	    CHECK_EQUAL(blocked, par_info.blocked, "blocking");
	    CHECK_EQUAL(header, par_info.header, "head");
	    CHECK_EQUAL(deflated, par_info.deflated, "deflation");
	    CHECK_EQUAL_VECTORS(ndim, npoints, par_info.npoints, "npts");

	    if (blocked)
		CHECK_EQUAL_VECTORS(ndim, block_size, par_info.block_size, "block");
	}
    }

    ref = par_info.ref;

    return  OK;
}

static Status input_list_parse(Generic_ptr *var, String error_msg)
{
    String file = (String) (var[0]);
    int i;
    Line line;
    FILE *fp;

    if (input_found)
	FOUND_TWICE("input_list");

    CHECK_OPEN_FOR_READING(fp, file);

    for (ninput_files = 0; fgets(line, LINE_SIZE, fp); ninput_files++)
	;

    FCLOSE(fp);

    if (ninput_files < 2)
	RETURN_ERROR_MSG("need at least two files on list");

    CHECK_STATUS(allocate_memory1(error_msg));

    CHECK_OPEN_FOR_READING(fp, file);

    for (i = 0; i < ninput_files; i++)
    {
	if (!fgets(line, LINE_SIZE, fp)) /* should not happen */
	{
	    sprintf(error_msg, "could not read name of file #%d on list", i+1);
	    return  ERROR;
	}

	STRIP_LEADING_SPACE(line);
	STRIP_TRAILING_SPACE(line);

	strcpy(input_files[i], line);
    }

    FCLOSE(fp);

    CHECK_STATUS(read_par_files(error_msg));

    input_found = TRUE;

    ncomponents = ninput_files / 10;
    ncomponents = MAX(2, ncomponents);

    return  OK;
}

static Status output_data_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output_data", "input_list");

    if (output_data_found)
	FOUND_TWICE("output_data");

    strcpy(output_data_prefix, name);

    output_data_found = TRUE;

    return  OK;
}

static Status output_weights_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (!input_found)
	FOUND_BEFORE("output_weights", "input_list");

    if (output_weights_found)
	FOUND_TWICE("output_weights");

    strcpy(output_weights_file, name);

    output_weights_found = TRUE;

    return  OK;
}

static Status components_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("components", "input_list");

    if (components_found)
	FOUND_TWICE("components");

    components_found = TRUE;

    ncomponents = *((int *) var[0]);

    if ((ncomponents < 1) || (ncomponents > ninput_files))
    {
	sprintf(error_msg,
	    "in \"%s\" have 'components %d', must be in range (1, %d)",
				components_file, ncomponents, ninput_files);
	return  ERROR;
    }

    return  OK;
}

static Status fraction_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("fraction", "input_list");

    if (fraction_found)
	FOUND_TWICE("fraction");

    fraction_found = TRUE;

    fraction = *((float *) var[0]);

    if ((fraction < 0) || (fraction > 0.999))
    {
	sprintf(error_msg,
	    "in \"%s\" have 'fraction %4.3f', must be in range [0, 0.999]",
					components_file, fraction);
	return  ERROR;
    }

    return  OK;
}

static Status correlation_parse(Generic_ptr *var, String error_msg)
{
    if (!input_found)
	FOUND_BEFORE("correlation", "input_list");

    if (correlation_found)
	FOUND_TWICE("correlation");

    correlation_found = TRUE;

    correlation = TRUE;

    return  OK;
}

static Parse_line components_table[] =
{
    { "input_list",	1,	parse_string,		input_list_parse },
    { "output_data",	1,	parse_string,		output_data_parse },
    { "output_weights",	1,	parse_string,		output_weights_parse },
    { "components",	1,	parse_int,		components_parse },
    { "fraction",	1,	parse_float,		fraction_parse },
    { "correlation",	0,	(int *) NULL,		correlation_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

static Status read_components_file(String error_msg)
{
    input_found = FALSE;
    output_data_found = FALSE;
    output_weights_found = FALSE;
    components_found = FALSE;
    correlation_found = FALSE;
    fraction_found = FALSE;

    correlation = FALSE;
    fraction = 0.95;

    CHECK_STATUS(parse_file(components_file, components_table, TRUE, error_msg));

    if (!input_found)
	NOT_FOUND("input_list");

    if (!output_data_found && !output_weights_found)
	NOT_FOUND("output_data/output_weights");

    return  OK;
}

static void print_components_info()
{
    printf("number of dimensions of data = %d\n", ndim);
    printf("number of input files = %d\n", ninput_files);
    printf("minimum number of components being found = %d\n", ncomponents);
    printf("minimum fraction of total weight being found = %4.3f\n", fraction);
    printf("using %s calculation\n",
		correlation ? "correlation" : "covariance");

    if (output_data_found)
	printf("will output data\n");

    if (output_weights_found)
	printf("will output weights\n");
}

static void output_weights(FILE *fp)
{
    int i, j, o;

    fprintf(fp, "component");

    for (i = 0; i < ncomponents; i++)
	fprintf(fp, "\t%d", i+1);

    fprintf(fp, "\n");

    for (i = 0; i < ncomponents; i++)
    {
	o = *order[i];
	fprintf(fp, "\t%4.3f", eigenvalues[o]);
    }

    fprintf(fp, "\n");

    fprintf(fp, "name\n");

    for (i = 0; i < ninput_files; i++)
    {
	fprintf(fp, "%s", input_files[i]);

	for (j = 0; j < ncomponents; j++)
	{
	    o = *order[j];
	    fprintf(fp, "\t%4.3f", matrix[i][o]);
	}

	fprintf(fp, "\n");
    }
}

void main(int argc, char **argv)
{
    int i;
    Line error_msg;
    Size_info size_info;
    Store_info store_info;
    File_info file_info;
    Components_info components_info;
    Par_info par_info;
    String par_file;

    printf(product);

    if (help_request(argc, argv, help_table))
	exit (0);

    if (argc != 2)
    {
        sprintf(error_msg, "correct usage: %s <components file>", argv[0]);
        ERROR_AND_EXIT(error_msg);
    }

    components_file = argv[1];

    if (read_components_file(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    determine_params();

    if (allocate_memory2(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    for (i = 0; i < ninput_files; i++)
    {
	if (OPEN_FOR_BINARY_READING(file_in[i], input_files[i]))
	{
	    sprintf(error_msg, "opening \"%s\" for reading", input_files[i]);
	    ERROR_AND_EXIT(error_msg);
	}
    }

    size_info.ndim = ndim;
    size_info.block_size = block_size;
    size_info.npoints = npoints;

    store_info.store_in = store_in;
    store_info.matrix = matrix;
    store_info.eigenvalues = eigenvalues;
    store_info.work = work;
    store_info.order = order;

    file_info.ninput_files = ninput_files;
    file_info.input_files = input_files;
    file_info.file_in = file_in;
    file_info.swapped = swapped;
    file_info.integer = integer;
    file_info.blocked = TRUE;  /* even if not blocked */
    file_info.header = header;

    components_info.ncomponents = ncomponents;
    components_info.fraction = fraction;
    components_info.correlation = correlation;

    print_components_info();
    FLUSH;

    if (block_process(&size_info, &store_info, &file_info,
				&components_info, error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);

    ncomponents = components_info.ncomponents;

    printf("found %d components\n", ncomponents);
    printf("found %4.3f fraction of components\n", components_info.fraction);

    if (output_data_found)
    {
	noutput_files = ncomponents;

	if (allocate_memory3(error_msg) == ERROR)
            ERROR_AND_EXIT(error_msg);

	store_info.store_out = store_out;

	file_info.noutput_files = noutput_files;
	file_info.output_files = output_data_files;
	file_info.file_out = file_out;

	for (i = 0; i < noutput_files; i++)
	{
	    if (OPEN_FOR_BINARY_WRITING(file_out[i], output_data_files[i]))
	    {
		sprintf(error_msg,
			"opening \"%s\" for writing", output_data_files[i]);
		ERROR_AND_EXIT(error_msg);
	    }
	}

	for (i = 0; i < ninput_files; i++)
	    rewind(file_in[i]);

	if (block_output(&size_info, &store_info, &file_info,
				&components_info, error_msg) == ERROR)
	    ERROR_AND_EXIT(error_msg);

	for (i = 0; i < noutput_files; i++)
	{
	    FCLOSE(file_out[i]);

	    par_info.file = output_data_files[i];
	    par_info.ndim = ndim;
	    par_info.npoints = npoints;
	    par_info.block_size = block_size;
	    par_info.ref = ref;
	    par_info.blocked = blocked;
	    par_info.deflated = deflated;
	    par_info.level = level;
	    par_info.param_dim = -1;

	    par_file = NULL;

	    if (write_par_file(par_file, &par_info, error_msg) == ERROR)
		ERROR_AND_EXIT(error_msg);
	}
    }

    if (output_weights_found)
    {
	if (OPEN_FOR_WRITING(file_weights, output_weights_file))
	{
	    sprintf(error_msg,
			"opening \"%s\" for writing", output_weights_file);
	    ERROR_AND_EXIT(error_msg);
	}

	printf("outputting weights\n");

	output_weights(file_weights);

	FCLOSE(file_weights);
    }

    for (i = 0; i < ninput_files; i++)
	FCLOSE(file_in[i]);
}
