/*  Note, program below assumes that npoints_out and block_size_out are
    equal to npoints_in and block_size_in in the non-combined dimensions  */

#include "block.h"

#include "block_io.h"

#define  MAX_NINPUT_FILES  4

static int ndim;
static int *npoints_in;
static int *npoints_out;
static int *block_size_in;
static int *block_size_out;

static float **row_in;
static float *row_out;
static float **store_in;
static float *store_out;

static int ninput_files;

static int dim;
static int type;
static int step;
static int ndim_rest;
static int nchunks;
static int nrows;
static int size_of_block_in;
static int size_of_block_out;
static int row_step_in;
static int row_step_out;
static int block_step_in;
static int block_step_out;

static int dim_rest[MAX_NDIM];
static int nblocks_in[MAX_NDIM];
static int nblocks_out[MAX_NDIM];
static int blocks[MAX_NDIM];
static int points[MAX_NDIM];
static int end_points[MAX_NDIM];
static int cum_nblocks[MAX_NDIM];
static int cum_nblocks_in[MAX_NDIM];
static int cum_nblocks_out[MAX_NDIM];
static int cum_block_size[MAX_NDIM];
static int cum_block_size_in[MAX_NDIM];
static int cum_block_size_out[MAX_NDIM];
static int array[MAX_NDIM];

static Block_IO block_io_in[MAX_NINPUT_FILES];
static Block_IO block_io_out;

static Combine_func combine_func;

static void init_arrays()
{
    int i, j, n;

    ndim_rest = ndim - 1;

    j = 0;
    for (i = 0; i < ndim; i++)
    {
	if (i == dim)
	    continue;

	dim_rest[j++] = i;
    }

    BLOCKS(nblocks_in, npoints_in, block_size_in, ndim);
    BLOCKS(nblocks_out, npoints_out, block_size_out, ndim);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	array[i] = nblocks_in[j];
    }

    CUMULATIVE(cum_nblocks, array, nchunks, ndim_rest);

    CUMULATIVE(array, nblocks_in, n, ndim);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	cum_nblocks_in[i] = array[j];
    }

    block_step_in = array[dim];

    CUMULATIVE(array, nblocks_out, n, ndim);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	cum_nblocks_out[i] = array[j];
    }

    block_step_out = array[dim];

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	array[i] = block_size_in[j];
    }

    if (type == COMPLEX_DATA)
	step = 2;
    else
	step = 1;

    array[0] /= step;

    CUMULATIVE(cum_block_size, array, nrows, ndim_rest);

    CUMULATIVE(array, block_size_in, size_of_block_in, ndim);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	cum_block_size_in[i] = array[j];
    }

    row_step_in = array[dim];

    CUMULATIVE(array, block_size_out, size_of_block_out, ndim);

    for (i = 0; i < ndim_rest; i++)
    {
	j = dim_rest[i];
	cum_block_size_out[i] = array[j];
    }

    row_step_out = array[dim];
}

static Status disk_to_store(int block, int store_block, String error_msg)
{
    int i;
    float *s;
    char *msg;

    for (i = 0; i < ninput_files; i++)
    {
	s = store_in[i] + size_of_block_in * store_block;

	sprintf(error_msg, "block %d: ", i+1);
	msg = error_msg + strlen(error_msg);
	CHECK_STATUS(read_file_block(&(block_io_in[i]), block, s, msg));
    }

    return  OK;
}

static Status store_to_disk(int block, int store_block, String error_msg)
{
    float *s;

    s = store_out + size_of_block_out * store_block;

    CHECK_STATUS(write_file_block(&block_io_out, block, s, error_msg));

    return  OK;
}

static void store_to_row(int point)
{
    int i, j, k, n, p, ind;

    for (i = 0; i < ninput_files; i++)
    {
	ind = 0;
	for (j = 0; j < nblocks_in[dim]-1; j++)
	{
	    p = point + j*size_of_block_in;

	    for (k = 0; k < block_size_in[dim]; k++)
	    {
		row_in[i][ind++] = store_in[i][p];

		if (step == 2)
		    row_in[i][ind++] = store_in[i][p+1];

		p += row_step_in;
	    }
	}

	n = 1 + (npoints_in[dim]-1) % block_size_in[dim];
	p = point + j*size_of_block_in;

	for (k = 0; k < n; k++)
	{
	    row_in[i][ind++] = store_in[i][p];

	    if (step == 2)
		row_in[i][ind++] = store_in[i][p+1];

	    p += row_step_in;
	}
    }
}

static void row_to_store(int point)
{
    int j, k, /*n,*/ p, ind;

    ind = 0;
    for (j = 0; j < nblocks_out[dim]-1; j++)
    {
	p = point + j*size_of_block_out;

	for (k = 0; k < block_size_out[dim]; k++)
	{
	    store_out[p] = row_out[ind++];

	    if (step == 2)
		store_out[p+1] = row_out[ind++];

	    p += row_step_out;
	}
    }

/*
    n = 1 + (npoints_out[dim]-1) % block_size_out[dim];
*/
    p = point + j*size_of_block_out;

    for (k = 0; k < block_size_out[dim]; k++)
    {
	store_out[p] = row_out[ind++];

	if (step == 2)
	    store_out[p+1] = row_out[ind++];

	p += row_step_out;
    }
}

static void find_end_points()
{
    int i, j;

    for (j = 0; j < ndim_rest; j++)
    {
	i = dim_rest[j];

	if (blocks[j] == (nblocks_in[i]-1))
	    end_points[j] = 1 + (npoints_in[i]-1) % block_size_in[i];
	else
	    end_points[j] = block_size_in[i];
    }
}

static Bool do_row()
{
    int i;

    for (i = 0; i < ndim_rest; i++)
	if (points[i] >= end_points[i])
	    return  FALSE;

    return  TRUE;
}

Status block_process(Size_info *size_info, Store_info *store_info,
	File_info *file_info, Combine_info *combine_info, String error_msg)
{
    int i, j, block, store_block, point;
    char *msg;

    ndim = size_info->ndim;
    npoints_in = size_info->npoints_in;
    npoints_out = size_info->npoints_out;
    block_size_in = size_info->block_size_in;
    block_size_out = size_info->block_size_out;

    row_in = store_info->row_in;
    row_out = store_info->row_out;
    store_in = store_info->store_in;
    store_out = store_info->store_out;

    ninput_files = file_info->ninput_files;

    dim = combine_info->dim;
    type = combine_info->type;
    combine_func = combine_info->combine_func;

    if (!(file_info->blocked))
	RETURN_ERROR_MSG("input file must be blocked");

    init_arrays();

    for (i = 0; i < ninput_files; i++)
    {
	block_io_in[i].name = file_info->input_file[i];
	block_io_in[i].file = file_info->file_in[i];
	block_io_in[i].swapped = file_info->swapped;
	block_io_in[i].integer = file_info->integer;
	block_io_in[i].deflated = FALSE;
	block_io_in[i].header = file_info->header;
	block_io_in[i].dir_size = 0;
	block_io_in[i].directory = (int *) NULL;
	block_io_in[i].block_size = size_of_block_in;
	block_io_in[i].byte_size = BYTES_PER_WORD;

	CHECK_STATUS(init_block_read(&(block_io_in[i]), error_msg));
    }

    block_io_out.name = file_info->output_file;
    block_io_out.file = file_info->file_out;
    block_io_out.block_size = size_of_block_out;
    block_io_out.deflated = FALSE;

    CHECK_STATUS(init_block_write(&block_io_out, error_msg));

    for (i = 0; i < nchunks; i++)
    {
	printf("\t... working on chunk %d of %d\n", i+1, nchunks);
	FLUSH;

	sprintf(error_msg, "chunk %d: ", i+1);
	msg = error_msg + strlen(error_msg);

	ARRAY_OF_INDEX(blocks, i, cum_nblocks, ndim_rest);

	store_block = 0;
	INDEX_OF_ARRAY(block, blocks, cum_nblocks_in, ndim_rest);

	for (j = 0; j < nblocks_in[dim]; j++)
	{
	    CHECK_STATUS(disk_to_store(block, store_block, msg));

	    block += block_step_in;
	    store_block++;
	}

	find_end_points();

	for (j = 0; j < nrows; j++)
	{
	    ARRAY_OF_INDEX(points, j, cum_block_size, ndim_rest);
	    points[0] *= step;

	    if (do_row())
	    {
		INDEX_OF_ARRAY(point, points, cum_block_size_in, ndim_rest);
		store_to_row(point);
		(*combine_func)(row_in, row_out);
		INDEX_OF_ARRAY(point, points, cum_block_size_out, ndim_rest);
		row_to_store(point);
	    }
	}

	store_block = 0;
	INDEX_OF_ARRAY(block, blocks, cum_nblocks_out, ndim_rest);

	for (j = 0; j < nblocks_out[dim]; j++)
	{
	    CHECK_STATUS(store_to_disk(block, store_block, msg));

	    block += block_step_out;
	    store_block++;
	}
    }

    return  OK;
}
