/******************************************************************************
This file is part of the argtable library. It contains a simple example program
showing how to use the argtable library to process a variable number of command
line arguments.

Copyright (C) 1998, 1999 Stewart Heitmann (Stewart.Heitmann@tip.csiro.au)

The argtable library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This software is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
USA.


How it works
------------
This program gives an example of setting up an argument table for handling
arguments that may occur multiple times. It takes one or more double-type
arguments, each representing the radius of a circle. It calculates the area
of each of the circles and prints it to stdout. If given no arguments it 
displays the usage,
    example2 radius1 [radius2] ... [radiusN]

The problem is that each argument in the argument table is restricted to
having only one value. However we can create an argument table with duplicate
entries of the same argument construct, thus achieving the desired effect.
We only need ensure that we define a sufficient number of duplicates to cover
all anticipated uses, and that all the duplicate arguments are given default
values. The default values are necessary to make presence of the duplicate 
arguments on the command line optional instead of mandatory.

Next, we would like to prevent the arg_syntax() and arg_glossary() functions
from generating unwanted repeat descriptions for each of the duplicate
arguments.
This is achieved by defining the name strings of all but the first duplicate
arguments as an empty string "", and the description strings as NULL. 
(Remember that if we used NULL for the name string, the arg_syntax() function
will automatically use a description of the arguments data type in its place.)
We can now write the name and description strings of the first argument
in such a way that they appear to refer to multiple arguments. A neat trick.

On a final note. If you require so many duplicates that the argument table
would be impossibly long, or if you just hate the idea of coding long 
repetitive tables in your program, then you may prefer to allocate the 
argument table dynamically at run time. There is no reason that the argument
table has to be defined as a static array, as long as the argument table is
assigned the necessary argument specifications before it is used then all
will be well. The arg_record() function will help with this.
******************************************************************************/

#include "argtable.h"

/*-- This example program assumes a maximum of 10 radius arguments. Their --*/
/*-- values will  be written sequentially into this array of doubles.     --*/
double radius[10];


/*-- This function calculates and print the areas of each circle  --*/
/*-- whose radius[] value does not have the default value of -1.0 --*/
void calc_areas(void)
  {
  int i;
  for (i=0; i<10; i++)
     {
     if (radius[i] != -1.0)
        printf("radius=%f, area=%f\n", radius[i], radius[i]*radius[i]*3.14159);
     else
        break;
     }
  }


int main(int argc, char**argv)
  {
  arg_rec argtable[] =
    {
    /*-tag--name---------type-----storage---default---descrip --*/
    { NULL, "radius1 [radius2] ... [radius10]",
                         arg_dbl,  radius,   NULL,    " are all <double> values" },
    { NULL, "",          arg_dbl,  radius+1, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+2, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+3, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+4, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+5, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+6, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+7, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+8, "-1.0",  NULL },
    { NULL, "",          arg_dbl,  radius+9, "-1.0",  NULL }
    };
  const size_t narg = sizeof(argtable)/sizeof(arg_rec);

  /*-- process the command line args --*/
  if (argc==1)
     {
     /*-- no command line arguments, so display program usage and exit. --*/
     printf("This program calculates the area of up to 10 circles.\n");
     printf("Usage: %s %s\n",argv[0],arg_syntax(argtable,narg));
     printf("%s\n",arg_glossary(argtable,narg,"   "));
     return 0;
     }
  else
     {
     /*-- scan command line arguments from argv[] as per argtable[] specs--*/
     char cmdline[200], errmsg[200], errmark[200]; 
     if (!arg_scanargv(argc,argv,argtable,narg,cmdline,errmsg,errmark))
	{
	/*-- scan error occurred, print error message and exit --*/
        printf("ERROR: %s\n", cmdline);
        printf("       %s %s\n", errmark, errmsg);
        return 1;
	}
     }

  /*-- now calculate the areas of the circles --*/  
  calc_areas();
 
  return 0; 
  }


