// Copyright 2015-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License"). You may
// not use this file except in compliance with the License. A copy of the
// License is located at
//
//	http://aws.amazon.com/apache2.0/
//
// or in the "license" file accompanying this file. This file is distributed
// on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
// express or implied. See the License for the specific language governing
// permissions and limitations under the License.

package regcredio

/* ----------------- INPUT types ----------------- */

// ECSRegCredsInput contains registry cred entries for creation and/or use in a task execution role
type ECSRegCredsInput struct {
	Version             string
	RegistryCredentials RegistryCreds `yaml:"registry_credentials"`
}

// RegistryCreds is a map of registry names to RegCredEntry structs
type RegistryCreds map[string]RegistryCredEntry

// RegistryCredEntry contains info needed to create an AWS Secrets Manager secret and match it to an ECS container(s)
type RegistryCredEntry struct {
	SecretManagerARN string   `yaml:"secrets_manager_arn"`
	Username         string   `yaml:"username"`
	Password         string   `yaml:"password"`
	KmsKeyID         string   `yaml:"kms_key_id"`
	ContainerNames   []string `yaml:"container_names"`
}

// HasRequiredFields indicates whether the entry has the fields required to create or use registry credentials
func (e RegistryCredEntry) HasRequiredFields() bool {
	return e.SecretManagerARN != "" || e.HasCredPair()
}

// HasCredPair indicates whether the entry contains a username + password
func (e RegistryCredEntry) HasCredPair() bool {
	return e.Username != "" && e.Password != ""
}

/* ----------------- OUTPUT types ----------------- */

// ECSRegistryCredsOutput contains the content of the output file
type ECSRegistryCredsOutput struct {
	Version             string
	CredentialResources CredResources `yaml:"registry_credential_outputs"`
}

// CredResources contains the credential resources generated by "registry-creds up"
type CredResources struct {
	TaskExecutionRole    string                      `yaml:"task_execution_role"`
	ContainerCredentials map[string]CredsOutputEntry `yaml:"container_credentials"`
}

// CredsOutputEntry contains the credential ARN, key, and associated container names for a single registry
type CredsOutputEntry struct {
	CredentialARN  string   `yaml:"credentials_parameter"` //TODO: rename 'CredentialARN' to 'CredentialsParam' ?
	KMSKeyID       string   `yaml:"kms_key_id,omitempty"`
	ContainerNames []string `yaml:"container_names"`
}
