/*  SpiralSynth
 *  Copyleft (C) 2000 David Griffiths <dave@pawfal.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#define	_ISOC9X_SOURCE	1
#define _ISOC99_SOURCE	1
#include  <math.h>
#include "Output.h"

#define CHECK_AND_REPORT_ERROR	if (result<0)         \
								{                     \
									perror("Sound device did not accept settings"); \
									m_OutputOk=false; \
									return;           \
								}


//////////////////////////////////////////////////////////////////////
OSSOutput* OSSOutput::m_Singleton = NULL;
//////////////////////////////////////////////////////////////////////

OSSOutput::OSSOutput() :
m_Amp(0.5),
m_Channels(2),
m_ReadBufferNum(0),
m_WriteBufferNum(0),
m_OutputOk(true)
{ 
	m_Buffer[0]=NULL;
	m_Buffer[1]=NULL;
	m_InBuffer[0]=NULL;
	m_InBuffer[1]=NULL;

	OpenWrite();
	AllocateBuffer(); 
}

//////////////////////////////////////////////////////////////////////

OSSOutput::~OSSOutput()
{
	Close();
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::AllocateBuffer()
{
	if (m_Buffer[0]==NULL)
	{
		m_BufSizeBytes=SpiralInfo::BUFSIZE*m_Channels*2;
		
		// initialise for stereo
		m_Buffer[0] = (short*) calloc(m_BufSizeBytes/2,m_BufSizeBytes);
		m_Buffer[1] = (short*) calloc(m_BufSizeBytes/2,m_BufSizeBytes);
		m_InBuffer[0] = (short*) calloc(m_BufSizeBytes/2,m_BufSizeBytes);
		m_InBuffer[1] = (short*) calloc(m_BufSizeBytes/2,m_BufSizeBytes);
	}
	
	m_Wav.SetSamplerate(SpiralInfo::SAMPLERATE);
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::SendStereo(const Sample *ldata,const Sample *rdata) 
{
	if (m_Channels!=2) return;
	
	float ChannelLevel=1.0f/SpiralInfo::POLY;
	
	int on=0;
	float t;
	for (int n=0; n<SpiralInfo::BUFSIZE; n++)
	{
		// stereo channels - interleave	
		if (ldata) 
		{
			t=(*ldata)[n]*m_Amp;
			if (t>1) t=1;
			if (t<-1) t=-1;
			t*=ChannelLevel;
			m_Buffer[m_WriteBufferNum][on]+=lrintf(t*SHRT_MAX);
		}
		on++;
		 
		if (rdata) 
		{
			t=(*rdata)[n]*m_Amp;
			if (t>1) t=1;
			if (t<-1) t=-1;
			t*=ChannelLevel;
			m_Buffer[m_WriteBufferNum][on]+=lrintf(t*SHRT_MAX);
		}		
		on++;
	}
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::Play()
{
    int BufferToSend=!m_WriteBufferNum;
	
    #if __BYTE_ORDER == BIG_ENDIAN
    for (int n=0; n<SpiralInfo::BUFSIZE*m_Channels; n++)
    {		
		m_Buffer[BufferToSend][n]=(short)((m_Buffer[BufferToSend][n]<<8)&0xff00)|
		                                 ((m_Buffer[BufferToSend][n]>>8)&0xff);
    }
    #endif	
    
	if (m_OutputOk)
		write(m_Dspfd,m_Buffer[BufferToSend],m_BufSizeBytes);
		
    if(m_Wav.Recording()) 
    {
        m_Wav.Save(m_Buffer[BufferToSend],m_BufSizeBytes);
    }

	memset(m_Buffer[BufferToSend],0,m_BufSizeBytes);
	m_WriteBufferNum=BufferToSend;
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::GetStereo(Sample *ldata,Sample *rdata) 
{
	if (m_Channels!=2) return;
	
	int on=0;
	for (int n=0; n<SpiralInfo::BUFSIZE; n++)
	{
		// stereo channels - interleave	
		if (ldata) ldata->Set(n,(m_InBuffer[m_ReadBufferNum][on]*m_Amp)/(float)SHRT_MAX);
		on++;
		if (rdata) rdata->Set(n,(m_InBuffer[m_ReadBufferNum][on]*m_Amp)/(float)SHRT_MAX);
		on++;
	}
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::Read()
{
    int BufferToRead=!m_ReadBufferNum;

	if (m_OutputOk)
		read(m_Dspfd,m_InBuffer[BufferToRead],m_BufSizeBytes);
	
    #if __BYTE_ORDER == BIG_ENDIAN
    for (int n=0; n<SpiralInfo::BUFSIZE*m_Channels; n++)
    {		
		m_InBuffer[BufferToRead][n]=(short)((m_InBuffer[BufferToRead][n]<<8)&0xff00)|
		                                   ((m_InBuffer[BufferToRead][n]>>8)&0xff);
    }
    #endif	
    
	m_ReadBufferNum=BufferToRead;
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::Close()
{ 	
	cerr<<"Closing dsp output"<<endl;
	close(m_Dspfd);
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::OpenWrite()
{ 	
	int result,val;
	cerr<<"Opening dsp output ["<<SpiralInfo::OUTPUTFILE.c_str()<<"]"<<endl;
  
	m_Dspfd = open(SpiralInfo::OUTPUTFILE.c_str(),O_WRONLY);  
	if(m_Dspfd<0) 
	{
		fprintf(stderr,"Can't open audio driver for writing.\n");
		m_OutputOk=false;
		return;
	}
   
	result = ioctl(m_Dspfd,SNDCTL_DSP_RESET,NULL);

	CHECK_AND_REPORT_ERROR;
			 
	short fgmtsize=0;
	int numfgmts=SpiralInfo::FRAGCOUNT;	 	 
	if (SpiralInfo::FRAGCOUNT==-1) numfgmts=0x7fff;
	
	for (int i=0; i<32; i++)
	{	 	
		if ((SpiralInfo::FRAGSIZE)==(1<<i)) 
		{
			fgmtsize=i; 
			break;
		}
	}

	if (fgmtsize==0)
	{
		cerr<<"Fragment size ["<<SpiralInfo::FRAGSIZE<<"] must be power of two!"<<endl;
		fgmtsize=256;
	}
 	 	
	numfgmts=numfgmts<<16;
	val=numfgmts|(int)fgmtsize;
	result=ioctl(m_Dspfd, SNDCTL_DSP_SETFRAGMENT, &val);
	CHECK_AND_REPORT_ERROR;
	
	val = 1;
	result = ioctl(m_Dspfd, SOUND_PCM_WRITE_CHANNELS, &val);
	CHECK_AND_REPORT_ERROR;
	
	val = AFMT_S16_LE;
	result = ioctl(m_Dspfd,SNDCTL_DSP_SETFMT,&val);	 
	CHECK_AND_REPORT_ERROR;

	if (m_Channels==2) val=1;
	else val=0;
	result = ioctl(m_Dspfd,SNDCTL_DSP_STEREO,&val);
	CHECK_AND_REPORT_ERROR;
		
	val = SpiralInfo::SAMPLERATE;
	result = ioctl(m_Dspfd,SNDCTL_DSP_SPEED,&val);
	CHECK_AND_REPORT_ERROR;
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::OpenRead()
{ 	
	int result,val;
  
	cerr<<"Opening dsp input ["<<SpiralInfo::OUTPUTFILE.c_str()<<"]"<<endl;

	m_Dspfd = open(SpiralInfo::OUTPUTFILE.c_str(),O_RDONLY);  
	if(m_Dspfd<0) 
	{
		fprintf(stderr,"Can't open audio driver for reading.\n");
		m_OutputOk=false;
		return;
	}
   
	result = ioctl(m_Dspfd,SNDCTL_DSP_RESET,NULL);
	CHECK_AND_REPORT_ERROR;
	
	val = 1;
	result = ioctl(m_Dspfd, SOUND_PCM_READ_CHANNELS, &val);
	CHECK_AND_REPORT_ERROR;
	
	val = AFMT_S16_LE;
	result = ioctl(m_Dspfd,SNDCTL_DSP_SETFMT,&val);	 
	CHECK_AND_REPORT_ERROR;
	
	val = SpiralInfo::SAMPLERATE;
	result = ioctl(m_Dspfd,SNDCTL_DSP_SPEED,&val);
	CHECK_AND_REPORT_ERROR;
}

//////////////////////////////////////////////////////////////////////

void OSSOutput::OpenReadWrite()
{ 	
	int result,val;
	cerr<<"Opening dsp output (duplex)"<<endl;
  
	m_Dspfd = open(SpiralInfo::OUTPUTFILE.c_str(),O_RDWR);  
	if(m_Dspfd<0) 
	{
		fprintf(stderr,"Can't open audio driver for writing.\n");
		m_OutputOk=false;
		return;
	}
   
	result = ioctl(m_Dspfd,SNDCTL_DSP_RESET,NULL);

	CHECK_AND_REPORT_ERROR;
			 
	short fgmtsize=0;
	int numfgmts=SpiralInfo::FRAGCOUNT;	 	 
	if (SpiralInfo::FRAGCOUNT==-1) numfgmts=0x7fff;
	
	for (int i=0; i<32; i++)
	{	 	
		if ((SpiralInfo::FRAGSIZE)==(1<<i)) 
		{
			fgmtsize=i; 
			break;
		}
	}

	if (fgmtsize==0)
	{
		cerr<<"Fragment size ["<<SpiralInfo::FRAGSIZE<<"] must be power of two!"<<endl;
		fgmtsize=256;
	}
 	 	
	numfgmts=numfgmts<<16;
	val=numfgmts|(int)fgmtsize;
	result=ioctl(m_Dspfd, SNDCTL_DSP_SETFRAGMENT, &val);
	CHECK_AND_REPORT_ERROR;
	
	val = 1;
	result = ioctl(m_Dspfd, SOUND_PCM_WRITE_CHANNELS, &val);
	CHECK_AND_REPORT_ERROR;
	
	val = AFMT_S16_LE;
	result = ioctl(m_Dspfd,SNDCTL_DSP_SETFMT,&val);	 
	CHECK_AND_REPORT_ERROR;

	if (m_Channels==2) val=1;
	else val=0;
	result = ioctl(m_Dspfd,SNDCTL_DSP_STEREO,&val);
	CHECK_AND_REPORT_ERROR;
		
	val = SpiralInfo::SAMPLERATE;
	result = ioctl(m_Dspfd,SNDCTL_DSP_SPEED,&val);
	CHECK_AND_REPORT_ERROR;
}


